/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes.elements;

import java.lang.reflect.InvocationTargetException;
import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.HashMap;
import java.awt.EventQueue;

import org.openide.nodes.*;
import org.openide.src.ElementProperties;
import org.openide.util.Utilities;
import org.openide.ErrorManager;
import org.netbeans.jmi.javamodel.Constructor;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.api.mdr.events.MDRChangeListener;
import org.netbeans.api.mdr.events.MDRChangeEvent;
import org.netbeans.api.mdr.events.MDRChangeSource;
import org.netbeans.api.mdr.events.AttributeEvent;

import javax.jmi.reflect.JmiException;

/** Node representing a constructor.
* @see org.openide.src.ConstructorElement
* @author Petr Hamernik, Jan Pokorsky
*/
public final class ConstructorNode extends ElementNode {

    private static final Map mapAttributeName;
    
    static {
        mapAttributeName = new HashMap();
        mapAttributeName.put(PROP_MODIFIERS, PROP_MODIFIERS);
        mapAttributeName.put(ElementProperties2.PROP_NAME, ElementProperties2.PROP_NAME);
        mapAttributeName.put(PROP_PARAMETERS, PROP_PARAMETERS);
        mapAttributeName.put("exceptionNames", PROP_EXCEPTIONS); // NOI18N
    }
    
    /** Create a new constructor node.
    * @param element constructor element to represent
    * @param writeable <code>true</code> to be writable
    */
    public ConstructorNode(Constructor element, boolean writeable) {
        super(element, Children.LEAF, writeable);
        setElementFormat0(getElementFormatProperty());
        superSetName(((JavaClass) element.getDeclaringClass()).getSimpleName());
        NameListener.getInstance(this, element);
    }

    /* Resolve the current icon base.
    * @return icon base string.
    */
    protected String resolveIconBase() {
        return IconResolver.getIconBaseForConstructor(getConstructor());
    }

    protected ElementFormat getElementFormatProperty() {
        return getSourceOptions().getConstructorElementFormat();
    }

    /* This method resolve the appropriate hint format for the type
    * of the element. It defines the short description.
    */
    protected ElementFormat getHintElementFormat() {
        return getSourceOptions().getConstructorElementLongFormat();
    }

    protected Map getAttributeNameMap() {
        return mapAttributeName;
    }

    /* Creates property set for this node */
    protected Sheet createSheet () {
        Sheet sheet = Sheet.createDefault();
        Sheet.Set ps = sheet.get(Sheet.PROPERTIES);
        ps.put(createModifiersProperty(writeable));
        ps.put(createNameProperty());
        ps.put(createParametersProperty(false));
        ps.put(createExceptionsProperty(writeable));
        return sheet;
    }

    /** Indicate that this node cannot be renamed.
    * An constructor must have the same name like class
    * @return <code>false</code>
    */
    public boolean canRename() {
        return false;
    }
    
    private Node.Property createNameProperty() {
        
        return new ElementNode.ElementProp(ElementProperties.PROP_NAME, String.class, false) {
            public Object getValue() throws
                    IllegalAccessException, InvocationTargetException {

                try {
                    JavaMetamodel.getDefaultRepository().beginTrans(false);
                    try {
                        String name;
                        ClassDefinition cd = getConstructor().getDeclaringClass();
                        name = cd instanceof JavaClass? ((JavaClass) cd).getSimpleName(): cd.getName();
                        return name;
                    } finally {
                        JavaMetamodel.getDefaultRepository().endTrans();
                    }
                } catch (JmiException e) {
                    throw new InvocationTargetException(e);
                }
            }
        };
    }

    /** Create a node property for constructor parameters.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    private Node.Property createParametersProperty(boolean canW) {
        // this should be read-olny -> refactoring job
        Node.Property p = createParametersProperty(getConstructor(), canW);
        p.setValue("changeImmediate" /* PropertyEnv.PROP_CHANGE_IMMEDIATE */,Boolean.FALSE); // NOI18N
        return p;
    }

    /** Create a node property for constructor exceptions.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    private Node.Property createExceptionsProperty(boolean canW) {
        Node.Property p = createExceptionsProperty(getConstructor(), canW);
        p.setValue("changeImmediate" /* PropertyEnv.PROP_CHANGE_IMMEDIATE */,Boolean.FALSE); // NOI18N
        return p;
    }
    
    private Constructor getConstructor() {
        return (Constructor) this.element;
    }

    /**
     * Since {@link Constructor} does not notify about changes of its name
     * it is necessary to add this listener to the declaring class.
     */ 
    private static final class NameListener extends WeakReference implements MDRChangeListener, Runnable {
        
        private final JavaClass declClass;
        
        public static NameListener getInstance(ConstructorNode node, Constructor c) {
            JavaClass cd = (JavaClass) c.getDeclaringClass();
            MDRChangeSource source = (MDRChangeSource) cd;
            NameListener l = new NameListener(node, cd);
            source.addListener(l, AttributeEvent.EVENTMASK_ATTRIBUTE);
            return l;
        }
        
        private NameListener(ConstructorNode referent, JavaClass declClass) {
            super(referent, Utilities.activeReferenceQueue());
            this.declClass = declClass;
        }

        public void change(MDRChangeEvent e) {
            if (!(e instanceof AttributeEvent)) {
                return;
            }
            
            final ConstructorNode node = (ConstructorNode) get();
            AttributeEvent ae = (AttributeEvent) e;
            String attrName = ae.getAttributeName();
            ElementFormat format = null;
            if (node != null && (attrName == null || ElementProperties2.PROP_NAME.equals(attrName)) &&
                    (format = node.getElementFormat()).dependsOnProperty(attrName)) {
                final String[] names = new String[2]; // [displayName, name]
                JavaMetamodel.getDefaultRepository().beginTrans(false);
                try {
                    try {
                        if (declClass.isValid()) {
                            names[0] = format.format(node.element);
                            names[1] = declClass.getSimpleName();
                        }
                    } finally {
                        JavaMetamodel.getDefaultRepository().endTrans();
                    }
                } catch (JmiException ex) {
                    ErrorManager.getDefault().notify(ErrorManager.WARNING, ex);
                }
                if (names[0] != null) {
                    EventQueue.invokeLater(new Runnable() {

                        public void run() {
                            node.setDisplayName(names[0]);
                            node.superSetName(names[1]);
                            node.superPropertyChange(ElementProperties2.PROP_NAME, null, null);
                        }
                    });
                }
            }
        }

        public void run() {
            ((MDRChangeSource) this.declClass).removeListener(this);
        }
    }
}
