/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes.editors;

import java.beans.*;
import java.util.*;

import javax.swing.*;
import javax.jmi.reflect.JmiException;

import org.openide.*;
import org.openide.util.Utilities;
import org.openide.util.NbBundle;
import org.openide.explorer.propertysheet.ExPropertyEditor;
import org.openide.explorer.propertysheet.PropertyEnv;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.PrimitiveType;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.java.ui.nodes.elements.ElementNode;


/** Property editors for array of org.netbeans.jmi.javamodel.MultipartId
*
* @author Petr Hamernik, Jan Pokorsky
*/
public class IdentifierArrayEditor extends PropertyEditorSupport implements ExPropertyEditor {
        
    public static final String JAVA_LANG_OBJECT = "java.lang.Object"; // NOI18N

    /** Custom property editor Component. */
    IdentifierArrayPanel panel;

    /** Flag for prevention of cycle in firing
    * of the properties changes.
    */
    boolean ignoreEditor = false;

    /** Flag for prevention of cycle in firing
    * of the properties changes.
    */
    boolean ignorePanel = false;

    PropertyEnv env;
    
    private JavaModelPackage model;

    /** @return text representation of the value */
    public String getAsText() {
        MultipartId[] id = (MultipartId[]) getValue();
        if ( id == null )
            return ""; // NOI18N
            
        StringBuffer buf = new StringBuffer();

        for (int i = 0; i < id.length; i++) {
            if (i > 0)
                buf.append(", "); // NOI18N
            buf.append(multipartIdToName(id[i]));
        }

        return buf.toString();
    }
    
    public static String multipartIdToName(MultipartId id) {
        LinkedList list = new LinkedList();
        while (id != null) {
            String stringRep;
            if (id.getTypeArguments().size() > 0) {
                JavaModelPackage pkg = (JavaModelPackage) id.refImmediatePackage();
                NamedElement e = id.getElement();
                stringRep = e.getName();
                stringRep += '<';
                for (Iterator tIt = id.getTypeArguments().iterator(); ;) {
                    stringRep += ((NamedElement) tIt.next()).getName();
                    if (tIt.hasNext()) {
                        stringRep += ", "; // NOI18N
                    } else {
                        break;
                    }
                }
                stringRep += '>';
            } else {
                stringRep = id.getName();
            }
            list.addFirst(stringRep);
            id = id.getParent();
        }
        StringBuffer buf = new StringBuffer();
        for (Iterator iter = list.iterator(); iter.hasNext();) {
            buf.append((String)iter.next());
            if (iter.hasNext())
                buf.append('.');
        }
        return buf.toString();
    }

    /** Sets the value as the text */
    public void setAsText(String text) throws IllegalArgumentException {
        List list = Collections.EMPTY_LIST;

        try {
            JavaMetamodel.getDefaultRepository().beginTrans(false);
            try {
                list = resolveIdentifiers(text);
            } finally {
                JavaMetamodel.getDefaultRepository().endTrans();
            }
        } catch (JmiException ex) {
            ErrorManager.getDefault().notify(ex);
        }

        MultipartId[] ret = new MultipartId[list.size()];
        list.toArray(ret);
        setValue(ret);
    }

    /**
     * parses txt to return list of MultipartIds
     */ 
    private List/*<MultipartId>*/ resolveIdentifiers(String txt) {
        StringTokenizer tukac = new StringTokenizer(txt, ", ", false); // NOI18N
        List list = new ArrayList();
        while (tukac.hasMoreTokens()) {
            String id = tukac.nextToken();
            JavaModelPackage jmp = model;
            Type t = jmp.getType().resolve(id);
            
            // Assume any Java identifier can serve as a reference type name:
            if (t instanceof PrimitiveType) {
                IllegalArgumentException ex = new IllegalArgumentException();
                String msg = java.text.MessageFormat.format(
                        getString("MSG_InvalidIdentifier"), // NOI18N
                        new Object[] { id });
                ErrorManager.getDefault().annotate(ex, ErrorManager.USER, null, msg, null, null);
                throw ex;
            }
            
            MultipartId mid = jmp.getMultipartId().createMultipartId(id, null, null);
            list.add(mid);
        }
        return list;
    }

    /** Set new value */
    public void setValue(Object o) {
        ignoreEditor = true;
        boolean saveIgnorePanel = ignorePanel;
        
        ignorePanel = false;
        super.setValue(o);
        if ((panel != null) & !saveIgnorePanel) {
            panel.setIdentifiers((MultipartId[]) o);
        }
        ignoreEditor = false;
    }

    /** @return <CODE>true</CODE> */
    public boolean supportsCustomEditor () {
        return true;
    }

    /** Create new panel for this property editor.
    * @return the visual component for editing the property
    */
    public java.awt.Component getCustomEditor () {
        if (panel == null) {
            panel = new IdentifierArrayPanel(model);
            panel.setIdentifiers((MultipartId[]) getValue());
            panel.setMnemonics(env);
            panel.addPropertyChangeListener(new PropertyChangeListener() {
                                                public void propertyChange(PropertyChangeEvent evt) {
                                                    if (!ignoreEditor && IdentifierArrayPanel.PROP_IDENTIFIERS.equals(evt.getPropertyName())) {
                                                        ignorePanel = true;
                                                        setValue(evt.getNewValue());
                                                        ignorePanel = false;
                                                    }
                                                }
                                            });
        }
        return panel;
    }

    /**
     * This method is called by the IDE to pass
     * the environment to the property editor.
     */
    public void attachEnv(PropertyEnv env) {
        this.env = env;
        model = ElementNode.getModel(env.getFeatureDescriptor());
    }
    
    /** Implementation of the abstract ObjectArrayPanel2 class.
    * It is used for editing of arrays of Identifier objects.
    */
    static class IdentifierArrayPanel extends ObjectArrayPanel2 {

        /** Name of the 'identifiers' property. */
        public static final String PROP_IDENTIFIERS = "identifiers"; // NOI18N

        /** Previous value */
        MultipartId[] prevValue;
        
        private final JavaModelPackage javaModel;
        
        /** Constructor */
        public IdentifierArrayPanel(JavaModelPackage javaModel) {
            this.javaModel = javaModel;
            prevValue = new MultipartId[0];

            this.getListComponent().setCellRenderer(new DefaultListCellRenderer() {
                                                        public java.awt.Component getListCellRendererComponent(JList list,
                                                                Object value, int index, boolean isSelected, boolean cellHasFocus) {
                                                            java.awt.Component comp = super.getListCellRendererComponent(list,
                                                                                      value, index, isSelected, cellHasFocus);
                                                            if (comp == this) {
                                                                setText(multipartIdToName((MultipartId) value));
                                                            }
                                                            return comp;
                                                        }
                                                    });
        }

        /** @return the current value */
        public MultipartId[] getIdentifiers() {
            MultipartId[] ret = new MultipartId[model.size()];
            model.copyInto(ret);
            return ret;
        }

        /** Set new value.
        */
        public void setIdentifiers(MultipartId[] data) {
            model = new DefaultListModel();
            if (data != null) {
                for (int i = 0; i < data.length; i++)
                    model.addElement(data[i]);
            }
            this.getListComponent().setModel(model);
            modelChanged();
        }

        /** Fire the 'identifiers' property change. */
        protected void modelChanged() {
            MultipartId[] newValue = getIdentifiers();
            firePropertyChange(PROP_IDENTIFIERS, prevValue, newValue);
            prevValue = newValue;
        }

        /** Ask user for new value.
        * @return new value or <CODE>null</CODE> when 
        *    operation was canceled.
        */
        protected Object insertNewValue() {
            return openInputDialog(null);
        }

        /** Ask user for edit value.
        * @param oldValue The previous value to be edited
        * @return new value or <CODE>null</CODE> when 
        *    operation was canceled.
        */
        protected Object editValue(Object oldValue) {
            return openInputDialog((MultipartId) oldValue);
        }

        /** Show dialog and allow user to enter new name.
        */
        protected MultipartId openInputDialog(MultipartId origValue) {
            NotifyDescriptor.InputLine input = new NotifyDescriptor.InputLine(
                                                   getString("LAB_NewName"), // NOI18N
                                                   getString("LAB_NewIdentifier") // NOI18N
                                               );
            if (origValue != null)
                input.setInputText(multipartIdToName(origValue));

            for (;;) {
                Object ret = DialogDisplayer.getDefault().notify(input);
                if (ret == NotifyDescriptor.OK_OPTION) {
                    String retValue = input.getInputText();
                    if (retValue != null && !"".equals(retValue)) { // NOI18N
                        if (!retValue.startsWith(".") && !retValue.endsWith(".") && // NOI18N
                                (retValue.indexOf("..") == -1)) { // NOI18N
                            boolean ok = true;
                            StringTokenizer tokenizer = new StringTokenizer(retValue, ".", false); // NOI18N
                            while (tokenizer.hasMoreTokens()) {
                                String token = tokenizer.nextToken();
                                if (!Utilities.isJavaIdentifier(token)) {
                                    ok = false;
                                    break;
                                }
                            }
                            if (ok)
                                return javaModel.getMultipartId().createMultipartId(retValue, null, null);
                        }
                    }
                    DialogDisplayer.getDefault().notify(new NotifyDescriptor.Message(getString("MSG_NotValidID"))); // NOI18N
                } else {
                    return null;
                }
            }
        }
    }
         
    private static String getString(String key) {
        return NbBundle.getMessage(IdentifierArrayEditor.class, key);
    }
         
}
