/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes;

import org.openide.src.*;
import org.openide.src.Element;
import org.openide.loaders.DataObject;
import org.openide.cookies.SourceCookie;
import org.openide.ErrorManager;
import org.netbeans.modules.java.bridge.ElementImpl;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.jmi.javamodel.*;

import javax.jmi.reflect.RefObject;
import javax.jmi.reflect.JmiException;

/**
 * Helper for bridging between source hierarchy and jmi
 */
public final class BridgeUtils {
    /**
     * maps source hierarchy element to jmi object
     * @param element source hierarchy element
     * @return jmi object
     */ 
    public static RefObject getJavaObject(Element element) {
        ElementImpl impl = (ElementImpl) element.getCookie(Element.Impl.class);
        return impl.getJavaElement();
    }

    /**
     * maps jmi object to class element that contains the object.
     * @param member jmi object
     * @return source hierarchy element or null
     */ 
    private static ClassElement getClassElement(ClassMember member) {
        JavaClass jc = (JavaClass) member.getDeclaringClass();
        if (jc == null) {
            throwIllegalState(member, "none declaring class"); // NOI18N
        }
        return getElement(jc);
    }

    public static Element getElement(ClassMember member) {
        Element e = null;
        if (member instanceof JavaClass)
            e = getElement((JavaClass) member);
        else if (member instanceof Constructor)
            e = getElement((Constructor) member);
        else if (member instanceof Field)
            e = getElement((Field) member);
        else if (member instanceof Initializer)
            e = getElement((Initializer) member);
        else if (member instanceof Method)
            e = getElement((Method) member);
        else
            throw new IllegalArgumentException("Unknown member: " + member); //NOI18N
        
        return e;
    }
    
    /**
     * @see #getElementForCookieSet
     */ 
    private static final ThreadLocal isForCookieSet = new ThreadLocal();
    
    /**
     * workaround for issue #48997; see also issue #48595. It is replacement for
     * {@link #getElement(ClassMember)} in case you would invoke it
     * inside Node.getCookie
     */ 
    static Element getElementForCookieSet(ClassMember member) {
        try {
            isForCookieSet.set(Boolean.TRUE);
            return getElement(member);
        } finally {
            isForCookieSet.set(Boolean.FALSE);
        }
    }
    
    /**
     * maps JavaClass to ClassElement
     * @param member jmi object
     * @return source hierarchy element
     */ 
    public static ClassElement getElement(JavaClass member) {
        Resource r = member.getResource();
        if (r == null) {
            throwIllegalState(member, "missing resource"); // NOI18N
        }
        DataObject d = JavaMetamodel.getManager().getDataObject(r);
        if (d == null) {
            throwIllegalState(member, "missing data object"); // NOI18N
        }
        SourceCookie sc = (SourceCookie) d.getCookie(SourceCookie.class);
        if (sc == null) {
            if (Boolean.TRUE.equals(isForCookieSet.get())) {
                return null;
            } else {
                throwIllegalState(member, "missing source cookie"); // NOI18N
            }
        }
        SourceElement source = sc.getSource();
        ClassElement[] els;
       
        if (member.refImmediateComposite() instanceof JavaClass)
            els = source.getAllClasses();
        else
            els = source.getClasses();
        return (ClassElement) chooseElement(els, member);
    }

    /**
     * maps Constructor to ConstructorElement
     * @param member jmi object
     * @return source hierarchy element
     */ 
    public static ConstructorElement getElement(Constructor member) {
        ClassElement ce = getClassElement(member);
        if (ce == null) // #48997
            return null;
        ConstructorElement[] els = ce.getConstructors();
        return (ConstructorElement) chooseElement(els, member);
    }

    /**
     * maps Field to FieldElement
     * @param member jmi object
     * @return source hierarchy element
     */ 
    public static FieldElement getElement(Field member) {
        ClassElement ce = getClassElement(member);
        if (ce == null) // #48997
            return null;
        FieldElement[] els = ce.getFields();
        return (FieldElement) chooseElement(els, member);
    }

    /**
     * maps Method to MethodElement
     * @param member jmi object
     * @return source hierarchy element
     */ 
    public static MethodElement getElement(Method member) {
        ClassElement ce = getClassElement(member);
        if (ce == null) // #48997
            return null;
        MethodElement[] els = ce.getMethods();
        return (MethodElement) chooseElement(els, member);
    }

    /**
     * maps Initializer to FieldElement
     * @param member jmi object
     * @return source hierarchy element
     */ 
    public static InitializerElement getElement(Initializer member) {
        ClassElement ce = getClassElement(member);
        if (ce == null) // #48997
            return null;
        InitializerElement[] els = ce.getInitializers();
        return (InitializerElement) chooseElement(els, member);
    }

    private static Element chooseElement(Element[] els, ClassMember member) {
        for (int i = 0; i < els.length; i++) {
            Element e = els[i];
            RefObject ref = getJavaObject(e);
            if (member.equals(ref)) {
                return e;
            }
        }
        throwIllegalState(member, "cannot map class member to source hierarchy element"); // NOI18N
        return null;
    }
    
    /**
     * throws annotated illegal state exception
     * @param member
     * @param message
     * @throws IllegalStateException
     */ 
    private static void throwIllegalState(ClassMember member, String message) throws IllegalStateException {
        IllegalStateException ise = new IllegalStateException(message);
        ErrorManager.getDefault().annotate(ise, ErrorManager.INFORMATIONAL, member.getClass().toString(), null, null, null);
        try {
            ErrorManager.getDefault().annotate(
                    ise, ErrorManager.INFORMATIONAL, "name: " + member.getName(), null, null, null); // NOI18N
            ErrorManager.getDefault().annotate(
                    ise, ErrorManager.INFORMATIONAL, "member: " + member.toString(), null, null, null); // NOI18N
        } catch (JmiException e) {
            ErrorManager.getDefault().annotate(
                    ise, ErrorManager.INFORMATIONAL, null, null, e, null); // NOI18N
        }
        throw ise;
    }
    
}
