/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.tools;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.reflect.Modifier;
import java.util.*;
import javax.jmi.reflect.InvalidObjectException;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.jmi.javamodel.ParameterizedType;
import org.netbeans.jmi.javamodel.PrimitiveType;
import org.netbeans.jmi.javamodel.PrimitiveTypeKindEnum;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.modules.java.settings.JavaSynchronizationSettings;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MethodImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ParameterizedTypeImpl;
import org.openide.ErrorManager;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;
import org.openide.util.SharedClassObject;
import org.openide.util.Task;
import org.openide.util.TaskListener;
import org.openide.loaders.DataObject;

public class JMIInheritanceSupport implements Runnable, TaskListener {
    
    public static final String PROP_FINISHED = "JMIInheritanceSupport.finished"; //NOI18N

    private static String getString(String key) {
        return NbBundle.getMessage(JMIInheritanceSupport.class, key);
    }
    
    private ClassDefinition root;
    
    private JavaModelPackage modelPackage;
    
    private Resource resource;

    private Set classes;
    
    // <MethodElement.Key, MethodElement>
    private Map methods;
    
    // <Identifier, Set<MethodElement.Key>>, (alias <class, Set<method>>)
    private Map tree;
    
    private List classesList;
    
    private Set finalMethodsKeys;
    
    private PropertyChangeSupport pcs = new PropertyChangeSupport(this);
    private RequestProcessor.Task task = RequestProcessor.getDefault().create(this);
    
    /** used for storing info about position where to add next feature */
    private int nextPosition = -1;
    
    /** Creates a new instance of JMIInheritanceSupport. The root must be a class. */
    public JMIInheritanceSupport(ClassDefinition root) {
        DataObject dobj = (DataObject) JavaMetamodel.getManager().getDataObject(root.getResource());
        assert dobj != null;
        this.root = root;
        this.resource = root.getResource();
        this.modelPackage = (JavaModelPackage)root.refOutermostPackage();
        this.task.addTaskListener(this);
    }

    public ClassDefinition getRootClass() {
        return root;
    }
    
    public void setRootClass(ClassDefinition root) {
        this.root = root;
    }
    
    public Collection getClasses(Collection result, boolean cls, boolean ifc) {
        if (!task.isFinished()) {
            task.waitFinished();
        }

        for (Iterator i = classesList.iterator(); i.hasNext(); ) {
            JavaClass jc = (JavaClass) i.next();
            boolean isInterface = jc.isInterface();
            if ((!isInterface && cls) || (isInterface && ifc)) {
                result.add(jc);
            }
        }
        
        return result;
    }
    
    public Collection getAllMethods(Collection result, boolean abstractOnly) {
        if (!task.isFinished()) {
            task.waitFinished();
        }

        for (Iterator i = methods.values().iterator(); i.hasNext(); ) {
            Method mte = (Method) i.next();
            if (!abstractOnly || (isAbstract(mte) && !isImplemented(mte))) {
                result.add(mte);
            }
        }
        
        return result;
    }
    
    public Collection getMethods(Collection result, ClassDefinition cd, boolean abstractOnly) {
        if (!task.isFinished()) {
            task.waitFinished();
        }

        Set meths = (Set) tree.get(cd);
        for (Iterator i = meths.iterator(); i.hasNext(); ) {
            MethodKey key = (MethodKey) i.next();
            Method m = (Method) methods.get(key);
            if (!abstractOnly || (isAbstract(m) && !isImplemented(m))) {
                result.add(m);
            }
        }
        
        return result;
    }

    /**
     * Adds method that overrides <code>source</code>.
     * @param source method to be overridden
     * @param superCall generate super call?
     * @param javadoc generate super javadoc?
     * @return reference to method created in source file
     * @throws SourceException impossible to add the method
     */ 
    public Method overrideMethod(Method m, boolean superCall, boolean javadoc) {        
        try {
            String bodyText = "";

            // remove ABSTRACT and NATIVE since the user probably wants to write the method body ;-)
            // SYNCHRONIZED is removed for safety reasons: if the user wants the method to be synchronized,
            // she will add the modifier.
            int mods = m.getModifiers() & ~(Modifier.NATIVE | Modifier.ABSTRACT | Modifier.SYNCHRONIZED);
            // to be formatted nicely in the source code...
            if (superCall) {
                bodyText = createSuperCall(m, isImplementedInSuper(root, m, false));
            }
            MethodImpl miOrig;
            
            if (m instanceof MethodImpl)
                miOrig = (MethodImpl) m;
            else
                miOrig = ((MethodImpl) ((ParameterizedTypeImpl.Wrapper) m).getWrappedObject());
            
            MethodImpl newMethod = (MethodImpl) miOrig.duplicate((JavaModelPackage) root.refImmediatePackage());
            newMethod.setJavadocText(javadoc?miOrig.getJavadocText():null);
            newMethod.setModifiers(mods);
            newMethod.setBody(null);
            newMethod.fixImports(root, m);
            newMethod.setBodyText(bodyText);
            addMethod(newMethod);
            return newMethod;
        } catch (InvalidObjectException e) {
            ErrorManager.getDefault().notify(e);
            return null;
        }
    }
    
    private void addMethod(Method m) {
        List contents = root.getContents();
        if (nextPosition != -1) {
            contents.add(nextPosition, m);
            nextPosition++;
            return;
        }
        int index = -1;
        ListIterator iter = contents.listIterator();
        for (int x = 0; iter.hasNext(); x++) {
            Object obj = iter.next();
            if (!(obj instanceof JavaClass)) {
                index = x;
            }
        }
        if (index == -1) {
            contents.add(m);
            nextPosition = 1;
            return;
        }
        
        JavaMetamodel manager = JavaMetamodel.getManager();
        iter = contents.listIterator(index + 1);
        while (iter.hasPrevious()) {
            NamedElement elem = (NamedElement) iter.previous();
            if (!manager.isElementGuarded(elem)) {
                iter.next();
                iter.add(m);
                nextPosition = contents.indexOf(m) + 1;
                return;
            }
        }
        
        contents.add(0, m);
        nextPosition = 1;
    }
    
    public void addPropertyChangeListener(PropertyChangeListener l) {
        pcs.addPropertyChangeListener(l);
    }

    public void removePropertyChangeListener(PropertyChangeListener l) {
        pcs.removePropertyChangeListener(l);
    }

    public boolean isFinished() {
        return task.isFinished();
    }
    
    public void reset() {
        classes = null;
        classesList = null;
        methods = null;
        tree = null;
        finalMethodsKeys = null;
        task.schedule(0);
    }

    // ---- Runnable implementation
    
    public void run() {
        classes = new HashSet();
        classesList = new ArrayList();
        methods = new HashMap();
        tree = new HashMap();
        finalMethodsKeys = new HashSet();

        MDRepository repository = JavaMetamodel.getDefaultRepository();
        repository.beginTrans(false);
        try {
            JavaMetamodel model = JavaMetamodel.getManager();
            model.setClassPath(model.getFileObject(root.getResource()), true);
            
            // traverse all super classes
            JavaClass superClass = root.getSuperClass();
            while (superClass != null && classes.add(superClass)) {
                classesList.add(superClass);
                HashSet meths = new HashSet();
                collectMethods(meths, superClass);
                tree.put(superClass, meths);
                superClass = superClass.getSuperClass();
            }

            // traverse all interfaces
            traverseInterfaces(classes, classesList, tree, root);
        } finally {
            repository.endTrans();
        }
    }

    private void traverseInterfaces(Set classes, List classesList, Map tree, ClassDefinition root) {
        List interfaces = root.getInterfaces();
        for (Iterator iter = interfaces.iterator(); iter.hasNext(); ) {
            JavaClass jc = (JavaClass) iter.next();
            if (classes.add(jc)) {
                classesList.add(jc);
                HashSet meths = new HashSet();
                collectMethods(meths, jc);
                tree.put(jc, meths);
                traverseInterfaces(classes, classesList, tree, jc);
            }
        }
    }
    
    // ---- TaskListener implementation
    
    public void taskFinished(Task task) {
        pcs.firePropertyChange(PROP_FINISHED, Boolean.FALSE, Boolean.TRUE);
    }
    
    // ---- private implementation

    private boolean isAbstract(Method mte) {
        ClassDefinition cd = mte.getDeclaringClass();
        return Modifier.isAbstract(mte.getModifiers()) || (cd instanceof JavaClass && ((JavaClass)cd).isInterface());
    }
        
    /** Determines, if the given method is accessible from the root class. */
    private boolean isAccessibleMethod(Method m) {
        ClassDefinition cls = m.getDeclaringClass();
        int modifs = cls instanceof JavaClass ? ((JavaClass)cls).getModifiers() : 0;
        
        if ((modifs & Modifier.PRIVATE) != 0 ||
            ((modifs & (Modifier.PUBLIC | Modifier.PROTECTED | Modifier.PRIVATE)) == 0 &&
             !isSamePackage(cls))) {
            return false;
        }
        if ((cls instanceof JavaClass) && ((JavaClass)cls).isInterface())
            return true;
        modifs = m.getModifiers();
        if ((modifs & Modifier.PRIVATE) != 0) {
            return false;
        }
        if ((modifs & (Modifier.PUBLIC | Modifier.PROTECTED)) != 0)
            return true;
        return isSamePackage(cls);
    }

    /** Determines if the method is implemented in some super class of the
     * passed class.
     */
    private boolean isImplementedInSuper(ClassDefinition rootCl, Method m, boolean includeClass) {
        String name = m.getName();
        List params = m.getParameters();
        List paramTypes = new ArrayList(params.size());
        for (Iterator iter = params.iterator(); iter.hasNext(); ) {
            paramTypes.add(((Parameter)iter.next()).getType());
        }
        
        // traverse all super classes
        ClassDefinition jc = includeClass ? rootCl : rootCl.getSuperClass();
        HashSet visited = new HashSet();
        while (jc != null && visited.add(jc)) {
            Method mte = jc.getMethod(name, paramTypes, false);
            if (mte != null && !Modifier.isAbstract(mte.getModifiers())) {
                return true;
            }
            jc = jc.getSuperClass();
        }
        return false;
    }
    
    private boolean isImplemented(Method m) {
        String name = m.getName();
        List params = m.getParameters();
        List paramTypes = new ArrayList(params.size());
        for (Iterator iter = params.iterator(); iter.hasNext(); ) {
            paramTypes.add(((Parameter)iter.next()).getType());
        }
        
        // traverse all super classes
        ClassDefinition jc = root;
        ClassDefinition realJc = jc instanceof ParameterizedType ? ((ParameterizedType) jc).getDefinition() : jc;
        HashSet visited = new HashSet();
        visited.add(m.getDeclaringClass());
        // String dcName = m.getDeclaringClass().getName();
        while (jc != null && visited.add(realJc)) {
            Method mte = jc.getMethod(name, paramTypes, false);
            if (mte != null && !Modifier.isAbstract(mte.getModifiers())) {
                return true;
            }
            jc = jc.getSuperClass();
            realJc = jc instanceof ParameterizedType ? ((ParameterizedType) jc).getDefinition() : jc;
        }
        return false;
    }
    
    /**
     * Determines if the given method is overriden in the hierarchy.
     *
     * @param  method  method for which overriding methods are looked for
     * @return true, if given method is overriden in subclasses
     */
    private boolean isOverriden(Method m) {
        List params = m.getParameters();
        String name = m.getName();
        List paramTypes = new ArrayList(params.size());
        for (Iterator iter = params.iterator(); iter.hasNext(); ) {
            paramTypes.add(((Parameter)iter.next()).getType());
        }
        ClassDefinition cd = root;
        HashSet visited = new HashSet();
        while (cd != null && visited.add(cd)) {
            Method meth = root.getMethod(name, paramTypes, false);
            if (meth == m)
                return false;
            else if (meth != null)
                return true;
            cd = cd.getSuperClass();
        }
        return false;
    }
    
    /** Deteremines if the class is in the same package as the root class */
    private boolean isSamePackage(ClassDefinition cd) {
        String p1 = root.getResource().getPackageName();
        String p2 = cd.getResource().getPackageName();
        return p1 == null ? p2 == null : p1.equals(p2);
    }

    /** Collects all methods declared by the class and all its interfaces. It doesn't
     * traverse super classes.
     */
    private void collectMethods(Set meths, JavaClass jc) {
        for (Iterator iter = jc.getContents().iterator(); iter.hasNext(); ) {
            Object obj = iter.next();
            if (!(obj instanceof Method))
                continue;
            Method m = (Method)obj;
            int mods = m.getModifiers();
            boolean modFlag = !Modifier.isStatic(mods) && (Modifier.isPublic(mods) || Modifier.isProtected(mods));
            boolean isFinal = Modifier.isFinal(mods);
            if (modFlag && !isFinal && isAccessibleMethod(m) && !isOverriden(m)) {
                MethodKey key = new MethodKey(m);
                if (!finalMethodsKeys.contains(key) && methods.get(key) == null) {
                    methods.put(key, m);
                    meths.add(key);
                }
            } else {
                if (isFinal && modFlag) {
                    MethodKey key = new MethodKey(m);
                    finalMethodsKeys.add(key);
                }
            }
        }
        
        for (Iterator iter = jc.getInterfaces().iterator(); iter.hasNext(); ) {
            collectMethods(meths, (JavaClass) iter.next());
        }
    }

    private String createSuperCall(Method target, boolean hasSuper) {
        JavaSynchronizationSettings syncSettings = (JavaSynchronizationSettings)SharedClassObject.findObject(JavaSynchronizationSettings.class, true);

        String body;
        
        if (hasSuper) {
            // create a parameter list for the superclass' call:
            String format;
            
            StringBuffer str = new StringBuffer();
            for (Iterator iter = target.getParameters().iterator(); iter.hasNext(); ) {
                Parameter par = (Parameter) iter.next();
                if (str.length() > 0)
                    str.append(", "); // NOI18N
                str.append(par.getName());
            }
            Type t = target.getType();
            if (t instanceof PrimitiveType && PrimitiveTypeKindEnum.VOID.equals(((PrimitiveType)t).getKind())) {
                format = NbBundle.getMessage(JMIInheritanceSupport.class, "FMT_CallSuper"); // NOI18N
            } else {
                format = NbBundle.getMessage(JMIInheritanceSupport.class, "FMT_ReturnCallSuper"); // NOI18N
            }
            body = java.text.MessageFormat.format(format,
                new Object[] {
                    JavaModelUtil.resolveImportsForType(root, target.getType()).getName(),
                    target.getName(),
                    str
            });
        } else {
            // no super, let JavaSyncSettings generate a default return:
            body = syncSettings.getGenerateReturnAsString(target.getType());
        }
        return body;
    }
    
    // ..........................................................................
    
    public static class MethodKey extends Object {
        
        private String name;
        private String[] paramTypes;
        
        public MethodKey (Method m) {
            name = m.getName();
            List params = m.getParameters();
            paramTypes = new String[params.size()];
            Iterator iter = params.iterator();
            for (int x = 0; iter.hasNext(); x++) {
                paramTypes[x] = ((Parameter)iter.next()).getType().getName();
            }
        }

        /* Returns true if parameters are the same */
        public boolean equals (Object obj) {
            if (!(obj instanceof MethodKey)) return false;
            return name.equals(((MethodKey)obj).name) && Arrays.equals(paramTypes, ((MethodKey)obj).paramTypes);
        }

        /* Computes hashcode as exclusive or of first and
        * last parameter's names
        * (or only from the first or return some constant
        * for special cases) */
        public int hashCode () {
            int length = paramTypes.length;
            if (length == 0) return 0;
            if (length == 1) return paramTypes[0].hashCode();
            return paramTypes[0].hashCode() ^
                   paramTypes[length - 1].hashCode();
        }

    }
}
