/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.settings;

import java.io.*;
import java.util.ResourceBundle;
import java.util.Properties;
import java.util.List;
import java.util.Iterator;
import java.util.Collection;

import java.util.HashMap;
import java.util.Map;
import org.netbeans.modules.javacore.JMManager;

import org.openide.options.ContextSystemOption;
import org.openide.util.HelpCtx;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;

/** Settings for java data loader and java source parser
*
* @author Ales Novak, Petr Hamernik
*/
public class JavaSettings extends ContextSystemOption {
    private static final int currentVersion = 1;

    /** serial uid */
    static final long serialVersionUID = -8522143676848697297L;

    public static final String PROP_SHOW_OVERRIDING = "showOverriding"; //NOI18N

    public static final String PROP_SHOW_COMPILE_STATUS = "showCompileStatus"; // NOI18N

    public static final String PROP_REPLACEABLE_STRINGS_TABLE = "replaceableStringsTable"; // NOI18N

    public static final String PROP_AUTO_PARSING_DELAY = "autoParsingDelay"; // NOI18N
    public static final String PROP_PARSING_ERRORS = "parsingErrors"; // NOI18N
            
    public static final int DEFAULT_AUTO_PARSING_DELAY = 2000;
    public static final int DEFAULT_PARSING_ERRORS = 100;
    private static final boolean DEFAULT_SHOW_OVERRIDING = true;

    /**
     * Current version of the settings object. If < currentVersion,
     * a settings upgrade is made.
     */
    private int version;
        
    private static JavaSettings javaSettings;
    
    /** If true then external execution is used */
    public String displayName () {
        return getString("CTL_Java_option");
    }

    public HelpCtx getHelpCtx () {
        return new HelpCtx (JavaSettings.class);
    }

    public JavaSettings() {
        addOption(getJavaSynchronizationSettings());        
    }

    /** Sets the replaceable strings table - used during instantiating
    * from template.
    */
    public void setReplaceableStringsTable(String table) {
        String t = getReplaceableStringsTable();
        if (t.equals(table))
            return;
        putProperty(PROP_REPLACEABLE_STRINGS_TABLE, table, true);
    }

    /** Gets the replacable strings table - used during instantiating
    * from template.
    */
    public String getReplaceableStringsTable() {
        String table = (String)getProperty(PROP_REPLACEABLE_STRINGS_TABLE);
        if (table == null) {
            return "USER="+System.getProperty("user.name")+"\n"; // NOI18N
        } else {
            return table;
        }
    }

    /** Gets the replaceable table as the Properties class.
    * @return the properties
    */
    public Properties getReplaceableStringsProps() {
        Properties props = new Properties();
        String propString = getReplaceableStringsTable();
        int i,len = propString.length();
        StringBuffer unicodeString = new StringBuffer(len);
        
        for (i=0;i<len;i++) {
            char aChar = propString.charAt(i);
            
            if (aChar > 0x00ff) {
                String hex = Integer.toHexString(aChar);
                
                unicodeString.append("\\u"); // NOI18N
                if (aChar < 0x1000)
                    unicodeString.append("0"); // NOI18N
                unicodeString.append(hex);
            } else {
                unicodeString.append(aChar);
            }
        }
        try {
            props.load(new ByteArrayInputStream(unicodeString.toString().getBytes()));
        }
        catch (IOException e) {
        }
        return props;
    }

    /** Gets the delay time for the start of the parsing.
    * @return The time in milis
    */
    public int getAutoParsingDelay() {
        Integer delay = (Integer)getProperty(PROP_AUTO_PARSING_DELAY);
        if (delay == null)
            return DEFAULT_AUTO_PARSING_DELAY;
        return delay.intValue();
    }

    /** Sets the number of errors returned by the parsing and displayed as 
     * annotations in source text
    * @param number of errors, 0 means disable 
    */
    public void setParsingErrors(int errors) {
        if (errors < 0)
            throw new IllegalArgumentException();
        putProperty(PROP_PARSING_ERRORS, new Integer(errors),true); // fire property change
    }

    /** Gets the number of errors returned by the parsing and displayed as 
     * annotations in source text
     * @return number of errors
    */
    public int getParsingErrors() {
        Integer errors = (Integer)getProperty(PROP_PARSING_ERRORS);
        if (errors == null)
            return DEFAULT_PARSING_ERRORS;
        return errors.intValue();
    }

    public boolean getShowOverriding () {
        Boolean value = (Boolean)getProperty (PROP_SHOW_OVERRIDING);
        if (value == null)
            return DEFAULT_SHOW_OVERRIDING;
        return value.booleanValue();
    }

    public void setShowOverriding (boolean value) {
        Boolean b = (Boolean)getProperty(PROP_SHOW_OVERRIDING);
        if (b != null && b.booleanValue() == value)
            return;
        putProperty (PROP_SHOW_OVERRIDING, value? Boolean.TRUE : Boolean.FALSE,true);
    }

    /** Sets the delay time for the start of the parsing.
    * @param delay The time in milis
    */
    public void setAutoParsingDelay(int delay) {
        if (delay < 0)
            throw new IllegalArgumentException();
        putProperty(PROP_AUTO_PARSING_DELAY, new Integer(delay));
    }

    
    public boolean isCompileStatusEnabled() {
        Boolean b = (Boolean)getProperty(PROP_SHOW_COMPILE_STATUS);
        if (b == null)
            return true;
        return b.booleanValue();
    }
    
    public void enableCompileStatus(boolean b) {
        if (isCompileStatusEnabled() == b)
            return;
        putProperty(PROP_SHOW_COMPILE_STATUS, b ? Boolean.TRUE : Boolean.FALSE, true);
    }
    
    /** @return localized string */
    static String getString(String s) {
        return NbBundle.getMessage(JavaSettings.class, s);
    }

    private static JavaSynchronizationSettings getJavaSynchronizationSettings() {
        return (JavaSynchronizationSettings) JavaSynchronizationSettings.findObject(JavaSynchronizationSettings.class, true);
    }

    public void readExternal(ObjectInput in) 
    throws java.io.IOException, ClassNotFoundException {
        super.readExternal(in);
        if (in.available() > 0) {
            version = in.readInt();
        }
        if (version < currentVersion) {
            // assume version == 0, for now
            
            version = currentVersion;
        }
        JMManager.setDefaultEncoding(defaultEncoding);
    }
    
    public void writeExternal(ObjectOutput out) throws IOException {
        super.writeExternal(out);
        out.writeInt(version);
    }
    
    public static final JavaSettings getDefault() {
        if (javaSettings==null)
            javaSettings=(JavaSettings) JavaSettings.findObject(JavaSettings.class, true);
        return javaSettings;
    }
    
    /* ------------ Support for text I/O ---------------------- */
    
    /**
     * Name of the "default encoding" property
     */
    public static final String PROP_DEFAULT_ENCODING = "defaultEncoding"; // NOI18N
    
    /**
     * Platform - default encoding. No specific encoding will be passed to readers
     * or writers. It is specified as empty string so it won't produce the (null) text
     * for the user.
     */
    public static final String ENCODING_PLATFORM_DEFAULT = ""; // NOI18N
    
    /**
     * Holds the name of the default encoding which should be used when reading 
     * or writing .java files. Can be one of the tagged values ENCODING_* manifested above.
     */
    private String defaultEncoding = ENCODING_PLATFORM_DEFAULT;
    
    public String getDefaultEncoding() {
        return defaultEncoding;
    }
    
    public void setDefaultEncoding(String enc) {
        String e = defaultEncoding;
        defaultEncoding = enc;
        JMManager.setDefaultEncoding(enc);
        firePropertyChange(PROP_DEFAULT_ENCODING, e, enc);
    }
    
}
