/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.codegen;

import java.util.*;

import javax.swing.text.Position;

import org.openide.src.*;
import org.openide.text.CloneableEditorSupport;
import org.openide.text.PositionBounds;
import org.openide.text.PositionRef;

import org.netbeans.modules.java.bridge.Binding;

/**
 * Specialized document binding for a ClassElement.
 *
 * @author  Svatopluk Dedic <mailto:sdedic@netbeans.org>
 * @version 0.1
 */
class Clazz extends Member implements Binding.Class, TextBinding.Container {
    /**
     * Support for the containment of member elements.
     */
    ContainerSupport    container;
    
    public Clazz(ClassElement el, SourceText s) {
        super(el, s);
    }
    
    public TextBinding.Container getContainer() {
        return this;
    }
    
    public boolean isEmpty() {
        return container == null ||  container.isEmpty();
    }
    
    public void updateChildren(Collection c) {
        if (container == null && c.isEmpty())
            return;
        initializeContainer();
        container.updateChildren(c);
    }
    
    /** Changes class into an interface and vice-versa. Although classes and interfaces
     * are clearly separated in the language specs, they represent the same concept with
     * some minor differencies.
     */
    public void changeClassType(boolean properClass) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        
        ClassElement el = cloneClass();
        el.setClassOrInterface(properClass);
        regenerateHeader(el);
    }
    
    /** Rewrites some interfaces from class' implements property.
     */
    public void changeInterfaces(Identifier[] replaceWith) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        
        ClassElement el = cloneClass();
        el.setInterfaces(replaceWith);
        regenerateHeader(el);
    }
    
    /**
     * Changes the superclass' name.
     */
    public void changeSuperclass(Identifier id) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        
        ClassElement el = cloneClass();
        el.setSuperclass(id);
        regenerateHeader(el);
    }

    /** The clone implementation will clone, for efficiency reasons only the class itself,
     * not its children, as ClassElement.clone() normally does.
     */
    protected Element cloneElement() {
        ClassElement x = new ClassElement();
        ClassElement my = (ClassElement)getElement();
        try {
            x.setName(my.getName());
            x.setModifiers(my.getModifiers());
            x.setSuperclass(my.getSuperclass());
            x.setInterfaces(my.getInterfaces());
            x.setClassOrInterface(my.isClassOrInterface());
            if (!my.getJavaDoc().isEmpty()) {
                x.getJavaDoc().setRawText(my.getJavaDoc().getRawText());
            }
        } catch (SourceException ex) {
            // should NOT happen.
        }
        return x;
    }
    
    private ClassElement cloneClass() {
        return (ClassElement)cloneElement();
    }
    
    protected int classifyProperty(String name) {
        return CLASS_HEADER;
    }
    
    public ElementBinding findBindingAt(int pos) {
        ElementBinding b = super.findBindingAt(pos);
        if (b == this) {
            ElementBinding b2 = container.findBindingAt(pos);
            if (b2 != null)
                return b2;
        }
        return b;
    }

    /* ============== CONTAINER MANAGEMENT METHODS ================= */
    
    public boolean canInsertAfter(Binding b) {
        if (container == null) {
            return source.canWriteInside(bodyBounds);
        }
        return container.canInsertAfter(b);
    }

    /** The map contains mapping from target places to their new contents.
     */
    public void reorder(Map fromToMap) throws SourceException {
        container.reorder(fromToMap);
    }
    
    /** Replaces the slot contents with another element (different type permitted ?)
     */
    public void replace(Binding oldBinding, Binding newBinding) throws SourceException {
        container.replace(oldBinding, newBinding);
    }
    
    public void changeMembers(MultiPropertyChangeEvent evt) throws SourceException {
        if (container == null) {
            int etype = evt.getEventType();
            if (etype == evt.TYPE_ADD || etype == evt.TYPE_COMPOUND)
                initializeContainer();
            else
                return;
        }
        container.changeMembers(evt);
    }
    
    /**
     * Initializes a new binding for the element so the element is stored after the
     * `previous' binding, if that matters to the binding implementation.
     * @param toInitialize the binding that is being initialized & bound to the storage.
     * @param previous marker spot, the binding that should precede the new one.
     */
    public void insert(Binding toInitialize,Binding previous) throws SourceException {
        initializeContainer();
        container.insert(toInitialize, previous);
    }
    
    protected void initializeContainer() {
        if (container != null)
            return;
        container = new ContainerSupport(this.source, this);
    }
    
    public void regenerateWhole(Element el) {
    }
    
    PositionBounds findContainerBounds(TextBinding.Container cont) {
        return bodyBounds;
    }
    
    public void create(PositionBounds bounds) throws SourceException {
        ClassElement c = cloneClass();
        ClassElement orig = (ClassElement)getElement();

        // create the class definition.
        wholeBounds = bounds;
        super.regenerateWhole(c, true);
        
        // initialize the container with the members:
        PositionRef r = bodyBounds.getBegin();
        boolean empty = true;
        ElementBinding prevBinding= null;
        
        for (int kind = 0; kind < 5; kind++) {
            Element[] models;
            
            switch (kind) {
                case 0:
                    models = orig.getFields();
                    break;
                case 1:
                    models = orig.getInitializers();
                    break;
                case 2:
                    models = orig.getConstructors();
                    break;
                case 3:
                    models = orig.getMethods();
                    break;
                case 4:
                    models = orig.getClasses();
                    break;
                default:
                    models = null;
            }
            if (empty && models.length > 0) {
                initializeContainer();
                empty = false;
            }
            
            for (int i = 0; i < models.length; i++) {
                ElementBinding b = source.findBinding(models[i]);
                container.insertChild(b, prevBinding, null);
                prevBinding = b;
            }
        }
    }
}
