/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.bridge;

import java.lang.reflect.Modifier;
import java.beans.*;

import org.openide.src.*;
import org.netbeans.api.mdr.events.*;

import org.netbeans.jmi.javamodel.Initializer;

/**
 * The class implements property handling for a class' initializer. In fact, the initializer
 * has only two properties: the body and the optional static modifier.
 *
 * @author  Svatopluk Dedic, Petr Hamernik
 * @version 0.1
 * @since 24/11/2000
 */
class InitializerElementImpl extends MemberElementImpl 
    implements InitializerElement.Impl, Element.Impl2 {

    private transient boolean   bodyInited;
    private transient String    cachedBody;
    
    private transient long      bodyHash = -1;
    
    private ElementImpl.ElementListener initializerListener;
    
    private static final long serialVersionUID = 3490993892523739282L;
    
    InitializerElementImpl(DefaultLangModel model, Initializer initializer) {
        super(model, initializer);
    }
    
    public void connectListener () {
        initializerListener = new InitializerListener (this);
        initializerListener.connect ();
    }
    
    protected void createFromModel(Element el) throws SourceException {
        InitializerElement i = (InitializerElement)el;
        setBody(i.getBody());
        setStatic(i.isStatic());
    }
    
    public void attachedToElement(Element el) {
        super.attachedToElement(el);
    }

    public boolean isStatic() {
        repository.beginTrans(false);
        try {
            if (javaElement.isValid()) {
                setClassPath();
                return (getModifiers() & Modifier.STATIC) > 0;
            } else {
                return false;
            }
        } finally {
            repository.endTrans(false);
        }
    }
    
    void copyBody(String s) {
    }
        
    public final String getBody() {
        repository.beginTrans(false);
        try {
            if (javaElement.isValid()) {
                setClassPath();
                return ((Initializer)javaElement).getBodyText();
            } else {
                return null;
            }
        } finally {
            repository.endTrans(false);
        }
    }
    
    public JavaDoc getJavaDoc() {
        return null;
    }

    public boolean isValid() {
        return super.isValid() && (getDeclaringImpl() == null ||
            getDeclaringImpl().isValid());
    }
    
    void updateBody(String content) {
        /*
        if (content == null) {
            bodyHash = -1;
            return;
        }
        long newHash = computeHash(content);
        if (newHash == bodyHash) 
            return;
        bodyHash = newHash;
        // we don't have info about the previous body's contents.
        addPropertyChange(new PropertyChangeEvent(getElement(), PROP_BODY, null, null));
         */
    }
    
    // Change operations.
    ///////////////////////////////////////////////////////////////////////////////
    public void setStatic(boolean enable) throws SourceException {
        checkWritable(false);
        checkDocument();
        boolean failed = true;
        repository.beginTrans (true);        
        try {
            if (javaElement.isValid()) {
                PropertyChangeEvent evt;

                setClassPath();
                int mod = getModifiers ();
                if (((mod & Modifier.STATIC) > 0) == enable) {
                    failed = false;
                    return;
                }
                evt = new PropertyChangeEvent(getElement(), PROP_STATIC, enable ? Boolean.FALSE : Boolean.TRUE, 
                    enable ? Boolean.TRUE : Boolean.FALSE);

                checkVetoablePropertyChange(evt);

                ((Initializer) javaElement).setModifiers (enable ? Modifier.STATIC : 0);
                failed = false;
            } else {
                failed = false;
                throwIsInvalid ();
            }
        } finally {
            repository.endTrans (failed);
        }
    }
    
    /** Sets the body. The entire functionality is delegated to the storage binding object.
     * Note that the text sent out in the PropertyChangeEvent may NOT match the text
     * passed in the vetoable change event since the text may be transformed by an 
     * indentation engine.
     * @param body text of the initializer's body.
     */
    public void setBody(String body) throws SourceException {
        checkWritable(false);
        checkDocument();
        repository.beginTrans (true);
        boolean rollback = true;
        try {
            if (javaElement.isValid()) {
                PropertyChangeEvent evt;

                setClassPath();
                String cachedBody = getBody ();
                if (cachedBody == body ||
                    (cachedBody != null && body != null && cachedBody.equals(body))) {
                    checkIsValid ();
                    return;
                }
                evt = new PropertyChangeEvent(getElement(), PROP_BODY, cachedBody, body);
                checkVetoablePropertyChange(evt); 

                InitializerElement elem = (InitializerElement) cloneSelf ();            
                ((Initializer)javaElement).setBodyText(body);
                rollback = false;
            } else {
                rollback = false;
                throwIsInvalid ();
            }
        } finally {
            repository.endTrans (rollback);
        }
    }
    
    public void fireModifiersChange (Integer oldValue, Integer newValue) {
        Boolean oldStatic, newStatic;
        boolean flag = (oldValue.intValue () & Modifier.STATIC) > 0;
        if (flag) {
            oldStatic = Boolean.TRUE;
            newStatic = Boolean.FALSE;
        } else {
            oldStatic = Boolean.FALSE;
            newStatic = Boolean.TRUE;
        }
        
        PropertyChangeEvent evt = new PropertyChangeEvent(getElement(), PROP_STATIC, oldStatic, newStatic);
        fireOwnPropertyChange(evt);
        
        InitializerElement old = (InitializerElement) cloneSelf ();
        try {
            old.setStatic (flag);
        } catch (SourceException e) {
            e.printStackTrace ();
        }
        notifyConnectionChange (old);
    }
    
    public void fireBodyChange (String oldVal, String newVal) {
        PropertyChangeEvent evt = new PropertyChangeEvent(getElement(), PROP_BODY, oldVal, newVal);
        fireOwnPropertyChange(evt);
    }
    
    // Utility methods.
    //////////////////////////////////////////////////////////////////////////////////    
    protected final SourceElementImpl findSource() {
        return getDeclaringImpl().findSource();
    }
    
    protected Element cloneSelf() {
        InitializerElement el = new InitializerElement();
        try {
            el.setStatic(isStatic());
        } catch (SourceException ex) {
        }
        return el;
    }
    
    // Serialization support
    //////////////////////////////////////////////////////////////////////////////////
    public Object readResolve() {
        return null;
    }
    
    // ...........................................................................
    
    static class InitializerListener extends MemberElementImpl.MemberElementListener {
                
        InitializerListener (InitializerElementImpl impl) {
            super (impl);
        }

        public void doChange(MDRChangeEvent event) {
            super.doChange (event);
            if (event instanceof AttributeEvent) {
                AttributeEvent attrEv = (AttributeEvent) event;
                if (attrEv.getAttributeName ().equals ("bodyText")) { // NOI18N
                    ((InitializerElementImpl) impl).fireBodyChange (
                        (String) attrEv.getOldElement (),
                        (String) attrEv.getNewElement ()
                     );
                }
            }
        }
        
    }
    
}
