/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.java;

import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import javax.swing.text.StyledDocument;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.cookies.SourceCookie;
import org.openide.loaders.DataObject;
import org.openide.text.Line;
import org.openide.text.NbDocument;
import org.openide.util.NbBundle;

class OverrideAnnotation extends LineSetAnnotation {


    private static final String OVERRIDE_ANNOTATION_TYPE = "org-netbeans-modules-java-override_annotation"; // NOI18N
    private static final String IMPLEMENT_ANNOTATION_TYPE = "org-netbeans-modules-java-implement_annotation"; // NOI18N

    private Descriptor descriptor;
    private int attachedToLine = -1;

    private OverrideAnnotation (Descriptor descriptor) {
        this.descriptor = descriptor;
    }


    public String getAnnotationType() {
        return this.descriptor.isOverriding()? OVERRIDE_ANNOTATION_TYPE : IMPLEMENT_ANNOTATION_TYPE;
    }

    public String getShortDescription() {
        String bundleKey = this.descriptor.isOverriding() ? "TXT_Overrides" : "TXT_Implements";    //NOI18N
        return MessageFormat.format(NbBundle.getMessage(OverrideAnnotation.class,bundleKey),
                new Object[] {this.descriptor.getMethodName (), this.descriptor.getOriginalClassQName ()});
    }

    public void attachToLineSet(Line.Set lines) {
        attachToLineSet(lines, this.descriptor.getLine());
    }
    
    private void attachToLineSet(Line.Set lines, int line) {
        if (line < 0) return;
        Line docline = lines.getCurrent(line);
        Line.Part part=docline.createPart(0,0);
        attach(part);
        attachedToLine = line;
    }
    
    public void updateLine(Line.Set lines) {
        int line = this.descriptor.getLine();
        if (line != attachedToLine) {
            detach();
            attachToLineSet(lines, line);
        }
    }

    public boolean equals (Object other) {
        if (other instanceof OverrideAnnotation) {
            OverrideAnnotation otherAnnotation = (OverrideAnnotation) other;
            return this.descriptor.equals (otherAnnotation.descriptor);
        }
        return false;
    }

    Descriptor getDescriptor () {
        return this.descriptor;
    }

    public static OverrideAnnotation forDescriptor (Descriptor desc) {
        return new OverrideAnnotation (desc);
    }

    public static class Descriptor {

        private final String overridenMethod;
        private final String originalMethod;
        private final boolean isImplementing;

        public Descriptor (Method originalMethod, Method overridenMethod) {
            this.originalMethod = originalMethod.refMofId();
            this.overridenMethod = overridenMethod.refMofId();
            JavaClass declaring = (JavaClass) originalMethod.getDeclaringClass();
            isImplementing = declaring != null && (declaring.isInterface() ||
                    Modifier.isAbstract (originalMethod.getModifiers()));
        }

        public final Method getOriginalMethod () {
            return (Method) JavaMetamodel.getDefaultRepository().getByMofId(this.originalMethod);
        }

        public final Method getOverridenMethod () {
            return (Method) JavaMetamodel.getDefaultRepository().getByMofId(this.overridenMethod);
        }

        public final boolean isOverriding () {
            return ! this.isImplementing();
        }

        public final boolean isImplementing () {
            return isImplementing;
        }

        public final String getMethodName () {
            try {
                return getOverridenMethod().getName();
            } catch (NullPointerException e) {
                return "";
            }
        }

        public final String getOriginalClassQName () {
            try {
                return getOriginalMethod().getDeclaringClass().getName();
            } catch (NullPointerException e) {
                return "";
            }
        }

        public final boolean equals (Object other) {
            if (other instanceof OverrideAnnotation.Descriptor) {
                OverrideAnnotation.Descriptor oa = (OverrideAnnotation.Descriptor) other;
                return (this.originalMethod == null ? oa.originalMethod == null : this.originalMethod.equals (oa.originalMethod)) &&
                       (this.overridenMethod == null ? oa.overridenMethod == null : this.overridenMethod.equals (oa.overridenMethod));
            }
            return false;
        }

        public final int hashCode () {
            return this.overridenMethod == null ? 0 : this.originalMethod.hashCode();
        }

        /** returns -1 if the document is not accessible */
        final int getLine () {
            Method overridenMethod = getOverridenMethod();
            if (overridenMethod == null) return -1;
            DataObject dobj = JavaMetamodel.getManager().getDataObject(overridenMethod.getResource());
            if (dobj == null) return -1;
            SourceCookie.Editor seditor = (SourceCookie.Editor) dobj.getCookie (SourceCookie.Editor.class);
            if (seditor == null) return -1;
            StyledDocument doc = seditor.getDocument();
            if (doc == null) return -1;

            int offset = JavaMetamodel.getManager().getElementPosition(overridenMethod).getBegin().getOffset();
            return NbDocument.findLineNumber(doc,offset);
        }

    }
}
