/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
 
package org.netbeans.modules.java;

import java.io.IOException;
import java.util.*;

import org.openide.ErrorManager;
import org.openide.cookies.InstanceCookie;
import org.openide.ErrorManager;
import org.openide.loaders.DataFolder;
import org.openide.loaders.FolderInstance;
import org.openide.src.nodes.FilterFactory;
import org.openide.src.nodes.ElementNodeFactory;
import org.netbeans.modules.java.ui.nodes.*;

/**
 * A pool for registering FilterSourceNodeFactories for the object browser and the
 * explorer. The implementation is backward compatible for FilterFactories.
 *
 * @author  Svatopluk Dedic
 * 
 */
final class NodeFactoryPool extends FolderInstance {

    /**
     * Base factory, which serves as a default and the tail of the chain.
     */
    ElementNodeFactory  base;
    
    /**
     * Factories, which were explicitly added by calls to {@link #addFactory}
     */
    LinkedList          explicit;
    
    /**
     * State of the underlying folder. Contains list of factories registered
     * through the SFS.
     */
    List     factories = Collections.EMPTY_LIST;
    
    /**
     * Computed head of the factory chain.
     */
    ElementNodeFactory  head;
    
    /**
     * True, if the folder scan was triggered at least once.
     */
    boolean             initialized;
    
    NodeFactoryPool(DataFolder storage, ElementNodeFactory base) {
        super(storage);
        this.base = base;
        head = base;
    }
    
    final Object sync() {
        return base;
    }

    /**
     * Returns the head of the current factory list. Except for the initialization,
     * the method does not block.
     */
    ElementNodeFactory getHead() {
        // force folder scan the first time the Pool is queried
        if (!initialized) {
            recreate();
            waitFinished();
            initialized = true;
        }
        return head;
    }

    /**
     * Creates an array of factories from the underlying folder. The "product" of
     * the method is the head of the factory list.
     */
    protected Object createInstance(InstanceCookie[] cookies) 
    throws java.io.IOException, ClassNotFoundException {
        List l = new ArrayList(cookies.length + 1);
        JavaSourceNodeFactory jsnf = new JavaSourceNodeFactory(); 
        jsnf.setGenerateForTree(((JavaElementNodeFactory) base).getGenerateForTree());
        l.add(jsnf);
        for (int i = 0; i < cookies.length; i++) {
            try {
                Object o = cookies[i].instanceCreate();
                if (o instanceof FilterSourceNodeFactory) {
                    l.add(o);
                } else if (o instanceof FilterFactory) { // backward compatibility
                    ExFilterFactory eff = new ExFilterFactory((ElementNodeFactory) o);
                    l.add(eff);
                }
            } catch (IOException ex) {
                logError(ex);
            } catch (ClassNotFoundException ex) {
                logError(ex);
            }
        }
        synchronized (sync()) {
            ElementNodeFactory f = relinkFactories(l);
            this.factories = l;
            return head = f;
        }
    }

    /**
     * Reattaches factories in the logicall factory chain to each other.
     */
    private ElementNodeFactory relinkFactories(Collection first) {
        Object previousFactory = new ExFilterFactory(base);
        Iterator it;
        Collection next = explicit;
        
        if (first == null)
            first = factories;
        
        for (it = first.iterator(); it.hasNext(); ) {
            Object obj = it.next();
            attachFactories(obj, previousFactory);
            previousFactory = obj;
        }
        if (next != null) {
            for (it = next.iterator(); it.hasNext(); ) {
                Object obj = it.next(); // FilterFactory
                ExFilterFactory eff = new ExFilterFactory((ElementNodeFactory) obj);
                attachFactories(previousFactory, eff);
                previousFactory = obj;
            }
        }
        return previousFactory instanceof ElementNodeFactory?
                (ElementNodeFactory) previousFactory:
                SourceNodes.createElementNodeFactory((FilterSourceNodeFactory) previousFactory);
    }
    
    /**
     * attaches factory f2 to f1
     * @param f1 FilterSourceNodeFactory or ExFilterFactory instance
     * @param f2 SourceNodeFactory or ExElementNodeFactory instance
     */ 
    private static void attachFactories(Object f1, Object f2) {
        if (f1 instanceof FilterSourceNodeFactory) {
            FilterSourceNodeFactory f1SNF = (FilterSourceNodeFactory) f1;
            if (f2 instanceof SourceNodeFactory) {
                f1SNF.attach((SourceNodeFactory) f2);
            } else if (f2 instanceof ExElementNodeFactory) {
                f1SNF.attach(SourceNodes.createSourceNodeFactory((ExElementNodeFactory) f2));
            } else {
                ErrorManager.getDefault().notify(
                        ErrorManager.INFORMATIONAL, new IllegalStateException("Unsupported factory: " + f2)); // NOI18N
            }
        } else if (f1 instanceof ExFilterFactory) {
            ExFilterFactory f1EFF = (ExFilterFactory) f1;
            if (f2 instanceof ExElementNodeFactory) {
                f1EFF.attach((ExElementNodeFactory) f2);
            } else if (f2 instanceof SourceNodeFactory){
                f1EFF.attach(SourceNodes.createElementNodeFactory((SourceNodeFactory) f2));
            } else {
                ErrorManager.getDefault().notify(
                        ErrorManager.INFORMATIONAL, new IllegalStateException("Unsupported factory: " + f2)); // NOI18N
            }
        } else {
            ErrorManager.getDefault().notify(
                    ErrorManager.INFORMATIONAL, new IllegalStateException("Unsupported factory: " + f1)); // NOI18N
        }
    }

    /**
     * Adds an explicit factory and the head of the chain. Relinks the entire
     * chain as well.
     */
    void addFactory(FilterFactory f) {
        synchronized (sync()) {
            if (explicit == null) {
                explicit = new LinkedList();
            }
            explicit.add(f);
            head = relinkFactories(null);
        }
    }

    /**
     * Removes one factory from the explicit list. Relinks the chain, if the
     * factory was, actually, on the list.
     */
    void removeFactory(FilterFactory f) {
        synchronized (sync()) {
            if (!explicit.remove(f))
                return;
            relinkFactories(null);
        }
    }
    
    void logError(Exception ex) {
        ErrorManager.getDefault().notify(ex);
    }
    
}
