/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java;

import org.openide.src.*;

/** Builder interface for creating parse results.
 * This inteface allows a parser engine to create some items of implementation
 * unknown to the engine. The engine is provided with opaque Item reference
 * and can communicate with the factory using those references.
 * The ElementFactory is focused on creating and connecting Java Source
 * elements and provides the only way how to access JavaLoader's internals from
 * the parser engine.
 */
public interface ElementFactory {
    /* ======================= Item creator methods ========================== */
    
    /** Creates a new class element.
	@param isInterface true if the element should in fact correspond to an interface rather than to a class.
	@param modifiers union of Modifier.* values composing access and other modifiers for the element.
	@param superclass Identifier of superclass, or null if there isn't any.
	@param interfaces all interfaces implemented on this class.
    */
    public Item createClass(boolean isInterface, int modifiers, Identifier name, Identifier superclass, Identifier[] interfaces);
    
    /** Creates an element for a method.
	@param modifiers Modifiers for the method.
	@param name name of the method
	@param returnType return type for the method.
	@param params array of parameters for the method.
	@param exceptions array of exceptions declared to be thrown from the method.
    */
    public Item createMethod(int modifiers, Identifier name, Type returnType, MethodParameter[] params, Identifier[] exceptions);
    
    /** Creates an element for a field.
	@param modifiers Modifiers for the field element
	@param name Field's name
	@param type Field's type
	@param initializer Initial value for the field (with equal sign excluded) or null if the field is not initialized.
    */
    public Item createField(int modifiers, Identifier name, Type type, String initializer);
    
    /** Creates an element for a constructor.
	For parameters see {@link #createMethod}
    */
    public Item createConstructor(int modifiers, Identifier id, MethodParameter[] params, Identifier[] exceptions);
    
    /** Creates an element for an initializer.
	@param modifiers Modifiers for the initializer.
    */
    public Item createInitializer(int modifiers);
    
    /** Creates data for import statement found in the source file.
	@param im Import object that is declared in the source.
	@param begin offset of the import statement start (position of `import' keyword)
	@param end offset of the end of the import statement (after semicolon)
    */
    public void	createImport(Import im, int begin, int end);
    
    /** Creates data for package declaration found in the source.
	@param name name of the declared package
	@param begin character index of the package declarator start
	@param end character index of the package decalarator end (after the semicolon)
    */
    public void	createPackage(Identifier name, int begin, int end);
    
    /** Binds two Items together in a parent-child relationship.
	@param child Child item to be inserted into the parent
	@param parent Parent item
    */
    public void setParent(Item child, Item parent);
    
    /** Sets bounds for the whole element. Begin is offset of first character of the element,
    end is the offset of the last one.
    */
    public void setBounds(Item item, int begin, int end);

    /** Sets bounds for the body of the element.
    */
    public void setBodyBounds(Item item, int begin, int end);

    public void setHeaderBounds(Item item, int begin, int end);

    /** Sets a documentation for the element.
    @param begin offset of doc comment start
    @param end offset of doc comment end
    @param text documentation comment content
    */
    public void setDocumentation(Item item, int begin, int end, String text);
    
    /** Sets name of the field that precedes this one in a declaration statement.
    */
    public void setPrecedingField(Item item, Item previous);

    /** Sets bounds for the identifier only.
    */
    public void setFieldTypeBounds(Item item, int begin, int end);
    
    public void markError(Item item);

    /** Only marker interface
    */
    public interface Item {
    }
}
