/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.java.navigation.strings;

import org.netbeans.modules.java.navigation.spi.strings.WeightedStringPainter;
import org.netbeans.modules.java.navigation.spi.strings.WeightedString;

/**
 * Implementation of WeightedString.
 *
 * @author Tim Boudreau
 */
public final class WeightedStringImpl extends WeightedString {
    private static final WeightedStringPainter painter =
            new WeightedStringPainterImpl ();

    private Object[] chars = new Object[ 15 ];
    private float[] importances = new float[ 15 ];

    private int count = -1;
    int length = 0;
    private float minImportance = 10;
    private float maxImportance = 0;

    private final Markup markup = new Markup ();

    public WeightedStringImpl () {
    }
    
    public char[] getCharsSegment (int seg) {
        return (char[])chars[seg];
    }

    public int allMarkupTypes () {
        return markup.allMarkupTypes ();
    }

    public void startMarkupRun (int bitmask) {
        markup.start ( count + 1, bitmask );
    }

    public void endMarkupRun () {
        markup.end ( count + 1 );
    }

    public int getMarkupFor (int seg) {
        return markup.get ( seg );
    }

    public float minImportance () {
        return minImportance == 2.0 ? 0 : minImportance;
    }

    public float maxImportance () {
        return maxImportance;
    }

    public WeightedStringPainter getPainter () {
        return painter;
    }

    public void append (String s, float importance) {
        append ( s.toCharArray (), importance );
    }

    /**
     * Append a string of the specified importance to this string. <i>Note:  For performance reasons, there is no
     * defensive copying of the char array passed here.  Do not modify the char array passed after calling this method
     * unless you intend to change what is displayed.</i>
     */
    public void append (char[] str, float importance) {
        if ( str.length == 0 ) {
            return;
        }
        if ( importance < 0.0f || importance > 1.0f ) {
            throw new IllegalArgumentException ( "Importance must be between " +
                    "0.0f and 1.0f but received " + importance );
        }
        count++;
        ensureCapacity ( count + 1 );
        chars[ count ] = str;
        importances[ count ] = importance;

        length += str.length;
        minImportance = Math.min ( minImportance, importance );
        maxImportance = Math.max ( maxImportance, importance );
    }

    public void clear () {
        count = -1;
        length = 0;
        minImportance = 2.0f;
        maxImportance = 0;
        markup.clear ();
    }

    private void ensureCapacity (int val) {
        if ( chars.length < val + 1 ) {
            Object[] oldChars = chars;
            chars = new Object[ val * 2 ];
            System.arraycopy ( oldChars, 0, chars, 0, oldChars.length );

            float[] oldImportances = importances;
            importances = new float[ val * 2 ];
            System.arraycopy ( oldImportances, 0, importances, 0, oldImportances.length );
        }
    }

    int getCount () {
        return count;
    }

    float getImportance (int seg) {
        return importances[ seg ];
    }
    
    public String toString() {
        StringBuffer sb = new StringBuffer (length + 1);
        for (int i=0; i <= count; i++) {
            sb.append ((char[]) chars[i]);
        }
        return sb.toString();
    }
}
