/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
/*
 * Change.java
 *
 * Created on September 18, 2004, 6:42 PM
 */

package org.netbeans.modules.java.navigation.spi.diff;

import javax.swing.event.*;

/**
 * Immutable class representing a single transformation to a data range in a list indicating the addition, removal or
 * modification of a range of indices.
 *
 * @author Tim Boudreau
 * @see Diff
 */
public final class Change {
    /**
     * Insertion type.  For convenience, this is the same value as ListDataEvent.INTERVAL_ADDED.
     */
    public static final int INSERT = ListDataEvent.INTERVAL_ADDED;
    /**
     * Deletion type.  For convenience, this is the same value as ListDataEvent.INTERVAL_REMOVED.
     */
    public static final int DELETE = ListDataEvent.INTERVAL_REMOVED;
    /**
     * Change type.  For convenience, this is the same value as ListDataEvent.CONTENTS_CHANGED.
     */
    public static final int CHANGE = ListDataEvent.CONTENTS_CHANGED;

    private final int type;
    private final int start;
    private int end;

    /**
     * Create a new Change object with the given start, end and type
     */
    public Change (int start, int end, int type) {
        this.type = type;
        this.start = start;
        this.end = end;

        //Sanity check
        if ( end < start ) {
            throw new IllegalArgumentException ( "Start " + start //NOI18N
                    + " > " + end ); //NOI18N
        }
        if ( end < 0 || start < 0 ) {
            throw new IllegalArgumentException ( "Negative start " + //NOI18N
                    start + " or end " + end ); //NOI18N
        }
        if ( type != DELETE && type != CHANGE && type != INSERT ) {
            throw new IllegalArgumentException ( "Unknown change type " + type ); //NOI18N
        }
    }

    /**
     * Constructor used by ListDiff
     */
    Change (int start, int type) {
        this.start = start;
        end = start;
        this.type = type;
        assert ( type == DELETE || type == CHANGE || type == INSERT ) : "" + type;
    }

    /**
     * Grow the endpoint of the Change by one
     */
    void inc () {
        end++;
    }

    /**
     * Set the endpoint of the Change
     */
    void setEnd (int end) {
        assert end >= start;
        this.end = end;
    }

    /**
     * Get the change type
     */
    public final int getType () {
        return type;
    }

    /**
     * Get the start index
     *
     * @return the first affected index in the list
     */
    public final int getStart () {
        return start;
    }

    /**
     * Get the end index (inclusive)
     *
     * @return the last affected index in the list
     */
    public final int getEnd () {
        return end;
    }

    /**
     * Get a string representation of this change.
     *
     * @return a string
     */
    public final String toString () {
        StringBuffer sb = new StringBuffer ();
        switch ( type ) {
            case INSERT:
                sb.append ( "INSERT " ); //NOI18N
                break;
            case DELETE:
                sb.append ( "DELETE " ); //NOI18N
                break;
            case CHANGE:
                sb.append ( "CHANGE " ); //NOI18N
                break;
            default :
                assert false;
        }
        sb.append ( start );
        sb.append ( '-' ); //NOI18N
        sb.append ( end );
        return sb.toString ();
    }
}
