/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
/*
 * TreeModelSupport.java
 *
 * Created on September 18, 2004, 10:28 PM
 */

package org.netbeans.modules.java.navigation.spi;

import javax.swing.JComponent;
import org.openide.util.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import java.util.*;
import org.netbeans.modules.java.navigation.spi.diff.Change;
import org.netbeans.modules.java.navigation.spi.diff.Diff;
import org.netbeans.modules.java.navigation.spi.strings.WeightedString;

/**
 * A simple tree model which uses nested <code>ListModelSupport</code>s to handle children. The resulting tree can be 2
 * or 3 levels deep.  If more levels are needed, implement <code>NavigatorTreeModel</code> directly - this class is
 * designed for simplicity and efficiency in the common case.
 * <p/>
 * <b>General usage</b><br> The primary method to implement is <code>loadContents()</code> - implement this to return a
 * list of children of the tree root, and <code>createListModelFor()</code>, which will be called on demand to create
 * list models for the children of the objects listed by <code>loadContents</code>.
 * <p/>
 * Beyond that, it is necessary only to override the methods for fetching icons, actions and names to have a basic,
 * functioning tree model.
 * <p/>
 * To implement a basic list model, implement the following methods: <ul> <li><code>loadContents</code> - create a list
 * of children of the root node</li> <li><code>createListModelFor</code> - to create ListModelSupports for objects
 * returned by <code>loadContents</code></li> <li><code>getDisplayName</code> or <code>assembleName</code> to provide
 * labels for objects</li> <li><code>getIcon()</code> to provide icons for objects. </ul> To handle structural changes
 * to the root's children, simply call <code>change()</code>. To notify about non-structural changes (all the objects in
 * the new list of children equal the ones in the old one, but some property like a display name has changed), call one
 * of the other <code>change</code> methods.  Note that if the embedded <code>ListModelSupport</code>s listen correctly
 * for changes, there is no need for the TreeModelSupport to process those changes - the <code>TreeModelSupport</code>
 * will automatically receive the changes from the <code>ListModelSupport</code> instances it embeds.
 *
 * @author Tim Boudreau
 */
public abstract class TreeModelSupport extends AbstractModel implements NavigatorTreeModel {
    private final Roots roots;
    protected final Object root;
    private HashMap mdls2roots = new HashMap ();
    private HashMap roots2mdls = new HashMap ();
    private List listeners = Collections.synchronizedList ( new ArrayList () );

    /**
     * Creates a new instance of TreeModelSupport
     */
    public TreeModelSupport (Object root) {
        this.root = root;
        roots = new Roots( this );
        roots.setTree(this);
    }

    //DisplayProvider methods >>
    public final void doAddNotify () {
        roots.setTree ( this );
        roots.addNotify ();
    }

    public final void removeNotify () {
        List old = getList();
        roots.removeNotify ();
        suppressChildFiring = true;
        try {
            for ( Iterator i = mdls2roots.keySet ().iterator (); i.hasNext (); ) {
                AbstractModel supp = (AbstractModel) i.next ();
                if (supp.isActive()) {
                    supp.removeNotify ();
                }
            }
            
        } finally {
            suppressChildFiring = false;
        }
    }

    /**
     * Returns true if the model is currently being displayed in the UI (addNotify has been called and removeNotify
     * hasn't been called since).
     */
    protected final boolean isActive () {
        return roots.isActive ();
    }

    public boolean acceptsStringFilter () {
        return false;
    }

    /**
     * Get a task which should complete before loadContents can be called, if any.
     */
    protected Task getWaitTask () {
        return null;
    }

    /**
     * Cancels the wait task returned by <code>getWaitTask()</code>, if the wait task was non-null and implements
     * <code>Cancellable</code>. <i> If you did not create the task returned by <code>getWaitTask()</code>, don't cancel
     * it!</code></i>
     * <p/>
     * This method will be called if <code>removeNotify()</code> has been called before the wait task has completed.
     *
     * @param waitTask The task returned by <code>getWaitTask()</code>
     * @see org.openide.util.Cancellable
     */
    protected void cancelWaitTask (Cancellable waitTask) {
        waitTask.cancel ();
    }

    public abstract Action[] getActions (Object o);

    /**
     * Convenience method for cases where no markup is needed - simply return a String that is a user-displayable name
     * for the object.  The default implementation returns null.
     * <p/>
     * Return null from this method if you need to include markup, and override <code>assembleName()</code> to do that.
     *
     * @param o The object to be named
     */
    protected String getDisplayName (Object o) {
        return null;
    }

    /**
     * Called if getName() returns null for an object.  The default implementation assembles a string with some warning
     * markup and the value of <code>o.toString()</code> to indicate that <code>getName()</code> returned null for the
     * object, but this method has not been overridden to provide markup.
     *
     * @return markup string
     */
    protected WeightedString assembleName (WeightedString ws, Object o) {
        ws.startMarkupRun ( ws.ERROR_EMPHASIS );
        ws.append ( "Unknown: ", 0.09f ); //NOI18N
        ws.endMarkupRun ();
        ws.append ( o.toString (), 0.09f );
        return ws;
    }

    public final WeightedString getName (Object o) {
        WeightedString result = WeightedString.instance ();
        String name = getDisplayName ( o );
        if ( name != null ) {
            result.append ( name, 0.09f );
        } else {
            result = assembleName ( result, o );
        }
        return result;
    }

    /**
     * Override to supply icons for objects in the model
     */
    public abstract Icon getIcon (Object o);

    /**
     * Returns null.  Override to provide tooltips for objects in the model.
     */
    public String getTooltip (Object o) {
        return null;
    }

    /**
     * Return a default action for an object.  Typically this opens the object in the editor.  Do not include the result
     * of this call in the array returned by <code>getActions()</code>.
     */
    public abstract Action getDefaultAction (Object o);

    /**
     * Returns false.  Generally single click access is not a great idea for trees, since it can get confused with
     * expansion logic.
     */
    public final boolean isDefaultActionInstant () {
        return false;
    }


    // << DisplayProvider

    // TreeModelSupport >>

    /**
     * Notifies the model that a path is about to be expanded or collapsed. This hook can be used to discard data not
     * needed after a collapse, or to record expanded paths that should be reexpanded if the same object is displayed
     * again.
     * <p/>
     * The default implementation does nothing
     */
    public void notifyExpandedState (TreePath path, boolean expanded) {
        AbstractModel mdl = findModelFor ( path, true );
        if ( mdl.isActive() != expanded ) {
            if (expanded) {
                mdl.addNotify();
            } else {
//                mdl.removeNotify();
            }
        }
    }

    AbstractModel findModelFor (TreePath path, boolean create) {
        //XXX more efficient to use the object at root + 1 and only scan
        //mdls2roots.
        return findModelFor ( path.getLastPathComponent(), create );
    }    
    
    AbstractModel findModelFor (Object o, boolean create) {
        AbstractModel result = modelForRootObject ( o, create );
//        if (result == roots) {
//            return this;
//        }
        if (result == null) {
            for (Iterator i=mdls2roots.keySet().iterator(); i.hasNext();) {
                AbstractModel am = (AbstractModel) i.next();
                if (am.owns(o)) {
                    //result = am;
                    result = am.findModelFor(o, create);
                }
            }
        }
        return result;
    }
    
    void notifyChildModelAddNotify (AbstractModel mdl) {
        if (mdl != roots) {
            List l = mdl.getLoadingTempList();
            TreeModelEvent tme = new TreeModelEvent (this, 
                mdl.getPath(), new int[] { 0 }, new Object[] { l.get(0) });
                
            childTreeFires (Change.INSERT, tme, mdl);
        }
    }

    /**
     * Get an array of tree paths that should be expanded when the model is first displayed in a tree.  Views such as
     * structural views may want to automatically expand some elements by default.
     *
     * @return an empty array unless overridden
     */
    public TreePath[] getDefaultExpandedPaths () {
        return new TreePath[ 0 ];
    }

    /**
     * Determine whether the root node of this model should be visible.
     *
     * @return true unless overridden
     */
    public boolean isShowRoot () {
        return true;
    }

    // << TreeModelSupport

    /**
     * Implement this method to generate a list of the immediate children of the root object.
     */
    protected abstract List loadContents ();
    
    protected void updated() {
        //do nothing
    }

    /**
     * Implement this method to create list models modelling the grandchildren of the root.
     */
    protected abstract AbstractModel createModelFor (Object o);

    /**
     * Override to begin listening for changes on the root of this model. It is not necessary to call this method
     * directly - it will be called after <code>loadContents()</code>
     */
    protected void startListening () {
        //do nothing
    }

    /**
     * Override to stop listening for changes on the root of this model. It is not necessary to call this method
     * directly - it will be called after <code>removeNotify()</code>.
     */
    protected void stopListening () {
        //do nothing
    }

    /**
     * Returns true if the model is loaded and functional.  Models which load data on a background thread should return
     * false from this property and fire a change event when they become loaded.
     * <p/>
     * Objects which will listen on the model should defer attaching listeners other than change listeners until the
     * model is ready.
     */
    public boolean isReady () {
        return roots.isReady ();
    }

    /**
     * Add a change listener.  The following states are of interest - isReady, and if the model implements Reorderable,
     * isReorderable().
     */
    public void addChangeListener (ChangeListener cl) throws TooManyListenersException {
        roots.addChangeListener ( cl );
    }

    /**
     * Remove a change listener.  The following states are of interest - isReady, and if the model implements
     * Reorderable, isReorderable().
     */
    public void removeChangeListener (ChangeListener cl) {
        roots.removeChangeListener ( cl );
    }

    /**
     * Get the current list, as generated in the last call to <code>loadContents</code>. This method may only be called
     * from the event dispatch thread.  Note that this method returns the list wrapped via
     * <code>Collections.unmodifiableList</code>, so it is not precisely the same instance that was last created.
     */
    protected final List getList () {
        assert SwingUtilities.isEventDispatchThread () : "getList may only be" + //NOI18N
                "called on the dispatch thread"; //NOI18N

        return roots.getList ();
    }

    public final Object getRoot () {
        return root;
    }

    public final int getIndexOfChild (Object parent, Object child) {
        if ( root.equals ( parent ) ) {
            return roots.indexOf ( child );
        } else {
            AbstractModel am = findModelFor (parent, true);
            return am == null ? -1 : am.indexOf (child);
        }
    }

    public final Object getChild (Object o, int index) {
        AbstractModel supp = findModelFor ( o, true );
        if ( supp != null ) {
            return supp.getElementAt ( index );
        } else {
            return null;
        }
    }

    /**
     * Get the ListModelSupport that corresponds to a root object, possibly creating it.
     * Triggers triggering a call to
     * <code>createListModelFor()</code> if the model has not been created and
     * create is true.
     * @param o An object in the list last returned from <@link loadContent>
     * @param create true if the model should be created if not present
     */
    protected final AbstractModel modelForRootObject (Object o, boolean create) {
        if ( root.equals ( o ) ) {
            return roots;
        } else if ( roots.owns ( o ) ) {
            AbstractModel result = (AbstractModel) roots2mdls.get ( o );
            if ( result == null ) {
                result = createModelFor ( o );
                result.setPath (combine (getPath(), new TreePath(o)));
                result.setTree ( this );
                mdls2roots.put ( result, o );
                roots2mdls.put ( o, result );
                notifyChildModelAddNotify(result);
            }
            return result;
        }
        return null;
    }

    public final int getChildCount (Object o) {
        AbstractModel supp = findModelFor ( o, true );
        if (supp instanceof Roots && o != supp.getPath().getLastPathComponent()) {
            return 0;
        }
        if ( supp != null ) {
            return supp.getSize ();
        } else {
            return 0;
        }
    }
    
    
    // Internal API for dealing the same way with nested List or Tree models >>
    TreePath getPath() {
        TreePath result = super.getPath();
        if (result == null) {
            return new TreePath(getRoot());
        }
        return result;
    }
    
    boolean owns (Object o) {
        boolean result = roots.owns (o);
        if (!result) {
            for (Iterator i=mdls2roots.keySet().iterator(); i.hasNext();) {
                result |= ((AbstractModel) i.next()).owns(o);
                if (result) break;
            }
        }
        return result;
    }
    
    int getSize() {
        return roots.getSize();
    }
    
    int fullSize() {
        int result = getSize();
        for (Iterator i=mdls2roots.keySet().iterator(); i.hasNext();) {
            result += ((AbstractModel) i.next()).fullSize();
        }
        return result;
    }
    
    Object getElementAt (int i) {
        return roots.getElementAt(i);
    }
    
    int indexOf (Object o) {
        return roots.indexOf(o);
    }

    
    private boolean suppressChildFiring = false;
    void childTreeFires (int type, TreeModelEvent tme, AbstractModel childModel) {
        if (suppressChildFiring) {
            return;
        }
        TreePath nuPath = combine (getPath(), tme.getTreePath());
        TreeModelEvent nue = new TreeModelEvent (this, nuPath, 
            tme.getChildIndices(), tme.getChildren());
        
        fire (type, nue);
    }
    
    // << Internal API for dealing the same way with nested List or Tree models
    

    /**
     * Returns true if createModelFor returned null or the model created for this object has 0 elements.
     */
    public final boolean isLeaf (Object o) {
        AbstractModel supp = findModelFor ( o, true );
        
        if (supp instanceof ListModelSupport && !(supp instanceof Roots)) {
            Object root = supp.getPath().getLastPathComponent();
            return !o.equals (root);
        } else {
            return supp == null ? true : !supp.isActive() ? false : supp.getSize() == 0;
        }
    }

    /**
     * Throws an UnsupportedOperationException
     */
    public final void valueForPathChanged (TreePath treePath, Object obj) {
        throw new UnsupportedOperationException ();
    }

    public final void addTreeModelListener (TreeModelListener l) {
        listeners.add ( l );
    }

    public final void removeTreeModelListener (TreeModelListener l) {
        listeners.remove ( l );
    }

    /**
     * Call this method when a structural change is detected (via a listener) in the root node that affects the
     * immediate children of the root.
     */
    protected final void change () {
        roots.change ();
    }

    /**
     * Call this method when a non-structural change has been detected across one or more contiguous children of the
     * root.
     */
    protected final void change (int start, int end, int changeType) {
        roots.change ( start, end, changeType );
    }

    /**
     * Call this method to notify non-structural changes across a possibly discontiguous range of children of the root
     */
    protected final void change (List changes) {
        roots.change ( changes );
    }

    /**
     * Callback that allows list models to message their owning TreeModelSupport about changes.  Since TreeModelEvents
     * can be discontiguous, this makes more sense than converting first into ListDataEvents and then back into
     * TreeModelEvents.
     */
    void processDiffs (ListModelSupport list, Diff diff[]) {
        assert SwingUtilities.isEventDispatchThread ();

        Object rootForList = list == roots ? root :
                (Object) mdls2roots.get ( list );

        TreePath rootPath = list == roots ? getPath() : list.getPath();
           
        for ( int t = 0; t < diff.length; t++ ) {
            List changes = diff[ t ].getChanges ();

            if ( rootForList != null && !changes.isEmpty () ) {
                int currType = -1;
                List indices = new ArrayList ();
                List affected = new ArrayList ();

                Change[] ch = (Change[]) changes.toArray ( new Change[ 0 ] );

                for ( int i = 0; i < ch.length; i++ ) {
                    Change change = ch[ i ];
                    if ( currType == change.getType () || currType == -1 ) {
                        currType = change.getType ();
                        addIndicesForChange ( change, indices, affected,
                                diff[ t ].getOld (), diff[ t ].getNew () );
                    }

                    if ( i == ch.length - 1 || ch[ i + 1 ].getType () != currType ) {
                        int[] childIndices = (int[]) Utilities.toPrimitiveArray
                                ( (Integer[]) indices.toArray ( new Integer[ indices.size () ] ) );


                        Object[] childObjects = affected.toArray ();

                        indices.clear ();
                        affected.clear ();
                        TreeModelEvent tme = new TreeModelEvent ( this, rootPath, childIndices, childObjects );

                        fire ( currType, tme );
                        currType = -1;
                    }
                }
            }
        }
    }

    private void addIndicesForChange (Change change, List indices, List affected, List oldValues, List newValues) {
        for ( int j = change.getStart (); j <= change.getEnd (); j++ ) {
            indices.add ( new Integer ( j ) );
            // #67203: add newValues, not oldValues to affected list if Change.CHANGE
            affected.add ( change.getType () != Change.DELETE ? newValues.get ( j ) : oldValues.get ( j ) );
        }
    }

    /**
     * Fire an event.
     */
    private void fire (int type, TreeModelEvent tme) {
        TreeModelListener[] l = (TreeModelListener[]) listeners.toArray ( new TreeModelListener[ 0 ] );
        for ( int i = 0; i < l.length; i++ ) {
            switch ( type ) {
                case Change.CHANGE:
                    l[ i ].treeNodesChanged ( tme );
                    break;
                case Change.INSERT:
                    l[ i ].treeNodesInserted ( tme );
                    break;
                case Change.DELETE:
                    l[ i ].treeNodesRemoved ( tme );
                    break;
                default :
                    assert false;
            }
        }
        TreeModelSupport rootTree = getRootTree();
        if (rootTree != this) {
            rootTree.childTreeFires (type, tme, this);
        }
        
        //For unit tests
        synchronized (this) {
            notifyAll();
        }        
    }
    
    public String toString() {
        return "[TME for " + getRoot() + " path " + getPath() + "]";
    }

    /**
     * A ListModelSupport implementation that delegates to its equivalent methods in the outer class.
     */
    private static final class Roots extends ListModelSupport {
        Roots (TreeModelSupport supp) {
            setTree(supp);
        }
        
        public Action[] getActions (Object o) {
            return getTree().getActions ( o );
        }

        public Action getDefaultAction (Object o) {
            return getTree().getDefaultAction ( o );
        }

        public Icon getIcon (Object o) {
            return getTree().getIcon ( o );
        }

        protected List loadContents () {
            return getTree().loadContents ();
        }

        protected Task getWaitTask () {
            return getTree().getWaitTask ();
        }

        protected void cancelWaitTask (Cancellable waitTask) {
            getTree().cancelWaitTask ( waitTask );
        }

        protected void startListening () {
            getTree().startListening ();
        }

        protected void stopListening () {
            getTree().stopListening ();
        }
        
        protected void updated() {
            getTree().updated();
        }
        
        TreePath getPath() {
            return getTree().getPath();
        }
        
        public JComponent getFilters () {
            return getTree().getFilters();
        }
        
    }
}
