/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
/*
 * RelatedItemProvider.java
 *
 * Created on September 23, 2004, 2:18 PM
 */

package org.netbeans.modules.java.navigation.spi;



import java.util.*;
import org.netbeans.modules.java.navigation.spi.diff.ListDiff;

/**
 * Models which implement this interface can provide information about objects which are in some way connected to a
 * given object.  For example, say that whenever the editor caret enters a method in a java class, you want all the
 * methods which call that one to be highlighted in the Navigator UI.  You would implement RelatedItemProvider to track
 * the caret position (or activated node), build a list of such methods, and fire an event when it changes.
 * <p/>
 * This interface makes no assumptions about what type of listening or what types of objects should be supplied (other
 * than that the list of objects should be objects actually in the model that provides it).
 * <p/>
 * Implementations of NavigatorTreeModel are expected to return a list of TreePath objects.
 *
 * @author Tim Boudreau
 */
public interface RelatedItemProvider {
    /**
     * Determine if an object being painted is related, and should be highlighted as such.
     */
    public boolean isRelated (Object o);

    /**
     * Determine if an object is the primary item, which the related objects are related to.
     */
    public boolean isPrimary (Object o);

    /**
     * Get the primary item, which should appear selected or otherwise indicated in the GUI.
     */
    public Object getPrimaryItem ();

    /**
     * Add a listener.
     */
    public void addRelatedItemListener (RelatedItemListener l) throws TooManyListenersException;

    /**
     * Remove a listener.
     */
    public void removeRelatedItemListener (RelatedItemListener l);
    
    /**
     * Tell this provider that its content is probably invalid;  should 
     * cause the list of related items to immediately become empty.  Used
     * for cases where the search for related items is lengthy and may occur
     * on a background thread, but it is no longer appropriate for the last
     * known set of items to be displayed;  i.e. the user changed the selection
     * in the list, and a new set of related items is about to be calculated.
     */
    public void reset();

    /**
     * Listener interface for related items
     */
    public static interface RelatedItemListener extends EventListener {
        public void itemsChanged (RelatedItemEvent evt);

        public void itemsCleared (RelatedItemEvent evt);
    }

    /**
     * Events for changes in the related items.  Events from a RelatedItemProvider <i>must</i> be thrown only on the AWT
     * event dispatch thread.
     */
    public static final class RelatedItemEvent extends EventObject {
        private final Collection oldItems;
        private final Collection newItems;
        private final Object oldPrimary;
        private final Object newPrimary;

        /**
         * Create an event.
         *
         * @param source - the provider
         * @param oldItems - the collection of related items before the event. May be null.
         * @param newItems - the collection of related items now.  May be null.
         */
        public RelatedItemEvent (Object source, Collection oldItems, Collection newItems, Object oldPrimary,
                                 Object newPrimary) {
            super ( source );
            this.oldItems = oldItems == null ? Collections.EMPTY_SET : oldItems;
            this.newItems = newItems == null ? Collections.EMPTY_SET : newItems;
            this.oldPrimary = oldPrimary;
            this.newPrimary = newPrimary;
        }

        /**
         * Get the collection of items which are now related.
         */
        public Collection getRelatedItems () {
            return newItems;
        }

        public Collection getPreviousRelatedItems () {
            return oldItems;
        }

        public Object getOldPrimary () {
            return oldPrimary;
        }

        public Object getNewPrimary () {
            return newPrimary;
        }

        public String toString () {
            StringBuffer sb = new StringBuffer ();
            sb.append ( super.toString () );
            try {
                List l1 = oldItems instanceof List ? (List) oldItems :
                        new ArrayList ( oldItems );
                List l2 = newItems instanceof List ? (List) newItems :
                        new ArrayList ( newItems );
                sb.append ( ListDiff.createDiff ( l1, l2 ) );
            } catch ( Exception e ) {
                //InvalidObjectException, etc.  Legal.
                sb.append ( "Exception iterating the list: " + e );
            }
            return sb.toString ();
        }
    }
}
