/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.navigation.spi;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.TooManyListenersException;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import org.netbeans.modules.java.navigation.base.ModelBusyListener;
import org.netbeans.modules.java.navigation.spi.diff.Change;
import org.netbeans.modules.java.navigation.spi.diff.Diff;
import org.netbeans.modules.java.navigation.spi.diff.ListDiff;
import org.netbeans.modules.java.navigation.spi.strings.WeightedString;
import org.openide.ErrorManager;
import org.openide.util.Cancellable;
import org.openide.util.Mutex;
import org.openide.util.RequestProcessor;
import org.openide.util.Task;
import org.openide.util.TaskListener;

/**
 * Support class to simplify implementing NavigatorListModel.  The implementor provides a {@link List} on
 * demand, which will contain the contents of the model.
 * <p>
 * This class is designed to handle asynchronous loading of data, and will return "Please Wait" as its contents while
 * loading.  There are several methods that affect threading.
 * <p>
 * Constructor arguments can determine whether the generation of the list happens on the AWT event thread or a dedicated
 * thread.
 * <p>
 * This class significantly simplifies the complexity and threading issues with implementing a list model.
 * <p>
 * To implement a basic list model, implement the following methods:
 * <dl>
 * <dt>{@link #loadContents} <dd>create a list
 * that is the model's contents
 * <dt>{@link #getDisplayName} or {@link #assembleName} <dd>to provide labels for objects
 * <dt>{@link #getIcon} <dd>to provide icons for objects
 * </dl>
 * <p>
 * To handle structural changes to the root's children, simply call {@link #change()}. To notify about
 * non-structural changes (all the objects in the new list of children equal the ones in the old one, but some property
 * like a display name has changed), call one of the other {@link #change} methods.  Note that if the embedded
 * {@link ListModelSupport}s listen correctly for changes, there is no need for the {@link TreeModelSupport} to process
 * those changes - the {@link TreeModelSupport} will automatically receive the changes from the
 * {@link ListModelSupport} instances it embeds.
 *
 * @author Tim Boudreau
 */
public abstract class ListModelSupport extends AbstractModel implements NavigatorListModel {
    private boolean invalid = false;

    private final boolean singleClick;
    List list = null;
    private volatile boolean active;
    private List listeners = Collections.synchronizedList ( new ArrayList ( 2 ) );
    private ChangeManager changeManager = null;

    private Loader loader = null;

    // XXX Javadoc?
    protected static final RequestProcessor rp =
            new RequestProcessor ( "ListModelSupport loader", 1 ); //NOI18N

    private static Thread rpThread = null;
    private static ListModelSupport waitingModel = null;

    private boolean updateAsync = false;
    private boolean readAsync = true;

    private int retryCount = 0;

    private ChangeListener changeListener = null;
    private ModelBusyListener busyListener = null;
    
    private static final Object START_STOP_LISTENING_LOCK = new Object();
    

    /**
     * Creates a new support which returns true from {@link #isDefaultActionInstant()}.
     */
    protected ListModelSupport () {
        this ( true );
    }

    /**
     * Creates a new support which returns the specified value from
     * {@link #isDefaultActionInstant}.
     *
     * @param singleClick whether or not a single click should cause the default action of clicked objects to be
     * invoked when this model is displayed in a gui component.
     */
    protected ListModelSupport (boolean singleClick) {
        this.singleClick = singleClick;
    }

    /**
     * Creates a new support.
     *
     * @param singleClick whether or not a single click should invoke the default action of objects in the model
     * @param readAsync whether or not {@link #loadContents()} may be slow and the initial populating of the model
     * should happen in a request processor thread
     * @param updateAsync whether or not later code to {@link #loadContents} in response to events (triggered by
     * calls to one of the {@link #change()} methods) may be slow and should be run in a request processor thread.
     * @see #isDefaultActionInstant
     */
    protected ListModelSupport (boolean singleClick, boolean readAsync, boolean updateAsync) {
        this ( singleClick );
        this.readAsync = readAsync;
        this.updateAsync = updateAsync;
    }

    /**
     * Get a list of objects in the model which match the passed string.
     * <p>
     * The default implementation simply searches the string form of the result of {@link #getDisplayName}.  It is
     * preferable to override this method to operate more efficiently, and handle cases where object names have leading
     * text which is less important and should not be what is matched against (i.e. a node <samp>SomeClass.getFoo()</samp>
     * should match on <samp>getF</samp>, not <samp>SomeClass.getF</samp>).
     *
     * @return A List of objects that are in this model
     */
    public List getSearchResults (String partial) {
        if ( list == null ) {
            return Collections.EMPTY_LIST;
        }
        ArrayList result = new ArrayList ();
        for ( Iterator i = list.iterator (); i.hasNext (); ) {
            Object o = i.next ();
            String s = getName ( o ).toString ().toUpperCase ();
            if ( s.startsWith ( partial ) ) {
                result.add ( o );
            }
        }
        return result;
    }

    /**
     * Get the current list, as generated in the last call to {@link #loadContents}. This method may only be called
     * from the event dispatch thread.  Note that this method returns the list wrapped via
     * {@link Collections#unmodifiableList}, so it is not precisely the same instance that was last created.
     */
    protected final List getList () {
        assert SwingUtilities.isEventDispatchThread () : "getList may only be" + //NOI18N
                "called on the dispatch thread"; //NOI18N

        return list == null ? Collections.EMPTY_LIST : Collections.unmodifiableList ( list );
    }

    /**
     * Used by TreeModelSupport
     */
    boolean owns (Object o) {
        return list == null || invalid ? false : list.contains ( o );
    }

    /**
     * Get the index of an object in the model.
     * <p>
     * It should typically not be necessary to override this method, but in cases such as using wrapper objects, where a
     * RelatedItemProvider will not know about the wrappers and try to look up a wrapper object's index, it may be.
     *
     * @return The index of the passed object in the last returned list from {@link #loadContents}, or -1 if the
     *         list is null or the model is invalid
     */
    public int indexOf (Object o) {
        return list == null || invalid ? -1 : list.indexOf ( o );
    }

    /**
     * Determine if updates - calls to {@link #loadContents} in response to change events - should also run in a non-AWT
     * thread.  In some cases, the initial loading of data is slow, but later changes to the data do not have the same
     * amount of overhead, and don't need to be run asynchronously.
     *
     * @return if not specified in the constructor, returns false
     * @see #isReadAsync
     */
    protected final boolean isUpdateAsync () {
        return updateAsync;
    }

    /**
     * Determines if fetching the list is a slow enough operation that it should not be done in the AWT event thread.
     *
     * @return If not specified in the constructor, returns true
     * @see #isUpdateAsync
     */
    protected final boolean isReadAsync () {
        return readAsync;
    }

    /**
     * Retrieve a task which must complete before {@link #loadContents()} can be called.  If, at the time
     * {@link #addNotify()} is called, there is an externally running task (for example, the Java parser's parse
     * task) which must complete before the contents can be read, return it from this method - the model will listen for
     * the task's completion, and not try to load data until it is complete.
     *
     * @return null by default
     */
    protected Task getWaitTask () {
        return null;
    }

    /**
     * Indicates that the model should begin listening for changes in whatever object(s) it represents.  The default
     * implementation does nothing.
     */
    protected void startListening () {
        //do nothing
    }

    /**
     * Indicates that the model should stop listening for changes in whatever object(s) it represents, and ensure no
     * external objects hold references to this model instance - it is probably not going to be used again. The default
     * implementation does nothing.
     */
    protected void stopListening () {
        //do nothing
    }

    /**
     * Create a {@link List} of the contents of this model.  Note that if filters are supported, the list returned
     * should be filtered by the currently set filter, if any.
     * <p>
     * This method will be called after {@link #addNotify} is called.
     * <p>
     * <b>Threading:</b> If {@link #isReadAsync} returns true, then the initial call to populate the model will be
     * run in a thread that is not the AWT event thread, but calls triggered by {@link #change} to update the
     * model will be made from the event thread.  If {@link #isUpdateAsync} returns false (the default), subsequent
     * calls to refresh the model will run in the event thread.  If both methods return false, this method will always
     * run in the event thread.
     * <p>
     * Subclasses should never directly call this method - rather, call {@link #change} which trigger loading in
     * accordance with the threading model specified, and fire events as needed.
     * <p>
     * Once a list has been returned from this method, it should be considered immutable and its contents should not be
     * changed.
     *
     * @return a list that is the current contents of the model
     */
    protected abstract List loadContents ();
    
    /**
     * Hook method called when the contents are actually set (in the AWT thread).
     * If there is supplementary data other than the list of objects which should
     * be updated in a thread-safe way when the contents are actually set, handle
     * that here. 
     * <div class="nonnormative">
     * <p>
     * Example - say that we are processing a properties file, generating a list
     * of the keys in {@link #loadContents}.  We also want to keep the 
     * <code>Properties</code> object around to fetch values from, for 
     * tooltips and such.  But {@link #loadContents} runs in a different
     * thread.  So we don't want to set an instance field for the <code>Properties</code>
     * object from {@link #loadContents} - a paint could be happening at the
     * same time in the AWT thread, and try to access the properties field while
     * it is only half-written.
     * <p>
     * Here is the simple way to handle this:
     * <pre>
     * private Properties myProperties = null;
     * private Properties myNewProperties = null;
     *
     * protected List loadContents() {
     *     myNewProperties = // find/create the Properties object
     *     return new ArrayList(p.keySet()); // or something such
     * }
     *
     * protected void updated() {
     *     myProperties = myNewProperties;
     *     myNewProperties = null;
     * }
     * </pre>
     * </div>
     * The return value of {@link #getList()} is guaranteed to be the 
     * newly generated list at the time this method is called.
     *
     */
    protected void updated () {
        //do nothing
    }

    /**
     * Cancels the wait task returned by {@link #getWaitTask()}, if the wait task was non-null and implements
     * {@link Cancellable}.
     * <p>
     * <strong>If you did not create the task returned by {@link #getWaitTask()}, do not cancel it!</strong>
     * <p>
     * This method will be called if {@link #removeNotify()} has been called before the wait task has completed.
     *
     * @param waitTask the task returned by {@link #getWaitTask()}
     */
    protected void cancelWaitTask (Cancellable waitTask) {
        waitTask.cancel ();
    }

    /**
     * Returns true if {@link #removeNotify} has not been called since the last call to {@link #addNotify}, false if {@link #removeNotify} has been
     * called or {@link #addNotify} has not yet been called.
     * If long running operations are being performed in a background
     * thread, it may be worth checking this value to decide whether to continue or not.
     */
    protected final boolean isActive () {
        return active;
    }

    /**
     * Returns true if the model is loaded and functional.  Models which load data on a background thread should return
     * false from this property and fire a change event when they become loaded.
     */
    public final boolean isReady () {
        return list != null;
    }

    /**
     * Add a change listener.
     * The following states are of interest: {@link #isReady}; and if the model implements {@link Reorderable},
     * {@link Reorderable#isReorderable}.
     * <p>
     * Objects which will listen on the model should defer attaching listeners other than change listeners until the
     * model is ready.
     */
    public final void addChangeListener (ChangeListener cl) throws TooManyListenersException {
        if ( changeListener != null ) {
            throw new TooManyListenersException ( cl.toString () );
        }
        changeListener = cl;
    }

    /**
     * Remove a change listener.
     * See {@link #addChangeListener} for details.
     */
    public final void removeChangeListener (ChangeListener cl) {
        if ( changeListener != cl ) {
            throw new IllegalArgumentException ( "Wrong listener " + cl );
        }
        changeListener = null;
    }
    
    /**
     * Fire a change event to the change listener, if any
     */
    protected final void fireChange () {
        if ( changeListener != null ) {
            changeListener.stateChanged ( new ChangeEvent ( this ) );
        }
    }
    
    public final void addBusyListener(ModelBusyListener bl) throws TooManyListenersException {
        if ( busyListener != null ) {
            throw new TooManyListenersException ( bl.toString () );
        }
        busyListener = bl;
    }
    
    public final void removeBusyListener (ModelBusyListener bl) {
        if ( busyListener != bl ) {
            throw new IllegalArgumentException ( "Wrong listener " + bl );
        }
        busyListener = null;
    }
    
    /** fires change of businnes state of this model. True means start of
     * being busy, false means ready (not busy) again.
     */
    protected final void fireBusyChange (boolean value) {
        if (busyListener == null) {
            return;
        }
        if (value) {
            busyListener.busyStart();
        } else {
            busyListener.busyEnd();
        }
    }
    
    private void fireNewContentReady () {
        if (busyListener == null) {
            return;
        }
        busyListener.newContentReady();
    }

    /**
     * Mark this model as invalid, meaning its former contents should no longer be read or touched - or if it was
     * invalid, allow it to resume functioning.
     */
    protected final void setInvalid (boolean invalid) {
        if ( invalid != this.invalid ) {
            this.invalid = invalid;
            if ( invalid ) {
                stopListening ();
                Mutex.EVENT.readAccess ( new Runnable () {
                    public void run () {
                        setContents ( Arrays.asList ( new Object[]{ AbstractModel.createInvalidMarker()} ) );
                    }
                } );
                loader.cancel ();
            } else {
                list = null;
                loader = new Loader ( getWaitTask () );
            }
        }
    }

    /**
     * Call this method if a change has happened whose nature is unknown, and the contents should be reloaded and diffed
     * against the previous contents.
     */
    protected final void change () {
        if ( invalid ) return;
        if ( loader == null ) {
            throw new IllegalStateException ( "Change called before addNotify" );
        }
        loader.start ();
    }

    /**
     * This method can be called from {@link #loadContents}, if the data to be loaded is temporarily invalid, and a reread should
     * be attempted after a delay. This method simply puts the loader thread to sleep. If the loader thread is needed by
     * another model, it may be interrupted before the timeout to handle that model's needs.  There is a hard limit of
     * 10 retries<!-- XXX no such method! after which setInvalid(true) will be called-->.  This method may only be called from the
     * loader thread (meaning that the model must use asynchronous loading), and may only be called from within
     * {@link #loadContents()}.
     *
     * @param delay - a delay in milliseconds
     */
    protected void retryLoad (long delay) {
        //We will only put the loader thread to sleep
        if ( !rp.isRequestProcessorThread () ) {
            throw new IllegalStateException ( "retryLoad must be called from " +
                    "the loader thread" );
        }

        //Do not let this method be called from anywhere except a call to
        //loadContents
        if ( !Thread.holdsLock ( loader ) ) {
            throw new IllegalStateException ( "retryLoad may only be called " +
                    "from loadContents" );
        }

        //Arbitrary, but prohibits really deranged stuff
        if ( delay > 90000 ) {
            throw new IllegalArgumentException ( "Abusive retry delay - must be" +
                    " less than 1.5 minutes" );
        }

        //See if it's time to give up
        if ( retryCount > 10 ) {
            ErrorManager.getDefault ().log ( ErrorManager.INFORMATIONAL, "Too many" +
                    " retries on " + this + " - marking it invalid" );
            setInvalid ( true );
            return;
        }
        ErrorManager.getDefault ().log ( ErrorManager.INFORMATIONAL, "Putting " +
                "loader thread to sleep for " + delay + " milliseconds due to error" );

        //Okay, put the loader thread to sleep.
        try {
            retryCount++;
            //Record who and what is sleeping - add/removeNotify will use these
            //to interrupt the thread if needed
            synchronized (ListModelSupport.class) {
                rpThread = Thread.currentThread ();
                waitingModel = this;
            }

            Thread.currentThread ().sleep ( delay );
        } catch ( InterruptedException e ) {
            //May legitimately be interrupted if another model needs
            //the thread.  Post a delayed retry.
            if ( isActive () && !invalid ) {
                //Post the loader into rp, don't call loader.start(), which could
                //execute synchronously and we could end up right back here
                //before handling the reason we were interrupted.
                loader.retryStart ( delay );
                return;
            }
        } finally {
            rpThread = null;
            waitingModel = null;
        }
        if ( isActive () && !invalid ) {
            //Try to post the load request again
            loader.retryStart ( 0 );
        }
    }

    /**
     * Set the contents of the model.
     */
    private void setContents (final List l) {
        final List oldList = ListModelSupport.this.list;
        Mutex.EVENT.writeAccess ( new Runnable () {
            public void run () {
                if ( !active ) {
                    return;
                }
                boolean isNewContent = ListModelSupport.this.list == null;
                ListModelSupport.this.list = l;
                
                try {
                    updated();
                } catch (RuntimeException e) {
                    ErrorManager.getDefault().notify (e);
                }
                
                fireChange ();
                if ( listeners.size () > 0 || getTree() != null ) {
                    if ( isNewContent ) {
                        List pseudoContent =
                                Arrays.asList ( new Object[]{ AbstractModel.createInvalidMarker() } );

                        if ( listeners.size () > 0 ) {
                            ExtListDataEvent e = new ExtListDataEvent ( ListModelSupport.this,
                                    ListDataEvent.INTERVAL_REMOVED,
                                    0,
                                    0 );
                            e.setContents ( pseudoContent, null );
                            fire ( e );
                        }

                        if ( l.size () > 0 && listeners.size () > 0 ) {
                            ExtListDataEvent e = new ExtListDataEvent ( ListModelSupport.this,
                                    ListDataEvent.INTERVAL_ADDED, 0, l.size () );
                            e.setContents ( Collections.EMPTY_LIST, l );
                            fire ( e );
                        }

                        if ( getTree() != null ) {
                            getTree().processDiffs ( ListModelSupport.this,
                                    new Diff[]{ListDiff.createDiff ( pseudoContent,
                                            list )} );
                        }
                                
                        // #59934: Allow components to keep selection
                        fireNewContentReady();        
                    }
                }
            }
        } );
        if (oldList != null) {
            change (oldList, l);
        }
    }

    /**
     * Convenience method for firing a set of changes.
     *
     * @param changes A {@link List} of {@link Change} objects.
     */
    protected final void change (List changes) {
        queueChanges ( ListDiff.createFixed ( list, list, changes ) );
    }

    /**
     * Convenience method for firing a change of a given type to an inclusive range of elements.
     *
     * @param start The first element
     * @param end The last element >= start
     * @param changeType one of the change types defined in {@link Change}
     */
    protected final void change (int start, int end, int changeType) {
        Change change = new Change ( start, end, changeType );
        change ( Arrays.asList ( new Object[]{change} ) );
    }

    /**
     * Creates and queues change events for the diff of the old and new list
     */
    private final void change (List old, List nue) {
        if ( !active ) {
            return;
        }
        Diff diff = ListDiff.createDiff ( old, nue );
        diff.getChanges ();
        queueChanges ( diff );
    }

    public final Object getElementAt (int idx) {
        Object result;
        if (list == null) {
            if (invalid) {
                result = AbstractModel.createInvalidMarker();
            } else {
                // waiting marker
                result = getLoadingTempList().get(0);
            }
        } else {
            if (!list.isEmpty()) {
                result = list.get ( idx );
            } else {
                // XXX - here should be empty marker
                result = getLoadingTempList().get(0);
            }
        }
        if ( !AbstractModel.isInvalidMarker(result) &&
             !AbstractModel.isWaitMarker(result) &&
             !isObjectValid ( result ) ) {
            result = AbstractModel.createInvalidMarker();
        }
        return result;
    }

    /**
     * Used to determine if an object in the list is valid.  If it is possible for objects displayed to expire
     * unexpectedly due to some catastrophe, return false from this method, and the "invalid" marker object
     * will be displayed instead, so no methods on the invalid object will be called when painting.
     */
    protected boolean isObjectValid (Object o) {
        return true;
    }

    public final int getSize () {
        return list == null || invalid ? 1 : list.size ();
    }

    /**
     * Returns the constructor argument {@link #singleClick}, which is false if the no-argument constructor was
     * used.
     */
    public final boolean isDefaultActionInstant () {
        return singleClick;
    }

    /**
     * Returns null.
     */
    public String getTooltip (Object o) {
        return null;
    }
    
    public String getTooltip (Object tooltipFor, int x, int y) {
        return null;
    }

    /**
     * Convenience method for cases where no markup is needed - simply return a string that is a user-displayable name
     * for the object.  The default implementation returns null.
     * <p>
     * Return null from this method if you need to include markup, and override {@link #assembleName} to do that.
     *
     * @param o The object to be named
     */
    protected String getDisplayName (Object o) {
        return null;
    }

    /**
     * Called if {@link #getDisplayName} returns null for an object.  The default implementation assembles a string with some warning
     * markup and the value of <code>o.toString()</code> to indicate that {@link #getDisplayName} returned null for the
     * object, but this method has not been overridden to provide markup.
     *
     * @return markup string
     */
    protected WeightedString assembleName (WeightedString str, Object o) {
        str.startMarkupRun ( str.ERROR_EMPHASIS );
        str.append ( "Unknown: ", 0.09f );
        str.endMarkupRun ();
        str.append ( o.toString (), 0.09f );
        return str;
    }

static long cumulateTime = 0;    
    
    /**
     * Implemented to first call {@link #getDisplayName(Object)} with the object in question.  If that returns null,
     * then it will call assembleName() with the same argument.
     */
    public final WeightedString getName (Object o) {
long a = System.currentTimeMillis();        
        WeightedString result = WeightedString.instance ();
        String name = getDisplayName ( o );
        if ( name != null ) {
            result.append ( name, 0.09f );
        } else {
            result = assembleName ( result, o );
        }
cumulateTime += System.currentTimeMillis() - a;
        return result;
    }


    //Change event support >>

    /**
     * Enqueue a diff of changes to run on the event queue
     */
    private synchronized void queueChanges (Diff diff) {
        if ( !active ) {
            return;
        }
        if ( changeManager == null ) {
            changeManager = new ChangeManager ();
        }
        if ( changeManager.isBusy () ) {
            changeManager = new ChangeManager ().enqueue ( diff );
        } else {
            changeManager.enqueue ( diff );
        }
    }

    /**
     * Add a list data listener.  This method is thread safe.
     */
    public final void addListDataListener (ListDataListener l) {
        listeners.add ( l );
    }

    /**
     * Remove a list data listener.  This method is thread safe.
     */
    public final void removeListDataListener (ListDataListener l) {
        listeners.remove ( l );
    }

    /**
     * Fires a list data event to listeners
     */
    private void fire (ListDataEvent e) {
        if ( !active ) {
            return;
        }
        ListDataListener[] l = new ListDataListener[ listeners.size () ];
        l = (ListDataListener[]) listeners.toArray ( l );
        for ( int i = 0; i < l.length; i++ ) {
            switch ( e.getType () ) {
                case ListDataEvent.CONTENTS_CHANGED:
                    l[ i ].contentsChanged ( e );
                    break;
                case ListDataEvent.INTERVAL_ADDED:
                    l[ i ].intervalAdded ( e );
                    break;
                case ListDataEvent.INTERVAL_REMOVED:
                    l[ i ].intervalRemoved ( e );
                    break;
                default :
                    throw new IllegalArgumentException ( "Bad event type: " + e.getType () ); //NOI18N
            }
        }
    }

    /**
     * Runnable class which batches changes and ensures they are fired on the event queue; is responsible for handling
     * diffs and firing the appropriate events.
     */
    private final class ChangeManager implements Runnable {
        private boolean enqueued = false;
        private final List diffs = Collections.synchronizedList ( new ArrayList ( 2 ) );
        private volatile boolean busy = false;

        /**
         * Enqueue one diff
         */
        ChangeManager enqueue (Diff diff) {
            diffs.add ( diff );
            if ( !enqueued ) {
                enqueued = true;
                Mutex.EVENT.readAccess ( this );
            }
            return this;
        }

        /**
         * Determine if we are processing diffs at this moment
         */
        boolean isBusy () {
            return busy;
        }

        /**
         * Processes all queued diffs
         */
        public void run () {
            if ( !active ) {
                return;
            }
            assert SwingUtilities.isEventDispatchThread ();
            busy = true;
            try {
                Diff diffs[] = null;
                synchronized (this.diffs) {
                    diffs = (Diff[]) this.diffs.toArray ( new Diff[ this.diffs.size () ] );
                    this.diffs.clear ();
                }
                processDiffs ( diffs );
            } finally {
                enqueued = false;
                busy = false;
            }
        }

        /**
         * Process an array of diffs
         */
        private void processDiffs (Diff[] diffs) {
            assert SwingUtilities.isEventDispatchThread ();
            if ( !isActive () ) {
                return;
            }

            if ( !listeners.isEmpty () ) {
                for ( int i = 0; i < diffs.length; i++ ) {
                    processDiff ( diffs[ i ] );
                }
            }
            if ( getTree() != null ) {
                getTree().processDiffs ( ListModelSupport.this, diffs );
            }
        }

        /**
         * Process an individual diff by generating and firing a ListDataEvent for each Change object in the diff
         */
        private void processDiff (Diff diff) {
            for ( Iterator i = diff.getChanges ().iterator (); i.hasNext (); ) {
                Change change = (Change) i.next ();
                ExtListDataEvent evt = new ExtListDataEvent ( ListModelSupport.this,
                        change.getType (),
                        change.getStart (),
                        change.getEnd () );
                evt.setContents ( diff.getOld (), diff.getNew () );
                fire ( evt );
            }
        }
    }

    //<< Change event support


    //Asynchronous/synchronous loading support >>

    /**
     * Starts loading data after the wait task (if any) has finished, loading it on or off the event thread as
     * determined by {@link #isReadAsync()}. Sets the active flag to true.  Until data is loaded, the model will
     * contain a single item: "Please wait".
     */
    public final void doAddNotify () {
        if ( active ) {
            throw new IllegalStateException ( "Already active" );
        }
        if ( invalid ) {
            return;
        }
        active = true;
        list = null;
        if ( rpThread != null ) {
            rpThread.interrupt ();
            rpThread = null;
            waitingModel = null;
        }
        loader = new Loader ( getWaitTask () );
    }

    /**
     * Cancels and pending data loading, unregisters listeners.
     */
    public final void removeNotify () {
        if ( loader == null ) {
            throw new IllegalStateException ( "removeNotify called before addNotify" );
        }
        // #68859: ensure correct order of startListening/stopListening order
        synchronized (START_STOP_LISTENING_LOCK) {
            loader.cancel ();
            stopListening ();
        }
        active = false;

        synchronized (ListModelSupport.class) {
            if ( rpThread != null && waitingModel == this ) {
                rpThread.interrupt ();
                rpThread = null;
                waitingModel = null;
            }
        }
    }

    /**
     * Class responsible for loading on the appropriate thread, waiting for the wait task to finish before loading,
     * etc.
     */
    private final class Loader implements Runnable, TaskListener, Cancellable {
        private boolean cancelled = false;
        private boolean finished = true;
        private final Task waitTask;

        public Loader (Task waitTask) {
            this.waitTask = waitTask;
            if ( waitTask != null && !waitTask.isFinished () ) {
                waitTask.addTaskListener ( this );
            } else {
                start ();
            }
        }

        public boolean cancel () {
            cancelled = true;
            if ( waitTask != null && waitTask instanceof Cancellable ) {
                cancelWaitTask ( (Cancellable) waitTask );
            }
            return true;
        }

        public void start () {
            if ( !finished ) {
                return;
            }
            finished = false;
            cancelled = false;
            boolean async = list == null ? isReadAsync () : isUpdateAsync ();
            if ( rp.isRequestProcessorThread () && async ) {
                run ();
            } else if ( SwingUtilities.isEventDispatchThread () && !async ) {
                run ();
            } else {
                if ( async ) {
                    rp.post ( this );
                } else {
                    SwingUtilities.invokeLater ( this );
                }
            }
        }

        public void retryStart (long delay) {
            rp.post ( this, (int) delay );
        }

        /**
         * Guaranteed to run in either the request processor thread or the event queue.
         */
        public void run () {
            assert SwingUtilities.isEventDispatchThread () || rp.isRequestProcessorThread ();
            if ( cancelled || !active || invalid ) {
                return;
            }
            try {
                boolean needListen = list == null;
                //We synchronize here to enforce the Thread.holdsLock() test in
                //retryLoad()
                synchronized (this) {
                    setContents ( loadContents () );
                }
                // #68859: ensure correct order of startListening/stopListening order
                synchronized (START_STOP_LISTENING_LOCK) {
                    if ( cancelled || !active || invalid ) {
                        return;
                    }
                    if ( needListen ) {
                        startListening ();
                    }
                }
            } finally {
                finished = true;
                cancelled = false;
            }
        }

        public void taskFinished (Task task) {
            task.removeTaskListener ( this );
            start ();
        }
    }

    //<< Asynchronous/synchronous loading support
}
