/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 *  You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 *  The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 *  Microsystems, Inc. All Rights Reserved.
 *
 */

package org.netbeans.modules.java.navigation.base;

import org.openide.util.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.List;
import org.netbeans.modules.java.navigation.spi.NavigatorListModel;
import org.netbeans.modules.java.navigation.spi.NavigatorTreeModel;

/**
 * Hastily adapted from TreeView.
 *
 * @author tim
 */
public final class SearchPanel {
    
    private JPanel searchpanel = null;
    private SearchTextField searchTextField = new SearchTextField ();

    private JTree treeTarget = null;
    private JList listTarget = null;

    private SearchKeyProcessor keyProcessor = new SearchKeyProcessor ();
    private SearchFieldListener searchFieldListener = new SearchFieldListener ();

    private List results = null;

    /**
     * Creates a new instance of SearchPanel
     */
    public SearchPanel () {
        // Create a the "multi-event" listener for the text field. Instead of
        // adding separate instances of each needed listener, we're using a
        // class which implements them all. This approach is used in order
        // to avoid the creation of 4 instances which takes some time
        searchTextField.addKeyListener ( searchFieldListener );
        searchTextField.addFocusListener ( searchFieldListener );
        searchTextField.getDocument ().addDocumentListener ( searchFieldListener );
    }

    public void prepare (JComponent component) {
        //XXX doesn't this hose default keyboard behavior?

        // Remove the default key listeners
        KeyListener[] keyListeners = (KeyListener[]) component.getListeners ( KeyListener.class );

        for ( int i = 0; i < keyListeners.length; i++ ) {
            component.removeKeyListener ( keyListeners[ i ] );
        }
        // Add new key listeners
        component.addKeyListener ( keyProcessor );
    }

    private void invoke (JTree target) {
        this.treeTarget = target;
        this.listTarget = null;
        show ();
    }

    private void invoke (JList target) {
        this.treeTarget = null;
        this.listTarget = target;
        show ();
    }

    private void detach () {
        if ( searchpanel.isShowing () ) {
            JComponent c = (JComponent) searchpanel.getParent ();
            searchpanel.getParent ().remove ( searchpanel );
            c.repaint ( searchpanel.getBounds () );
        }
        results = null;
    }

    private JComponent getTarget () {
        return treeTarget == null ? (JComponent) listTarget : (JComponent) treeTarget;
    }

    private void show () {
        prepareSearchPanel ();

        JComponent target = (JComponent) getTarget ().getParent ().getParent ();
        JLayeredPane dest = target.getRootPane ().getLayeredPane ();

        Point loc = new Point ( target.getWidth () / 2,  5 );

        loc = SwingUtilities.convertPoint ( target, loc, dest );
        int width = target.getWidth () / 2;
        int height = getTarget () instanceof JTree ? ( (JTree) getTarget () ).getRowHeight () + 8 :
                ( (JList) getTarget () ).getFixedCellHeight () + 8;
        if ( width < 120 ) {
            //too narrow
            width = 160;
            loc.x -= 160;
        }
        if ( loc.x < 0 ) {
            loc.x += target.getWidth ();
        }
        searchTextField.setFont ( getTarget ().getFont () );
        searchpanel.setBounds ( loc.x, loc.y, width, height );

        dest.add ( searchpanel, JLayeredPane.POPUP_LAYER );
        searchpanel.setVisible ( true );
        searchTextField.requestFocus ();
    }


    // searchTextField manages focus because it handles VK_TAB key
    private class SearchTextField extends JTextField {
        public boolean isManagingFocus () {
            return true;
        }

        public void processKeyEvent (KeyEvent ke) {
            //override the default handling so that
            //the parent will never receive the escape key
            if ( ke.getKeyCode () == ke.VK_ESCAPE ) {
                final JComponent target = getTarget ();
                detach ();
                ke.consume();
                // bugfix #32909, reqest focus when search field is removed
                SwingUtilities.invokeLater ( new Runnable () {
                    //additional bugfix - do focus change later or removing
                    //the component while it's focused will cause focus to
                    //get transferred to the next component in the
                    //parent focusTraversalPolicy *after* our request
                    //focus completes, so focus goes into a black hole - Tim
                    public void run () {
                        target.requestFocus ();
                    }
                } );
            } else {
                super.processKeyEvent ( ke );
            }
        }
    }

    private void prepareSearchPanel () {
        if ( searchpanel == null ) {
            searchpanel = new JPanel ();
            JLabel lbl = new JLabel ( NbBundle.getMessage ( SearchPanel.class,
                    "LBL_QUICKSEARCH" ) ); //NOI18N
            searchpanel.setLayout ( new BoxLayout ( searchpanel, BoxLayout.X_AXIS ) );
            searchpanel.add ( lbl );
            searchpanel.add ( searchTextField );
            lbl.setLabelFor ( searchTextField );
            searchpanel.setBorder ( BorderFactory.createRaisedBevelBorder () );
            lbl.setBorder ( BorderFactory.createEmptyBorder ( 0, 0, 0, 5 ) );
        }
    }

    private class SearchKeyProcessor extends KeyAdapter {
        private boolean armed = false;

        public void keyPressed (KeyEvent e) {
            int modifiers = e.getModifiers ();
            int keyCode = e.getKeyCode ();
            //#43617 - don't eat + and -
            if ( keyCode == KeyEvent.VK_PLUS ||
                    keyCode == KeyEvent.VK_MINUS ||
                    keyCode == KeyEvent.VK_ADD ||
                    keyCode == KeyEvent.VK_SUBTRACT ) {
                return;
            }

            if ( ( modifiers > 0 && modifiers != KeyEvent.SHIFT_MASK ) || e.isActionKey () )
                return;
            char c = e.getKeyChar ();
            if ( !Character.isISOControl ( c ) && keyCode != KeyEvent.VK_SHIFT && keyCode != KeyEvent.VK_ESCAPE ) {
                armed = true;
                e.consume ();
            }
        }

        public void keyTyped (KeyEvent e) {
            if ( armed ) {
                final KeyStroke stroke = KeyStroke.getKeyStrokeForEvent ( e );
                if ( e.getSource () instanceof JTree ) {
                    invoke ( (JTree) e.getSource () );
                } else {
                    invoke ( (JList) e.getSource () );
                }
                searchTextField.setText ( String.valueOf ( stroke.getKeyChar () ) );
                e.consume ();
                armed = false;
            }
        }
    }

    private class SearchFieldListener extends KeyAdapter
            implements DocumentListener, FocusListener {
        SearchFieldListener () {
        }

        private List results = new ArrayList ();
        private int currentSelectionIndex;

        public void changedUpdate (DocumentEvent e) {
            searchForNode ();
        }

        public void insertUpdate (DocumentEvent e) {
            searchForNode ();
        }

        public void removeUpdate (DocumentEvent e) {
            searchForNode ();
        }

        public void keyPressed (KeyEvent e) {
            int keyCode = e.getKeyCode ();
            if ( keyCode == KeyEvent.VK_ESCAPE ) {
                detach ();
                e.consume();
                getTarget ().requestFocus ();
            } else if ( keyCode == KeyEvent.VK_UP ) {
                currentSelectionIndex--;
                displaySearchResult ();
                // Stop processing the event here. Otherwise it's dispatched
                // to the tree too (which scrolls)
                e.consume ();
            } else if ( keyCode == KeyEvent.VK_DOWN ) {
                currentSelectionIndex++;
                displaySearchResult ();
                // Stop processing the event here. Otherwise it's dispatched
                // to the tree too (which scrolls)
                e.consume ();
            } else if ( keyCode == KeyEvent.VK_TAB ) {
/*                if (maxPrefix != null)
                    searchTextField.setText(maxPrefix);
 */
                //Hmm, can't really do anything here, since the search match may
                //*not* be a match for the displayed text.
                e.consume ();
            } else if ( keyCode == KeyEvent.VK_ENTER ) {
                detach ();
                performAction ();
            }
        }

        private void performAction () {
            JComponent jc = getTarget ();
            if ( jc instanceof JTree ) {
                performActionTree ();
            } else {
                performActionList ();
            }
        }

        private void performActionTree () {
            JTree tree = (JTree) getTarget ();
            NavigatorTreeModel mdl = (NavigatorTreeModel) tree.getModel ();
            if (!(tree.getModel() instanceof NavigatorTreeModel)) {
                //This will happen in sliding mode - removeNotify has
                //already replaced the model
                return;
            }
            TreePath selectedTPath = tree.getSelectionPath ();
            if ( selectedTPath != null ) {
                Object o = (TreeNode) selectedTPath.getLastPathComponent ();
                Action a = mdl.getDefaultAction ( o );
                if ( a.isEnabled () ) {
                    a.actionPerformed ( new ActionEvent ( tree,
                            ActionEvent.ACTION_PERFORMED, null ) );
                }
            }
        }

        private void performActionList () {
            JList list = (JList) getTarget ();
            if (!(list.getModel() instanceof NavigatorListModel)) {
                //This will happen in sliding mode - removeNotify has
                //already replaced the model
                return;
            }
            NavigatorListModel mdl = (NavigatorListModel) list.getModel ();
            int idx = list.getSelectedIndex ();
            if ( idx != -1 ) {
                Object o = mdl.getElementAt ( idx );
                Action a = mdl.getDefaultAction ( o );
                if ( a.isEnabled () ) {
                    a.actionPerformed ( new ActionEvent ( list,
                            ActionEvent.ACTION_PERFORMED, null ) );
                }
            }
        }

        private void searchForNode () {
            currentSelectionIndex = 0;
            results = null;
            String text = searchTextField.getText ();
            if ( text.length () > 0 ) {
                results = doSearch ( text );
                displaySearchResult ();
            }
        }


        private void displaySearchResult () {
            if (results == null) {
                return;
            }
            JComponent jc = getTarget ();
            if ( jc instanceof JList ) {
                displaySearchResultList ();
            } else {
                displaySearchResultTree ();
            }
        }

        private void displaySearchResultList () {
            JList list = (JList) getTarget ();
            NavigatorListModel mdl = (NavigatorListModel) list.getModel ();
            int sz = results.size ();
            if ( sz > 0 ) {
                if ( currentSelectionIndex < 0 ) {
                    currentSelectionIndex = sz - 1;
                } else if ( currentSelectionIndex >= sz ) {
                    currentSelectionIndex = 0;
                }
                Object node = results.get ( currentSelectionIndex );
                int index = mdl.indexOf(node);
                list.setSelectedIndex(index);
                list.ensureIndexIsVisible(index);
            }
        }

        private void displaySearchResultTree () {
            JTree tree = (JTree) getTarget ();
            if ( tree == null ) {
                System.err.println ( "SOMETHING IS VERY WRONG" );
                return;
            }
            int sz = results.size ();
            if ( sz > 0 ) {
                if ( currentSelectionIndex < 0 ) {
                    currentSelectionIndex = sz - 1;
                } else if ( currentSelectionIndex >= sz ) {
                    currentSelectionIndex = 0;
                }
                TreePath path = (TreePath) results.get ( currentSelectionIndex );
                tree.setSelectionPath ( path );
                tree.scrollPathToVisible ( path );
            } else {
                tree.clearSelection ();
            }
        }

        public void focusGained (FocusEvent e) {
            // Do nothing
        }

        public void focusLost (FocusEvent e) {
            detach ();
        }
    }

    private List doSearch (String prefix) {
        JComponent jc = getTarget ();
        if ( jc == null ) {
            System.err.println ( "SOMETHING IS WRONG!!!" );
            return Collections.EMPTY_LIST;
        }
        List result;
        if ( jc instanceof JList ) {
            result = ( (NavigatorListModel) ( (JList) jc ).getModel () ).getSearchResults ( prefix );
        } else {
            result = ( (NavigatorTreeModel) ( (JTree) jc ).getModel () ).getSearchResults ( prefix );
        }
        return result;
    }
    /*
    private String findMaxPrefix(String str1, String str2) {
        String res = null;
        for (int i = 0; str1.regionMatches(true, 0, str2, 0, i); i++) {
            res = str1.substring(0, i);
        }
        return res;
    }
     */


}
