/*
* The contents of this file are subject to the terms of the Common Development
* and Distribution License (the License). You may not use this file except in
* compliance with the License.
*
* You can obtain a copy of the License at http://www.netbeans.org/cddl.html
* or http://www.netbeans.org/cddl.txt.
* 
* When distributing Covered Code, include this CDDL Header Notice in each file
* and include the License file at http://www.netbeans.org/cddl.txt.
* If applicable, add the following below the CDDL Header, with the fields
* enclosed by brackets [] replaced by your own identifying information:
* "Portions Copyrighted [year] [name of copyright owner]"
*
* The Original Software is NetBeans. The Initial Developer of the Original
* Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
* Microsystems, Inc. All Rights Reserved.
*/
/*
 * NavigatorJList.java
 *
 * Created on September 24, 2004, 10:27 PM
 */

package org.netbeans.modules.java.navigation.base;
import java.awt.event.MouseEvent;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.EventObject;
import org.netbeans.modules.java.navigation.spi.AbstractModel;
import org.netbeans.modules.java.navigation.spi.NavigatorListModel;
import org.openide.util.NbBundle;

/**
 * A JList with some custom handling for tooltip positioning.
 *
 * @author Tim Boudreau
 */
public final class NavigatorJList extends JList implements MouseMotionListener, MouseListener {
    /**
     * We need the JLabel in a cell renderer pane so it has a graphics context to allow it to figure out its preferred
     * size.
     */
    static CellRendererPane cellpane = new CellRendererPane ();
    /**
     * A JLabel we will configure with the tooltip before displaying it, so we know how wide it needs to be, to
     * determine the offset for the tooltip display position
     */
    static JLabel widthTestLabel = new JLabel ();
    
    {
        cellpane.add ( widthTestLabel );
    }
    
    /** holds reference to component which visually encloses this list */
    private JComponent enclosingComp; 
    /** Stores current mouse pointer location */
    private Point curMouseLoc = null;
    /** true when mouse pointer inside component, false otherwise */
    private boolean isMouseInside = false;
    /** helper, true when list should ignore next key event, false otherwise */
    private boolean ignoreNextKey = false;
    
    private static final boolean DND = Boolean.getBoolean (
            "nb.navigator.reordering"); //NOI18N
    
    public NavigatorJList (JComponent enclosingComp) {
        this.enclosingComp = enclosingComp;
        getAccessibleContext().setAccessibleName(
                NbBundle.getMessage(NavigatorJList.class, "ACC_NavListName") //NOI18N
                );
        getAccessibleContext().setAccessibleDescription(
                NbBundle.getMessage(NavigatorJList.class, "ACC_NavListDesc") //NOI18N
                );
        
        setBorder (BorderFactory.createEmptyBorder());
        
        if (DND) {
            ReorderDragAndDropHandler handler = new ReorderDragAndDropHandler();
            addMouseListener(handler);
            addMouseMotionListener(handler);
        }        
    }
    
    private static final boolean NO_SCROLL = Boolean.getBoolean ("nb.navigator.noscroll"); //NOI18N
    public void scrollRectToVisible(Rectangle aRect) {
        if (!NO_SCROLL) {
            super.scrollRectToVisible(aRect);
        } else {
            //Probably this should really be the default - having navigator dance
            //as the caret moves is very distracting;  the value of showing the
            //selection is much less than the negative value of the dancing
            //navigator.  Below we still want this method to do something for
            //drag and drop, but only then.
            EventObject eo = EventQueue.getCurrentEvent();
            if (eo instanceof MouseEvent && ((MouseEvent) eo).getID() == MouseEvent.MOUSE_DRAGGED) {
                super.scrollRectToVisible(aRect);
            }
        }
    }
    

    public void addNotify () {
        super.addNotify ();
        ToolTipManager.sharedInstance ().registerComponent ( this );
        addMouseMotionListener(this);
        addMouseListener(this);
    }

    public void removeNotify () {
        super.removeNotify ();
        ToolTipManager.sharedInstance ().unregisterComponent ( this );
        removeMouseMotionListener(this);
        removeMouseListener(this);
        curMouseLoc = null;
    }
    
    /** Always select some element when focused */
    public boolean requestFocusInWindow () {
        boolean result = super.requestFocusInWindow();
        assureSelection();
        return result;
    }
    
    /** Selects first element in list if possible */
    public void assureSelection () {
        if (getSelectedIndex() == -1 && getModel().getSize() > 0) {
            setSelectedIndex(0);
        }
    }
    
    public String getToolTipText (MouseEvent me) {
        if ( getModel () instanceof NavigatorListModel ) {
            NavigatorListModel nlm = (NavigatorListModel) getModel ();
            int idx = indexAtLocation(this, me.getPoint());
            if ( idx >= 0 && idx <= nlm.getSize () ) {
                Object rep = nlm.getElementAt ( idx );
                
                if ( AbstractModel.isWaitMarker( rep ) || AbstractModel.isInvalidMarker ( rep ) ) {
                    return null;
                }
                return nlm.getTooltip ( rep, me.getX(), me.getY() );
            }
        }
        return "";
    }

    public Point getToolTipLocation (MouseEvent e) {
        Point result = e.getPoint ();
        Container c = getTopLevelAncestor ();
        Rectangle r = enclosingComp.getBounds ();
        Rectangle converted = SwingUtilities.convertRectangle ( this, r, c );
        boolean closerToLeft = converted.x + converted.width < c.getWidth () - converted.x;

        result = SwingUtilities.convertPoint ( this, result, enclosingComp );

        int approxWidth = approximateWidth ( getToolTipText ( e ) );
        int approxHeight = widthTestLabel.getPreferredSize ().height;
        if ( closerToLeft ) {
            result.x = r.x + r.width + 5;
        } else {
            result.x = r.x - ( approxWidth + 5 );
        }

        result = SwingUtilities.convertPoint ( enclosingComp, result, this );

        Point orig = e.getPoint ();
        int idx = locationToIndex ( orig );
        Rectangle r2 = getCellBounds ( idx, idx );
        if (r2 != null) {
            //#51396 - NPE if empty
            result.y = r2.y;
        }

        return result;
    }
    
    /** Hack to get tooltips working on Ctrl+F1. Always returns empty string, but
     * invokes tooltip by a hack on selected item.
     */
    public String getToolTipText() {
        final int index = getSelectedIndex();
        if (index >= 0) {
            final Rectangle rect = getCellBounds(index, index);
            if (rect != null) {
                SwingUtilities.invokeLater(new Runnable () {
                    public void run () {
                        TooltipHack.invokeTip(NavigatorJList.this,
                                (int)rect.getCenterX(), (int)rect.getCenterY(),
                                Integer.MAX_VALUE);
                    }
                });
            }
        }
            
        return "";
    }
    
    /** Due to asynchronous tooltip invocation, conditions may change and 
     * tooltip may not be valid sometimes. This method checks validity of
     * given tooltip location.
     *
     * @return true if tooltip position still valid, false otherwise
     */
    public boolean isTooltipLocValid (int x, int y) {
        if (curMouseLoc == null || !isMouseInside) {
            return false;
        }
        return indexAtLocation(this, curMouseLoc) == indexAtLocation(this, new Point(x, y));
    }
    

    private int approximateWidth (String s) {
        widthTestLabel.setText ( s );
        return widthTestLabel.getPreferredSize ().width;
    }
    
    /** @return Index of cell which contains given location point or -1 if no such
     * cell exists.
     * Note that similar method JList.locationToIndex return closest cell,
     * not neccessarily the one exactly containing the point.
     */
    public static int indexAtLocation (JList list, Point location) {
        int result = list.locationToIndex (location);
        Rectangle cellB = list.getCellBounds(result, result);
        // clear not exact match
        if ((result != -1) && !cellB.contains(location)) {
            result = -1;
        }
        return result;
    }
    
    /********* implementation of MouseMotionListener interface **********/
    
    public void mouseMoved(MouseEvent e) {
        curMouseLoc = e.getPoint();
    }
    
    public void mouseDragged(MouseEvent e) {
        // no operation
    }
    
    /********* implementation of MouseListener interface **********/
    
    public void mouseEntered(MouseEvent e) {
        isMouseInside = true;
    }

    public void mouseExited(MouseEvent e) {
        isMouseInside = false;
        // enforce tooltip hiding
        TooltipHack.invokeTip(this, 0, 0, 0);
    }
    
    public void mouseClicked(MouseEvent e) {
        // no operarion
    }

    public void mousePressed(MouseEvent e) {
        // no operarion
    }

    public void mouseReleased(MouseEvent e) {
        // no operarion
    }

    
    /*** a hack to stop Enter and Esc key events coming from popup menu
     * and triggering unwanted further actions */ 
   
    /** Tells this list to ignore next key events until key released come */
    public void ignoreNextKeyUntilReleased () {
        ignoreNextKey = true;
    }
    
    /** Overriden to block key events on request */
    protected void processKeyEvent (KeyEvent e) {
        if (ignoreNextKey) {
            if (e.getID() == KeyEvent.KEY_RELEASED) {
                ignoreNextKey = false;
            }
        } else {
            super.processKeyEvent(e);
        }
    }
    
    
}
