/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.navigation;

import java.awt.AWTEvent;
import java.awt.EventQueue;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JPopupMenu;
import javax.swing.KeyStroke;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import org.netbeans.modules.java.navigation.base.NavigatorJList;
import org.netbeans.modules.java.navigation.spi.DisplayProvider;
import org.netbeans.modules.java.navigation.spi.RelatedItemProvider;


/**
 * Controller for ClassMemberPanelUI. Handles user actions of members panel UI.
 *
 * @author Dafe Simonek
 */
final class ClassMemberController extends MouseAdapter implements PopupMenuListener {
    
    /** UI of members navigator panel */ 
    private ClassMemberPanelUI panelUI;

    private int origDelay = -1;

    public ClassMemberController (ClassMemberPanelUI panelUI) {
        this.panelUI = panelUI;
        initActions();
        initListeners();
    }
    
    private void initActions () {
        JList list = panelUI.getContent();
        
        // action on ENTER key
        Action keyAction = new OpenByKeyboardAction();
        list.getInputMap(JComponent.WHEN_FOCUSED).put(
                KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0, true), "open"); //NOI18N
        list.getActionMap().put( "open", keyAction); //NOI18N

        // popup menu via keyboard Shift+F10
        Action showPopupAction = new ShowPopupAction();
        list.getInputMap(JComponent.WHEN_FOCUSED).put(
                KeyStroke.getKeyStroke(KeyEvent.VK_F10, KeyEvent.SHIFT_DOWN_MASK, true),
                "popup" ); //NOI18N
        list.getActionMap().put( "popup", showPopupAction); //NOI18N

        /*
        Action switchViewAction = new PopupComboAction ( views );
        KeyStroke switchViewKey = Utilities.stringToKey ( NbBundle.getMessage ( Navigator.class, "KEY_SwitchView" ) );
        getInputMap ( WHEN_ANCESTOR_OF_FOCUSED_COMPONENT ).put ( switchViewKey, "switchView" ); //NOI18N
        getActionMap ().put ( "switchView", switchViewAction ); //NOI18N

        Action togglePanelsAction = new ToggleTapPanelsAction ();
        KeyStroke toggleKey = KeyStroke.getKeyStroke(KeyEvent.VK_T,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask());
        getInputMap ( WHEN_ANCESTOR_OF_FOCUSED_COMPONENT ).put ( toggleKey, "togglePanels" ); //NOI18N
        getActionMap ().put ( "togglePanels", togglePanelsAction ); //NOI18N

        Action returnAction = new ReturnAction ();
        KeyStroke returnKey = KeyStroke.getKeyStroke ( KeyEvent.VK_ESCAPE, 0, true );
        tree.getInputMap ( WHEN_FOCUSED ).put ( returnKey, "return" ); //NOI18N
        tree.getActionMap ().put ( "return", returnAction ); //NOI18N
        list.getInputMap ( WHEN_FOCUSED ).put ( returnKey, "return" ); //NOI18N
        list.getActionMap ().put ( "return", returnAction ); //NOI18N

         */
    }
    
    private void initListeners () {
        panelUI.getContent().addMouseListener(this);
    }
    
    /** Shows popup menu on given list at given point */
    private void showPopup (JList list, Point p) {
        // obtain current element
        DisplayProvider dp = (DisplayProvider)list.getModel();
        Object elem = null;
        int i = NavigatorJList.indexAtLocation(list, p);
        if (i >= 0) {
            elem = list.getModel().getElementAt(i);
        }
        //Do not use full width - no way to scroll back w/o horiz scrollbar
        list.scrollRectToVisible(new Rectangle( p.x, p.y + 16, 12, 16));

        JPopupMenu m = InheritanceTreeController.createPopupMenuForItem(list, dp, elem);
        if (m != null) {
            m.addPopupMenuListener(this);
            m.show(list, p.x, p.y);
        }

    }

    /** Runs default model action on double click (or single click eventually) */ 
    public void mouseClicked (MouseEvent e) {
        // react only on left mouse button click
        if (!SwingUtilities.isLeftMouseButton(e)) {
            return;
        }
        // obtain clicked element
        JList list = panelUI.getContent();
        Point p = e.getPoint ();
        int idx = NavigatorJList.indexAtLocation(list, p);
        Object clicked = null;
        if (idx != -1) {
            clicked = list.getModel().getElementAt(idx);
        } else {
            // no element clicked
            return;
        }
        // perform default action from model (DisplayProvider)
        DisplayProvider dp = (DisplayProvider)list.getModel();
        if (dp.isDefaultActionInstant() || e.getClickCount() > 1) {
            Action act = dp.getDefaultAction(clicked);
            if (act != null && act.isEnabled()) {
                ActionEvent ae = new ActionEvent(ClassMemberController.this,
                                ActionEvent.ACTION_PERFORMED, "click"); //NO18N
                //Immediately clear the list of related items if the user
                //has selected a new one
                if (ClassMemberRelatedItemProvider.USE_RELATED_ITEMS && 
                        list.getModel() instanceof RelatedItemProvider) {
                    
                    ((RelatedItemProvider) list.getModel()).reset();
                    list.repaint();
                }
                act.actionPerformed(ae);
            }
        }
    }
    
    /** Overriden to show popup if triggered */
    public void mousePressed (MouseEvent e) {
        if (e.isPopupTrigger()) {
            showPopup((JList)e.getSource(), e.getPoint());
            e.consume();
        }
    }

    /** Overriden to show popup if triggered */
    public void mouseReleased (MouseEvent e) {
        if (e.isPopupTrigger()) {
            showPopup((JList)e.getSource(), e.getPoint());
        }
    }
    
    public void mouseEntered(MouseEvent e) {
        ToolTipManager ttm = ToolTipManager.sharedInstance();
        origDelay = ttm.getDismissDelay();
        ttm.setDismissDelay(Integer.MAX_VALUE);
    }

    public void mouseExited(MouseEvent e) {
        if (origDelay > 0) {
            ToolTipManager.sharedInstance().setDismissDelay(origDelay);
            origDelay = -1;
        }
    }
    
    /***** implementation of PopupMenuListener */
    
    /** If popup menu is finishing with Enter or Esc key, tell the list underneath
     * to ignore the key event, to not double action.
     */ 
    public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
        ((JPopupMenu)e.getSource()).removePopupMenuListener(this);
        AWTEvent curEvt = EventQueue.getCurrentEvent();
        if (curEvt instanceof KeyEvent) {
            int key = ((KeyEvent)curEvt).getKeyCode();
            if (key == KeyEvent.VK_ENTER || key == KeyEvent.VK_ESCAPE) {
                panelUI.getContent().ignoreNextKeyUntilReleased();
            }
        }
    }

    public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
        // no operation
    }

    public void popupMenuCanceled(PopupMenuEvent e) {
        // no operation
    }

    
    /* Action that runs default action of list model on currently selected item
     */
    private class OpenByKeyboardAction extends AbstractAction {
        
        public boolean isEnabled () {
            return findDefaultAction() != null;
        }

        /** Returns default action for current selection or null */
        private Action findDefaultAction () {
            JList list = panelUI.getContent();
            ListModel model = list.getModel();
            Object sel = null;
            
            int index = list.getSelectedIndex();
            if (index != -1) {
                sel = model.getElementAt(index);
            }
            
            if (sel != null) {
                Action a = ((DisplayProvider)model).getDefaultAction(sel);
                if (a != null && a.isEnabled()) {
                    return a;
                }
            }
            
            return null;
        }

        /** Runs found default action if possible */
        public void actionPerformed (ActionEvent evt) {
            Action a = findDefaultAction();
            if (a != null) {
                InheritanceTreeController.runWithWaitCursor(panelUI.getContent(), a, evt);
            }
        }
        
    } // end of OpenByKeyboardAction

    
    /** Shows popup for currently selected item */
    private final class ShowPopupAction extends AbstractAction {
        
        public void actionPerformed (ActionEvent ae) {
            JList list = panelUI.getContent();
            int idx = list.getSelectedIndex ();
            Point p = list.indexToLocation(idx);
            p.x += 7;
            p.y += 14;
            showPopup(list, p);
        }
    }

    /*
    private class ToggleTapPanelsAction extends AbstractAction {
        public void actionPerformed (ActionEvent ae) {
            boolean val = filtersPanel.isExpanded ();
            synchronized (getTreeLock ()) {
                filtersPanel.setExpanded ( !val );
            }
            revalidate ();
            repaint ();
        }
    }*/

    
}
