/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.jmi.javamodel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.jmi.reflect.RefAssociation;
import javax.jmi.reflect.RefAssociationLink;

import junit.textui.TestRunner;

import org.netbeans.junit.NbTestCase;
import org.netbeans.junit.NbTestSuite;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ConstructorImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.IsOfTypeImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.JavaClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.SemiPersistentElement;
import org.netbeans.jmi.javamodel.codegen.Utility;
 

/** 
 * @author Vladimir Hudec
 * 
 * [TODO]: anonymous classes should be also considered
 */
public class IsOfTypeTest extends NbTestCase {
   
    /** Need to be defined because of JUnit */
    public IsOfTypeTest(String name) {
        super(name);
        
    } 
    
    public static NbTestSuite suite() {
        NbTestSuite suite = new NbTestSuite();
        suite.addTest(new IsOfTypeTest("testAllLinks"));
        suite.addTest(new IsOfTypeTest("testNullArguments"));
        suite.addTest(new IsOfTypeTest("testImproperArguments"));
        suite.addTest(new IsOfTypeTest("testAddRemoveAssociation"));
        suite.addTest(new IsOfTypeTest("testAddRemoveTypedElement"));
        suite.addTest(new IsOfTypeTest("testReferenceSet"));
        return suite;
    }
    
    /** Use for execution inside IDE */
    public static void main(java.lang.String[] args) {
        TestRunner.run(suite());
    }

    JavaModelPackage pkg;
    IsOfTypeImpl isOfTypeImpl;
    JavaClass class1, class2, interface2;
    Field field1, field2, field3;
    CallableFeature cfeature1, cfeature2;
    
    protected void setUp() {
        class1 = Utility.findClass("org.netbeans.test.classes.Class1");
        pkg = (JavaModelPackage) class1.refImmediatePackage();
        isOfTypeImpl = (IsOfTypeImpl) pkg.getIsOfType();
        try { Thread.sleep(2000); } catch (Exception ex) {}
        
        assertNotNull("Class1", class1);
        assertFalse("Class1 is instance of UnresolvedClass", class1 instanceof UnresolvedClass);
        class2 = Utility.findClass("org.netbeans.test.classes.Class2");
        assertNotNull("Class2", class2);
        assertFalse("Class2 is instance of UnresolvedClass", class2 instanceof UnresolvedClass);
        interface2 = Utility.findClass("org.netbeans.test.interfaces.Interface2");
        assertNotNull("Interface2", interface2);
        assertFalse("Interface2 is instance of UnresolvedClass", interface2 instanceof UnresolvedClass);
        
        field1 = class1.getField("field1_1", false);
        assertNotNull("class1.field1", field1);
        field2 = class1.getField("field1_2", false);
        assertNotNull("class1.field2", field1);
        field3 = class1.getField("field1_3", false);
        assertNotNull("class1.field1_3", field3);

        List list = new ArrayList();
        Type intType = pkg.getType().resolve("int"); list.add(intType);
        Type stringType = pkg.getType().resolve("String"); list.add(stringType);
        cfeature1 = class2.getMethod("method2_1", list, false);
        assertNotNull("class2.method2_1", cfeature1);
        list = new ArrayList(); list.add(intType);
        Type interface2Type = pkg.getType().resolve("org.netbeans.test.interfaces.Interface2"); list.add(interface2Type);
        cfeature2 = class2.getMethod("method2_2", list, false);
        assertNotNull("class2.method2_2", cfeature2);

        // Class1 implements Interface1
        // Class2 extends Class1 implements Interface2
        // Class3.Class4 implements Interface1
        // Interface3 extends Interface1, Interface2 ->
        //   Interface3 implements Interface1
        //   Interface3 implements Interface1
        // Exception1 extends Exception
        // Exception2 extends RuntimeException

        checkNumber("field1 type", true, field1.getType());
        checkNumber("field2 type", true, field2.getType());
        checkNumber("field3 type", true, field3.getType());
        checkNumber("method2_1 type", true, cfeature1.getType());
        checkNumber("method2_2 type", true, cfeature2.getType());
    }
    
    private static String getDesc(Object o) {
        String name = null;
                                                                                                                                           
        if (o instanceof Constructor) {
            JavaClass cls = (JavaClass) (((ConstructorImpl)o).getDeclaringClass());
            if (cls != null)
                name = cls.getName();
                                                                                                                                           
        } else if (o instanceof NamedElement) {
            name = ((NamedElement)o).getName();
        }
                                                                                                                                           
        if (name != null)
            return name + " / " + o.getClass();
        else
            return o.toString();
    }
                                                                                                                                           
    protected void printAllLinks(RefAssociation refAssociation) {
        Collection types = refAssociation.refAllLinks();
        int numOfLinks = 0;
        for (Iterator it = types.iterator(); it.hasNext(); ) {
            numOfLinks++;
            RefAssociationLink link = (RefAssociationLink) it.next();
            Type type = (Type) link.refFirstEnd();
            TypedElement typedElement = (TypedElement) link.refSecondEnd();
            System.out.println("  #"+numOfLinks+": "+getDesc(typedElement)+" is of type "+type.getName());
        }
    }
    
    protected void checkNumber(String msg, boolean expectedValue, Type actualValue) {
        assertEquals("Total number of "+msg, (expectedValue)?1:0, (actualValue!=null && !actualValue.getName().equals("java.lang.Object"))?1:0);
    }
    
    protected void checkNumber(String msg, int expectedValue, int actualValue) {
        assertEquals("Total number of "+msg, expectedValue, actualValue);
    }
    
    protected void checkNumber(String msg, int expectedValue, Collection coll) {
        if (expectedValue == 0) {
            assertTrue("Total number of "+msg, coll.isEmpty());
        }
        else {
            checkNumber(msg, expectedValue, coll.size());
        }
    }
    
    public void testAllLinks() {
        Utility.beginTrans(false);
        try {
            Collection types = isOfTypeImpl.refAllLinks();
            int numOfLinks = 0;
            int numOfInterfaceTypes = 0;
            
            String packageOfInterfaces = interface2.getResource().getPackageName();
            
            for (Iterator it = types.iterator(); it.hasNext(); ) {
                numOfLinks++;
                RefAssociationLink link = (RefAssociationLink) it.next();
                Type type = (Type) link.refFirstEnd();
                TypedElement typedElement = (TypedElement) link.refSecondEnd();
                if (typedElement.getType().getName().startsWith(packageOfInterfaces))
                    ++numOfInterfaceTypes;
                System.out.println("  #"+numOfLinks+": "+getDesc(typedElement)+" is of type "+type.getName());
            }
            
            System.out.println("  numOfInterfaceTypes "+numOfInterfaceTypes);
            checkNumber("elements with Class1 type", 9, numOfInterfaceTypes);
        }
        finally  {
            Utility.endTrans();
        }
    }
    
    public void testNullArguments() {
        String msg = "Association's operation with null argument should throw exception";
        boolean fail = true;
        Utility.beginTrans(true);
        try {
            try {
                isOfTypeImpl.exists(class1, null);
                fail(msg+" (exists)");
            }
            catch (NullPointerException ex) {
            }
            try {
                isOfTypeImpl.exists(null, field1);
                fail(msg+" (exists)");
            }
            catch (NullPointerException ex) {
            }
            try {
                isOfTypeImpl.add(class1, null);
                fail(msg+" (add)");
            }
            catch (NullPointerException ex) {
            }
            try {
                isOfTypeImpl.add(null, field1);
                fail(msg+" (add)");
            }
            catch (NullPointerException ex) {
            }
            try {
                isOfTypeImpl.remove(class1, null);
                fail(msg+" (remove)");
            }
            catch (NullPointerException ex) {
            }
            try {
                isOfTypeImpl.remove(null, field1);
                fail(msg+" (remove)");
            }
            catch (NullPointerException ex) {
            }
            try {
                isOfTypeImpl.getTypedElements(class1).add(null);
                fail(msg+" (getTypedElements.add)");
            }
            catch (NullPointerException ex) {
            }
            try {
                isOfTypeImpl.getTypedElements(class2).iterator().remove();
                fail(msg+" (getTypedElements.iterator.remove)");
            }
            catch (IllegalStateException ex) {
            }
            catch (NullPointerException ex) {
                // [TODO] - fix in ReferenceCol/ListWrapper
            }
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
    
    public void testImproperArguments() {
        String msg = "Association's operation with improper argument should throw exception";

        Object obj = new Object();
        boolean fail = true;
        Utility.beginTrans(true);
        try {
            try {
                isOfTypeImpl.getTypedElements(class2).add(obj);
                fail(msg+" (getTypedElements.add)");
            }
            catch (javax.jmi.reflect.TypeMismatchException ex) {
            }
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            try {
                isOfTypeImpl.add(class1, field3);
                fail(msg+" (Adding duplicate association)");
            }
            catch (javax.jmi.reflect.WrongSizeException ex) {
            }
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            boolean result = isOfTypeImpl.add(interface2, field3);
            assertFalse("Adding existing association", result);
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            boolean result = isOfTypeImpl.remove(class1, field3);
            assertFalse("Removing nonexisting association", result);
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
    
    public void testAddRemoveAssociation() {
        Type oldType;

        boolean fail = true;
        Utility.beginTrans(true);
        try {
            oldType = field1.getType();
            isOfTypeImpl.remove(oldType, field1);
            isOfTypeImpl.add(class1, field1);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 1, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            field1.setType(oldType);
            isOfTypeImpl.remove(class1, field1);
            isOfTypeImpl.add(oldType, field1);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 0, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
    
    public void testAddRemoveTypedElement() {
        Type oldType;

        boolean fail = true;
        Utility.beginTrans(true);
        try {
            oldType = field1.getType();
            isOfTypeImpl.remove(oldType, field1);
            Collection typedElements = isOfTypeImpl.getTypedElements(class1); 
            typedElements.add(field1);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 1, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            Collection typedElements = isOfTypeImpl.getTypedElements(class1); 
            typedElements.remove(field1);
            isOfTypeImpl.add(oldType, field1);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 0, isOfTypeImpl.getTypedElements(class1));

            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            isOfTypeImpl.remove(oldType, field1);
            Collection typedElements = isOfTypeImpl.getTypedElements(class1); 
            List list = new ArrayList();
            list.add(field1);
            typedElements.addAll(list);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 1, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            Collection typedElements = isOfTypeImpl.getTypedElements(class1); 
            List list = new ArrayList();
            list.add(field1);
            typedElements.removeAll(list);
            isOfTypeImpl.add(oldType, field1);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 0, isOfTypeImpl.getTypedElements(class1));

            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            isOfTypeImpl.remove(oldType, field1);
            Collection typedElements = isOfTypeImpl.getTypedElements(class1); 
            List list = new ArrayList();
            list.add(field1);
            typedElements.addAll(list);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 1, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            Collection typedElements = isOfTypeImpl.getTypedElements(class1); 
            Iterator it = typedElements.iterator();
            while (it.hasNext()) {
                TypedElement typedElement = (TypedElement) it.next();
                if (typedElement.equals(field1)) {
                    it.remove();
                    break;
                }
            }
            isOfTypeImpl.add(oldType, field1);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 0, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }

    public void testReferenceSet() {
        Type oldType;

        boolean fail = true;
        Utility.beginTrans(true);
        try {
            oldType = field1.getType();
            field1.setType(class1);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 1, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            field1.setType(oldType);
            
            checkNumber("field1 type", true, field1.getType());
            checkNumber("elements with Class1 type", 0, isOfTypeImpl.getTypedElements(class1));
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
}
