/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.jmi.javamodel;

import java.util.*;
import javax.jmi.reflect.RefAssociation;
import javax.jmi.reflect.RefAssociationLink;
import junit.textui.TestRunner;
import org.netbeans.junit.NbTestCase;
import org.netbeans.junit.NbTestSuite;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ExtendsImpl;
import org.netbeans.jmi.javamodel.codegen.Utility;
 

/** 
 * @author Vladimir Hudec
 * 
 * [TODO]: anonymous classes should be also considered
 */
public class ExtendsTest extends NbTestCase {
   
    /** Need to be defined because of JUnit */
    public ExtendsTest(String name) {
        super(name);
        
    } 
    
    public static NbTestSuite suite() {
        NbTestSuite suite = new NbTestSuite();
        suite.addTest(new ExtendsTest("testAllLinks"));
        suite.addTest(new ExtendsTest("testNullArguments"));
        suite.addTest(new ExtendsTest("testImproperArguments"));
        suite.addTest(new ExtendsTest("testAddRemoveAssociation"));
        suite.addTest(new ExtendsTest("testAddRemoveSubClass"));
        suite.addTest(new ExtendsTest("testReferenceSet"));
        return suite;
    }
    
    /** Use for execution inside IDE */
    public static void main(java.lang.String[] args) {
        TestRunner.run(suite());
    }

    JavaModelPackage pkg;
    ExtendsImpl extendsImpl;
    JavaClass class1, class2, class3, exception1, exception2;
    
    protected void setUp() {
        class1 = Utility.findClass("org.netbeans.test.classes.Class1");
        pkg = (JavaModelPackage) class1.refImmediatePackage();
        extendsImpl = (ExtendsImpl) pkg.getExtends();
        try { Thread.sleep(2000); } catch (Exception ex) {}
        
        class1 = Utility.findClass("org.netbeans.test.classes.Class1");
        assertNotNull("Class1", class1);
        assertFalse("Class1 is instance of UnresolvedClass", class1 instanceof UnresolvedClass);
        class2 = Utility.findClass("org.netbeans.test.classes.Class2");
        assertNotNull("Class2", class2);
        assertFalse("Class2 is instance of UnresolvedClass", class2 instanceof UnresolvedClass);
        class3 = Utility.findClass("org.netbeans.test.classes.Class3");
        assertNotNull("Class3", class3);
        assertFalse("Class3 is instance of UnresolvedClass", class3 instanceof UnresolvedClass);
        
        exception1 = Utility.findClass("org.netbeans.test.exceptions.Exception1");
        assertNotNull("Exception1", exception1);
        assertFalse("Exception1 is instance of UnresolvedClass", exception1 instanceof UnresolvedClass);
        exception2 = Utility.findClass("org.netbeans.test.exceptions.Exception2");
        assertNotNull("Exception2", exception2);
        assertFalse("Exception2 is instance of UnresolvedClass", exception2 instanceof UnresolvedClass);

        // Class1 implements Interface1
        // Class2 extends Class1 implements Interface2
        // Class3.Class4 implements Interface1
        // Interface3 extends Interface1, Interface2 ->
        //   Interface3 implements Interface1
        //   Interface3 implements Interface1
        // Exception1 extends Exception
        // Exception2 extends RuntimeException

        checkNumber("Class2 superclass", true, class2.getSuperClass());
        checkNumber("Exception1 superclass", true, exception1.getSuperClass());
        checkNumber("Exception2 superclass", true, exception2.getSuperClass());
        checkNumber("Class1 subclasses", 1, extendsImpl.getSubClasses(class1));
    }
    
    protected void printAllLinks(RefAssociation refAssociation) {
        Collection impls = refAssociation.refAllLinks();
        int numOfLinks = 0;
        for (Iterator it = impls.iterator(); it.hasNext(); ) {
            numOfLinks++;
            RefAssociationLink link = (RefAssociationLink) it.next();
            JavaClass subClass = (JavaClass) link.refFirstEnd();
            JavaClass superClass = (JavaClass) link.refSecondEnd();
            System.out.println("  #"+numOfLinks+": "+subClass.getName()+" extends "+superClass.getName());
        }
    }
    
    protected void checkNumber(String msg, boolean expectedValue, JavaClass actualValue) {
        assertEquals("Total number of "+msg, (expectedValue)?1:0, (actualValue!=null && !actualValue.getName().equals("java.lang.Object"))?1:0);
    }
    
    protected void checkNumber(String msg, int expectedValue, int actualValue) {
        assertEquals("Total number of "+msg, expectedValue, actualValue);
    }
    
    protected void checkNumber(String msg, int expectedValue, Collection coll) {
        if (expectedValue == 0) {
            assertTrue("Total number of "+msg, coll.isEmpty());
        }
        else {
            checkNumber(msg, expectedValue, coll.size());
        }
    }
    
    public void testAllLinks() {
        Utility.beginTrans(false);
        try {
            Collection impls = extendsImpl.refAllLinks();
            int numOfLinks = 0;
            int numOfSubClasses = 0;
            int numOfSuperClasses = 0;
            
            String packageOfClasses = class1.getResource().getPackageName();
            String packageOfExceptions = exception1.getResource().getPackageName();
            
            for (Iterator it = impls.iterator(); it.hasNext(); ) {
                numOfLinks++;
                RefAssociationLink link = (RefAssociationLink) it.next();
                JavaClass subClass = (JavaClass) link.refFirstEnd();
                JavaClass superClass = (JavaClass) link.refSecondEnd();
                if (subClass.getName().startsWith(packageOfClasses) || subClass.getName().startsWith(packageOfExceptions))
                    if (!subClass.getName().equals("java.lang.Object") && !superClass.getName().equals("java.lang.Object"))
                        ++numOfSubClasses;
                if (superClass.getName().startsWith(packageOfClasses) || superClass.getName().startsWith(packageOfExceptions))
                    if (!superClass.getName().equals("java.lang.Object"))
                        ++numOfSuperClasses;
                System.out.println("  #"+numOfLinks+": "+subClass.getName()+" extends "+superClass.getName());
            }
            
            checkNumber("sub classes", 3, numOfSubClasses);
            checkNumber("super classes", 1, numOfSuperClasses);
        }
        finally  {
            Utility.endTrans();
        }
    }
    
    public void testNullArguments() {
        String msg = "Association's operation with null argument should throw exception";
        boolean fail = true;
        Utility.beginTrans(true);
        try {
            try {
                extendsImpl.exists(null, class1);
                fail(msg+" (exists)");
            }
            catch (NullPointerException ex) {
            }
            try {
                extendsImpl.exists(class2, null);
                fail(msg+" (exists)");
            }
            catch (NullPointerException ex) {
            }
            try {
                extendsImpl.add(null, class1);
                fail(msg+" (add)");
            }
            catch (NullPointerException ex) {
            }
            try {
                extendsImpl.add(class2, null);
                fail(msg+" (add)");
            }
            catch (NullPointerException ex) {
            }
            try {
                extendsImpl.remove(null, class1);
                fail(msg+" (remove)");
            }
            catch (NullPointerException ex) {
            }
            try {
                extendsImpl.remove(class2, null);
                fail(msg+" (remove)");
            }
            catch (NullPointerException ex) {
            }
            try {
                extendsImpl.getSubClasses(class1).add(null);
                fail(msg+" (getSubClasses.add)");
            }
            catch (NullPointerException ex) {
            }
            try {
                extendsImpl.getSubClasses(class3).iterator().remove();
                fail(msg+" (getSubClasses.iterator.remove)");
            }
            catch (IllegalStateException ex) {
            }
            catch (NullPointerException ex) {
                // [TODO] - fix in ReferenceCol/ListWrapper
            }
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
    
    public void testImproperArguments() {
        String msg = "Association's operation with improper argument should throw exception";

        Object obj = new Object();
        boolean fail = true;
        Utility.beginTrans(true);
        try {
            try {
                extendsImpl.getSubClasses(class2).add(obj);
                fail(msg+" (getSubClasses.add)");
            }
            catch (javax.jmi.reflect.TypeMismatchException ex) {
            }
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            try {
                extendsImpl.add(class2, class3);
                fail(msg+" (Adding duplicate association)");
            }
            catch (javax.jmi.reflect.WrongSizeException ex) {
            }
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            boolean result = extendsImpl.add(class2, class1);
            assertFalse("Adding existing association", result);
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            boolean result = extendsImpl.remove(class2, class3);
            assertFalse("Trying remove nonexisting association", result);
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
    
    public void testAddRemoveAssociation() {
        JavaClass oldSuperClass;

        boolean fail = true;
        Utility.beginTrans(true);
        try {
            oldSuperClass = class3.getSuperClass();
            extendsImpl.remove(class3, oldSuperClass);
            extendsImpl.add(class3, class1);
            
            checkNumber("Class3 super classes", true, class3.getSuperClass());
            checkNumber("Class1 sub classes", 2, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            extendsImpl.remove(class3, class1);
            extendsImpl.add(class3, oldSuperClass);
            
            checkNumber("Class3 super classes", false, class3.getSuperClass());
            checkNumber("Class1 sub classes", 1, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
    
    public void testAddRemoveSubClass() {
        JavaClass oldSuperClass;

        boolean fail = true;
        Utility.beginTrans(true);
        try {
            oldSuperClass = class3.getSuperClass();
            extendsImpl.remove(class3, oldSuperClass);
            Collection subClasses = extendsImpl.getSubClasses(class1); 
            subClasses.add(class3);
            
            checkNumber("Class3 super classes", true, class3.getSuperClass());
            checkNumber("Class1 sub classes", 2, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            Collection subClasses = extendsImpl.getSubClasses(class1); 
            subClasses.remove(class3);
            extendsImpl.add(class3, oldSuperClass);
            
            checkNumber("Class3 super classes", false, class3.getSuperClass());
            checkNumber("Class1 sub classes", 1, class1.getSubClasses());

            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            extendsImpl.remove(class3, oldSuperClass);
            Collection subClasses = extendsImpl.getSubClasses(class1); 
            List list = new ArrayList();
            list.add(class3);
            subClasses.addAll(list);
            
            checkNumber("Class3 super classes", true, class3.getSuperClass());
            checkNumber("Class1 sub classes", 2, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            Collection subClasses = extendsImpl.getSubClasses(class1); 
            List list = new ArrayList();
            list.add(class3);
            subClasses.removeAll(list);
            extendsImpl.add(class3, oldSuperClass);
            
            checkNumber("Class3 super classes", false, class3.getSuperClass());
            checkNumber("Class1 sub classes", 1, class1.getSubClasses());

            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }

        fail = true;
        Utility.beginTrans(true);
        try {
            extendsImpl.remove(class3, oldSuperClass);
            Collection subClasses = extendsImpl.getSubClasses(class1); 
            List list = new ArrayList();
            list.add(class3);
            subClasses.addAll(list);
            
            checkNumber("Class3 super classes", true, class3.getSuperClass());
            checkNumber("Class1 sub classes", 2, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            Collection subClasses = extendsImpl.getSubClasses(class1); 
            Iterator it = subClasses.iterator();
            while (it.hasNext()) {
                JavaClass subClass = (JavaClass) it.next();
                if (subClass.equals(class3)) {
                    it.remove();
                    break;
                }
            }
            extendsImpl.add(class3, oldSuperClass);
            
            checkNumber("Class3 super classes", false, class3.getSuperClass());
            checkNumber("Class1 sub classes", 1, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }

    public void testReferenceSet() {
        JavaClass oldSuperClass;

        boolean fail = true;
        Utility.beginTrans(true);
        try {
            oldSuperClass = class3.getSuperClass();
            class3.setSuperClass(class1);
            
            checkNumber("Class3 super classes", true, class3.getSuperClass());
            checkNumber("Class1 sub classes", 2, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
        
        fail = true;
        Utility.beginTrans(true);
        try {
            class3.setSuperClass(oldSuperClass);
            
            checkNumber("Class3 super classes", false, class3.getSuperClass());
            checkNumber("Class1 sub classes", 1, class1.getSubClasses());
            
            fail = false;
        }
        finally  {
            Utility.endTrans(fail);
        }
    }
}
