/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.scanning;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.net.URL;
import java.net.URI;

import org.netbeans.jmi.javamodel.Codebase;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.ExclusiveMutex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.ProgressDisplayer;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.Utilities;

/**
 *
 * @author  Pavel Flaska
 */
public class FileScanner {
    
    private ClassUpdater classUpdater;
    private JavaUpdater javaUpdater;
    private File root;
    private String offset;
    private Codebase codebase;
    private JavaModelPackage mofPackage;
    private Collection resources;
    private Collection resourcesToScan;
    private boolean zipFile;
    private final ExclusiveMutex mutex;
    private static HashSet ignoredDirectories;
    private static HashSet ignoredPackages;
    private static HashSet eagerlyParse;
    private final boolean isRescan;
    
    private static final boolean DEBUG = false;
    
    /**
     * Creates new FileScanner
     * @param _root the URL of root which should be scanned, must be either 
     * file protocol of jar protocol with file path
     * @param sourceLevel 
     * @param cb codebase
     * @throws IllegalArgumentException when _root is not file protocol or jar protocol
     *  with file path URL
     *
     */
    public FileScanner(URL _root, String sourceLevel, Codebase cb) throws IllegalArgumentException {
        this(_root, sourceLevel, cb, false);
    }
    
    /**
     * Creates new FileScanner
     * @param _root the URL of root which should be scanned, must be either 
     * file protocol of jar protocol with file path
     * @param sourceLevel 
     * @param cb codebase
     * @throws IllegalArgumentException when _root is not file protocol or jar protocol
     *  with file path URL
     *
     */
    public FileScanner(URL _root, String sourceLevel, Codebase cb, boolean isRescan) throws IllegalArgumentException {
        synchronized (FileScanner.class) {
            if (ignoredPackages == null) {
                ignoredPackages = parseSet("org.netbeans.javacore.ignorePackages", "sun sunw"); // NOI18N
                ignoredDirectories = parseSet("org.netbeans.javacore.ignoreDirectories", "SCCS CVS"); // NOI18N
                eagerlyParse = parseSet("org.netbeans.javacore.eagerlyParse", "javax/swing/JFrame.java"); // NOI18N
            }
        }

        this.isRescan = isRescan;
        
        if ("jar".equals(_root.getProtocol())) {    //NOI18N
            String strUrl = _root.toExternalForm();
            int index = strUrl.lastIndexOf("!/")+2; //NOI18N
            assert index > 0 && index<=strUrl.length() :"Invalid jar protocol URL";    //NOI18N
            offset = index == strUrl.length() ? null : strUrl.substring (index);
            _root = FileUtil.getArchiveFile (_root);
            zipFile=true;
        }        
        if (!"file".equals (_root.getProtocol())) { //NOI18N
            throw new IllegalArgumentException ("The URL: "+_root.toExternalForm()+" has no file protocol.");   //NOI18N
        }
        URI uri = URI.create (_root.toExternalForm());
        root = new File (uri);
        codebase = cb;
        mofPackage=(JavaModelPackage)codebase.refImmediatePackage();
        mutex = JMManager.getTransactionMutex();
        if (!isRescan) {
            javaUpdater=new JavaUpdater(mofPackage,sourceLevel, this);
        }
        classUpdater=new ClassUpdater(mofPackage);
    }
    
    private static HashSet parseSet(String propertyName, String defaultValue) {
        StringTokenizer st = new StringTokenizer(System.getProperty(propertyName, defaultValue), " \t\n\r\f,-:+!");
        HashSet result = new HashSet();
        while (st.hasMoreTokens()) {
            result.add(st.nextToken());
        }
        return result;
    }
    
    void checkParseEagerly(Resource resource) {
        if (eagerlyParse.contains(resource.getName())) {
            if (resourcesToScan == null) {
                resourcesToScan = new ArrayList(eagerlyParse.size());
            }
            resourcesToScan.add(resource);
        }
    }
    
    public Resource[] scan() {
        NBMDRepositoryImpl repository = (NBMDRepositoryImpl)JavaMetamodel.getDefaultRepository();
        FileInfo rootInfo;
        // turn off events only in case it is not a rescan
        if (!isRescan) {
            repository.disableEvents();
        }
        resourcesToScan = null;
        ZipArchiveInfo zipInfo = null;
        try {
            ClassIndex index = null;
            if (zipFile) {
                long oldTime=codebase.getTimestamp();
                index = ClassIndex.getIndex(mofPackage);

                if (oldTime!=0) {
                    long jarTime=root.lastModified();

                    if (oldTime == jarTime && index.getTimestamp() >= jarTime) {
                        return new Resource[0]; // Jar is up to date
                    }
                }
                try {
                    zipInfo = new ZipArchiveInfo(root, offset);
                    rootInfo = zipInfo.getRootFileInfo();
                }catch (IOException ex) {
                    ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ex);
                    return null;
                }
            } else {
                rootInfo=new FileEntry(root,"");
            }
            resources = new HashSet(mofPackage.getResource().refAllOfClass());
            
            // get all subfolders (recursively) of filesystem root
            scanPackage(rootInfo, "", new HashSet());
            if (zipFile) {
                if (index != null) {
                    index.setTimestamp();
                }
                codebase.setTimestamp(root.lastModified());
                codebase.setLibrary(true);
            }
            removeFromRepository();
            return resourcesToScan == null ? new Resource[0] : (Resource[]) resourcesToScan.toArray(new Resource[resourcesToScan.size()]);
        } finally {
            try {
                if (zipInfo != null) zipInfo.close();
            } catch (IOException e) {
                ErrorManager.getDefault().notify(e);
            }
        }
    }
    
    private void removeFromRepository() {
        Iterator i = resources.iterator();
        
        while (i.hasNext()) {
            ((Resource) i.next()).refDelete(); 
        }
    }
    
    private static ProgressDisplayer progress;
    
    private void scanPackage(FileInfo directory, String pack, Set visited) {
        if (ignoredPackages.contains(pack)) return;
        
        String canonicalName = directory.getCanonicalName();
        if (canonicalName != null && !visited.add(canonicalName)) {
            return;
        }
        
        FileInfo files[] = directory.listFiles();
        Map javaFiles=new HashMap();
        Map classFiles=new HashMap();
        
        for(int i=0;i<files.length;i++) {
            FileInfo fo=files[i];
            String name=fo.getName();
            
            if (fo.isDirectory()) {
		if (ignoredDirectories.contains(name) || !Utilities.isJavaIdentifier(name))
		    continue;
                if (mutex.isSwingWaiting()) {
                    NBMDRepositoryImpl rep = (NBMDRepositoryImpl) JMManager.getDefaultRepository();
                    try {
                        if (DEBUG) System.err.println("FileScanner: Releasing transaction lock to allow event thread to continue."); // NOI18N
                        rep.endTrans();
                        while (mutex.isSwingWaiting()) {
                            Thread.sleep(100);
                        }
                    } catch (InterruptedException e) {
                        // ignore
                    } finally {
                        rep.beginTrans(true);
                        if (DEBUG) System.err.println("FileScanner: Re-acquiring transaction lock to continue with scanning.");    //NOI18N
                        if (!isRescan) {
                            rep.disableEvents();
                        }
                    }
                }
                String newpack=pack;
                
                if (pack.length()!=0) {
                    newpack=pack.concat("."); // NOI18N
                }
                newpack=newpack.concat(name);
                scanPackage(fo,newpack,visited);
            } else if (name!=null) {
                if (name.endsWith(".java")) {    //NOI18N
                    javaFiles.put(name,fo);
                } else if (name.endsWith(".class"))    //NOI18N
                    classFiles.put(name,fo);       
            }
        }
        if (!(javaFiles.isEmpty() && classFiles.isEmpty())) {
            if (isRescan) {
                // rescan
                Iterator resIt = javaFiles.values().iterator();
                List resList = new ArrayList();
                long indexTimestamp;

                while (resIt.hasNext()) {
                    FileInfo file = (FileInfo) resIt.next();
                    try {
                        String name = file.getPath();
                        long timestamp = file.lastModified();
                        ResourceImpl resource = (ResourceImpl) ((ResourceClassImpl) mofPackage.getResource()).resolveResource(name, true, false);

                        if (resource.getTimestamp() != timestamp) {
                            FileObject fobj = JavaMetamodel.getManager().getFileObject(resource);
                            if (fobj != null && fobj.isValid()) {
                                resources.remove(resource);
                                resource.updateFromFileObject(fobj, true);
                            }
                        } else {
                            resources.remove(resource);
                        }
                    } catch (Exception ex) {
                        ErrorManager.getDefault().notify(ex);
                    }
                }
            } else {
                // normal scan
                progress = ProgressDisplayer.getVisibleProgressDisplayer();
                if (progress != null) {
                    progress.updatePackage(pack);
                }
                resources.removeAll(javaUpdater.updateResources(javaFiles));
            }
            resources.removeAll(classUpdater.updateResources(javaFiles, classFiles));
        }
    }
}
