/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.parser;

import java.lang.reflect.Modifier;
import java.util.*;
import javax.jmi.reflect.RefFeatured;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.*;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.jmiimpl.javamodel.*;
import org.openide.filesystems.FileObject;
import org.openide.util.RequestProcessor;
import org.openide.ErrorManager;

/**
 *
 * @author  Tomas Hurka
 */
public class MDRParser extends ASTProvider {
    
    private Map semanticInfo;
    private ClassPath classPath;
    private JMManager manager;
    private JavaPackageClass jpckClass;
    private JavaClassClassImpl jclsClass;
    private TypeClass typeClass;
    private JavaEnumClassImpl enumClass;
    private AnnotationTypeClassImpl annotClass;
    private ClassDefinition currentClass;
    private Feature currentFeature;
    private String jpck;
    private Stack typeScopeInfo;
    private Stack variableScope;
    private Scope enumSwitchScope;
    private ClassIndex index;
    private TreeMap attributionToDo;
    private List localSuperInfo;
    private MDRepository rep;
    private Scope staticImpScope;
    private int javaFeatures;
    private boolean running;
    private boolean resolveClassName;
    private static final boolean DEBUG=false;
    private static Map superInfoMap=new HashMap();
    private static final ModifiersInfo NO_MODIFIERS = new ModifiersInfo(0, null);
    private static final RequestProcessor PARSE_AFTER_SCAN_RP = new RequestProcessor("Parse-After-Scan Request Processor"); // NOI18N

    // enum modifier
    public static final int M_ENUM = 0x00004000;
    // annotation modifier
    public static final int M_ANNOTATION = 0x00002000;
    // feature mofid to position bounds map
    public Map mofidToBounds;
    // start and end offsets of guarded blocks
    public int[] guardedBlocksBorders;

    public MDRParser(Resource r,FileObject fobj) {
        super(r,fobj);
        init();
    }

    public MDRParser(Resource r, FileObject fobj, String sourceText, boolean isFromDoc) {
        super(r,fobj,sourceText, isFromDoc);
        init();
    }
    
    private void init() {
        JavaModelPackage srcExtent=(JavaModelPackage)getResource().refImmediatePackage();

        jpckClass=srcExtent.getJavaPackage();
        jclsClass=(JavaClassClassImpl)srcExtent.getJavaClass();
        typeClass=srcExtent.getType();
        enumClass=(JavaEnumClassImpl)srcExtent.getJavaEnum();
        annotClass=(AnnotationTypeClassImpl)srcExtent.getAnnotationType();
        index=ClassIndex.getIndex(srcExtent);
        semanticInfo=new HashMap();
        typeScopeInfo=new Stack();
        variableScope=new Stack();
        attributionToDo = getAttributionToDoMap();
        localSuperInfo=new ArrayList();
        manager=(JMManager)JavaMetamodel.getManager();
        rep=JavaMetamodel.getDefaultRepository();
    }
    
    private static void reparseAfterScan(final FileObject fobj) {
        PARSE_AFTER_SCAN_RP.post(new Runnable() {
            public void run() {
                JMManager manager=(JMManager)JavaMetamodel.getManager();
                if (manager.waitScanFinished()) {
                    manager.addModified(fobj);
                    JavaMetamodel.getDefaultRepository().beginTrans(true); 
                    JavaMetamodel.getDefaultRepository().endTrans(false);
                }
            }
        });
    }
    
    private static TreeMap getAttributionToDoMap() {
        return new TreeMap(new Comparator() {
            public int compare(Object o1, Object o2) {
                int i1 = ((ASTree) o1).getFirstToken(), i2 = ((ASTree) o2).getFirstToken();
                if (i1 < i2)
                    return -1;
                if (i1 > i2)
                    return 1;
                // i1 == i2
                i1 = ((ASTree) o1).getLastToken();
                i2 = ((ASTree) o2).getLastToken();
                if (i1 < i2)
                    return 1;
                if (i1 > i2)
                    return -1;
                return 0;
                
            }
        });        
    }

    public Object getSemanticInfo(ASTree tree, Element element) {
        boolean wasSafe;
        Object info = semanticInfo.get(tree);
        
        if (info == null) {
            if (!attributionToDo.isEmpty()) {
                for (Iterator it = attributionToDo.keySet().iterator(); it.hasNext();) {
                    ASTree tmp = (ASTree) it.next();
                    if (tmp.getFirstToken() > tree.getFirstToken()) {
                        break;
                    }
                    if (tmp.getLastToken() >= tree.getLastToken()) {
                        doAttribution();
                        info = semanticInfo.get(tree);
                        break;
                    }
                }
            }
            if (info == null) {
                return null;
            }
        }
        wasSafe=JMManager.getTransactionMutex().isSafeTrans();
        manager.setSafeTrans(true);
        try {
            return getModelElement(tree,element,info);
        } finally {
            manager.setSafeTrans(wasSafe);
        }
    }
    
    private void doAttribution() {
        TreeMap map = attributionToDo;
        attributionToDo = getAttributionToDoMap();
        for (Iterator it = map.entrySet().iterator(); it.hasNext();) {
            Map.Entry entry = (Map.Entry) it.next();
            Element el = (Element) entry.getValue();
            
            if (el.isValid()) {
                enterBody(el, (ASTree) entry.getKey());
            }
        }
    }
    
    public void prepareForAttribution(MetadataElement feature, ASTree featureTree) {
        attributionToDo.put(featureTree, feature);
    }

    private synchronized void enterBody(Element element,ASTree elementTree) {
        boolean wasSafe;
        
        if (running) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL,new Exception("Recursion in enterBody"));   // NOI18N
            return;
        }
        running=true;
        wasSafe=JMManager.getTransactionMutex().isSafeTrans();
        manager.setSafeTrans(true);
        if (JMManager.PERF_DEBUG) try {
            Feature f=JavaModelUtil.getDeclaringFeature(element);
            System.err.println("Attributing method: " + f.getDeclaringClass().getName() + '.' + f.getName() + '.' + elementTree.getFirstToken()); // NOI18N
        } catch (NullPointerException e) {
            // ignore
        }
        long time = System.currentTimeMillis();
        try {
            Feature feature=JavaModelUtil.getDeclaringFeature(element);
            classPath=manager.getClassPath();
            currentClass=feature.getDeclaringClass();
            computeScope(currentClass);
            semanticInfo.put(elementTree, element);
            currentFeature=feature;
            processASTBody(elementTree);
            typeScopeInfo.pop();
            variableScope.pop();
        } finally {
            superInfoMap.keySet().removeAll(localSuperInfo);
            localSuperInfo.clear();
            typeScopeInfo.clear();
            variableScope.clear();
            currentFeature=null;
            currentClass=null;
            manager.setSafeTrans(wasSafe);
            running=false;
        }
        if (JMManager.PERF_DEBUG) System.out.println("    finished: " + (System.currentTimeMillis() - time) + "ms"); // NOI18N
    }
    
    public synchronized ResourceInfo enterMembers() {
        boolean wasSafe;

        if (running) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL,new Exception("Recursion in enterMembers"));   // NOI18N
            return null;
        }
        running=true;
        wasSafe=JMManager.getTransactionMutex().isSafeTrans();
        manager.setSafeTrans(true);
        try {
            classPath=manager.getClassPath();
            ASTree topNode=getASTree();
 
            if (topNode!=null) {
                if (DEBUG) System.out.println("Parsing :"+fobj.getPath()); // NOI18N
                if (getPackage("java.lang")==null) {        // no JDK found // NOI18N
                    reparseAfterScan(getFileObject());
                }
                return (ResourceInfo) processAST(topNode);
            }
        } finally {
            superInfoMap.keySet().removeAll(localSuperInfo);
            localSuperInfo.clear();
            manager.setSafeTrans(wasSafe);
            running=false;
        }
        return null;
    }
    
    public int getJavaFeatures() {
        return javaFeatures;
    }
    
    private void computeScope(ClassDefinition jcls) {
        Scope classScope;
        Scope varScope;
        
        if (jcls==null) {
            Resource rsc=getResource();
            
            classScope=Scope.createTypeScope(rsc,classPath);
            varScope=new Scope(null);
            staticImpScope=Scope.createStaticImpScope(rsc);
        } else if (jcls instanceof JavaClassImpl && ((JavaClassImpl)jcls).isTransient() && ((JavaClassImpl)jcls).getDeclaringClass()==null) {  // local class
           ASTree[] jclsSubASTs=((MetadataElement)jcls).getASTree().getSubTrees(); // Local class AST parts
           Token firstToken=getToken(jclsSubASTs[jclsSubASTs.length-1].getFirstToken()); // first token '{' of local class 
           Scope localscope[]=(Scope[])semanticInfo.get(firstToken);
           
           classScope=new Scope(localscope[0]);
           varScope=new Scope(localscope[1]);
           classScope.addMember(Scope.createMemberTypeScope(jcls,this));
           varScope.addMember(Scope.createFieldScope(jcls));
        } else if (jcls instanceof JavaClass) {
            JavaClass javaClass=((JavaClass)jcls);
            ClassDefinition decl=javaClass.getDeclaringClass();
            Iterator typeParIt;
            
            computeScope(decl);
            classScope=new Scope((Scope)typeScopeInfo.pop());
            varScope=new Scope((Scope)variableScope.pop());

            classScope.addMember(Scope.createMemberTypeScope(jcls,this));
            varScope.addMember(Scope.createFieldScope(jcls));
            typeParIt=javaClass.getTypeParameters().iterator();
            while(typeParIt.hasNext()) {
                TypeParameter tp=(TypeParameter)typeParIt.next();
                
                classScope.addMember(new TypeParamScope(tp));
            }
        } else {
           ASTree parentAST=((MetadataElement)jcls).getASTree(); // CLASS_BODY_DECLARATIONS
           Token firstToken=getToken(parentAST.getFirstToken()); // first token '{' of anonymous class 
           Scope anonscope[]=(Scope[])semanticInfo.get(firstToken);
           
           classScope=new Scope(anonscope[0]);
           varScope=new Scope(anonscope[1]);
           classScope.addMember(Scope.createMemberTypeScope(jcls,this));
           varScope.addMember(Scope.createFieldScope(jcls));
        }
        typeScopeInfo.push(classScope);
        variableScope.push(varScope);
    }
    
    private JavaPackage getPackage(String packId) {
        return jpckClass.resolvePackage(packId);
    }

    Object processAST(ASTree tree) {
        return processAST(tree,null);
    }
    
    Object processAST(ASTree tree,String fqn) {
        ASTree parts[];
        int i,treeType;
        
        if (tree==null)
            return null;
        parts=tree.getSubTrees();
        treeType=tree.getType();
        switch (treeType) {
            case COMPILATION_UNIT: {
                Object topClasses;
                ElementInfo[] importList;
                
                typeScopeInfo.push(new Scope(null));
                processAST(parts[0]);
                jpck=getResource().getPackageName();
                importList=(ElementInfo[]) processAST(parts[1]);  // imports;
                typeScopeInfo.pop();
                typeScopeInfo.push(Scope.createTypeScope(jpck,classPath,importList));
                topClasses=processAST(parts[2],jpck); // declarations
                typeScopeInfo.pop();
                return new ResourceInfo(tree, treeType, getResource(), (ClassInfo[]) topClasses, importList);
            }
            case PACKAGE_DECLARATION:
                return resolveTypeName(parts[1]);
            case SINGLE_TYPE_IMPORT:
            case TYPE_IMPORT_ON_DEMAND: {
                String id;
                
                if (treeType==SINGLE_TYPE_IMPORT && parts[0]!=null) { // static single type import
                    id=resolveStaticImport(parts[1]);
                } else {
                    id=((NameRef)resolveTypeName(parts[1])).name;
                }
                return new ElementInfo(tree, treeType, id);
            }
            case IMPORT_DECLARATIONS: {
                ElementInfo imports[] = null;
                if (parts != null) {
                    imports = new ElementInfo[parts.length];
                    for (i = 0; i < imports.length; i++) {
                        imports[i] = (ElementInfo) processAST(parts[i]);
                    }
                }
                return imports;
            }
            case TYPE_DECLARATIONS: {
                ClassInfo classes[]=null;
                
                if (parts!=null) {
                    ASTree filtered[]=filterParts(parts);
                    
                    for (i=0;i<filtered.length;i++) {
                        resolveSuperTypes(filtered[i],fqn);
                    }
                    classes=new ClassInfo[filtered.length];
                    for (i=0;i<classes.length;i++) {
                        classes[i]=(ClassInfo)processAST(filtered[i],fqn);
                    }
                }
                return classes;
            }
            case ENUM_CONSTANT: {
                return new FeatureInfo(tree, treeType, getText(parts[0]), 0, null);
            }
            case ENUM_CONSTANTS: {
                ElementInfo constants[] = null;
                if (parts != null) {
                    ASTree filtered[] = filterParts(parts);

                    constants = new ElementInfo[filtered.length];
                    for (i=0;i<filtered.length;i++) {
                        constants[i] = (ElementInfo) processAST(filtered[i], fqn);
                    }
                }
                return constants;
            }
            case ANNOTATION_TYPE_BODY_DECLARATIONS:
            case ENUM_BODY_DECLARATIONS:
            case INTERFACE_MEMBER_DECLARATIONS:
            case CLASS_BODY_DECLARATIONS: {
                FeatureInfo features[] = null;
                if (parts != null) {
                    ASTree filtered[]=filterParts(parts);
                    
                    for (i=0;i<filtered.length;i++) 
                        resolveSuperTypes(filtered[i],fqn);
                    ArrayList tempFeatures = new ArrayList(filtered.length);
                    for (i=0;i<filtered.length;i++) {
                        Object temp = processAST(filtered[i], fqn);
                        if (temp instanceof FieldInfo[]) {
                            tempFeatures.addAll(Arrays.asList((Object[]) temp));
                        } else {
                            tempFeatures.add(temp);
                        }
                    }
                    features = (FeatureInfo[]) tempFeatures.toArray(new FeatureInfo[tempFeatures.size()]);
                }
                return features;
            }
            case ENUM_DECLARATION: {
                ModifiersInfo modInfo=getModifiers(parts[0]);   // modifiers and annotations
                int modifiers= modInfo != NO_MODIFIERS ? modInfo.modifiers : getDeprecated(tree);
                ASTree body=parts[3];   // class body
                ASTree ifacesAST=parts[2];
                EnumInfo enumInfo;
                FeatureInfo features[];
                ElementInfo constants[];
                String currentFqn;
                NameRef interfaces[]=null;
                Scope classScope=new Scope((Scope)typeScopeInfo.peek());

                JavaEnum jcls = (JavaEnum) semanticInfo.get(tree);

                currentFqn=jcls.getName();
                classScope.addMember(Scope.createMemberTypeScope(jcls,this));

                typeScopeInfo.push(classScope);
                modifiers |= M_ENUM;
                javaFeatures|=ResourceImpl.HAS_ENUMS;
                if (ifacesAST != null)
                    interfaces = (NameRef[]) semanticInfo.get(ifacesAST);
                ASTree enumConsts = body.getSubTrees()[0];
                ASTree bodyDecls = body.getSubTrees()[1];
                constants = enumConsts == null ? null : (ElementInfo[]) processAST(enumConsts, currentFqn);
                features = bodyDecls == null ? null : (FeatureInfo[]) processAST(bodyDecls, currentFqn);
                enumInfo = new EnumInfo(tree, treeType, currentFqn, modifiers, features, interfaces, constants, modInfo.annotations);
                typeScopeInfo.pop();
                return enumInfo;
            }
            case ANNOTATION_TYPE_DECLARATION: {
                ModifiersInfo modInfo=getModifiers(parts[0]);   // modifiers and annotations
                int modifiers= modInfo != NO_MODIFIERS ? modInfo.modifiers : getDeprecated(tree);
                ASTree body=parts[2];   // class body
                AnnotationTypeInfo annTypeInfo;
                FeatureInfo features[];
                String currentFqn;
                String interfaces[]=null;
                Scope classScope=new Scope((Scope)typeScopeInfo.peek());
                AnnotationType annType = (AnnotationType) semanticInfo.get(tree);
                
                javaFeatures|=ResourceImpl.HAS_ANNOTATIONTYPES;
                currentFqn = annType.getName();
                classScope.addMember(Scope.createMemberTypeScope(annType, this));
                typeScopeInfo.push(classScope);
                features = (FeatureInfo[]) processAST(body, currentFqn);
                annTypeInfo = new AnnotationTypeInfo(tree, treeType, currentFqn, modifiers, features, modInfo.annotations);
                typeScopeInfo.pop();
                return annTypeInfo;
            }
            case INTERFACE_DECLARATION:
            case CLASS_DECLARATION: {
                ModifiersInfo modInfo=getModifiers(parts[0]);   // modifiers and annotations
                int modifiers= modInfo != NO_MODIFIERS ? modInfo.modifiers : getDeprecated(tree);
                ASTree body=parts[5];   // class body
                ASTree ifacesAST=parts[4];
                TypeParamInfo[] typeParams=(TypeParamInfo[])processAST(parts[2]);
                ClassInfo clsInfo;
                FeatureInfo features[];
                String currentFqn;
                NameRef sclass=(NameRef)semanticInfo.get(parts[1]);
                NameRef interfaces[]=null;
                Scope classScope=new Scope((Scope)typeScopeInfo.peek());
                
                JavaClass jcls=(JavaClass)semanticInfo.get(tree);

                currentFqn=jcls.getName();
                classScope.addMember(Scope.createMemberTypeScope(jcls,this));

                typeScopeInfo.push(classScope);
                if (treeType==INTERFACE_DECLARATION)
                    modifiers |= Modifier.INTERFACE;
                if (ifacesAST!=null) 
                    interfaces=(NameRef[])semanticInfo.get(ifacesAST);
                typeParams = (TypeParamInfo[]) processAST(parts[2],currentFqn);
                features=(FeatureInfo[])processAST(body,currentFqn);
                clsInfo=new ClassInfo(tree, treeType, currentFqn, modifiers, features, sclass, interfaces, typeParams, modInfo.annotations);
                typeScopeInfo.pop();
                return clsInfo;
            }
            case CONSTRUCTOR_DECLARATION:
            case METHOD_DECLARATION: {
                ModifiersInfo modInfo=getModifiers(parts[0]);   // modifiers and annotations
                int modifiers= modInfo != NO_MODIFIERS ? modInfo.modifiers : getDeprecated(tree);
                TypeRef type;
                Object header[];
                TypeParamRef exceptions[]=null;

//                semanticInfo.put(tree, bhFeature);
                TypeParamInfo[] typeParams = (TypeParamInfo[]) processAST(parts[1]);
                type=resolveTypeName(parts[2]);  // type
                header=(Object[])processAST(parts[3]);  // header 0 - ID, 1 - params, 2 - dims
                ASTree exsAST=parts[4]; // exceptions
                String name=null;
                ParameterInfo pars[]=(ParameterInfo[])header[1];
                
                if (treeType==METHOD_DECLARATION) {
                    name=(String)header[0];
                }
                if (exsAST!=null) {
                    ASTree[] exIds=exsAST.getSubTrees();
                    
                    exceptions=new TypeParamRef[exIds.length];
                    for (i=0;i<exIds.length;i++) {
                        exceptions[i]=(TypeParamRef)resolveTypeName(exIds[i]);
                    }
                }
                return new MethodInfo(tree, treeType, name, modifiers, type == null ? null : fullType(type, header[2]), pars, exceptions, typeParams, modInfo.annotations);
            }
            case CONSTRUCTOR_DECLARATOR:
            case VARIABLE_DECLARATOR: {
                return new Object[] {processAST(parts[0]),processAST(parts[1])};
            }
            case METHOD_DECLARATOR: {
                return new Object[] {processAST(parts[0]), processAST(parts[1]), processAST(parts[2])};
            } case FORMAL_PARAMETER_LIST: {
                ParameterInfo pars[]=new ParameterInfo[parts.length];
                
                for (i=0;i<pars.length;i++) 
                    pars[i]=(ParameterInfo)processAST(parts[i],fqn);
                return pars;
            }
            case FORMAL_PARAMETER: {
                ModifiersInfo modInfo=getModifiers(parts[0]);   // final and annotations
                boolean isFinal=false;
                TypeRef type=resolveTypeName(parts[1]);
                Object decl[]=(Object[])processAST(parts[3]); // VARIABLE_DECLARATOR
                TypeRef varType=fullType(type,decl[1]); // dims
                String name=(String)decl[0];
                
                if (modInfo!=null)
                    isFinal=Modifier.isFinal(modInfo.modifiers);
                return new ParameterInfo(tree, treeType, name, isFinal, varType, parts[2] != null,  modInfo.annotations);
            }
            case TYPE_PARAMETER_LIST: {
                TypeParamInfo typeParams[] = new TypeParamInfo[parts.length];
                
                javaFeatures|=ResourceImpl.HAS_GENERICS;
                createTypeParametrScope(tree);
                for (i = 0; i < typeParams.length; i++) {
                    typeParams[i] = (TypeParamInfo) processAST(parts[i]);
                }
                return typeParams;
            }
            case TYPE_PARAMETER: {
                String name = (String) processAST(parts[0]);
                TypeParamRef[] bounds = (TypeParamRef[]) processAST(parts[1]);
                
                return new TypeParamInfo(tree, tree.getType(), name, bounds);
            }
            case BOUND_LIST: {
                TypeParamRef[] result = new TypeParamRef[parts.length];
                for (i = 0; i < parts.length; i++) {
                    result[i] = (TypeParamRef)resolveTypeName(parts[i]);
                }
                return result;
            }
            case TYPE_ARGUMENTS: {
                Object[] result=new Object[parts.length];
                
                javaFeatures|=ResourceImpl.HAS_GENERICS;
                for (i=0;i<parts.length;i++) {
                    ASTree argAST=parts[i];
                    
                    if (argAST.getType()==IDENTIFIER)
                        result[i]=resolveTypeName(argAST);
                    else
                        result[i]=processAST(argAST);
                }
                return result;
            }
            case FIELD_DECLARATION: {
                ModifiersInfo modInfo=getModifiers(parts[0]);   // modifiers and annotations
                int modifiers= modInfo != NO_MODIFIERS ? modInfo.modifiers : getDeprecated(tree);
                TypeRef type=resolveTypeName(parts[1]);  // type
                ASTree varDeclsTree=parts[2];
                Object decls[]=(Object[])processAST(varDeclsTree); // VARIABLE_DECLARATORS

                if (varDeclsTree.getType()==VARIABLE_DECLARATOR) {
                    TypeRef varType=fullType(type,decls[1]);
                    String name=(String)decls[0];
                    return new FieldInfo(tree, treeType, name, modifiers, varType, FieldInfo.SINGLE_FIELD_INDEX, modInfo.annotations);
                } else { // VARIABLE_DECLARATORS
                    FieldInfo[] fields = new FieldInfo[decls.length];
                    ASTree[] children = varDeclsTree.getSubTrees();
                    for (i=0;i<decls.length;i++) {
                        Object decl[]=(Object[])decls[i];
                        fields[i] = new FieldInfo(children[i], FIELD_DECLARATION, (String) decl[0], modifiers, fullType(type,decl[1]), i, modInfo.annotations);
                    }
                    return new FieldGroupInfo(tree, VARIABLE_DECLARATORS, modifiers, type, fields, modInfo.annotations);
                }
            }
            case ANNOTATION_ATTRIBUTE_DECLARATION: {
                ModifiersInfo modInfo=getModifiers(parts[0]);   // modifiers and annotations
                int modifiers= modInfo != NO_MODIFIERS ? modInfo.modifiers : getDeprecated(tree);
                TypeRef type=resolveTypeName(parts[1]);  // type
                String name=(String)processAST(parts[2]); // name
                AnnotationValueInfo defaultValue=(AnnotationValueInfo)processAST(parts[3]);
                return new AttributeInfo(tree, treeType, name, modifiers, type, defaultValue, modInfo.annotations);
            }
            case INSTANCE_INITIALIZER:
            case STATIC_INITIALIZER: {
                ASTree modTree=parts[0]; // modifiers
                ModifiersInfo modInfo=getModifiers(parts[0]);   // modifiers and annotations
                int modifiers=modInfo.modifiers;
                
//                semanticInfo.put(tree,initializer);
                return new FeatureInfo(tree, treeType, null, modifiers, modInfo.annotations);
            }
            case ANNOTATION: {
                NameRef name=(NameRef)resolveTypeName(parts[0]);
                ASTree pAST=parts[1];
                ASTree[] pairsAST=pAST!=null?pAST.getSubTrees():new ASTree[0];
                AnnotationValueInfo[] attributePairs=new AnnotationValueInfo[pairsAST.length];
                
                javaFeatures|=ResourceImpl.HAS_ANNOTATION;
                for (i=0;i<attributePairs.length;i++) {
                    ASTree[] valueASTs=pairsAST[i].getSubTrees();
                    String pairName=(String)processAST(valueASTs[0]);
                    ASTree valueAST=valueASTs[1];
                    int valueType=valueAST.getType();
                    AnnotationValueInfo attributePair;
                    
                    if (valueType==ASTreeTypes.ANNOTATION) {
                        Object ann=processAST(valueAST);
                        attributePair=new AnnotationValueInfo(pairsAST[i],AnnotationValueInfo.ANNOTATIONVALUE_ANNOTATION,pairName,ann);
                    } else if (valueType==ASTreeTypes.ARRAY_INITIALIZER) {
                        Object[] ann=getArray(processAST(valueAST));
                        attributePair=new AnnotationValueInfo(pairsAST[i],AnnotationValueInfo.ANNOTATIONVALUE_ARRAY,pairName,ann);
                    } else {
                        String ann=getText(valueAST);
                        attributePair=new AnnotationValueInfo(pairsAST[i],AnnotationValueInfo.ANNOTATIONVALUE_STRING,pairName,ann);
                    }
                    attributePairs[i]=attributePair;
                }
                return new AnnotationInfo(tree,treeType,name,attributePairs);
            }
            case REFERENCE_TYPE: {
                ASTree dim=parts[1];
                ASTree typeAST=parts[0];
                TypeRef t;
                
                if (typeAST.getType()!=PRIMITIVE_TYPE)
                    t=resolveTypeName(typeAST);
                else
                    t=(TypeRef)processAST(typeAST);
                if (dim!=null && t!=null) {
                    int dimension=((Integer)processAST(dim)).intValue();
                    
                    t=new ArrayRef((PrimitiveTypeRef)t,dimension);
                }
                return t;
            }
            case PRIMITIVE_TYPE: {
                return getPrimitiveType(parts[0]);
            }
            
            case DIMS: {
                return new Integer((tree.getLastToken()-tree.getFirstToken()+1)/2);
            }
            case WILDCARD: {
                boolean isLower;
                ASTree lowerAST=parts[0];
                TypeRef bound;
                
                if (lowerAST==null) 
                    return new WildCardRef(false,null);
                isLower=lowerAST.getType()==EXTENDS;
                bound=resolveTypeName(parts[1]);
                return new WildCardRef(isLower,bound);
            }
            case MULTI_PART_ID: {
                ASTree parent=parts[0];
                ASTree nameAST=parts[1];
                String name=(String)processAST(nameAST);
                TypeParamRef parentRef;
                Object sInfo=null;
                Object[] args;
                TypeParamRef ref=null;
                
                if (parent!=null) {
                    Object parentInfo;
                    
                    if (parent.getType()==IDENTIFIER)
                        parentRef=(TypeParamRef)resolveTypeName(parent);
                    else
                        parentRef=(TypeParamRef)processAST(parent);
                    parentInfo=semanticInfo.get(parent);
                    if (parentInfo instanceof String && ((String)parentInfo).charAt(0)=='*') {    // package
                        NameRef pckRef=(NameRef)parentRef;
                        String pckName=pckRef.name.concat(".").concat(name); // NOI18N
                        
                        sInfo=getPackage(pckName);
                        if (sInfo!=null)
                            ref=new NameRef(pckName);
                    }
                    args=getArray(processAST(parts[2]));
                    if (sInfo==null) {
                        TypeRef[] argRef=new TypeRef[args.length];

                        System.arraycopy(args, 0, argRef, 0, args.length);
                        sInfo=ref=new NameRef(name,parentRef,argRef);
                    }
                    storeSemanticInfo(nameAST, sInfo);
                } else {
                    args=getArray(processAST(parts[2]));
                    TypeParamRef nameRef=(TypeParamRef)resolveTypeName(nameAST);
                    TypeRef[] argRef=new TypeRef[args.length];

                    System.arraycopy(args, 0, argRef, 0, args.length);
                    sInfo=ref=new NameRef(nameRef.name,null,argRef);
                }
                storeSemanticInfo(tree, sInfo);
                return ref;
            }
            case IDENTIFIER:
                return ((Token)tree).getValue();
                
            case MODIFIERS: {
                int modifiers=0,annCount=0;
                AnnotationInfo[] infos = new AnnotationInfo[parts.length];
                for (i=0;i<parts.length;i++) {
                    ASTree modTree=parts[i];
                    if (modTree.getType()==ASTreeTypes.ANNOTATION) {
                        AnnotationInfo ai = (AnnotationInfo) processAST(modTree);
                        infos[annCount++] = ai;
                        if (ai.type.name.equals("Deprecated") || ai.type.name.equals("java.lang.Deprecated")) // NOI18N
                            modifiers|=FeatureImpl.DEPRECATED;
                    }
                    else {
                        Token tok = (Token)modTree;
                        modifiers|=((Integer)(tok).getValue()).intValue();
                        if (tok.getDeprecatedFlag())
                            modifiers|=FeatureImpl.DEPRECATED;
                    }
                }
                AnnotationInfo[] annArray=new AnnotationInfo[annCount];
                System.arraycopy(infos,0,annArray,0,annCount);
                return new ModifiersInfo(modifiers,annArray);
            } case DEFAULT_VALUE: {
                return new AnnotationValueInfo(tree, AnnotationValueInfo.ANNOTATIONVALUE_STRING, null, null);
            }
                
            default: {
                if (parts!=null) {
                    Object val=null;
                    Object valArr[]=null;
                    int arrIndex=0;
                    
                    for (i=0;i<parts.length;i++) {
                        Object ret=processAST(parts[i]);
                        
                        if (ret!=null) {
                            if (val==null)
                                val=ret;
                            else if (arrIndex==0) {
                                valArr=new Object[parts.length-i+1];
                                valArr[0]=val;
                                valArr[1]=ret;
                                arrIndex=2;
                            } else
                                valArr[arrIndex++]=ret;
                        }
                    }
                    return arrIndex==0?val:valArr;
                }
            }
        }
        return null;
    }

    private String resolveStaticImport(ASTree mpid) {
        String id=null;

        if (mpid!=null) {
            ASTree[] mpiparts=mpid.getSubTrees();

            if (mpiparts!=null) {
                NameRef parentRef=(NameRef)resolveTypeName(mpiparts[0]);
                ASTree nameAST=mpiparts[1];
                String name=(String)processAST(nameAST);
                JavaClass jcls=(JavaClass)((SemiPersistentElement)getResource()).resolveType(parentRef);
                Feature feature=null;

                if (jcls!=null) {
                    Object[] features=jcls.getFeatures().toArray();

                    for(int i=0;i<features.length;i++) {
                        Feature f=(Feature)features[i];

                        if (name.equals(f.getName())) {
                            feature=f;
                            break;
                        }
                    }
                }
                storeSemanticInfo(mpid, feature);
                storeSemanticInfo(nameAST, feature);
                id=parentRef.name.concat(".").concat(name); // NOI18N
            }
        }
        return id;
    }
    
    private Object getModelElement(ASTree tree,Element mdrElement,Object symbolInfo) {
        if (symbolInfo instanceof String) {
            String strInfo=(String)symbolInfo;
            char firstChar=strInfo.charAt(0);
            String name;
            
            switch(firstChar) {
                case '*':
                case '@':
                case '<':
                case '^':
                    name=strInfo.substring(1);
                    break;
                default:
                    return typeClass.resolve(strInfo);
            }
            switch (firstChar) {
                case '*':
                    return getPackage(name);
                case '@':
                    return resolveClass(name);
                case '<':
                    return resolveTypeParameter(tree,name);
                case '^':
                    return rep.getByMofId(name);
            }
            return null;
        }
        if (symbolInfo.getClass().equals(TypeParamRef.class)) {
            TypeParamRef tpRef=(TypeParamRef)symbolInfo;
            
            return ((MetadataElement)mdrElement).resolveType(tpRef);
        }
        if (symbolInfo instanceof TypeRef) {
            if (mdrElement==null)
                mdrElement=getResource();
            Type el=((MetadataElement)mdrElement).resolveType((TypeRef)symbolInfo);
            
            if (el instanceof ParameterizedType)
                return ((ParameterizedType)el).getDefinition();
            if (el instanceof ParameterizedTypeImpl.Wrapper) {
                ParameterizedTypeImpl.Wrapper w=(ParameterizedTypeImpl.Wrapper)el;
                return (Element)w.getWrappedObject();
            }
            return el;
        }
        if (symbolInfo instanceof FieldRefInfo) {
            FieldRefInfo fref=(FieldRefInfo)symbolInfo;
            ClassDefinition javaClass=(ClassDefinition)getModelElement(null,null,fref.declClass);
            Feature f;
            
            if (javaClass==null)
                return null;
            f=(Feature)rep.getByMofId(fref.mofid);
            if (f!=null) {
                JavaClass declClass=(JavaClass)f.getDeclaringClass();

                if (declClass.equals(javaClass))
                    return f;
            }
            
            String name=fref.name;
            if (fref instanceof CallableRefInfo) {
                CallableRefInfo c=(CallableRefInfo)fref;
                Object features[]=javaClass.getContents().toArray();
                if (name==null) { // CONSTRUCTOR IMPLEMENTATION
                    for (int i=0;i<features.length;i++) {
                        ClassMember element = (ClassMember)features[i];
                        if ((element instanceof Constructor) && (parametersMatch((CallableFeature) element, c.parTypes))) {
                            return element;
                        }
                    }
                } else {    // Method
                    for (int i=0;i<features.length;i++) {
                        ClassMember element = (ClassMember)features[i];
                        if (!(element instanceof Method)) {
                            continue;
                        }
                        if (!name.equals(element.getName())) {
                            continue;
                        }
                        if (parametersMatch((CallableFeature) element, c.parTypes)) {
                            return element;
                        }
                    }
                }
            } else {    // Field
                Object features[]=javaClass.getFeatures().toArray();
                for (int i=0;i<features.length;i++) {
                    ClassMember element = (ClassMember)features[i];
                    if (!(element instanceof Field)) {
                        continue;
                    }
                    if (name.equals(element.getName())) {
                        return element;
                    }
                }
            }
            ErrorManager.getDefault().log(ErrorManager.WARNING, fref+" not found in "+getResource().getName());
            return null;
        }
        if (symbolInfo instanceof LocalVarRefInfo) {
            LocalVarRefInfo localvar=(LocalVarRefInfo)symbolInfo;
            
            if (localvar.var==null) {
                Collection topCollection=Collections.singletonList(getResource());
                localvar.var=(Variable)getModelElement(topCollection,localvar.varDeclAST);
            }
            return localvar.var;
        }
        if (!(symbolInfo instanceof MetadataElement) || ((MetadataElement)symbolInfo).isTransient())
            return symbolInfo;
        JMManager.getLog().log("Invalid type :"+symbolInfo); // NOI18N
        return null;
    }

    private TypeParameter resolveTypeParameter(ASTree tree,String name) {
        int firstToken=tree.getFirstToken();
        int lastToken=tree.getLastToken();
        int i;
        TypeParameter tp=null;
        Iterator elIterator=Collections.singletonList(getResource()).iterator();
        
        while (elIterator.hasNext()) {
            MetadataElement el=(MetadataElement)elIterator.next();
            ASTree t=el.getASTree();
            
            if (t==tree)
                return tp;
            if (firstToken>=t.getFirstToken() && lastToken<=t.getLastToken()) {
                if (el instanceof GenericElement) {
                    Iterator tpIt=((GenericElement)el).getTypeParameters().iterator();
                    
                    while(tpIt.hasNext()) {
                        TypeParameter p=(TypeParameter)tpIt.next();
                        
                        if (p.getName().equals(name)) {
                            tp=p;
                        }
                    }
                }
                elIterator=el.getChildren().iterator();
            }
        }
        return null;
    }
    
    private JavaClass resolveClass(TypeParamRef name) {
        SemiPersistentElement res=(SemiPersistentElement)getResource();
        JavaClass jcls=(JavaClass)res.resolveType(name);
        
        if (jcls instanceof ParameterizedType)
            return ((ParameterizedType)jcls).getDefinition();
        return jcls;
    }
    
    private JavaClass resolveClass(String name) {
        JavaClass javaClass = (JavaClass) jclsClass.resolveClass(name, true);
        if (javaClass == null) {
            // it is acceptable that the class was not resolved in case of opening a single file (from unopened project)
            // this is because in that case javac uses its standard class reader to do the attribution
            // let's log a message but do not throw an exception
            if (DEBUG) ErrorManager.getDefault().log(ErrorManager.USER, "Class " + name + " could not be resolved.");
        }
        return javaClass;
    }

    /**
     * Looks for the model element for the specified ASTree. Method is called
     * recursively to its subtrees.
     *
     * @param   elements     root object where to start find
     * @param   findingTree  tree for which we looking for the element
     * @return  found element for the tree
     */
    private MetadataElement getModelElement(Collection elements, ASTree findingTree) {
        if (elements == null || elements.isEmpty())
            return null;
        int firstToken = findingTree.getFirstToken();
        int lastToken = findingTree.getLastToken();
        for (Iterator it = elements.iterator(); it.hasNext(); ) {
            MetadataElement element = (MetadataElement) it.next();
            ASTree elementTree = element.getASTree();
            if (elementTree == findingTree) {
                // we have found it!
                return element;
            } else if (elementTree.getFirstToken() <= firstToken && elementTree.getLastToken() >= lastToken) {
                // the element is in child tree, look for it recursively
                MetadataElement result = getModelElement(element.getChildren(), findingTree);
                if (result == null) {
                    // seems that for the provided collection of elements there isn't
                    // tree provided in findingTree parameter.
                    throw new IllegalArgumentException("Tree not found! (Tree type = " + // NOI18N
                        findingTree.getType() + "; Bounds: " + firstToken + ", " + lastToken + // NOI18N
                        "; Element type: " + elementTree.getType() + "; Bounds: " + elementTree.getFirstToken() + ", " + elementTree.getLastToken() + // NOI18N
                        ")"); // NOI18N
                }
                return result;
            }
        }
        return null;
    }

    public boolean isVariableAccess(ASTree tree) {
        if (tree == null) {
            //66918
            return false;
        }
        if (!attributionToDo.isEmpty()) {
            doAttribution();
        }
        Object info = semanticInfo.get(tree);
        if (info == null) {
            if (tree.getType() == MULTI_PART_ID) {
                return isVariableAccess(tree.getSubTrees()[0]);
            } else {
                return false;
            }
        }
        if (info.getClass().equals(FieldRefInfo.class) || info instanceof LocalVarRefInfo)
            return true;
        if (info instanceof String) {
            String id=(String)info;
            Object jmiObject;
            
            if (id.charAt(0)!='^')
                return false;
            jmiObject=getSemanticInfo(tree, null);
            return jmiObject instanceof Variable;
        }
        return false;
    }

    /**
     * Checks wheter parameters names match. Takes callable feature parameters
     * and symbolInfo parameters and compares them. If all the type names and
     * they order is the same, returns true. Otherwise it returns false.
     *
     * @param   callableFeature  feature with parameters.
     * @param   symbolInfo       info with array of string representing
     *                           parameter type names.
     * @return  true, parameters' types match
     */
    static boolean parametersMatch(CallableFeature callableFeature, TypeRef[] parTypes) {
        Object[] params=callableFeature.getParameters().toArray();
        
        if (params.length != parTypes.length)
            return false; // if the size of list and array differs, continue
        for (int i=0;i<params.length;i++) {
            // check the parameters that they are the same
            ParameterImpl par = (ParameterImpl)params[i];
            ParameterInfo parInfo = (ParameterInfo) par.getElementInfo();
            if (!parTypes[i++].equals(parInfo.type)) {
                return false;
            }
        }
        return true;
    }
    
    void createTypeParametrScope(ASTree typeParsAST) {
        ASTree[] typeParams;
        int i;
        Scope currentScope;
        
        if (typeParsAST==null)
            return;
        if (typeParsAST.getType()!=TYPE_PARAMETER_LIST)
            throw new IllegalArgumentException("Type "+typeParsAST.getType()); // NOI18N
        typeParams=typeParsAST.getSubTrees();
        currentScope=(Scope)typeScopeInfo.peek();
        for (i = 0; i < typeParams.length; i++) {
            ASTree typeParAST=typeParams[i];
            String name = (String) processAST(typeParAST.getSubTrees()[0]);

            currentScope.addMember(new TypeParamScope(name));
        }        
    }
    
    void resolveSuperTypes(ASTree classTree,String fqn) {
        int treeType=classTree.getType();

        typeScopeInfo.push(new Scope((Scope)typeScopeInfo.peek()));        
        switch (treeType) {
            case ENUM_DECLARATION: {
                int i;
                ASTree parts[]=classTree.getSubTrees();
                String name=(String)processAST(parts[1]);  // class name
                ASTree ifacesAST=parts[2];  // interfaces
                NameRef interfaces[];
                List interJcls=Collections.EMPTY_LIST;
                NameRef sclass;
                String currentFqn=fqn.length()==0?name:fqn.concat(".").concat(name); // NOI18N
                JavaClass superJcls = null;
                
                sclass=NameRef.java_lang_Enum;
                semanticInfo.put(parts[1],sclass);
                if (ifacesAST!=null) {
                    ASTree ifaces[]=ifacesAST.getSubTrees();
                    
                    interfaces=new NameRef[ifaces.length];
                    interJcls=new ArrayList();

                    for (i=0;i<ifaces.length;i++) {
                        interfaces[i]=(NameRef)resolveTypeName(ifaces[i]);
                        interJcls.add(resolveClass(interfaces[i]));
                    }
                    semanticInfo.put(ifacesAST,interfaces);
                }
                if (semanticInfo.get(classTree)==null) {
                    JavaClass jcls = getClassByFqn(currentFqn);

                    // INDEX INCONSISTENCY RECOVERY
                    if (jcls == null) {
                        JMManager.getLog().log("Enum not found in index: " + currentFqn + ". Recovering...");
                        Thread.dumpStack();
                        jcls = enumClass.create(currentFqn, 0, false);
                        setParent(jcls,fqn);
                    } else if (!(jcls instanceof JavaEnum)) {
                        JMManager.getLog().log(ErrorManager.INFORMATIONAL,"Wrong type of object found in index. Expected: JavaEnum, found: " + jcls.getClass().getName());
                        JMManager.getLog().log(ErrorManager.INFORMATIONAL,"Recovering...");
                        Thread.dumpStack();
                        JavaClass oldCls = jcls;
                        jcls = enumClass.create(currentFqn, oldCls.getModifiers(), false);
                        swapChild(oldCls, jcls);
                    }
                    // -----------------------------
                    semanticInfo.put(classTree, jcls);
                }
                if (sclass!=null)
                    superJcls=resolveClass(sclass);
                superInfoMap.put(currentFqn,new SuperInfo(superJcls,interJcls));
                localSuperInfo.add(currentFqn);
                break;
            }
            case ANNOTATION_TYPE_DECLARATION:
            case INTERFACE_DECLARATION:
            case CLASS_DECLARATION:
                int i;
                ASTree parts[]=classTree.getSubTrees();
                String name=(String)processAST(parts[1]);  // class name
                ASTree superclassAST; // extends
                ASTree ifacesAST;  // interfaces
                NameRef interfaces[];
                List interJcls=Collections.EMPTY_LIST;
                NameRef sclass=null;
                String currentFqn=fqn.length()==0?name:fqn.concat(".").concat(name); // NOI18N
                JavaClass superJcls=null;

                if (treeType==ANNOTATION_TYPE_DECLARATION) {
                    superclassAST=null;
                    ifacesAST=null;
                    sclass=NameRef.java_lang_Annotation;
                } else {
                    createTypeParametrScope(parts[2]);      // type parameters
                    superclassAST=parts[3]; // extends
                    ifacesAST=parts[4];  // interfaces
                    if (superclassAST!=null) {
                        sclass=(NameRef)resolveTypeName(superclassAST.getSubTrees()[0]);
                    } else if (!(name.equals("Object") && jpck.equals("java.lang")) && treeType==CLASS_DECLARATION) // NOI18N
                        sclass=NameRef.java_lang_Object;
                }
                if (sclass!=null) {
                    semanticInfo.put(parts[1],sclass);
                }
                if (ifacesAST!=null) {
                    ASTree[] ifaces=ifacesAST.getSubTrees();
                    
                    interfaces=new NameRef[ifaces.length];
                    interJcls=new ArrayList();
                    
                    for (i=0;i<ifaces.length;i++) {                        
                        interfaces[i]=(NameRef)resolveTypeName(ifaces[i]);
                        interJcls.add(resolveClass(interfaces[i]));
                    }
                    semanticInfo.put(ifacesAST,interfaces);
                }
                if (semanticInfo.get(classTree)==null) {
                    JavaClass jcls = getClassByFqn(currentFqn);

                    // INDEX INCONSISTENCY RECOVERY
                    if (jcls == null) {
                        JMManager.getLog().notify(ErrorManager.INFORMATIONAL,new Exception((treeType == ANNOTATION_TYPE_DECLARATION ? "Annotation" : "Class") + " not found in index: " + currentFqn + ". Recovering...")); // NOI18N
                        if (treeType == ANNOTATION_TYPE_DECLARATION) {
                            jcls = annotClass.create(currentFqn, 0, false);
                        } else {
                            jcls = jclsClass.create(currentFqn, 0, null, null, false);
                        }
                        setParent(jcls,fqn);
                    } else if (treeType == ANNOTATION_TYPE_DECLARATION && !(jcls instanceof AnnotationType)) {
                        JMManager.getLog().log(ErrorManager.WARNING, "Wrong type of object found in index. Expected: AnnotationType, found: " + jcls.getClass().getName());
                        JMManager.getLog().notify(ErrorManager.INFORMATIONAL,new Exception("Recovering...")); // NOI18N
                        JavaClass oldCls = jcls;
                        jcls = annotClass.create(currentFqn, oldCls.getModifiers(), false);
                        swapChild(oldCls, jcls);
                    } else if (treeType != ANNOTATION_TYPE_DECLARATION && (jcls instanceof AnnotationType || jcls instanceof JavaEnum)) {
                        JMManager.getLog().log(ErrorManager.WARNING, "Wrong type of object found in index. Expected: JavaClass, found: " + jcls.getClass().getName());
                        JMManager.getLog().notify(ErrorManager.INFORMATIONAL,new Exception("Recovering...")); // NOI18N
                        JavaClass oldCls = jcls;
                        jcls = jclsClass.create(currentFqn, oldCls.getModifiers(), null, null, false);
                        swapChild(oldCls, jcls);
                    }
                    // -----------------------------

                    semanticInfo.put(classTree, jcls);
                }
                if (sclass!=null)
                    superJcls=resolveClass(sclass);
                superInfoMap.put(currentFqn,new SuperInfo(superJcls,interJcls));
                localSuperInfo.add(currentFqn);
        }
        typeScopeInfo.pop();
    }
    
    // ---------------- METHODS USED FOR INDEX INCONSISTENCY RECOVERY -----------------------------
    private JavaClass getClassByFqn(String currentFqn) {
        JavaClass jcls = null;
        Collection classes = index.getClassesByFqn(currentFqn);
        for (Iterator it = classes.iterator(); it.hasNext();) {
            JavaClass tmp = (JavaClass) it.next();
            if (tmp.isValid() && getResource().equals(tmp.getResource())) {
                jcls = tmp;
                break;
            }
        }
        return jcls;
    }
    
    
    private void swapChild(JavaClass oldType, JavaClass newType) {
        RefFeatured parent = oldType.refImmediateComposite();
        if (parent instanceof ResourceImpl) {
            ListIterator it;
            for (it = ((ResourceImpl) parent).getPersistentClassifiers().listIterator(); it.hasNext();) {
                JavaClass cls = (JavaClass) it.next();
                if (cls.equals(oldType)) {
                    it.set(newType);
                    oldType.refDelete();
                    if (((ResourceImpl) parent).classifiersInited()) {
                        ((ResourceImpl) parent).reinitClassifiers();
                    }
                    return;
                }
            }
            JMManager.getLog().log("Old type wasn't found in classifiers of its parent resource. Adding newType to the end of the resource classifiers list...");
            it.add(newType);
            oldType.refDelete();
            return;
        } else if (parent instanceof JavaClassImpl) {
            JavaClassImpl enclosingClass = (JavaClassImpl) parent;
            boolean contentsInited = enclosingClass.contentsInited();
            boolean contentsEmpty = enclosingClass.getPersistentContents().isEmpty();
            if (contentsInited || !contentsEmpty) {
                ListIterator it;
                for (it = enclosingClass.getPersistentContents().listIterator(); it.hasNext();) {
                    Element cls = (Element) it.next();
                    if (cls.equals(oldType)) {
                        it.set(newType);
                        oldType.refDelete();
                        if (contentsInited) {
                            enclosingClass.reinitContents();
                        }
                        return;
                    }
                }
                JMManager.getLog().log("Old type wasn't found in classifiers of its parent resource. Adding newType to the end of the resource classifiers list...");
                it.add(newType);
            } else {
                ((JavaClassImpl) newType).setParentClass(enclosingClass);
            }
            ((JavaClassImpl) oldType).setParentClass(null);
            oldType.refDelete();
            return;
        }
        JMManager.getLog().log("Parent of newType has not been set since parent of the oldType is: " + parent.getClass().getName());
    }
    
    private void setParent(JavaClass jcls,String fqn) {
        if (fqn.equals(jpck)) {
            ResourceImpl res = (ResourceImpl) getResource();
            res.getPersistentClassifiers().add(jcls);
            if (res.classifiersInited()) {
                res.reinitClassifiers();
            }
        } else {
            JavaClassImpl enclosingClass = (JavaClassImpl) getClassByFqn(fqn);
            if (enclosingClass == null) {
                JMManager.getLog().log("Could not find parent class: " + fqn);
                JMManager.getLog().log("Setting resource as a parent...");
                ResourceImpl res = (ResourceImpl) getResource();
                res.getPersistentClassifiers().add(jcls);
                if (res.classifiersInited()) {
                    res.reinitClassifiers();
                }
            } else {
                boolean contentsInited = enclosingClass.contentsInited();
                boolean contentsEmpty = enclosingClass.getPersistentContents().isEmpty();
                if (contentsInited || !contentsEmpty) {
                    enclosingClass.getPersistentContents().add(jcls);
                    if (contentsInited) {
                        enclosingClass.reinitContents();
                    }
                } else {
                    ((JavaClassImpl) jcls).setParentClass(enclosingClass);
                }
            }
        }
    }
    // --------------------------------------------------------------------------------------------
    
    ModifiersInfo getModifiers(ASTree t) {
        ModifiersInfo nfo = (ModifiersInfo)processAST(t);
        return nfo == null ? NO_MODIFIERS : nfo;
    }
    
    /** Returns deprecated modifier if first tree token has deprecated flag. */
    private int getDeprecated(ASTree t) {
        Token tok = getToken(t.getFirstToken());
        return tok.getDeprecatedFlag() ? FeatureImpl.DEPRECATED : 0;
    }
    
    public static TypeRef fullType(Object t,Object dims) {
        TypeRef td=(TypeRef)t;
        
        if (dims!=null) {
            int dimension=((Integer)dims).intValue();
            
            if (td instanceof ArrayRef) {
                ArrayRef aref=(ArrayRef)td;
                return new ArrayRef(aref.parent,aref.dimCount+dimension);
            }
            else
                return new ArrayRef((PrimitiveTypeRef)td,dimension);
        }
        return td;
    }

    Type fullType(Type td,Object dims) {        
        if (dims!=null) {
            int i;
            int dimension=((Integer)dims).intValue();
            ArrayClass arrayClass=((JavaModelPackage)td.refImmediatePackage()).getArray();
            
            for (i=0;i<dimension;i++)
                td=arrayClass.resolveArray(td);
        }
        return td;
    }
    
    private static Object[] getArray(Object obj) {
        if (obj==null)
            return new Object[0];
        if (obj.getClass().isArray())
            return (Object[])obj;
        return new Object[]{obj};
    }
    
    TypeRef resolveTypeName(ASTree typeAST) {
        if (typeAST!=null) {
            int treeType=typeAST.getType();

            if (treeType==IDENTIFIER) {
                Object name=resolveTypeName((String)processAST(typeAST));
                String fqn;
                
                storeSemanticInfo(typeAST,name);
                if (name instanceof TypeRef)
                    return (TypeRef)name;
                if (name instanceof Type)
                    return SemiPersistentElement.typeToTypeRef((Type)name);
                if (name instanceof JavaPackage)
                    fqn=((JavaPackage)name).getName();
                else
                    fqn=(String)name;
                return new NameRef(fqn,null,null);
            } else {
                return (TypeRef)processAST(typeAST);
            }
        }
        return null;
    }
    
    Object resolveTypeName(String simpleName) {
        Object ref;
        Scope currentScope=(Scope)typeScopeInfo.peek();
        String name;
        
        ref=currentScope.lookup(simpleName);
        if (ref==null) { // package
            JavaPackage jpck=getPackage(simpleName);
            
            if (jpck!=null)
                return jpck;
            return simpleName; // -- unresolved
        }
        return ref;
    }
    
    private PrimitiveTypeRef getPrimitiveType(ASTree tree) {
        switch(tree.getType()) {
            case BYTE:
                return PrimitiveTypeRef.BYTE;
            case SHORT:
                return PrimitiveTypeRef.SHORT;
            case INT:
                return PrimitiveTypeRef.INT;
            case LONG:
                return PrimitiveTypeRef.LONG;
            case CHAR:
                return PrimitiveTypeRef.CHAR;
            case FLOAT:
                return PrimitiveTypeRef.FLOAT;
            case DOUBLE:
                return PrimitiveTypeRef.DOUBLE;
            case BOOLEAN:
                return PrimitiveTypeRef.BOOLEAN;
            case VOID:
                return PrimitiveTypeRef.VOID;
        }
        ErrorManager.getDefault().log(ErrorManager.ERROR, "Unknown type "+tree.getType()); // NOI18N
        return null;
    }
    
    JavaClass getSuperClass(ClassDefinition jcls) {
        SuperInfo info=(SuperInfo)superInfoMap.get(jcls.getName());
        JavaClass superClass;
        
        if (info!=null)
            return info.superClass;
        if (getResource().equals(jcls.getResource())) {
            JMManager.getLog().log(ErrorManager.INFORMATIONAL,"Unresolved superclass for "+jcls.getName()); // NOI18N
            return null;
        }
        superClass=jcls.getSuperClass();
        if(superClass instanceof ParameterizedType) {
            return ((ParameterizedType)superClass).getDefinition();
        }
        return superClass;
    }
    
    Collection getInterfaces(ClassDefinition jcls) {
        SuperInfo info=(SuperInfo)superInfoMap.get(jcls.getName());
        
        if (info!=null)
            return info.interfaces;
        if (getResource().equals(jcls.getResource())) {
            JMManager.getLog().log(ErrorManager.INFORMATIONAL,"Unresolved Interface for "+jcls.getName()); // NOI18N
            return Collections.EMPTY_LIST;
        }
        return jcls.getInterfaces();
    }    

    private Type processASTBody(ASTree tree) {
        ASTree parts[];
        int i,treeType;
        Type td=null;
        
        if (tree==null)
            return null;
        parts=tree.getSubTrees();
        treeType=tree.getType();
        switch (treeType) {
            case COMPILATION_UNIT:
                typeScopeInfo.push(Scope.createTypeScope(getResource(),classPath));
                processASTBody(parts[2]); // declarations
                typeScopeInfo.pop();
                break;
            case ANNOTATION_TYPE_DECLARATION:
            case ENUM_DECLARATION:
            case INTERFACE_DECLARATION:
            case CLASS_DECLARATION: {
                ClassDefinition savedCurrentClass=currentClass;
                
                currentClass=(ClassDefinition)semanticInfo.get(tree);
                if (currentClass!=null) {
                    Scope classScope=new Scope((Scope)typeScopeInfo.peek());
                    Scope varScope=new Scope((Scope)variableScope.peek());

                    classScope.addMember(Scope.createMemberTypeScope(currentClass,this));
                    varScope.addMember(Scope.createFieldScope(currentClass));
                    typeScopeInfo.push(classScope);
                    variableScope.push(varScope);
                    if (treeType == ENUM_DECLARATION) {
                        ASTree enumBodyDecls = parts[3].getSubTrees()[1];
                        if (enumBodyDecls != null) {
                            processASTBody(enumBodyDecls);
                        }
                    } else if (treeType == ANNOTATION_TYPE_DECLARATION) {
                        processASTBody(parts[2]);
                    } else {
                        processASTBody(parts[5]);   // body
                    }
                    variableScope.pop();
                    typeScopeInfo.pop();
                } else {
                    Scope currTypeScope=(Scope)typeScopeInfo.peek();             
                    Scope currVarScope=(Scope)variableScope.peek();
                    JavaClassImpl locClass=createTransientClasses(tree,null);
                    ClassInfo locClassInfo;
                    Token firstBodyToken;       //    left bracket "{" // NOI18N
                    
                    currTypeScope.addMember(new LocalClassScope(locClass));
                    resolveSuperTypes(tree, "");
                    locClassInfo=(ClassInfo)processAST(tree);
                    locClass.updatePersistent(locClassInfo);
                    locClass.setElementInfo(locClassInfo);
                    firstBodyToken=getToken(parts[parts.length-1].getFirstToken());
                    semanticInfo.put(firstBodyToken,new Scope[]{(Scope)currTypeScope.clone(),(Scope)currVarScope.clone()});
                }
                currentClass=savedCurrentClass;
                break;
            }
            case FIELD_DECLARATION: {
                ASTree varDeclsTree=parts[2];
                
                if (varDeclsTree.getType()==VARIABLE_DECLARATOR) {
                    processASTBody(varDeclsTree.getSubTrees()[2]);  //initializer
                } else {
                    ASTree varDeclsTreeArray[]=varDeclsTree.getSubTrees();
                    
                    for (i=0;i<varDeclsTreeArray.length;i++) {
                        ASTree varDeclTree=varDeclsTreeArray[i];
                        
                        processASTBody(varDeclTree.getSubTrees()[2]);  //initializer
                    }
                }
                break;
            }
            case ENUM_CONSTANT: {
                ASTree identifier = parts[0];
                ASTree initValue = parts[1];
                ASTree anonymousClass = parts[2];
                List parTypes;
                
                parTypes=getArgumetsTypes(initValue);    // arguments
                usesMethod(null,currentClass,parTypes,identifier,false);
                if (anonymousClass!=null) {
                    createAnonClass(anonymousClass, (JavaClass) currentClass, null);
                }
                break;
            }
            case CONSTRUCTOR_DECLARATION:
            case METHOD_DECLARATION: {
                CallableFeature bhFeature=(CallableFeature)semanticInfo.get(tree);
                Scope methodVarScope=new Scope((Scope)variableScope.peek());
                Scope methodTypeScope=new Scope((Scope)typeScopeInfo.peek());
                Iterator parIt=bhFeature.getParameters().iterator();
                Iterator typeParIt=bhFeature.getTypeParameters().iterator();

                while(parIt.hasNext()) {
                    Parameter fp=(Parameter)parIt.next();
                    
                    methodVarScope.addMember(new LocalVarScope(fp));
                }
                while(typeParIt.hasNext()) {
                    TypeParameter tp=(TypeParameter)typeParIt.next();

                    methodTypeScope.addMember(new TypeParamScope(tp));
                }
                variableScope.push(methodVarScope);
                typeScopeInfo.push(methodTypeScope);
                processASTBody(parts[5]);   // body
                typeScopeInfo.pop();
                variableScope.pop();
                break;
            }
            case INSTANCE_INITIALIZER:
            case STATIC_INITIALIZER: {                
                variableScope.push(new Scope((Scope)variableScope.peek()));
                typeScopeInfo.push(new Scope((Scope)typeScopeInfo.peek()));
                processASTBody(parts[1]);   // body
                typeScopeInfo.pop();
                variableScope.pop();
                break;
            }
            case FORMAL_PARAMETER: {
                ParameterInfo par=(ParameterInfo)processAST(tree);
                Scope currVarScope=(Scope)variableScope.peek();
                Type type=((SemiPersistentElement)currentFeature).resolveType(par.type);
                LocalVarRefInfo ref=new LocalVarRefInfo(tree,type);
                currVarScope.addMember(new LocalVarScope(par.name,ref));
                storeSemanticInfo(tree,par);
                break;
            }
            case SWITCH_BLOCK:
            case CATCH_CLAUSE:
            case BLOCK_STATEMENTS:
                if (parts!=null && parts.length>0) {
                    boolean createVarScope=treeType!=BLOCK_STATEMENTS 
			|| getToken(tree.getFirstToken()).getType()==L_CURLY;
                    
                    if (createVarScope)
                        variableScope.push(((Scope)variableScope.peek()).clone());
                    for (i=0;i<parts.length;i++)
                        processASTBody(parts[i]);
                    if (createVarScope)
                        variableScope.pop();
                }
                break;
            case LOCAL_VARIABLE_DECLARATION: {
                Scope currVarScope=(Scope)variableScope.peek();
                ASTree modifiers=parts[0];
                ModifiersInfo modInfo=getModifiers(modifiers);   // final and annotations
                Type type=resolveClassName(parts[1]);  // type
                ASTree varDeclsTree=parts[2];
                Object names[]=getArray(processAST(varDeclsTree));
                ASTree varDeclsTreeArray[];
                
                if (modifiers!=null) {
                    Boolean isFinal=Boolean.valueOf(Modifier.isFinal(modInfo.modifiers));
                    AnnotationInfo[] ann=modInfo.annotations;

                    semanticInfo.put(modifiers, new Object[]{isFinal,ann});
                }
                usesType(type,tree);
                if (varDeclsTree.getType()==VARIABLE_DECLARATOR) {
                    varDeclsTreeArray=new ASTree[]{varDeclsTree};
                    names=new Object[]{names};
                } else
                    varDeclsTreeArray=varDeclsTree.getSubTrees();
                for (i=0;i<names.length;i++) {
                    Object name[]=(Object[])names[i];
                    Type varType=fullType(type,name[1]); // dims
                    String varName=(String)name[0];
                    ASTree varDecl=varDeclsTreeArray[i];
                    
                    if (names.length>1)
                        usesType(varType, varDecl);
                    currVarScope.addMember(new LocalVarScope(varName,new LocalVarRefInfo(varDecl,varType)));
                    processASTBody(varDecl.getSubTrees()[2]);  //initializer
                }
                break;
            }
            case FOR_STATEMENT: {
                ASTree initAST=parts[0];
                Scope forScope=null;
                
                if (initAST!=null && initAST.getType()==LOCAL_VARIABLE_DECLARATION) {
                    forScope=new Scope((Scope)variableScope.peek());
                    variableScope.push(forScope);
                }
                for (i=0;i<parts.length;i++)
                    processASTBody(parts[i]);
                if (forScope!=null)
                    variableScope.pop();
                break;
            }
            case FOR_EACH_STATEMENT: {
                variableScope.push(((Scope)variableScope.peek()).clone());
                for (i=0;i<parts.length;i++)
                    processASTBody(parts[i]);
                variableScope.pop();
                break;
            }
            case SWITCH_STATEMENT: {
                Type switchExpression=TypeClassImpl.getRawType(processASTBody(parts[0]));
                boolean enumSwitch=switchExpression instanceof JavaEnum;
                Scope oldEnumScope=enumSwitchScope;
                
                if (enumSwitch) {
                    enumSwitchScope=Scope.createFieldScope((JavaEnum)switchExpression);
                } else
                    enumSwitchScope=null;
                processASTBody(parts[1]);
                enumSwitchScope=oldEnumScope;
                break;
            }
            case SWITCH_LABEL: {
                if (enumSwitchScope!=null)
                   variableScope.push(enumSwitchScope); 
                processASTBody(parts[0]); 
                if (enumSwitchScope!=null)
                   variableScope.pop();                 
                break;
            }
            case PRIMARY_CLASS: {
                Type classType=resolveClassName(parts[0]);

                return jclsClass.resolveClass("java.lang.Class", false); // NOI18N
            }
            case PRIMARY_THIS: {                
                if (parts!=null)
                    td=resolveClassName(parts[0]);
                else
                    td=currentClass;
                break;
            }
            case INT_LIT:
            case FLOAT_LIT:
            case BOOL_LIT:
            case CHAR_LIT:
            case STRING_LIT:
            case NULL_LIT: {
                String typeText;
                Object value=((Token)tree).getValue();
                
                switch (treeType) {
                    case INT_LIT:
                        if (value instanceof Long)
                            typeText="long";  // NOI18N
                        else
                            typeText="int"; // NOI18N
                        break;
                    case FLOAT_LIT:
                        if (value instanceof Double)
                            typeText="double";  // NOI18N
                        else
                            typeText="float"; // NOI18N
                        break;
                    case BOOL_LIT:
                        typeText="boolean"; // NOI18N
                        break;
                    case CHAR_LIT:
                        typeText="char"; // NOI18N
                        break;
                    case STRING_LIT:
                        return resolveClass("java.lang.String"); // NOI18N
                    case NULL_LIT:
                        return resolveClass("java.lang.Object"); // NOI18N
                    default:
                        ErrorManager.getDefault().log(ErrorManager.ERROR, "Unknown Literal: "+treeType); // NOI18N
                        return null;
                }
                return typeClass.resolve(typeText);
            }
            case CLASS_INSTANCE_CREATION_EXPRESSION: {
                ASTree primaryAST=parts[0];
                ASTree classIdAST=parts[2];
                ASTree anonymousClass=parts[4];
                List parTypes;
                NameRef cls = null;
                
                if (primaryAST==null) {
                    td=resolveClassName(classIdAST);
                    if (classIdAST.getType()==MULTI_PART_ID) {
                        cls = (NameRef) processAST(classIdAST);
                    }
                } else {
                    JavaClass jcls=(JavaClass)resolvePrimaryWithSuper(primaryAST,false, treeType);
                    
                    if (jcls!=null) {
                        Object type=getMemberOf(jcls, (Token)classIdAST);
                        
                        if (type instanceof JavaClass) {
                            td=(JavaClass)type;
                            usesType(td, classIdAST);
                        }                        
                    }
                }
                parTypes=getArgumetsTypes(parts[3]);    // arguments
                if (td!=null) {
                    usesMethod(null,(JavaClass)td,parTypes,tree,false);
                    if (anonymousClass!=null) {
                        createAnonClass(anonymousClass, (JavaClass) td, cls);
                    }
                }
                break;
            }
            case ARRAY_CREATION_EXPRESSION: {
                ASTree elemTypeAST=parts[0];
                Type elemType=resolveClassName(elemTypeAST);
                
                if (elemType!=null) {
                    ASTree dimsExprAST=parts[1];
                    ASTree dimsAST=parts[2];
                    int dims=getDimExprsDimenstion(dimsExprAST);
                    Integer dimsVal=(Integer)processAST(dimsAST);
                    
                    if (dimsVal!=null)
                        dims+=dimsVal.intValue();
                    if (dims>0) {
                        ArrayClass arrayClass=((JavaModelPackage)elemType.refImmediatePackage()).getArray();
                   
                        for (i=0;i<dims;i++)
                            elemType=arrayClass.resolveArray(elemType);
                    }
                    usesType(elemType,tree);
                }
                for (i=1;i<parts.length;i++)
                    processASTBody(parts[i]);
                td=elemType;
                break;
            }
            case FIELD_ACCESS: {
                ClassDefinition jcls;
                Token fieldId=(Token)parts[2];
                String fieldName=(String)fieldId.getValue();
                Field field;
                boolean hasSuper=parts[1]!=null;
                
                if (DEBUG) System.out.print("\nField "+fieldName); // NOI18N
                jcls=resolvePrimaryWithSuper(parts[0],hasSuper, treeType);
                if (jcls==null || jcls instanceof UnresolvedClass)
                    return null;
                if (DEBUG && jcls instanceof JavaClass) System.out.print(" of class "+jcls.getName()+" is of type "); // NOI18N
                field=(Field)Scope.createFieldScope(jcls).lookup(fieldName);
                if (field!=null) {
                    td=field.getType();
                    usesField(field, fieldId);
                    storeSemanticInfo(tree,field);
                    if (td instanceof JavaClass)
                        if (DEBUG) System.out.print(td.getName());
                        else if (td instanceof PrimitiveType)
                            if (DEBUG) System.out.print(((PrimitiveType)td).getKind());
                }
                break;
            }
            case METHOD_INVOCATION: {
                ASTree pTree=parts[0];
                Token methodId=(Token)parts[3];
                String methodName=(String)methodId.getValue();
                boolean hasSuper=parts[1]!=null;
                ClassDefinition jcls;
                List parTypes;
                
                if (DEBUG) System.out.print("\nMethod "+methodName); // NOI18N
                jcls=resolvePrimaryWithSuper(pTree,hasSuper, treeType);
                if (jcls!=null)
                    if (DEBUG && jcls instanceof JavaClass) System.out.print(" of class "+jcls.getName()); // NOI18N
                parTypes=getArgumetsTypes(parts[4]);    // arguments
                if (jcls!=null) {
                    boolean impicitThis= pTree==null && !hasSuper;
                    td=usesMethod(methodName,jcls,parTypes,tree,impicitThis);
                }
                break;
            }
            case EXPLICIT_CONSTRUCTOR_INVOCATION: {
                ASTree pTree=parts[0];
                boolean hasSuper=parts[2].getType()==SUPER;
                ClassDefinition jcls;
                List parTypes;
                
                jcls=resolvePrimaryWithSuper(pTree,hasSuper, treeType);
                if (jcls!=null)
                    if (DEBUG && jcls instanceof JavaClass) System.out.print(" of class "+jcls.getName()); // NOI18N
                parTypes=getArgumetsTypes(parts[3]);    // arguments
                if (jcls!=null) {
                    td=usesMethod(null,jcls,parTypes,tree,false);
                }
                break;
            }
            case ARRAY_ACCESS: {
                ASTree arrTypeAST=parts[0];
                
                td=processASTBody(arrTypeAST); // array type
                processASTBody(parts[1]);   // expression
                if (td!=null) {
                    if (td instanceof UnresolvedClass)
                        td=null;
                    else if (td instanceof Array)
                        td=((Array)td).getType();
                }
                break;
            }
            case POSTFIX_EXPRESSION: {
                ASTree fieldAccess=parts[0];
                
                td=processASTBody(fieldAccess);
                for (i=1;i<parts.length;i++)
                    processASTBody(parts[i]);
                break;
            }
            case PREFIX_EXPRESSION:
                td=processASTBody(parts[1]);
                break;
            case COMPLEX_EXPRESSION:
                td=processASTBody(parts[0]);
                break;
            case CAST_EXPRESSION: {
                ASTree typeTree=parts[0];   // type
                int type=typeTree.getType();
                
                if (type==POSTFIX_EXPRESSION) {
                    typeTree=typeTree.getSubTrees()[0];
                    type=typeTree.getType();
                }
                if (type!=PRIMITIVE_TYPE && type!=MULTI_PART_ID && type!=IDENTIFIER && type!=REFERENCE_TYPE)
                    return null;    // invalid typecast
                td=resolveClassName(typeTree); // type
                processASTBody(parts[1]); // expression
                break;
            }
            case INFIX_EXPRESSION: {
                int operatorType=parts[1].getType();
                Type leftType=processASTBody(parts[0]);
                Type rightType;
                
                if (operatorType==INSTANCEOF)
                    rightType=resolveClassName(parts[2]);
                else
                    rightType=processASTBody(parts[2]);
                switch (operatorType) {
                    case LT:
                    case LTE:
                    case GT:
                    case GTE:
                    case EQ:
                    case NEQ:
                    case BOOL_AND:
                    case BOOL_OR:
                    case INSTANCEOF:
                        td=typeClass.resolve("boolean"); // NOI18N
                        break;
                    case PLUS:
                        // String +
                        if (leftType instanceof ClassDefinition || rightType instanceof ClassDefinition) {
                            td=resolveClass("java.lang.String"); // NOI18N
                            break;
                        }
                        // fall through
                    case MULTI:
                    case DIV:
                    case MOD:
                    case MINUS:
                    case AND:
                    case OR:
                    case XOR:
                        td=computeType(leftType,rightType);
                        break;
                    case L_SHIFT:
                    case R_SHIFT:
                    case UR_SHIFT:
                        if (leftType==null || !"long".equals(leftType.getName())) // NOI18N
                            td=typeClass.resolve("int"); // NOI18N
                        else
                            td=typeClass.resolve("long"); // NOI18N
                        break;
                    default:
                        ErrorManager.getDefault().log(ErrorManager.WARNING,"Unknown operator "+operatorType+" in "+getText(tree));
                }
                break;
            }
            case CONDITIONAL_EXPRESSION: {
                processASTBody(parts[0]);
                td=processASTBody(parts[1]);
                processASTBody(parts[2]);
                break;
            }
            case ASSIGNMENT: {
                ASTree leftTree=parts[0];
                
                td=processASTBody(leftTree);
                processASTBody(parts[2]);
                break;
            }
            case REFERENCE_TYPE: {
                ASTree dim=parts[1];
                
                td=processASTBody(parts[0]);
                if (dim!=null) {
                    td=fullType(td,processAST(dim));
                }
                break;
            }
            case PRIMITIVE_TYPE:
                td=(Type)((SemiPersistentElement)getResource()).resolveType(getPrimitiveType(parts[0]));
                break;
            case MULTI_PART_ID:                
            case IDENTIFIER: {
                Object type=processMultiPartId(tree,parts);
                if (type instanceof Type)
                    td=(Type)type;
                else
                    return null;
                if (DEBUG) {
                    System.out.print(" is of type "); // NOI18N
                    if (td instanceof JavaClass)
                        System.out.print(td.getName());
                    else if (td instanceof PrimitiveType)
                        System.out.print(((PrimitiveType)td).getKind());
                }
                break;
            }
            default:
                if (parts!=null) {
                    for (i=0;i<parts.length;i++)
                        processASTBody(parts[i]);
                }
        }
        usesType(td,tree);
        return td;
    }
    
    private JavaClassImpl createTransientClasses(ASTree tree,JavaClass parent) {
        ASTree[] parts;
        int treeType;
        
        if (tree==null)
            return null;
        parts=tree.getSubTrees();
        treeType=tree.getType();
        switch (treeType) {
            case ENUM_DECLARATION:                
            case ANNOTATION_TYPE_DECLARATION:
            case INTERFACE_DECLARATION:
            case CLASS_DECLARATION: {
                String name=(String)processAST(parts[1]);  // class name
                String currentFqn=parent==null?name:parent.getName().concat(".").concat(name); // NOI18N
                ASTree body;
                JavaClass jcls;
                
                if (treeType==ENUM_DECLARATION) {
                    body=parts[3];
                    jcls = enumClass.create(currentFqn, M_ENUM, true);
                } else if (treeType==ANNOTATION_TYPE_DECLARATION) {
                    body=parts[2];
                    jcls = annotClass.create(currentFqn, M_ANNOTATION, true);
                } else {
                    int mods=0;
                    
                    if (treeType==INTERFACE_DECLARATION)
                        mods=Modifier.INTERFACE;
                    body=parts[5];
                    jcls = jclsClass.create(currentFqn, mods, null,null, true);
                }
                if (parent!=null)
                    ((JavaClassImpl)jcls).setParentClass((JavaClassImpl)parent);
                semanticInfo.put(tree,jcls);
                createTransientClasses(body,jcls);
                return (JavaClassImpl)jcls;
            }
            case ANNOTATION_TYPE_BODY_DECLARATIONS:
            case ENUM_BODY_DECLARATIONS:
            case INTERFACE_MEMBER_DECLARATIONS:
            case CLASS_BODY_DECLARATIONS: {
                int i;
                
                for (i=0;i<parts.length;i++)
                    createTransientClasses(parts[i],parent);
                break;
            }
        }
        return null;
    }
    
    private void createAnonClass(ASTree anonymousClass, JavaClass superClass, NameRef cls) {
        FeatureInfo features[];
        ClassInfo clsInfo;
        Scope classScope=new Scope((Scope)typeScopeInfo.peek());
        NameRef[] ifcs=null;
        Token firstToken=getToken(anonymousClass.getFirstToken());

        if (cls==null) {
            cls = (NameRef)SemiPersistentElement.typeToTypeRef(superClass);
        }        
        classScope.addMember(Scope.createMemberTypeScope(superClass,this));
        typeScopeInfo.push(classScope);
        createTransientClasses(anonymousClass, null);
        features=(FeatureInfo[])processAST(anonymousClass,""); 
        if (superClass.isInterface()) {
            ifcs = new NameRef[] {cls};
            cls = NameRef.java_lang_Object;
        }
        clsInfo=new ClassInfo(anonymousClass, anonymousClass.getType(), null, 0, features, cls, ifcs, null, ElementInfo.EMPTY_ANNOTATIONS); // todo-anotation
        semanticInfo.put(firstToken,new Scope[]{classScope,(Scope)variableScope.peek()});
        typeScopeInfo.pop();
        semanticInfo.put(anonymousClass,clsInfo);
    }
    
    private Object processIdentifier(Token identifier) {
        Object type=resolveName(identifier);
        Variable refField=null;

        if (DEBUG) System.out.print("\nID:"+identifier.getValue()); // NOI18N
        if (type==null)
            return null;
        if (type instanceof Variable) {
            refField=(Variable)type;
            type=refField.getType();
            usesField(refField,identifier);
        } else {
            storeSemanticInfo(identifier,type);
            if (type instanceof LocalVarRefInfo) {
                type=((LocalVarRefInfo)type).type;
            }
        }
        return type;
    }
    
    private Object processMultiPartId(ASTree tree,ASTree[] parts) {
        if (tree.getType()==IDENTIFIER)
            return processIdentifier((Token)tree);
        
        if (tree.getType() == ARRAY_CREATION_EXPRESSION) {
            ASTree elemTypeAST=parts[0];
            Type elemType=resolveClassName(elemTypeAST);

            if (elemType!=null) {
                ASTree dimsExprAST=parts[1];
                ASTree dimsAST=parts[2];
                int dims=getDimExprsDimenstion(dimsExprAST);
                Integer dimsVal=(Integer)processAST(dimsAST);

                if (dimsVal!=null)
                    dims+=dimsVal.intValue();
                if (dims>0) {
                    ArrayClass arrayClass=((JavaModelPackage)elemType.refImmediatePackage()).getArray();

                    for (int i=0; i<dims; i++)
                        elemType=arrayClass.resolveArray(elemType);
                }
                usesType(elemType,tree);
            }
            for (int i=1;i<parts.length;i++)
                processASTBody(parts[i]);
            return elemType;
        }
        
        ASTree parent=parts[0];
        Object type;
        Token member=(Token)parts[1];
        Variable refField=null;
        
        if (parent!=null) {
            type=processMultiPartId(parent,parent.getSubTrees());
            if (!(type instanceof PrimitiveType)) {
                if (DEBUG) System.out.print("."+member.getValue()); // NOI18N
                type=getMemberOf(type,member);
            } else
                type=null;
        } else {
            type=processIdentifier(member);
        }
        if (type!=null) {
            if (type instanceof Field) {
                refField=(Field)type;
                type=refField.getType();
                usesField(refField,member);
            } else {
                storeSemanticInfo(member,type);
                if (type instanceof LocalVarRefInfo) {
                    type=((LocalVarRefInfo)type).type;
                }
            }
            if (refField!=null) {
                usesField(refField,tree);
            } else
                storeSemanticInfo(tree,type);
        }
        processASTBody(parts[2]);
        return type;
    }

    private Object getMemberOf(Object element,Token id) {
        String member=(String)id.getValue();
        
        if (element instanceof ClassDefinition) {
            ClassDefinition jcls=(ClassDefinition)element;
            Object memberFeature=null;
            
            if (!resolveClassName) {
                memberFeature=Scope.createFieldScope(jcls).lookup(member);
            }
            if (memberFeature==null) {
                memberFeature=Scope.createMemberTypeScope(jcls, this).lookup(member);                
                if (jcls!=null && memberFeature instanceof String)
                    memberFeature=resolveClass((String)memberFeature);
            }
            if (memberFeature==null && resolveClassName) {  // Unresolved class
                memberFeature=jclsClass.resolve(jcls.getName()+"."+member);
            }
            return memberFeature;
        }
        if (element instanceof JavaPackage) {
            String fqn=((JavaPackage)element).getName().concat(".").concat(member); // NOI18N
            JavaClass jcls;
            
            jcls=resolveClass(fqn);
            if (jcls!=null)
                return jcls;
            return getPackage(fqn);
        }
        if (element instanceof Array) {
            if (member.equals("length")) // NOI18N
                return typeClass.resolve("int"); // NOI18N
            return null;
        }
        return null;
    }
    
    private Object resolveName(Token id) {
        Object je;
        Scope typeScope;
        String simpleName=(String)id.getValue();
        
        if (!resolveClassName) {
            Scope varScope=(Scope)variableScope.peek();
        
            je=varScope.lookup(simpleName);
            if (je!=null)
                return je;
        }
        je=staticImpScope.lookup(simpleName);
        if (je!=null)
            return je;
        typeScope=(Scope)typeScopeInfo.peek();
        je=typeScope.lookup(simpleName);
        if (je!=null && je instanceof String)
            je=resolveClass((String)je);
        if (je==null)
            je=resolveClass(simpleName);
        if (je!=null)
            return je;
        je=getPackage(simpleName);
        if (je!=null)
            return je;
        if (DEBUG)
            System.out.println("Unresolved class: "+simpleName); // NOI18N
        return typeClass.resolve(simpleName); // simpleName unresolved
    }
        
    private ClassDefinition resolvePrimaryWithSuper(ASTree primaryTree,boolean hasSuper, int parentTreeType) {
        ClassDefinition jcls;
        
        if (primaryTree!=null) {
            Type type;
            
            if (!hasSuper || parentTreeType==EXPLICIT_CONSTRUCTOR_INVOCATION)
                type=processASTBody(primaryTree);
            else
                type=resolveClassName(primaryTree);
            if (type instanceof ClassDefinition) 
                jcls=(ClassDefinition)type;
            else {
                if (DEBUG)
                    System.out.println("Strange type: "+type+" "+type.getName()); // NOI18N
                jcls=jclsClass.resolveClass("java.lang.Object",false); // NOI18N
            }
        }
        else
            jcls=currentClass;
        if (hasSuper && jcls!=null) {
            jcls=jcls.getSuperClass();
        }
        return jcls;
    }
    
    private List getArgumetsTypes(ASTree argListTree) {
        List parTypes=new ArrayList();
        
        if (argListTree!=null) {
            if (argListTree.getType()==ARGUMENT_LIST) {
                ASTree argsAST[]=argListTree.getSubTrees();
                int i;
                
                for (i=0;i<argsAST.length;i++)
                    addArgumentToList(parTypes,argsAST[i]);
            }
            else
                addArgumentToList(parTypes,argListTree);
        }
        return parTypes;
    }
    
    private Type resolveClassName(ASTree tree) {
        Type type;

        resolveClassName=true;
        type=processASTBody(tree);
        resolveClassName=false;                
        return type;
    }
    
    private void addArgumentToList(List list,ASTree argExpr) {
        Type argType=processASTBody(argExpr);
        
        if (argType==null)
            argType=typeClass.resolve("void"); // NOI18N
        list.add(argType);
    }
    
    
    private int getDimExprsDimenstion(ASTree dimexprs) {
        if (dimexprs==null)
            return 0;
        if (dimexprs.getType()==DIM_EXPRS)
            return dimexprs.getSubTrees().length;
        return 1;
    }
    
    private Type computeType(Type left,Type right) {
        if (left==null) return right;
        if (right==null) return left;
        if ((left instanceof PrimitiveType) && (right instanceof PrimitiveType)) {
            PrimitiveTypeKind leftKind=((PrimitiveType)left).getKind();
            PrimitiveTypeKind rightKind=((PrimitiveType)right).getKind();
            
            if (leftKind.equals(PrimitiveTypeKindEnum.DOUBLE)) return left;
            if (rightKind.equals(PrimitiveTypeKindEnum.DOUBLE)) return right;
            if (leftKind.equals(PrimitiveTypeKindEnum.FLOAT)) return left;
            if (rightKind.equals(PrimitiveTypeKindEnum.FLOAT)) return right;
            if (leftKind.equals(PrimitiveTypeKindEnum.LONG)) return left;
            if (rightKind.equals(PrimitiveTypeKindEnum.LONG)) return right;
            return typeClass.resolve("int"); // NOI18N
        }
        ErrorManager.getDefault().log(ErrorManager.WARNING,"Invalid types "+left.getName()+" "+right.getName());
        return null;
    }
    
    private void usesField(Variable field,ASTree id) {        
        storeSemanticInfo(id,field);        
    }
    
    private void usesType(Type td,ASTree id) {
        storeSemanticInfo(id,td);
    }
    
    private CallableFeature findMethod(ClassDefinition jcls,String name,List parameters) {
        if (jcls!=null && !(jcls instanceof UnresolvedClass)) {
            Object methods=null;
            
            if (name==null) {     // construtor
                Object features[]=jcls.getContents().toArray();
                List constructors=new ArrayList();
                
                for (int i=0;i<features.length;i++) {
                    Object feature=features[i];
                    
                    if (feature instanceof Constructor) {
                        constructors.add(feature);
                        methods=feature;
                    }
                }
                if (constructors.size()>1)
                    methods=constructors;
            } else {
                MethodScope methodScope=Scope.createMethodScope(jcls);
            
                methods=methodScope.lookup(name);
            }
            if (methods instanceof List) {
                List methodList=(List)methods;
                CallableFeature[] methodArr=(CallableFeature[])methodList.toArray(new CallableFeature[methodList.size()]);
                CallableFeature closest=methodArr[0];
                Type[] parTypes=(Type[])parameters.toArray(new Type[parameters.size()]);
                int parDiff=computeParDiff(closest,parTypes.length);
                int parDistance=-1;
                int i;
                boolean isAccessible=false;
                
                if (parDiff==0)
                    isAccessible=isAccessible(closest);
                for (i=1;i<methodArr.length;i++) {
                    int localParDist;
                    CallableFeature candidate=methodArr[i];
                    int localParDiff=computeParDiff(candidate,parTypes.length);

                    if (localParDiff>0) {
                        if (localParDiff<parDiff) {
                            parDiff=localParDiff;
                            closest=candidate;
                        }
                        continue;
                    }
                    if (parDiff>0) {
                        parDiff=0;
                        closest=candidate;
                        isAccessible=isAccessible(candidate);
                        continue;
                    }
                    if (!isAccessible(candidate)) 
                        continue;                    
                    if (!isAccessible) {
                        closest=candidate;
                        isAccessible=true;
                        continue;
                    }
                    if (parDistance==-1) {
                        parDistance=isApplicable(closest,parTypes);
                        if (parDistance==0)
                            return closest;
                    }
                    localParDist=isApplicable(candidate,parTypes);
                    if (localParDist==0)
                        return candidate;
                    if (localParDist==Integer.MAX_VALUE)
                        continue;
                    if (localParDist<parDistance) {
                        parDistance=localParDist;
                        closest=candidate;
                        continue;
                    }
                    if (localParDist==parDistance) {
                        int canDiff=isApplicable(closest,candidate);
                        
                        if (canDiff==Integer.MAX_VALUE)
                            continue;
                        if (canDiff>0) {
                            closest=candidate;
                            continue;
                        }
                        if (!Modifier.isAbstract(candidate.getModifiers()) && Modifier.isAbstract(closest.getModifiers())) {
                            closest=candidate;
                            continue;
                        }
                    }
                }
                return closest;
            }
            return (CallableFeature)methods;
        }
        return null;
    }
    
    private int isApplicable(CallableFeature method,CallableFeature bestSoFar) {
        List parTypes=new ArrayList();
        Object[] pars=bestSoFar.getParameters().toArray();
        
        for (int i=0;i<pars.length;i++) {
            Parameter p=(Parameter)pars[i];
            
            parTypes.add(p.getType());
        }
        return isApplicable(method,(Type[])parTypes.toArray(new Type[parTypes.size()]));
    }
    
    private int isApplicable(CallableFeature method,Type[] pars) {
        Object mpars[]=method.getParameters().toArray();
        int p,diff=0;
        boolean varArg=false;
        Type methodType=null;
        
        for (p=0;p<pars.length;p++) {
            int localDiff;
            
            if (p<mpars.length) {
                Parameter par=(Parameter)mpars[p];
                methodType=par.getType();
            }
            localDiff=subType(pars[p],methodType);
            if (localDiff==Integer.MAX_VALUE)
                return localDiff;
            diff+=localDiff;
        }
        return diff;
    }
    
    private int subType(Type sub,Type type) {
        if (sub==type)
            return 0;
        if (type.getName().equals(sub.getName()))
            return 0;
        if (sub instanceof ClassDefinition && type instanceof ClassDefinition)
            return ((ClassDefinition)sub).isSubTypeOf((ClassDefinition)type)?1:Integer.MAX_VALUE;
        return Integer.MAX_VALUE;
    }
    
    private int computeParDiff(CallableFeature method,int parNum) {
        Object[] mpars=method.getParameters().toArray();
        int plen=mpars.length;
        boolean varArg=false;
        
        if (plen>0) {
            Parameter lastPar=(Parameter)mpars[plen-1];
            varArg=lastPar.isVarArg();
        }
        if (varArg && (plen<parNum || plen==(parNum+1)))
            return 0;
        return Math.abs(parNum-plen);
    }
    
    private boolean isAccessible(CallableFeature method) {
        int mods=method.getModifiers();
        boolean samePackage;
        
        if (Modifier.isPublic(mods))
            return true;
        if (Modifier.isPrivate(mods)) {
            return method.getResource().equals(getResource());
        }
        samePackage=method.getResource().getPackageName().equals(getResource().getPackageName());
        if (!Modifier.isProtected(mods))
            return samePackage;
        //protected
        if (samePackage)
            return true;
        return currentClass.isSubTypeOf(method.getDeclaringClass());
    }
    
    private Type usesMethod(String name,ClassDefinition desc,List parameters,ASTree id,boolean implicitThis) {
        CallableFeature ref=findMethod(desc,name,parameters);
        Type type=null;
        
        if (ref==null && implicitThis && desc!=null) {
            if (desc instanceof JavaClass) 
                type=usesMethod(name,((JavaClass)desc).getDeclaringClass(),parameters,id,true);
            else {
                Element immComposite=desc;
                
                do {
                    immComposite=(Element)immComposite.refImmediateComposite();
                } while (immComposite!=null && !(immComposite instanceof ClassDefinition));
                type=usesMethod(name,(ClassDefinition)immComposite,parameters,id,true);
            }
            if (type==null && name!=null) { // static import
                String className=(String)staticImpScope.lookup(name.concat("(")); // NOI18N
                
                if (className!=null) {
                    JavaClass jcls=resolveClass(className);
                    
                    if (jcls!=null)
                        return usesMethod(name,jcls,parameters,id,false);
                }
            }
            return type;
        }
        if (DEBUG) {
            if (desc instanceof JavaClass) System.out.print("Class "+desc.getName()+"."+name+"()"); // NOI18N
            if (ref!=null)
                System.out.println(" resolved to "+ref.getDeclaringClass().getName()+"."+ref.getName()+"()"); // NOI18N
            else 
                System.out.println(" unresloved"); // NOI18N
        }
        if (ref!=null) {
            storeSemanticInfo(id,ref);
            type=ref.getType();
        }
        return type;
    }

    private void storeSemanticInfo(ASTree tree,Object ref) {
        Object indirectRef;
        
        if (tree==null || ref==null || semanticInfo.get(tree)!=null)
            return;
        if (ref instanceof Element) 
            indirectRef=getElementId((Element)ref);
        else
            indirectRef=ref;
        semanticInfo.put(tree,indirectRef);
    }

    private Object getElementId(Element el) {
        String name=null;
        
        if (el instanceof ParameterizedType) {
            return getElementId(((ParameterizedType)el).getDefinition());
        }
        if (el instanceof ParameterizedTypeImpl.Wrapper) {
            ParameterizedTypeImpl.Wrapper w=(ParameterizedTypeImpl.Wrapper)el;
            return getElementId((Element)w.getWrappedObject());
        }
        // todo: MaM = make this code cleaner (these if statements are hard to read), should add LocalRef for referencing local classes
        if ((el instanceof UnresolvedClass) || (el instanceof TypeParameter) || (el instanceof Type && (!(el instanceof ClassDefinition && el instanceof MetadataElement && ((MetadataElement) el).isTransient())))) {
            SemiPersistentElement spel=(SemiPersistentElement)currentClass;
            
            if (spel==null)
                spel=(SemiPersistentElement)getResource();
            return spel.typeToTypeRef((Type)el);
        }
        if (el instanceof Parameter || (el instanceof MetadataElement && ((MetadataElement)el).isTransient())) {
            return "^".concat(el.refMofId()); // NOI18N
        }
        if (el instanceof NamedElement) {
            name=((NamedElement)el).getName();
        }
        if (el instanceof JavaPackage) {
            return "*".concat(name); // NOI18N
        }
        if (el instanceof Field) {
            Field f=(Field)el;
            return new FieldRefInfo((TypeRef)getElementId(f.getDeclaringClass()),name,f.refMofId());
        }
        if (el instanceof CallableFeature) {
            CallableFeature fe=(CallableFeature)el;
            Object[] pars=fe.getParameters().toArray();
            TypeRef parTypes[]=new TypeRef[pars.length];
            int i;
            
            for (i=0;i<parTypes.length;i++) {
                Parameter p=(Parameter)pars[i];
                
                parTypes[i]=(TypeRef)getElementId(p.getType());
            }
            return new CallableRefInfo((TypeRef)getElementId(fe.getDeclaringClass()),name,parTypes,fe.refMofId());
        }
        JMManager.getLog().log("Invalid type "+el); // NOI18N
        if (el instanceof NamedElement)
            JMManager.getLog().log("Name "+name); // NOI18N
        return null;
    }
    
    private static class FieldRefInfo {
        private TypeRef declClass;
        private String name;
        private String mofid;
        
        FieldRefInfo(TypeRef decl,String n,String id) {
            declClass=decl;
            name=n;
            mofid=id;
        }
    }
    
    private static class CallableRefInfo extends FieldRefInfo {
        private TypeRef[] parTypes;
        
        CallableRefInfo(TypeRef decl,String n,TypeRef[] pt,String id) {
            super(decl,n,id);
            parTypes=pt;
        }
    }
    
    private static class LocalVarRefInfo {
        private Type type;
        private ASTree varDeclAST;
        private Variable var;
        
        LocalVarRefInfo(ASTree decl,Type t) {
            varDeclAST=decl;
            type=t;
        }
    }
    
    private static class SuperInfo {
        private JavaClass superClass;
        private List interfaces;
        
        SuperInfo(JavaClass jcls,List ifaces) {
            superClass=jcls;
            interfaces=ifaces;
        }
    }
    
    private static class ModifiersInfo {
        private int modifiers;
        private AnnotationInfo[] annotations;
        
        ModifiersInfo(int mods,AnnotationInfo[] anns) {
            modifiers=mods;
            annotations=anns;
        }
    }
 }
