/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.parser;

import java.io.FileNotFoundException;
import java.io.Reader;
import org.netbeans.lib.java.parser.CompilerException;
import org.netbeans.lib.java.parser.ECRequestDesc;
import org.netbeans.lib.java.parser.ErrConsumer;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import java.util.*;
import java.net.URL;
import java.io.File;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.queries.SourceForBinaryQuery;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.loaders.DataObject;


/**
 *
 * @author  Pavel Flaska
 */
public class ECRequestDescImpl implements ECRequestDesc {
    private final String sourcePath;
    private final String classPath;
    private final String bootPath;
    private final String fileName;
    private final ASTProvider provider;
    private final ErrConsumer consumer;
    private Map modifiedMap;

    /** Creates a new instance of ECRequestDescImpl */
    public ECRequestDescImpl(String name, ASTProvider provider, ErrConsumer consumer) throws CompilerException {
        this.provider = provider;
        this.consumer = consumer;
        fileName = name;

        StringBuffer compileRoots=new StringBuffer(241);
        StringBuffer sourceRoots=new StringBuffer(246);
        FileObject fo=provider.getFileObject();
        ClassPath ccp = ClassPath.getClassPath (fo, ClassPath.COMPILE);
        ClassPath ecp = ClassPath.getClassPath (fo, ClassPath.EXECUTE);
        ClassPath scp = ClassPath.getClassPath (fo, ClassPath.SOURCE);
        if (!fo.isValid()) {
            throw new CompilerException(new RuntimeException("FileObject " + fo.getPath() + " is no more valid. EC request cancelled.")); //NOI18N
        }
        ClassPath extCcp = getCompileClassPathWithCompiledSrc (ccp, ecp, scp);
        Set cprootsSet = new HashSet();
        getCompileAndSourcePath(extCcp,sourceRoots,compileRoots, cprootsSet);
        getSourcePath(scp, sourceRoots, cprootsSet);
        sourcePath=sourceRoots.toString();
        classPath=compileRoots.toString();
        bootPath=getClassPathString(ClassPath.getClassPath(fo, ClassPath.BOOT));
    }
    
    private Map getModifiedMap() {
        // initialize map of modified JavaDataObjects that are on the classpath
        if (modifiedMap == null) {
            modifiedMap = new HashMap();
            DataObject[] dobjs = DataObject.getRegistry().getModified();
            for (int i = 0; i < dobjs.length; i++) {
                FileObject fobj = dobjs[i].getPrimaryFile();
                if (fobj != null && fobj.isValid()) {
                    String filename = fobj.getPath();
                    modifiedMap.put(filename, dobjs[i]);
                }
            }
        }
        return modifiedMap;
    }
    
    public String getBootClassPath() {
        return bootPath;
    }
    
    public String getClassPath() {
        return classPath;
    }
    
    public ErrConsumer getErrConsumer() {
        return consumer;
    }
    
    public String getFileName() {
        return fileName;
    }
    
    public Reader getReader() {
        try {
            return provider.getFileReader(false);
        } catch (FileNotFoundException fnfe) {
            //#63694
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, fnfe);
            return null;
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ex);
            return null;
        }
    }
    
    public String getSourceClassPath() {
        return sourcePath;
    }
    
    public String getSourceLevel() {
        return provider.getSourceLevel();
    }
    
    private void getCompileAndSourcePath(ClassPath classPath, StringBuffer sourceRoots, StringBuffer compileRoots, Set cprootsSet) {
        if (classPath != null) {
            for (Iterator it = classPath.entries().iterator(); it.hasNext(); ) {
                ClassPath.Entry entry = (ClassPath.Entry)it.next();
                FileObject root = entry.getRoot();
                if (root != null) {
                    cprootsSet.add(root);
                    getFileName(root,compileRoots);
                }
                FileObject[] sRoots = SourceForBinaryQuery.findSourceRoots(entry.getURL()).getRoots();
                for (int x = 0; x < sRoots.length; x++) {
                    FileObject sRoot=sRoots[x];
                    
                    if (!cprootsSet.contains(sRoot)) {
                        cprootsSet.add(sRoot);
                        getFileName(sRoot,sourceRoots);
                    } // if
                } // for
            } // for
        } // if
    }
    
    private void getSourcePath(ClassPath classPath, StringBuffer sourceRoots, Set cprootsSet) {
        if (classPath != null) {
            for (Iterator it = classPath.entries().iterator(); it.hasNext(); ) {
                ClassPath.Entry entry = (ClassPath.Entry)it.next();
                FileObject root = entry.getRoot();
                if (root != null && cprootsSet.add(root)) {
                    getFileName(root,sourceRoots);
                }
            } // for
        } // if
    }

    void getFileName(FileObject fo,StringBuffer buf) {
        try {
            URL url=fo.getURL();
            if (url.getProtocol().equals("jar")) { // NOI18N
                fo = FileUtil.getArchiveFile(fo);
            }
            File f = FileUtil.toFile(fo);
            
            if (buf.length()>0)
                buf.append(File.pathSeparatorChar);
            buf.append(f.getAbsolutePath());
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }
    
    String getClassPathString(ClassPath cp) {
        FileObject[] roots = cp.getRoots();
        if (roots.length == 0)
            return "";
        StringBuffer buf = new StringBuffer(237);
        for (int i = 0; i < roots.length; i++) {
            getFileName(roots[i],buf);
        }
        return buf.toString();
    }

    public Reader getReader(String filename) {
        filename = filename.replace('\\', '/');
        if (filename.startsWith("/")) { // NOI18N
            filename = filename.substring(1);
        }
        Reader reader = null;
        FileObject fo;
        DataObject dobj = (DataObject) getModifiedMap().get(filename);
        if (dobj == null) {
            // EC is asking reader for a file that is not modified
            fo=provider.getFileObject();
            ClassPath scp = ClassPath.getClassPath (fo, ClassPath.SOURCE);
            FileObject[] roots = scp.getRoots();
            for (int i = 0; i < roots.length; i++) {
                String root = roots[i].getPath();
                if (filename.startsWith(root)) {
                    filename = filename.substring(root.length() + 1);
                    break;
                }
            }
            fo = scp.findResource(filename);
        } else {
            fo = dobj.getPrimaryFile();
        }
        if (fo != null) {
            try {
                ASTProvider provider = new ASTProvider(null, fo);
                reader = provider.getFileReader(false);
            } catch (Exception e) {
                // ignore -> reader will be null
            }
        }
        return reader;
    }
    
    public boolean isModified(String filename) {
        filename = filename.replace('\\', '/');
        if (filename.startsWith("/")) { // NOI18N
            filename = filename.substring(1);
        }
        return getModifiedMap().containsKey(filename);
    }

    /**
     * Creates extended copile Classpath, the classpath is
     * ClassPath.COMPILE + build destination folder(s)
     * @param ccp - ClassPath.COMPILE
     * @param ecp - ClassPath.EXECUTE
     * @param scp - ClassPath.SOURCE
     * @return ClassPath
     */
    private static ClassPath getCompileClassPathWithCompiledSrc (final ClassPath ccp, final ClassPath ecp, final ClassPath scp) {
        assert scp != null : "Source path can not be null"; //Every Java file should have source path // NOI18N
        List buildRoots;
        if (ecp == null) {
            buildRoots = Collections.EMPTY_LIST;
        } else {
            buildRoots = new ArrayList (5);    //Mostly 1
            for (Iterator it = ecp.entries().iterator(); it.hasNext();) {
                URL url = ((ClassPath.Entry)it.next()).getURL();
                FileObject[] sourceRoots = SourceForBinaryQuery.findSourceRoots(url).getRoots();
                for (int i=0; i<sourceRoots.length; i++) {
                    if (scp.contains (sourceRoots[i])) {
                        buildRoots.add (ClassPathSupport.createResource(url));
                        break;
                    }
                }
            }
        }
        if (buildRoots.size()==0) {
            //No EXEC classpath for sources (DefaultClassPathProvider)
            //return just COMPILE classpath
            return ccp;
        } else {
            //Return COMPILE + buildRoots
            if (ccp == null) {
                //Not common case
                //but may be caused by some strange ClassPathProvider
                return ClassPathSupport.createClassPath (buildRoots);
            } else {
                return ClassPathSupport.createProxyClassPath( new ClassPath[] {
                    ccp,
                    ClassPathSupport.createClassPath (buildRoots),
                });
            }
        }
    }
}
