/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.parser;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.netbeans.lib.java.parser.*;

/**
 *
 * @author Martin Matula, Tomas Hurka
 */
public class ASTRepairer implements ASTreeTypes, ParserTokens {
    private final ASTProvider provider;
    private final JParser parser;
    
    ASTRepairer(ASTProvider provider, JParser parser) {
        this.parser = parser;
        this.provider = provider;
    }
    
    ASTree fixTree() {
        ASTree compUnit = parser.getASTree();
             
        if (compUnit != null && compUnit.getType() == COMPILATION_UNIT && compUnit.getSubTrees() != null) {
            fixTree(compUnit);
        } else {
            // let's create a fake AST
            compUnit = new ASTreeNode(COMPILATION_UNIT, 0, provider.getTokens().length - 1, new ASTree[] {null, null, null}); 
        }
        consistencyCheck(compUnit);
        return compUnit;
    }
    
    private void consistencyCheck(ASTree topLevel) {
        final int S_NORMAL = 0;
        final int S_TOP_TYPE = 1;
        final int S_PACKAGE = 2;
        
        ASTree typeDecls = topLevel.getSubTrees()[2];
        int index = 0;
        int blockDepth = 0;
        int state = S_NORMAL;
        int lastTokenType = 0;
        boolean inTopType = false;
        
        Token[] tokens = provider.getTokens();
        for (int i = 0; i < tokens.length; i++) {
            int tokenType = tokens[i].getType();
            switch (state) {
                case S_NORMAL:
                    switch (tokenType) {
                        case CLASS:
                        case INTERFACE:
                        case ENUM:
                            if (lastTokenType != ParserTokens.DOT && blockDepth == 0 && !inTopType) {
                                state = S_TOP_TYPE;
                            }
                            break;
                        case L_CURLY:
                            blockDepth++;
                            break;
                        case R_CURLY:
                            if (blockDepth > 0) {
                                blockDepth--;
                                if (blockDepth == 0) inTopType = false;
                            }
                            break;
                        case PACKAGE:
                            state = S_PACKAGE;
                            break;
                    }
                    break;
                case S_PACKAGE:
                    if (tokenType == SEMICOLON) state = S_NORMAL;
                    break;
                case S_TOP_TYPE:
                    if (tokenType == IDENTIFIER) {
                        int type;
                        if (lastTokenType == INTERFACE) {
                            if (i - 2 > 0 && tokens[i - 2].getType() == MONKEYS_AT) {
                                type = ANNOTATION_TYPE_DECLARATION;
                            } else {
                                type = INTERFACE_DECLARATION;
                            }
                        } else if (lastTokenType == ENUM) {
                            type = ENUM_DECLARATION;
                        } else {
                            type = CLASS_DECLARATION;
                        }
                        index = getNextChildIndex(typeDecls, index);
                        ASTree typeAST = getChild(typeDecls, index);
                        if (typeAST == null || typeAST.getType() != type || typeAST.getSubTrees()[1].getFirstToken() != i) {
                            typeDecls = addToTypeDecls(typeDecls, index, type, tokens[i], i);
                        }
                        index++;
                    }
                    state = S_NORMAL;
                    inTopType = true;
                    break;
            }
            lastTokenType = tokenType;
        }
        topLevel.getSubTrees()[2] = trimTypeDecls(typeDecls, index);
    }
    
    private ASTree trimTypeDecls(ASTree typeDecls, int index) {
        if (index > 0) {
            ASTree[] children = typeDecls.getSubTrees();
            if (children.length > index) {
                ASTree[] newChildren = new ASTree[index];
                System.arraycopy(children, 0, newChildren, 0, index);
                typeDecls = new ASTreeNode(TYPE_DECLARATIONS, newChildren[0].getFirstToken(), newChildren[index - 1].getLastToken(), newChildren);
            }
            return typeDecls;
        } else {
            return null;
        }
    }
    
    private ASTree addToTypeDecls(ASTree typeDecls, int index, int type, Token token, int lastTokenIndex) {
        ASTree[] children = typeDecls == null ? new ASTree[1] : typeDecls.getSubTrees();
        
        assert children.length >= index;
        
        if (children.length == index) {
            ASTree[] newChildren = new ASTree[index + 1];
            System.arraycopy(children, 0, newChildren, 0, children.length);
            children = newChildren;
        }
        
        int size;
        switch (type) {
            case CLASS_DECLARATION:
            case INTERFACE_DECLARATION:
                size = 6;
                break;
            case ENUM_DECLARATION:
                size = 4;
                break;
            case ANNOTATION_TYPE_DECLARATION:
                size = 3;
                break;
            default:
                throw new IllegalArgumentException("Unexpected ASTree type: " + type); // NOI18N
        }
        ASTree[] subtrees = new ASTree[size];
        subtrees[1] = token;
        children[index] = new ASTreeNode(type, lastTokenIndex - 1, lastTokenIndex, subtrees);
        
        if (typeDecls == null || typeDecls.getSubTrees() != children) {
            typeDecls = new ASTreeNode(TYPE_DECLARATIONS, children[0].getFirstToken(), children[children.length - 1].getLastToken(), children);
        }
        
        return typeDecls;
    }
    
    private ASTree getChild(ASTree tree, int index) {
        if (tree == null || tree.getSubTrees().length <= index) {
            return null;
        }
        return tree.getSubTrees()[index];
    }
    
    private int getNextChildIndex(ASTree tree, int index) {
        if (tree == null) {
            return index;
        }
        ASTree[] subTrees = tree.getSubTrees();
        while (subTrees.length > index && subTrees[index].getType() == SEMICOLON) {
            ++index;
        }
        return index;
    }
    
    private ASTree fixTree(ASTree tree) {
        if (tree == null) return null;
        int type = tree.getType();
        ASTree parts[] = tree.getSubTrees();
        
        switch (type) {
            case ENUM_BODY:
            case COMPILATION_UNIT: {
                fixChildren(parts);
                break;
            } case ENUM_CONSTANT: {
                if (parts == null) return null;
                parts[1] = parts[2] = null;
            } case TYPE_PARAMETER: 
            case SUPER_:
            case CONSTRUCTOR_DECLARATOR: {
                fixChildren(parts);
                if (parts == null || parts[0] == null || (parts[0].getType() != IDENTIFIER && parts[0].getType() != MULTI_PART_ID)) {
                    return null;
                }
                break;
            } case PACKAGE_DECLARATION: {
                fixChildren(parts);
                if (parts == null || parts[1] == null || (parts[1].getType() != IDENTIFIER && parts[1].getType() != MULTI_PART_ID)) {
                    return null;
                }
                break;
            } case PRIMITIVE_TYPE: {
                fixChildren(parts);
                if (parts == null || parts[0] == null) {
                    return null;
                }
                switch (parts[0].getType()) {
                    case BOOLEAN:
                    case BYTE:
                    case CHAR:
                    case DOUBLE:
                    case FLOAT:
                    case INT:
                    case LONG:
                    case SHORT:
                    case VOID:
                        break;
                    default:
                        return null;
                }
                break;
            } case METHOD_DECLARATOR: {
                fixChildren(parts);
                if (parts == null || parts[0] == null || parts[0].getType() != IDENTIFIER) {
                    return null;
                }
                break;
            } case MULTI_PART_ID: {
                if (parts == null) return null;
                fixChildren(parts);
                if ((parts[0] != null) && (parts[0].getType() != IDENTIFIER) && (parts[0].getType() != MULTI_PART_ID)) {
                    return null;
                }
                if (parts[1] == null || parts[1].getType() != IDENTIFIER) {
                    return null;
                }
                break;
            } case SINGLE_TYPE_IMPORT:
            case TYPE_IMPORT_ON_DEMAND: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[1] == null) {
                    return null;
                }
                break;
            } case REFERENCE_TYPE: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[0] == null || parts[1] == null) {
                    return null;
                }
                break;
            } case FIELD_DECLARATION:
            case ANNOTATION_ATTRIBUTE_DECLARATION: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[1] == null || parts[2] == null) {
                    return null;
                }
                break;
            } case FORMAL_PARAMETER: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[1] == null || parts[3] == null) {
                    return null;
                }
                break;
            } case CLASS_DECLARATION:
            case INTERFACE_DECLARATION: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[1] == null || parts[1].getType() != IDENTIFIER || parts[5] == null) {
                    return null;
                }
                break;
            } case ENUM_DECLARATION: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[1] == null || parts[1].getType() != IDENTIFIER || parts[3] == null) {
                    return null;
                }
                break;
            } case ANNOTATION_TYPE_DECLARATION: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[1] == null || parts[1].getType() != IDENTIFIER || parts[2] == null) {
                    return null;
                }
                break;
            } case CONSTRUCTOR_DECLARATION: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[3] == null || parts[5] == null || parts[5].getType() != BLOCK_STATEMENTS) {
                    return null;
                }
                break;
            } case METHOD_DECLARATION: {
                if (parts == null) return null;
                fixChildren(parts);
                if (parts[2] == null || parts[3] == null || parts[5] == null
                    || (parts[5].getType() != BLOCK_STATEMENTS && parts[5].getType() != SEMICOLON)) {
                    return null;
                }
                break;
            } case VARIABLE_DECLARATORS: {
                ASTree result = fixChildren(tree, type, parts, false);
                if (result != null && result.getSubTrees().length == 1) {
                    result = result.getSubTrees()[0];
                }
                return result;
            } case VARIABLE_DECLARATOR: {
                if (parts == null) return null;
                if (parts.length > 2) parts[2] = null;
                fixChildren(parts);
                if (parts[0] == null || parts[0].getType() != IDENTIFIER) {
                    return null;
                }
                break;
            } case TYPE_ARGUMENTS:
            case TYPE_DECLARATIONS:
            case MODIFIERS:
            case TYPE_PARAMETER_LIST:
            case FORMAL_PARAMETER_LIST:
            case ENUM_CONSTANTS:
            case IMPORT_DECLARATIONS: {
                return fixChildren(tree, type, parts, false);
            }
            case BOUND_LIST:
            case TYPE_LIST: { 
                return fixChildren(tree, type, parts, new int[] {MULTI_PART_ID, IDENTIFIER}, false);
            }
	    case ANNOTATION_TYPE_BODY_DECLARATIONS:
            case ENUM_BODY_DECLARATIONS:
            case INTERFACE_MEMBER_DECLARATIONS:
            case CLASS_BODY_DECLARATIONS: {
                return fixChildren(tree, type, parts, true);
            } case WILDCARD: {
                if (parts == null) return null;
                fixChildren(parts);
                if ((parts[0] == null) != (parts[1] == null)) {
                    return null;
                }
                break;
            } case BLOCK_STATEMENTS: {
                return new ASTreeNode(type, tree.getFirstToken(), tree.getLastToken(), ASTProvider.NULL_TREE);
            } case ERRONEOUS:
            case ANNOTATION:
            case INSTANCE_INITIALIZER:
            case STATIC_INITIALIZER:
            case DEFAULT_VALUE: {
                return null;
            }
        }
        return tree;
    }

    private ASTree fixChildren(ASTree tree, int type, ASTree[] parts, boolean canBeEmpty) {
	return fixChildren(tree, type, parts, null, canBeEmpty);
    }
    
    /** NOTE !! 
     * allowedTypes must be sorted
     */
    private ASTree fixChildren(ASTree tree, int type, ASTree[] parts,int[] allowedTypes, boolean canBeEmpty) {
        if (parts == null) return null;
        int i = 0, j = 0;
        for (; i < parts.length; i++) {
            parts[i] = fixTree(parts[i]);
            if (parts[i] != null) {
                if (allowedTypes==null || Arrays.binarySearch(allowedTypes,parts[i].getType())>=0) {
                    j++;
                } else {
                    parts[i] = null;
                }
            }
        }
        if (i == j) {
            return tree;
        } else if (j == 0) {
            return canBeEmpty ? new ASTreeNode(type, tree.getFirstToken(), tree.getLastToken(), ASTProvider.NULL_TREE) : null;
        } else {
            ASTree[] newParts = new ASTree[j];
            for (i = 0, j = 0; i < parts.length; i++) {
                if (parts[i] != null) {
                    newParts[j++] = parts[i];
                }
            }
            return new ASTreeNode(type, tree.getFirstToken(), tree.getLastToken(), newParts);
        }
    }

    private void fixChildren(ASTree[] parts) {
        if (parts == null) return;
        for (int i = 0; i < parts.length; i++) {
            parts[i] = fixTree(parts[i]);
        }
    }

    private class ASTreeNode implements ASTree {
        private final int type, firstToken, lastToken;
        private final ASTree[] subTrees;
        
        private ASTreeNode(int type, int firstToken, int lastToken, ASTree[] subTrees) {
            this.type = type;
            this.firstToken = firstToken;
            this.lastToken = lastToken;
            this.subTrees = subTrees;
        }

        public int getType() {
            return type;
        }

        public int getFirstToken() {
            return firstToken;
        }

        public int getLastToken() {
            return lastToken;
        }

        public ASTree[] getSubTrees() {
            return subTrees;
        }

        public SymbolInfo getSymbolInfo() {
            return null;
        }

        public ASTContext getASTContext() {
            return provider;
        }
    }
}
