/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.parser;

import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.Position.Bias;
import org.netbeans.api.java.queries.SourceLevelQuery;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.lib.java.parser.*;
import org.netbeans.modules.javacore.JMManager;
import org.openide.ErrorManager;
import org.openide.cookies.EditorCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.text.CloneableEditorSupport;
import org.openide.text.PositionBounds;
import org.openide.text.PositionRef;


/**
 *
 * @author  Tomas Hurka
 */
public class ASTProvider implements ParserTokens, ASTreeTypes, ASTContext {
    FileObject fobj;
    private CloneableEditorSupport editor;
    private ASTree topNode;
    private Token[] tokens;
    private String sourceText;
    private Resource rsc;
    static ASTree[] NULL_TREE=new ASTree[0];
    boolean documentPositions;
    boolean hasSyntaxErrors;
    
    /** Creates a new instance of ASTProvider */
    public ASTProvider(Resource r,FileObject obj) {
        rsc=r;
        fobj = obj;
        if (obj == null) throw new javax.jmi.reflect.InvalidObjectException(r);
    }
    
    protected ASTProvider(Resource r, FileObject obj, String sourceText, boolean isFromDoc) {
        this(r, obj);
        this.sourceText = sourceText;
        this.documentPositions = isFromDoc;
    }

    private void createASTree() {
        Reader r;
        int status=-1;
        hasSyntaxErrors = false;
        
        try {
            r=getReader();
        } catch (Exception ex) {
            JMManager.getLog().log(ErrorManager.WARNING, "Error opening stream for: " + getResource().getName() + "; " + ex.toString());
            ErrorManager.getDefault().notify(ex);
            return;
        }
        JParser parser=Factory.getDefault().getParser(this, r, fobj.getName());
        try {
            if (JMManager.PERF_DEBUG) {
                System.err.println("Parsing file: " + fobj.getName());
                Thread.dumpStack();
            }
            status=parser.parse(false);  // don't generate symbols
        } catch (Exception ex) {
            JMManager.getLog().notify(ErrorManager.INFORMATIONAL, ex);
            dumpSource(ex);
        }
        
	tokens=parser.getTokens();
	if (tokens == null)
	    tokens = new Token[0];
        
        if (status!=0) {
            hasSyntaxErrors = true;
            topNode = new ASTRepairer(this, parser).fixTree();
        } else {
            topNode = parser.getASTree();
        }
    }
    
    Token[] getTokens() {
        return tokens;
    }
    
    public Token getTokenByOffset(int offset) {
        if (offset >= sourceText.length()) return null;
        int lo = 0;
        int hi = tokens.length;
        int check;
        int index;
        while (true) {
            if (lo >= hi) return null;
            index = (hi - lo) / 2 + lo;
            check = checkToken(tokens[index], offset);
            if (check < 0) {
                hi = index;                
            } else if (check > 0) {
                lo = index + 1;
            } else {
                break;
            }
        }
        Token result = tokens[index];
        Token[] paddings = result.getPadding();
        if (paddings != null) {
            for (int i = paddings.length - 1; i >= 0; i--) {
                if (paddings[i].getEndOffset() > offset) {
                    result = paddings[i];
                }
            }
        }
        return result;
    }
    
    private int checkToken(Token token, int offset) {
        Token[] padding = token.getPadding();
        int startOffset = padding != null && padding.length > 0 ? padding[0].getStartOffset() : token.getStartOffset();
        int endOffset = token.getEndOffset();
        if (offset < startOffset) return -1;
        else if (offset >= endOffset) return 1;
        else return 0;
    }

    public ASTree getASTree() {
        //if (!dobj.isValid())
        //workaround for issue #54232
        if (!fobj.isValid())
            return null;
        if (topNode==null)
            createASTree();
        return topNode;
    }

    public boolean hasSyntaxError() {
        return hasSyntaxErrors;
    }
    
    FileObject getFileObject() {
        return fobj;
    }

    String getRealSource(boolean overridePositions) throws FileNotFoundException, UnsupportedEncodingException, IOException {
        EditorCookie editor;
        Document doc=null;
        DataObject dobj=Util.getModifiedDataObject(fobj);
        
        if (dobj!=null) {
            editor = (EditorCookie) dobj.getCookie(EditorCookie.class);
            if (editor!=null) {
                doc = editor.getDocument();
            }
            assert doc!=null:"Modified DO "+fobj.getPath()+" without document";
            if (overridePositions)
                documentPositions = true;
            // loading from the memory (Document)
            final String[] str = new String[1];
            // safely take the text from the document
            final Document doc2 = doc;
            Runnable run = new Runnable() {
                public void run() {
                    try {
                        str[0] = doc2.getText(0, doc2.getLength());
                    }
                    catch (BadLocationException e) {
                        // impossible
                    }
                }
            };
            doc.render(run);
            return str[0];
        } else {
            // loading from the file
            InputStream is = fobj.getInputStream();
            Reader fileReader;
            String encoding=Util.getFileEncoding(fobj);

            if (encoding == null) {
                fileReader = new InputStreamReader(is);
            } else {
                fileReader = new InputStreamReader(is, encoding);
            }
            try {
                return Util.readContents(fileReader,fobj.getSize());
            } finally {
                fileReader.close();
            }
        }
    }

    Reader getFileReader(boolean overridePositions) throws FileNotFoundException, UnsupportedEncodingException, IOException {
        return new StringReader(getRealSource(overridePositions));
    }

    /** Returns contents of DataObject obj. If the object is opened in the editor,
     * the function returns current contents of the edited document.
     * If the file is not opened in a JavaEditor, it is read from the disk and
     * guarded sections are filtered out.
     * @return contents of the file/editor document; guarded section markers are filtered out.
     *
     */
    public Reader getReader() throws FileNotFoundException, UnsupportedEncodingException, IOException {
        if (sourceText==null) {
            sourceText = getRealSource(true);
        }
        return new StringReader(sourceText);
    }
    
    public Resource getResource() {
        return rsc;
    }
    
    public String getJavaDoc(ASTree tree) {
        String javadoText=null;
        Token javaDocToken=getComment(tree);

        if (javaDocToken.getType()==DOC_COMMENT) {
            javadoText=getText(javaDocToken);
        }
        return removeJavadocStars(javadoText);
    }

    public ASTree findTree(ASTree parentTree, int firstToken, int lastToken, int type) {
        ASTree[] children = parentTree.getSubTrees();

        for (int i = 0; i < children.length; i++) {
            ASTree ch = children[i];
            if (ch != null) {
                int first = ch.getFirstToken();
                int last = ch.getLastToken();
                if (first == firstToken && last == lastToken && type == ch.getType()) {
                    return ch;
                }
                if (first <= firstToken && last >= lastToken) {
                    return findTree(ch, firstToken, lastToken, type);
                }
            }
        }
        String addMsg = ""; // NOI18N
        if (firstToken == lastToken) {
            Token tk = getToken(firstToken);
            if (tk != null) {
                addMsg = " found type: " + tk.getType(); // NOI18N
                try {
                    addMsg = addMsg + " found source text: " + getSourceText().substring(tk.getStartOffset(), tk.getEndOffset()); // NOI18N
                } catch (IndexOutOfBoundsException e) {
                }
            }
        }
        throw new IllegalArgumentException("Child tree not found (type: " + type + " firstToken: " + firstToken + " lastToken: " // NOI18N
                + lastToken + " resource: " + getResource().getName() + addMsg + ")"); // NOI18N
    }

    public Token getComment(ASTree t) {
        Token first=getToken(t.getFirstToken());
        Token pad[]=first.getPadding();
        int i;
        
        for(i=pad.length-1;i>=0;i--) {
            Token c=pad[i];
            
            if (c.getType()==DOC_COMMENT)
                return c;
        }
        return first;
    }
    
    ASTree[] check(ASTree tree) {
        if (tree==null || tree.getType()==SEMICOLON)
            return null;
        return NULL_TREE;
    }
    
    public ASTree[] filterParts(ASTree parts[]) {
        if (parts!=null) {
            List typeDecl=null;
            int i;
            
            for (i=parts.length-1;i>=0;i--) {
                ASTree tree=parts[i];
                ASTree checked[]=check(tree);
                
                if (checked!=NULL_TREE) {
                    if (typeDecl==null) {
                        typeDecl=new ArrayList(Arrays.asList((Object[]) parts));
                    }
                    if (checked==null)
                        typeDecl.remove(i);
                    else
                        typeDecl.addAll(i, Arrays.asList((Object[]) checked));
                }
            }
            if (typeDecl!=null) {
                return (ASTree[])typeDecl.toArray(new ASTree[typeDecl.size()]);
            }
        }
        return parts;
    }

    public String getText(ASTree tree) {
        if (tree == null) {
            return null;
        } else {
            return getText(tree,tree);
        }
    }
    
    public String getText(ASTree first,ASTree last) {
        int start=getToken(first.getFirstToken()).getStartOffset();
        int end=getToken(last.getLastToken()).getEndOffset();

        return sourceText.substring(start,end);
    }
    
    public String getText(Token token) {
        int start=token.getStartOffset();
        int end=token.getEndOffset();

        return sourceText.substring(start,end);
    }

    public Token getToken(int index) {
	return (index >= 0 && index < tokens.length) ? tokens[index] : null;
    }
    
    public ASTree getParent(ASTree tree) {
        ASTree parent=getASTree();
        int index=tree.getFirstToken();
        
        while(true) {
            ASTree parts[]=parent.getSubTrees();
            ASTree element=null;
            int i;
        
            for (i=0;i<parts.length;i++) {
                element=parts[i];
                if (element!=null && element.getFirstToken()<=index && element.getLastToken()>=index)
                    break;
            }
            if (i==parts.length)
                return parent;
            if (element==tree)
                return parent;
            parent=element;
        }
    }
    
    public boolean isFromDocument() {
        return documentPositions;
    }
    
    /**
     * Creates bounds for two ASTree elements. First tree represents begining,
     * second one represents end of the created bounds. If inclPads is true,
     * then begining is counted from the paddings of the first tree.
     *
     * @param startTree  start offset of first token of the tree is begin
     *                   position reference offset. It is valid only in case 
     *                   that inclDoc is false. 
     *                   Otherwise it takes the first token of javadoc comment.
     * @param endTree    end offset of last token of the tree is end position
     *                   reference offset.
     * @param inclDoc    if false, then start offset of first token of startTree
     *                   is beginning, otherwise it starts from its javadoc
     *                   comment.
     *
     * @return  element's bounds
     */
    public PositionBounds createBounds(ASTree startTree, ASTree endTree, boolean inclDoc) {
        Token startToken;
        if (inclDoc) {
            startToken = getComment(startTree);
        } else {
            startToken = getToken(startTree.getFirstToken());
        }
        Token endToken=getToken(endTree.getLastToken());
        
        if (startToken == null || endToken == null) {
            RuntimeException ex;
            String startTreeText = null;
            String endTreeText = null;
            try {
                startTreeText = startTree.toString();
                endTreeText = endTree.toString();
            } catch (RuntimeException e) {
                // ignore
            }
            if (startToken == null) {
                ex = new RuntimeException("First token is null for ASTree: " + startTreeText + " of type: " + startTree.getType() + // NOI18N
                    ";\ncalling startTree.getFirstToken() returns " + startTree.getFirstToken()); // NOI18N
            } else {
                ex = new RuntimeException("Last token is null for ASTree: " + endTreeText + " of type: " + endTree.getType() + // NOI18N
                    ";\ncalling endTree.getLastToken() returns " + endTree.getLastToken()); // NOI18N
            }
            ex.printStackTrace();
            dumpSource(ex);
        }
        
        return createBounds(new int[] {startToken.getStartOffset(), endToken.getEndOffset()});
    }
    
    public PositionBounds createBounds(int startOffset,int endOffset) {
        return createBounds(new int[] {startOffset, endOffset});
    }
    
    public PositionBounds createBounds(int[] oldOffsets) {
        if (editor == null) {
            DataObject dobj; 
            try {
                dobj=DataObject.find(fobj);
            } catch (DataObjectNotFoundException ex) {
                ErrorManager.getDefault().notify(ex);
                return null;
            }
            editor = Util.findCloneableEditorSupport(dobj);
        }
        int[] offsets = getDocumentOffsets(oldOffsets);
        
        PositionRef start=editor.createPositionRef(offsets[0], Bias.Forward);
        PositionRef end=editor.createPositionRef(offsets[1], Bias.Backward);
        
        return new PositionBounds(start,end);
    }
    
    // this method changes the array passed as argument
    // offsets must be ordered (from lowest to highest)
    public int[] getDocumentOffsets(int[] offsets) {
        if (!isFromDocument()) {
            return convertToDocumentOffsets(offsets);
        } else {
            return offsets;
        }
    }

    public int[] convertToDocumentOffsets(final int[] offsets) {
        int pos = -1;
        int firstOffset = 0;

        int[] newOffsets = (int[]) offsets.clone();
        int srcLastIdx = sourceText.length()-1;
        while (pos < offsets[offsets.length - 1]) {
            pos = sourceText.indexOf('\r',pos+1);
            if (pos == -1) break;
            if (pos < srcLastIdx && sourceText.charAt(pos+1) != '\n')
                continue;
            while ((firstOffset < offsets.length) && (offsets[firstOffset] <= pos)) {
                firstOffset++;
            }
            for (int i = firstOffset; i < offsets.length; i++) {
                newOffsets[i]--;
            }
        }
        return newOffsets;
    }

    public String getSourceText() {
        if (sourceText == null) {
            try {
                getReader();
            } catch (Exception ex) {
                JMManager.getLog().log(ErrorManager.WARNING, "Error opening stream for: " + getResource().getName() + "; " + ex.toString());
                ErrorManager.getDefault().notify(ex);
            }
        }
        return sourceText;
    }

    /** pattern for removing leading stars from javadoc comment */
    private static final Pattern docRegExp = Pattern.compile("\\r?\\n(\\s*\\*+)?"); // NOI18N
    
    private static String removeJavadocStars(String rawText) {
        if (rawText==null)
            return null;
        rawText="\n".concat(rawText.substring(1,rawText.length()-2)); // NOI18N
        String[] lines=docRegExp.split(rawText,Integer.MAX_VALUE);
        StringBuffer purgedComment = new StringBuffer(rawText.length());
        for (int i=1; i<lines.length; i++) {
            if ((i+1)==lines.length) {
                if (lines[i].trim().length() != 0) {
                    purgedComment.append(lines[i]);
                }
            }
            else {
                purgedComment.append(lines[i]).append('\n'); // NOI18N
            }
        }
        return purgedComment.toString();
    }

    // ASTContext methods.
    public ASTree getRootTree() {
	return getASTree();
    }

    public String getSourceLevel() {
        return SourceLevelQuery.getSourceLevel(fobj);
    }
    
    public String getClassPath() {
	return null;
    }
    
    public ErrConsumer getErrorConsumer() {
        return null;
    }

    /**
     * Dumps the source code to the file. Used for parser debugging. It uses
     * only five dump files: from source1.dump to source5.dump. If last
     * source5.dump file exists, methods doesn't dump anything.
     *
     * @param  src  source text which causes parser failure
     * @param  exc  exception to write to the end of dump file
     * @param  dumpDir directory where dump file will be stored
     * @return  name of created dump file or null if dump file was not created.
     */
    private void dumpSource(Exception exc) {
        String dumpDir = System.getProperty("netbeans.user") + "/var/log/"; //NOI18N
        String src = getSourceText();
        String rscName = getResource() == null ? "null" : getResource().getName();
        int pos = rscName.lastIndexOf('/');
        String origName;
        if (pos >= 0) {
            origName = rscName.substring(pos + 1);
        } else {
            origName = rscName;
        }
        String name = null;
        File f = new File(dumpDir + origName + ".dump"); // NOI18N
        int i = 1;
        while (i < 255) {
            if (!f.exists())
                break;
            f = new File(dumpDir + origName + '_' + i + ".dump"); // NOI18N
            i++;
        }
        if (!f.exists()) {
            try {
                OutputStream os = new FileOutputStream(f);
                PrintWriter writer = new PrintWriter(new OutputStreamWriter(os, "UTF-8")); // NOI18N
                try {
                    writer.println(src);
                    writer.println("----- Original exception ---------------------------------------------"); // NOI18N
                    exc.printStackTrace(writer);
                    name = f.getName();
                } finally {
                    writer.close();
                }
            } catch (IOException ioe) {
                ErrorManager.getDefault().annotate(ioe, ErrorManager.UNKNOWN, "Error when writing parser dump file!", null, null, null); // NOI18N
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ioe);
                name = null;
            }
        }
        if (name != null) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, new RuntimeException("Invalid AST returned from parser or error parsing \'" + rscName + "\'. Please report a bug against java module and attach dump file '"  // NOI18N
                + dumpDir + name + "'.")); // NOI18N
        } else {
            ErrorManager.getDefault().log(ErrorManager.WARNING,
                "Dump could not be written. Either dump file could not " + // NOI18N
                "be created or all dump files were already used. Please " + // NOI18N
                "check that you have write permission to '" + dumpDir + "' and " + // NOI18N
                "clean all *.dump files in that directory."); // NOI18N
        }
    }
    
    public String toString() {
        return "ASTProvider@" + System.identityHashCode(this) + " for \"" + getResource().getName() + "\"";
    }
}
