/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.*;
import javax.jmi.reflect.InvalidObjectException;
import javax.jmi.reflect.RefClass;
import javax.jmi.reflect.RefFeatured;
import javax.jmi.reflect.RefObject;
import org.netbeans.api.mdr.events.AttributeEvent;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.handlers.AttrCollWrapper;
import org.netbeans.mdr.handlers.AttrListWrapper;
import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.storagemodel.StorableFeatured;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.ExclusiveMutex;
import org.netbeans.modules.javacore.JMManager;

import org.netbeans.modules.javacore.parser.*;
import org.openide.ErrorManager;
import org.openide.util.Utilities;

/**
 * Superclass for the partialy persistent elements.
 *
 * @author  Martin Matula
 * @author  Pavel Flaska
 */
public abstract class SemiPersistentElement extends MetadataElement {
    private int modCount = 0;
    private boolean recentlyPersisted = false;
    private ElementInfo info;
    RefFeatured parent; // used for hard-referencing parent

    /** Creates a new instance of SemiPersistentElement */
    public SemiPersistentElement(StorableObject o) {
        super(o);
    }
    
    public boolean hasComposite() {
        return ((StorableObject) _getDelegate()).getImmediateCompositeId() != null;
    }
    
    public boolean isValid() {
        if (super.isValid() && !isSafeTrans()) {
            try {
                if (this instanceof ResourceImpl) {
                    this.refImmediatePackage();
                } else {
                    ResourceImpl rsc = (ResourceImpl) getResource();
                    if (rsc != null) {
                        rsc.checkUpToDate(false, false, false);
                    }
                }
            } catch (javax.jmi.reflect.InvalidObjectException e) {
                return false;
            }
        }
        return super.isValid();
    }

    protected final void fireAttrChange(String attrName, Object oldValue, Object newValue) {
        fireAttrChange(attrName, oldValue, newValue, AttributeEvent.POSITION_NONE);
    }
    
    protected final void fireAttrChange(String attrName, Object oldValue, Object newValue, int position) {
        if (_getMdrStorage().eventsEnabled()) {
            AttributeEvent event = new AttributeEvent(
                this, 
                AttributeEvent.EVENT_ATTRIBUTE_SET, 
                attrName, 
                oldValue, 
                newValue, 
                position
            );
            _getMdrStorage().getEventNotifier().INSTANCE.firePlannedChange(this, event);
        }
    }
    
    /** Should be called by a class proxy whenever a new metadata object is created via
     * standard API (it has no ASTInfo so a default ASTInfo is set into it)
     */
    protected final void setNew() {
        setChanged();
        if (this.info != null) {
            for (Iterator it = getInitedChildren().iterator(); it.hasNext();) {
                ((MetadataElement) it.next()).setNew();
            }
        }
        this.info = getDefaultInfo();
        setPersisted(true);
        childrenInited = true;
        // [TODO] uncomment this
//        ((ExclusiveMutex) _getMdrStorage().getRepositoryMutex()).addNew(this);
    }

    public void setPersisted(boolean persisted) {
    }

    protected final boolean isNew() {
        return this.info == getDefaultInfo();
    }
    
    protected final boolean infoIdenticalTo(ElementInfo info) {
        return this.info == info;
    }

    /** This method is called from parent to set ASTInfo of this element. It can be called
     * basically in two cases:
     * <ul>
     *  <li>object was not initialized - it needs AST info to be able to read value of some of its attributes
     *  <li>object was changed externally (in editor) - it gets an updated AST and needs to fire
     * match it with the original AST and fire appropriate change events
     * </ul>
     * This method should not be called for new objects. setNew should be called instead.
     */
    public final void setElementInfo(ElementInfo info) {
        if (infoIdenticalTo(info)) return;
        boolean changes = disableChanges;
        disableChanges = true;
        try {
            if (isInitialized()) {
                // whole ASTInfo should be matched
                matchElementInfo(info);
            }
            this.info = info;
            hardRefParent(true);
            // go down via containment hierarchy and force ASTInfo renewal
            if (childrenInited) {
                initChildren();
            }
        } finally {
            disableChanges = changes;
        }
        resetChange();
    }

    protected void hardRefParent(boolean enabled) {
        parent = enabled ? refImmediateComposite() : null;
    }

    protected void parentChanged() {
        parent = refImmediateComposite();
        if (parent == null || ((BaseObjectHandler) parent)._getDelegate() instanceof DeferredObject) {
            if (!(_getDelegate() instanceof DeferredObject)) mutate(true); // make object transient
            parent = null;
        } else {
            if (_getDelegate() instanceof DeferredObject) mutate(false); // make object persistent
        }
    }

    protected void mutate(boolean makeTransient) {
        StorableObject newDelegate;
        StorableObject oldDelegate = (StorableObject) _getDelegate();
        try {
            if (makeTransient) {
                newDelegate = new DeferredObject(oldDelegate);
            } else {
                newDelegate = new StorableObject(oldDelegate);
            }
            _setDelegate(newDelegate);
            oldDelegate.clearComposite();
            oldDelegate.deleteInstance();
        } catch (StorageException e) {
            throw (RuntimeException) ErrorManager.getDefault().annotate(new RuntimeException(), e);
        }
    }

    public final ElementInfo getElementInfo() {
        initCheck();
        if (info == null && (_getDelegate() instanceof StorableObject)) {
            String name;
            if (this instanceof JavaClass) {
                name = getName();
            } else {
                name = "<unknown>"; // NOI18N
            }
            JMManager.getLog().notify(ErrorManager.INFORMATIONAL, new Exception("***** Info is null for element: " + getName() + " (instanceof: " + getClass().getName() + " MOFID: " + refMofId() + ")")); // NOI18N
            if (this instanceof ResourceImpl) {
                JMManager.getLog().log(ErrorManager.WARNING, "isChanged: " + isChanged() + ", alreadyChecking: " + ((ResourceImpl) this).alreadyChecking);
                JMManager.getLog().log(ErrorManager.WARNING, "isValid: " + isValid());
                RuntimeException e = ((ResourceImpl) this).alreadyCheckingStackTrace;
                if (e != null) {
                    JMManager.getLog().log(ErrorManager.WARNING, "alreadyChecking stacktrace:");
                    JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                    throw new IllegalStateException("Recursive call to getElementInfo()");
                }
                setNew();
            } else if (this instanceof JavaClass) {
                JavaClass replacement = null;
                SemiPersistentElement composite = (SemiPersistentElement) refImmediateComposite();
                if (composite == null) {
                    JMManager.getLog().log(ErrorManager.WARNING, "    composite is null - need to find the class with the same name in index...");
                    JMManager.getLog().log(ErrorManager.WARNING, "    classindex classes for the same FQN in this extent:");
                    for (Iterator it = ClassIndex.getIndex((JavaModelPackage) refImmediatePackage()).getClassesByFqn(getName()).iterator(); it.hasNext();) {
                        JavaClass feat = (JavaClass) it.next();
                        if (!feat.isValid()) {
                            continue;
                        }
                        JMManager.getLog().log(ErrorManager.WARNING, "        " + feat.getName() + " (instanceof: " + feat.getClass().getName() + " MOFID: " + feat.refMofId() + ")" + (refMofId().equals(feat.refMofId()) && feat != this ? " !! same MOFID as me, but different object" : "") + (feat.refImmediateComposite() == null ? " !! composite is null" : ""));
                        if (!equals(feat)) {
                            replacement = feat;
                        }
                    }
                } else {
                    JMManager.getLog().log(ErrorManager.WARNING, "    composite contains the following features: ");
                    List features;
                    if (composite instanceof JavaClass) {
                        features = ((JavaClass) composite).getFeatures();
                    } else {
                        features = ((Resource) composite).getClassifiers();
                    }
                    for (Iterator it = features.iterator(); it.hasNext();) {
                        Feature feat = (Feature) it.next();
                        JMManager.getLog().log(ErrorManager.WARNING, "        " + feat.getName() + " (instanceof: " + feat.getClass().getName() + " MOFID: " + feat.refMofId() + ")" + (refMofId().equals(feat.refMofId()) && feat != this ? " !! same MOFID as me, but different object" : ""));
                        if ((feat instanceof JavaClass) && Utilities.compareObjects(feat.getName(), getName())) {
                            replacement = (JavaClass) feat;
                        }
                    }
                }
                if (replacement == null) {
                    JMManager.getLog().log(ErrorManager.WARNING, "No suitable replacement found in index - we need to create a fake info for class.");
                    setNew();
                } else {
                    JMManager.getLog().log(ErrorManager.WARNING, "Acquired info from a replacement class: " + replacement.refMofId());
                    this.info = ((SemiPersistentElement) replacement).info;
                }
            } else {
                setNew();
            }
            JMManager.getTransactionMutex().invalidateAtCommit(this);
        }
        return info;
    }

    public final boolean isInitialized() {
        return info != null;
    }

    protected void uninitialize() {
        info = null;
        hardRefParent(false);
    }
    
    protected void checkUpToDate() {
        if (checkAndUpdateModCount() && !(isSafeTrans() && isPersisted())) {
            if (this instanceof ResourceImpl) {
                ((ResourceImpl) this).checkUpToDate(true, false, true);
            } else {
                Object parent = refImmediateComposite();
                if (parent instanceof SemiPersistentElement) {
                    ((SemiPersistentElement) parent).checkUpToDate();
                    if (super.isValid() && !isPersisted()) {
                        initCheck();
                    }
                }
            }
        }
    }
    
    protected void persist() {
        if (!recentlyPersisted) {
            recentlyPersisted = true;
            Object parent = refImmediateComposite();
            if (parent instanceof SemiPersistentElement) {
                if (!((SemiPersistentElement) parent).recentlyPersisted) {
                    ((ExclusiveMutex) _getMdrStorage().getRepositoryMutex()).registerPersisted(this);
                }
            }
        }
    }
    
    public void clearPersist(boolean fail) {
        if (fail) {
            boolean persisted;
            try {
                persisted = isPersisted();
            } catch (InvalidObjectException e) {
                // object is not valid anymore -> return
                // [TODO] once isValid supports rollback, this should be changed
                // to simply check isValid
                return;
            }
            modCount = 0;
            if (!persisted) {
                if (info == null) {
                    recentlyPersisted = false;
                } else {
                    updatePersistent(info);
                }
            }
        } else {
            recentlyPersisted = false;
        }
    }
    
    /** Returns true if the modCount is different */
    public boolean checkModCount() {
        return this.modCount != JMManager.getTransactionMutex().getModCount();
    }
    
    /** Updates modCount for this object and returns true if the modCount was different. */
    public boolean checkAndUpdateModCount() {
        int modCount = JMManager.getTransactionMutex().getModCount();
        boolean result = modCount != this.modCount;
        this.modCount = modCount;
        return result;
    }
    
    protected final boolean isSafeTrans() {
        return JMManager.getTransactionMutex().isSafeTrans();
    }
    
    public boolean isPersisted() {
        return false;
    }

    /**
     * Find referenced resources of this element
     */
    Resource[] findReferencedResources() {
        return new Resource[] {getResource()};
    }
    
    public Collection getReferences() {
        if (this instanceof NamedElement) {
            Resource[] res = findReferencedResources();
            UsageFinder finder = new UsageFinder((NamedElement) this);
            return finder.getUsers(res);
        } else {
            return Collections.EMPTY_LIST;
        }
    }

    /** Should be overriden by elements that have persistent attributes.
     * They should implement this method to compare values in newly passed AST info with
     * values of the persistent attributes and if the values do not match, they should be
     * updated and proper events should be fired.
     */
    protected void matchPersistent(ElementInfo info) {
        modCount = JMManager.getTransactionMutex().getModCount();
        matchName(info);
    }
    
    public final void updatePersistent(ElementInfo info) {
        boolean changes = disableChanges;
        disableChanges = true;
        try {
            matchPersistent(info);
        } finally {
            disableChanges = changes;
        }
    }
    
    protected void matchName(ElementInfo info) {
        if ((this instanceof NamedElement) && !Utilities.compareObjects(info.name, this.getName())) {
            setName(info.name);
        }
    }

    protected abstract ElementInfo getDefaultInfo();

    /** The method has to make sure that the AST infos of children are also updated.
     */
    protected void matchElementInfo(ElementInfo newInfo) {
        if (info.infoType!=newInfo.infoType) {
            if ((!(info instanceof ClassInfo) || !(newInfo instanceof ClassInfo)) && 
                    !(info instanceof AnnotationValueInfo && newInfo instanceof AnnotationValueInfo))
                throw  new IllegalArgumentException("Illegal element type "+info.infoType+"!="+newInfo.infoType); // NOI18N
        }    
    }

    public void setName(String name) {
        objectChanged(CHANGED_NAME);
        super_setName(name);
    }

    protected abstract void super_setName(String name);
    public abstract String getName();

    protected static String getInnerIndentation(int level) {
        StringBuffer indentation = new StringBuffer();
        for (int i = 0; i < level; i++) {
            indentation.append(INDENTATION);
        }
        return indentation.toString();
    }

    /**
     * Return indentation of the element. If the element is new, it asks
     * for parent's indentation and appends its own.
     *
     * @return indentation
     */
    protected String getIndentation() {
        super.getIndentation();
        if (isNew()) {
            // For the new element it asks parent (immediateComposite)
            // for its indentation and appends its own indentation.
            return ((MetadataElement) refImmediateComposite()).getIndentation().concat(INDENTATION);
        }
        else {
            // For the existing element, it tries to analyze element to
            // get the indentation. It asks parent for its indentation,
            // takes its paddings. In the paddings, it looks for the nearest
            // end-line char to the first token. Difference between end-line
            // character and first char of the first token is parent's
            // indentation.
            Token firstToken = getParser().getToken(getASTree().getFirstToken());
            Token[] pad = firstToken.getPadding();
            Token endLine = null;
            for (int i = pad.length-1; i >= 0; i--) {
                if (pad[i].getType() == ParserTokens.EOL) {
                    endLine = pad[i];
                    break;
                }
            }
            if (endLine != null) {
                String src = getParser().getSourceText();
                return src.substring(endLine.getEndOffset(), firstToken.getStartOffset());
            }
            else {
                // What to do when there is no endline character
                // before the first token? -- We will indent it as a new element.
                // This branch of code is running e.g. when field group is 
                // splitted by changing of its javadoc. (Use e.g. Autocomment 
                // tool and comment one of the field in field group)
                return ((MetadataElement) refImmediateComposite()).getIndentation().concat(INDENTATION);
            }

        }
    }

    /**
     * Returns indentation of the body of element
     *
     * @return indentation of the body of element
     */
    protected String getBodyIndentation() {
        return getIndentation().concat(INDENTATION);
    }

    private void deleteCollection(Collection col) {
        for (Iterator it = col.iterator(); it.hasNext();) {
            RefObject obj = (RefObject) it.next();
            boolean del = (obj.refImmediateComposite() == this);
            it.remove();
            // if the object was contained by this before its removal
            // from the collection and after the removal its container is null,
            // it means that it was removed from the composite collection
            // i.e. the object needs to be deleted
            if (del && obj.refImmediateComposite() == null) {
                obj.refDelete();
            }
        }
    }

    private void deleteCollection(List mdrList,RefObject[] mdrObjects, List indexes) {
        Iterator it=indexes.iterator();

        while(it.hasNext()) {
            int index=((Integer)it.next()).intValue();
            RefObject obj=mdrObjects[index];

            if (mdrList instanceof LightAttrList || mdrList instanceof SemiPersistentAttrList
                    || mdrList instanceof DeferredAttrList || mdrList instanceof AttrCollWrapper) {
                obj.refDelete();
            } else {
                boolean del = (obj.refImmediateComposite() == this);
                mdrList.remove(obj);
                // if the object was contained by this before its removal
                // from the collection and after the removal its container is null,
                // it means that it was removed from the composite collection
                // i.e. the object needs to be deleted
                if (del && obj.refImmediateComposite() == null) {
                    obj.refDelete();
                }
            }
        }
    }
    
    public void processMembers(List mdrList,Object[] astMembers) {
        RefObject[] mdrMembers;
        ArrayMapper mapper;
        int[] map;
        int i;

        if (astMembers==null || astMembers.length==0) {
            deleteCollection(mdrList);
            return;
        }
        if (mdrList.isEmpty()) {
            for (i=0;i<astMembers.length;i++) {
                mdrList.add(createAnyElement(astMembers[i]));
            }
            return;
        }
        mdrMembers=(RefObject[])mdrList.toArray(new RefObject[0]);
        mapper=new ArrayMapper(mdrMembers, astMembers);
        map=mapper.getMap();
        if (!mapper.isIdentical()) {
            deleteCollection(mdrList,mdrMembers,mapper.getDeletedIndexes());
            if (mapper.hasPermutation()) {
                mdrList.clear();
                for (i=0;i<astMembers.length;i++) {
                    Object member=astMembers[i];
                    int oldIndex=map[i];
                    RefObject newObj;

                    if (oldIndex!=-1) { // update
                        newObj=mdrMembers[oldIndex];
                        if (newObj instanceof SemiPersistentElement) {
                            ((SemiPersistentElement)newObj).matchMember(member);
                        }
                    } else { // new object
                        newObj=createAnyElement(member);
                    }
                    mdrList.add(newObj);
                }
                return;
            }
        }
        for (i=0;i<astMembers.length;i++) {
            Object astMember=astMembers[i];
            int oldIndex=map[i];

            if (oldIndex==-1) {
                RefObject newObj=createAnyElement(astMember);

                mdrList.add(i,newObj);
            } else if (mdrMembers[oldIndex] instanceof SemiPersistentElement) {
                ((SemiPersistentElement)mdrMembers[oldIndex]).matchMember(astMember);
            }
        }
        return;
    }

    public void fixMembers(List mdrList, ElementInfo[] infos) {
        JMManager.getLog().notify(ErrorManager.INFORMATIONAL, new RuntimeException("Inconsistent storage. (mdrList.size == " + mdrList.size() + ", infos.size == " + infos.length + ")")); // NOI18N
        boolean changes = disableChanges;
        disableChanges = true;
        try {
            ArrayList oldElements = new ArrayList();
            for (Iterator it = mdrList.iterator(); it.hasNext();) {
                oldElements.add(it.next());
                it.remove();
            }
            for (int i = 0; i < infos.length; i++) {
                SemiPersistentElement element = findElement(oldElements, infos[i]);
                if (element == null) {
                    element = createElement(infos[i]);
                } else {
                    element.info = null; // to avoid matching
                    element.setElementInfo(infos[i]);
                }
                mdrList.add(element);
            }
            for (Iterator it = oldElements.iterator(); it.hasNext();) {
                SemiPersistentElement element = (SemiPersistentElement) it.next();
                element.refDelete();
            }
        } finally {
            disableChanges = changes;
        }
    }
    
    private SemiPersistentElement findElement(List elements, ElementInfo info) {
        JavaModelPackage extent = (JavaModelPackage) refImmediatePackage();
        RefClass classFactory;
        switch (info.infoType) {
            case FieldInfo.FIELD_TYPE:
                classFactory = extent.getField();
                break;
            case FieldGroupInfo.FIELDGROUP_TYPE:
                classFactory = extent.getFieldGroup();
                break;
            case ClassInfo.INTERFACE_TYPE:
            case ClassInfo.CLASS_TYPE:
                classFactory = extent.getJavaClass();
                break;
            case ClassInfo.ANON_CLASS_TYPE:
                classFactory = extent.getClassDefinition();
                break;
            case MethodInfo.METHOD_TYPE:
                classFactory = extent.getMethod();
                break;
            case FeatureInfo.INSTANCE_INITIALIZER_TYPE:
            case FeatureInfo.STATIC_INITIALIZER_TYPE:
                classFactory = extent.getInitializer();
                break;
            case MethodInfo.CONSTRUCTOR_TYPE:
                classFactory = extent.getConstructor();
                break;
            case ElementInfo.IMPORT_ON_DEMAND_TYPE:
            case ElementInfo.SINGLE_IMPORT_TYPE:
                classFactory = extent.getImport();
                break;
            case ParameterInfo.PARAMETER_TYPE:
                classFactory = extent.getParameter();
                break;
            case TypeParamInfo.TYPEPARAM_TYPE:
                classFactory = extent.getTypeParameter();
                break;
            case FeatureInfo.ENUM_CONSTANT_TYPE:
                classFactory = extent.getEnumConstant();
                break;
            case EnumInfo.ENUM_TYPE:
                classFactory = extent.getJavaEnum();
                break;
            case EnumInfo.ANNOTATIONTYPE_TYPE:
                classFactory = extent.getAnnotationType();
                break;
            case AttributeInfo.ATTRIBUTE_TYPE:
                classFactory = extent.getAttribute();
                break;
            case AnnotationInfo.ANNOTATION_TYPE:
                classFactory = extent.getAnnotation();
                break;
            case AnnotationValueInfo.ANNOTATIONVALUE_ANNOTATION:
            case AnnotationValueInfo.ANNOTATIONVALUE_ARRAY:
            case AnnotationValueInfo.ANNOTATIONVALUE_STRING:
                classFactory =  extent.getAttributeValue();                
                break;
            default:
                throw new IllegalArgumentException("Illegal type " + info.infoType); // NOI18N
        }
        
        for (Iterator it = elements.iterator(); it.hasNext();) {
            SemiPersistentElement element = (SemiPersistentElement) it.next();
            if (element.refClass().equals(classFactory)) {
                it.remove();
                return element;
            }
        }
        
        return null;
    }

    private void matchMember(Object member) {
        if (member instanceof ElementInfo) {
            updatePersistent((ElementInfo) member);
        }
    }
    
    private RefObject createAnyElement(Object elem) {
        if (elem instanceof TypeRef) {
            return resolveType((TypeRef)elem);
        }
        return createElement((ElementInfo) elem, false);
    }

    Type resolveTypeString(String name) {
        if (name.startsWith("<")) { // NOI18N
            String typeParamName=name.substring(1);
            SemiPersistentElement element=this;
            do {
                if (element instanceof GenericElement) {
                    GenericElement gel=(GenericElement)element;
                    if (!(gel instanceof CallableFeatureImpl) || ((CallableFeatureImpl)gel).hasTypeParameters()) {
                        Iterator typeParsIt=gel.getTypeParameters().iterator();

                        while(typeParsIt.hasNext()) {
                            TypeParameter typePar=(TypeParameter)typeParsIt.next();

                            if (typePar.getName().equals(typeParamName))
                                return typePar;
                        }
                    }
                }
                element=(SemiPersistentElement)element.refImmediateComposite();
            } while (!(element instanceof Resource));
            // type parameter not found, create unresolved class
        } 
        TypeClass typeClass = ((JavaModelPackage) refImmediatePackage()).getType();
        return typeClass.resolve(name);
    }
        
    public static TypeRef typeToTypeRef(Type type) {
        if (type == null) return null;
        
        if (type instanceof TypeParameter) {
            return new TypeParamRef(type.getName());
        } else if (type instanceof ParameterizedType) {
            ParameterizedTypeImpl paramType = (ParameterizedTypeImpl) type;
            Type typePars[] = (Type[])paramType.getParameters().toArray(new Type[0]);
            TypeRef args[] = new TypeRef[typePars.length];
            int i = 0;
            for (; i<typePars.length; i++) {
                args[i] = typeToTypeRef(typePars[i]);
                int status = paramType.getWildCardStatus(i);
                if (status != 0) {
                    args[i] = new WildCardRef(status == 1, status == 3 ? null : args[i]);
                }
            }
            NameRef parentName=(NameRef) typeToTypeRef(paramType.getDeclaringClass());
            JavaClass def=paramType.getDefinition();
            String name;
            
            if  (parentName==null)
                name=def.getName();
            else
                name=def.getSimpleName();
            return new NameRef(name, parentName, args);
        } else if (type instanceof Array) {
            int dimCount = 0;
            Type currType = type;
            while (currType instanceof Array) {
                dimCount++;
                currType = ((Array) currType).getType();
            }
            return new ArrayRef((PrimitiveTypeRef) typeToTypeRef(currType), dimCount);
        } else if (type instanceof JavaClass) {
            if (type instanceof JavaClassImpl && ((JavaClassImpl)type).isTransient())  // local class
                return new LocalClassNameRef((JavaClass)type);
            return new NameRef(type.getName());
        } else if (type instanceof PrimitiveType) {
            String kind = ((PrimitiveType) type).getKind().toString();
            PrimitiveTypeRef result = PrimitiveTypeRef.forName(kind);
            if (result != null) {
                return result;
            }
            throw new IllegalArgumentException("Unknown primitive type name: " + kind); // NOI18N
        }
        throw new IllegalArgumentException("Unable to convert to typeref: " + type.getClass().getName()); // NOI18N
    }
    
    static List typeReferencesToTypeRef(List typeReferences) {
        ArrayList result = new ArrayList(typeReferences.size());
        for (Iterator it = typeReferences.iterator(); it.hasNext();) {
            result.add(typeReferenceToTypeRef((Element) it.next(), 0));
        }
        return result;
    }
    
    /** @param dimCount Number of dimensions to be added to the typeRef
     * and make it an ArrayRef if necessary.
     */
    public static TypeRef typeReferenceToTypeRef(Element typeReference, int dimCount) {
        if (typeReference == null) return null;
        
        int resultDimCount = dimCount;
        TypeRef result;
        if (typeReference instanceof ArrayReference) {
            ArrayReference arrayReference = (ArrayReference) typeReference;
            resultDimCount += arrayReference.getDimCount();
            result = typeReferenceToTypeRef(arrayReference.getParent(), 0);
        } else if (typeReference instanceof MultipartId) {
            MultipartId mpi = (MultipartId) typeReference;
            NamedElement ne = mpi.getElement();
            if (ne instanceof PrimitiveType) {
                result = PrimitiveTypeRef.forName(ne.getName());
            } else {
                TypeArgument typeArgs[] = (TypeArgument[])mpi.getTypeArguments().toArray(new TypeArgument[0]);
                TypeRef args[] = new TypeRef[typeArgs.length];
                int i = 0;
                for (; i<typeArgs.length; i++) {
                    args[i] = typeReferenceToTypeRef(typeArgs[i], 0);
                }
                NameRef parent = (NameRef) typeReferenceToTypeRef(mpi.getParent(), 0);
                if (parent != null && parent.parent == null && parent.args.length == 0) {
                    parent = null;
                }
                String name;
                if (parent != null) {
                    name = ne.getName();
                } else {
                    name = mpi.getName();
                }
                result = new NameRef(name, parent, args);
            }
        } else if (typeReference instanceof WildCard) {
            WildCard wc = (WildCard) typeReference;
            result = new WildCardRef(wc.isLower(), typeReferenceToTypeRef(wc.getBoundName(), 0));
        } else {
            throw new IllegalStateException();
        }
        if ((resultDimCount > 0) && (result instanceof PrimitiveTypeRef)) {
            result = new ArrayRef((PrimitiveTypeRef) result, resultDimCount);
        }
        return result;
    }
    
    /** @param dimCount Number of dimensions to be subtracted from the typeRef
     * if it is an ArrayRef. It is a responsibility of the caller of this method
     * to make sure that if dimCount > 0, typeRef is instance of ArrayRef and
     * dimCount <= typeRef.dimCount.
     */
    public Element typeRefToTypeReference(TypeRef typeRef, int dimCount) {
        return typeRefToTypeReference((JavaModelPackage) refImmediatePackage(), typeRef, dimCount);       
    }
    
    /** @param dimCount Number of dimensions to be subtracted from the typeRef
     * if it is an ArrayRef. It is a responsibility of the caller of this method
     * to make sure that if dimCount > 0, typeRef is instance of ArrayRef and
     * dimCount <= typeRef.dimCount.
     */
    public static Element typeRefToTypeReference(JavaModelPackage extent, TypeRef typeRef, int dimCount) {
        if (typeRef == null) return null;
        
        if (typeRef instanceof NameRef) {
            NameRef nRef = (NameRef) typeRef;
            ArrayList args = new ArrayList(nRef.args.length);
            for (int i = 0; i < nRef.args.length; i++) {
                args.add(typeRefToTypeReference(extent, nRef.args[i], 0));
            }
            return extent.getMultipartId().createMultipartId(nRef.name, (MultipartId) typeRefToTypeReference(extent, nRef.parent, 0), args);
        } else if (typeRef instanceof PrimitiveTypeRef) {
            return extent.getMultipartId().createMultipartId(((PrimitiveTypeRef) typeRef).name, null, null);
        } else if (typeRef instanceof WildCardRef) {
            WildCardRef wcRef = (WildCardRef) typeRef;
            return extent.getWildCard().createWildCard(wcRef.isLower, (MultipartId) typeRefToTypeReference(extent, wcRef.bound, 0));
        } else if (typeRef instanceof ArrayRef) {
            ArrayRef aRef = (ArrayRef) typeRef;
            int resultDimCount = aRef.dimCount - dimCount;
            if (resultDimCount > 0) {
                StringBuffer name = new StringBuffer(aRef.parent.name);
                for (int i = 0; i < resultDimCount; i++) {
                    name.append("[]"); // NOI18N
                }
                return extent.getArrayReference().createArrayReference(name.toString(), (MultipartId) typeRefToTypeReference(extent, aRef.parent, 0), resultDimCount);
            } else {
                return typeRefToTypeReference(extent, aRef.parent, 0);
            }
        }
        throw new IllegalStateException();
    }

    public final MDRParser getParser() {
        ASTree tree = getASTree();
        return tree == null ? null : (MDRParser) tree.getASTContext();
    }

    protected final Object getInternalForm() {
        return getElementInfo();
    }

    protected void _delete() {
        super._delete();
        // --- delete this instance ---------------------------------------
        try {
            ((StorableObject) _getDelegate()).deleteInstance();
        } catch (StorageException e) {
            throw (GeneralException) ErrorManager.getDefault().annotate(new GeneralException(e.getMessage()), e);
        }
    }

    protected final LightAttrList createChildrenList(LightAttrList currentList, String attrName, ElementInfo[] infos, int changeMask, boolean rebuild) {
        DeferredAttrList deferredList;
        try {
            if (currentList != null) {
                deferredList = (DeferredAttrList) ((AttrListWrapper) currentList.getInnerList()).getInnerList();
                if (rebuild) {
                    for (Iterator it = deferredList.iterator(); it.hasNext();) {
                        MetadataElement temp = (MetadataElement) it.next();
                        it.remove();
                        temp.refDelete();
                    }
                } else {
                    if (!isNew() && (infos.length != currentList.size())) {
                        JMManager.getLog().log(ErrorManager.WARNING, "Inconsistent storage from attribute: " + attrName);
                        fixMembers(currentList, infos); 
                    } else {
                        int i = 0;
                        for (Iterator it = deferredList.iterator(); it.hasNext(); i++) {
                            SemiPersistentElement im = (SemiPersistentElement) it.next();
                            if (isNew()) {
                                // if an existing feature is added into a new element,
                                // it needs to be set to new
                                im.setNew();
                            } else {
                                if (checkElementType(infos[i], im)) {
                                    im.setElementInfo(infos[i]);
                                } else {
                                    JMManager.getLog().log(ErrorManager.WARNING, "Inconsistent storage - element types do not match.");
                                    fixMembers(currentList, infos);
                                    break;
                                }
                            }
                        }
                    }
                    deferredList = null;
                }
            } else {
                StorableFeatured storable = (StorableFeatured) _getDelegate();
                deferredList = new DeferredAttrList(storable, storable.getClassProxy().getAttrDesc(attrName), new ArrayList());
                currentList = createWrapper(attrName, deferredList, changeMask);
            }
            if (deferredList != null) {
                for (int i = 0; i < infos.length; i++) {
                    SemiPersistentElement s = createElement(infos[i]);
                    deferredList.add(s);
                }
            }
        } catch (StorageException e) {
            throw (GeneralException) ErrorManager.getDefault().annotate(new GeneralException(e.getMessage()), e);
        }
        return currentList;
    }
    
    protected final LightAttrList createChildrenList(String attrName, AttrListWrapper persistentList, Collection values, int changeMask) {
        if (values != null) {
            List innerList = persistentList.getInnerList();
            for (Iterator it = values.iterator(); it.hasNext();) {
                innerList.add(it.next());
            }
        }
        persistentList.setAttrName(attrName);
        return new LightAttrList(persistentList, this, changeMask, true);
    }
    
    protected final LightAttrList createChildrenList(LightAttrList currentList, String attrName, AttrListWrapper persistentList, ElementInfo[] infos, int changeMask) {
        List nakedCollection = persistentList.getInnerList();
        boolean needsUpdate = false;

        if (!isNew() && (nakedCollection.size() != infos.length)) {
            fixMembers(nakedCollection, infos);
            needsUpdate = true;
        }
        ListIterator it = nakedCollection.listIterator();
        for (int i = 0; i < infos.length; i++) {
            SemiPersistentElement element = (SemiPersistentElement) it.next();
            element.setElementInfo(infos[i]);
        }

        persistentList.setAttrName(attrName);
        if (currentList == null) {
            currentList = new LightAttrList(persistentList, this, changeMask, true);
        } else if (needsUpdate) {
            currentList.setInnerList(persistentList);
        }

        return currentList;
    }
    
    protected final AttrListWrapper getPersistentList(String attrName, List list) {
        AttrListWrapper result = (AttrListWrapper) list;
        result.setAttrName(attrName);
        return result;
    }
    
    protected final void persistChildren(AttrListWrapper persistentList, ElementInfo[] infos) {
        List nakedCollection = persistentList.getInnerList();
        if (!nakedCollection.isEmpty()) {
            JMManager.getLog().notify(ErrorManager.INFORMATIONAL, new Exception("Inconsistent storage - persistent children collection not empty although parent says it has not been persisted yet. Removing old children...")); // NOI18N
            processMembers(persistentList, infos);
        } else {
            for (int i = 0; i < infos.length; i++) {
                SemiPersistentElement element = createElement(infos[i], false);
                nakedCollection.add(element);
            }
        }
    }
    
    protected final void deleteChildren(String attrName, AttrListWrapper persistentList) {
        persistentList.setAttrName(attrName);
        for (Iterator it = persistentList.iterator(); it.hasNext();) {
            RefObject obj = (RefObject) it.next();
            it.remove();
            obj.refDelete();
        }
    }
}
