/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import javax.jmi.reflect.DuplicateException;
import javax.jmi.reflect.RefObject;

import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.storagemodel.StorableClass;
import org.netbeans.mdr.storagemodel.StorableFeatured;
import org.netbeans.mdr.util.DebugException;
import org.openide.ErrorManager;

public class SemiPersistentAttrList implements List {

    
    private List innerList;
    private final Set innerSet = new HashSet();
    private int modCount = 0;
    
    private final StorableFeatured mdrObject;
    private final StorableClass.AttributeDescriptor attrDesc;
    private final MOFID metaMofId;
    private final boolean isRefObject;
    
    private final SemiPersistentElement parent;
    private final int changeMask;

    
    public SemiPersistentAttrList(StorableFeatured mdrObject, StorableClass.AttributeDescriptor attrDesc, List innerList, SemiPersistentElement parent, int changeMask) {
        this.mdrObject = mdrObject;
        this.attrDesc = attrDesc;
        this.parent = parent;
        this.changeMask = changeMask;
        metaMofId = attrDesc.getMofId();
        isRefObject = RefObject.class.isAssignableFrom(attrDesc.getType());
        setInnerList(innerList);
    }

    public void setInnerList(List innerList) {
        modCount++;
        this.innerList = innerList;        
        innerSet.clear();
        Iterator iter = innerList.iterator();
        while(iter.hasNext()) {
            Object obj = iter.next();
            if (innerSet.contains(obj)) {
                throw new DuplicateException(obj, getMetaElement());
            } else {
                innerSet.add(obj);
                if (isRefObject) {
                    setAttribComposite((RefObject) obj);
                }
            }
        }
    }

    protected void objectChanged() {
        if (parent != null)
            parent.objectChanged(changeMask);
    }
    
    // ..........................................................................
    
    private void setAttribComposite(RefObject object) {
        try {
            DeferredObject storable = (DeferredObject) ((BaseObjectHandler) object)._getDelegate();
            storable.setComposite(mdrObject, storable.getMofId(), metaMofId);
        } catch (StorageException e) {
            ErrorManager.getDefault().notify(e);
        }
    }
    
    private void clearAttribComposite(RefObject object) {
        try {
            DeferredObject storable = (DeferredObject) ((BaseObjectHandler) object)._getDelegate();
            storable.clearComposite();
        } catch (StorageException e) {
            ErrorManager.getDefault().notify(e);
        }
    }
    
    private RefObject getMetaElement() {
        try {
            return (RefObject) mdrObject.getMdrStorage().getRepository().getHandler(mdrObject.getMdrStorage().getObject(metaMofId));
        } catch (Exception e) {
            return null;
        }
    }
    
    // ..........................................................................
    
    public boolean remove(Object obj) {
        boolean res = innerList.remove(obj);
        if (res) {
            innerSet.remove(obj);
            if (isRefObject) {
                clearAttribComposite((RefObject) obj);
            }
            objectChanged();
        }
        return res;
    }
    
    public Object set(int param, Object obj) {
        objectChanged();
        Object old = innerList.set(param, obj);
        if (isRefObject) {            
            clearAttribComposite((RefObject) old);
            setAttribComposite((RefObject) obj);
        }
        innerSet.remove(old);
        if (!innerSet.add(obj)) {
            throw new DuplicateException(obj, getMetaElement());
        }
        return old;
    }
    
    public Object remove(int param) {        
        objectChanged();
        Object old = innerList.remove(param);
        if (isRefObject) {
            clearAttribComposite ((RefObject) old);
        }
        innerSet.remove(old);
        return old;
    }
    
    public void add(int param, Object obj) {        
        objectChanged();
        innerList.add(param, obj);
        if (isRefObject) {
            setAttribComposite((RefObject) obj);
        }
        if (!innerSet.add(obj)) {
            throw new DuplicateException(obj, getMetaElement());
        }
    }
    
    public boolean add(Object obj) {        
        objectChanged();
        boolean res = innerList.add(obj);
        if (isRefObject && res) {
            setAttribComposite((RefObject) obj);
        }
        if (!innerSet.add(obj)) {
            throw new DuplicateException(obj, getMetaElement());
        }
        return res;
    }
    
    public ListIterator listIterator(int param) {
        return new SemiPersistentAttrListIterator(innerList.listIterator(param));
    }
    
    public Iterator iterator() {
        return new SemiPersistentAttrListIterator(innerList.listIterator());
    }
    
    public ListIterator listIterator() {
        return new SemiPersistentAttrListIterator(innerList.listIterator());
    }
    
    public List subList(int param, int param1) {
        return innerList.subList(param,  param1);
    }
    
    public boolean contains(Object obj) {
        return innerSet.contains(obj);
    }
    
    public boolean containsAll(Collection collection) {
        return innerSet.containsAll(collection);
    }
    
    public boolean addAll(Collection c) {
        // should not be called
        throw new DebugException();
    }
    
    public void clear() {
        // should not be called
        throw new DebugException();
    }
    
    public boolean isEmpty() {
        return innerList.isEmpty();
    }
    
    public boolean removeAll(Collection c) {
        // should not be called
        throw new DebugException();
    }
    
    public boolean retainAll(Collection c) {
        // should not be called
        throw new DebugException();
    }
    
    public int size() {
        return innerList.size();
    }
    
    public Object[] toArray() {
        return innerList.toArray();
    }
    
    public Object[] toArray(Object[] a) {
        return innerList.toArray(a);
    }
    
    public boolean addAll(int index, Collection c) {
        // should not be called
        throw new DebugException();
    }
    
    public Object get(int index) {
        return innerList.get(index);
    }
    
    public int indexOf(Object o) {
        return innerList.indexOf(o);
    }
    
    public int lastIndexOf(Object o) {
        return innerList.lastIndexOf(o);
    }

    // ListIterator .................................................
    class SemiPersistentAttrListIterator implements ListIterator {
        
        private Object lastRead;
        private ListIterator innerIterator;
        private final int modCount;
        
        SemiPersistentAttrListIterator(ListIterator iterator) {
            this.innerIterator = iterator;
            this.modCount = getParentModCount();
        }
        
        private int getParentModCount() {
            return SemiPersistentAttrList.this.modCount;
        }
        
        private void testModCount() throws ConcurrentModificationException {
            if (modCount != getParentModCount())
                throw new ConcurrentModificationException();
        }
        
        public void remove() {
            testModCount();
            objectChanged();
            innerIterator.remove();
            innerSet.remove(lastRead);
            if (isRefObject) {
                clearAttribComposite((RefObject) lastRead);
            }
        }
        
        public void add(Object obj) {            
            testModCount();
            objectChanged();
            innerIterator.add(obj);
            if (!innerSet.add(obj)) {
                throw new DuplicateException(obj, getMetaElement());
            }
            if (isRefObject) {
                setAttribComposite((RefObject) obj);
            }
        }
        
        public void set(Object obj) {
            testModCount();
            objectChanged();
            innerIterator.set(obj);
            innerSet.remove(lastRead);
            if (!innerSet.add(obj)) {
                throw new DuplicateException(obj, getMetaElement());
            }
            if (isRefObject) {
                clearAttribComposite((RefObject) lastRead);
                setAttribComposite((RefObject) obj);
            }
        }
        
        public boolean hasNext() {
            testModCount();
            return innerIterator.hasNext();
        }
        
        public boolean hasPrevious() {
            testModCount();
            return innerIterator.hasPrevious();
        }
        
        public Object next() {
            testModCount();
            return lastRead = innerIterator.next();
        }

        public int nextIndex() {
            testModCount();
            return innerIterator.nextIndex();
        }
        
        public Object previous() {
            testModCount();
            return lastRead = innerIterator.previous();
        }
        
        public int previousIndex() {
            testModCount();
            return innerIterator.previousIndex();
        }
    }
}
