/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Reader;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.regex.Pattern;
import javax.jmi.reflect.*;
import javax.swing.text.Position;
import javax.swing.text.StyledDocument;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.*;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.handlers.AttrListWrapper;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.*;
import org.netbeans.modules.javacore.parser.*;
import org.netbeans.modules.javacore.parser.Util;
import org.openide.ErrorManager;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.text.CloneableEditorSupport;
import org.openide.text.NbDocument;
import org.openide.text.PositionBounds;
import org.openide.text.PositionRef;
import org.netbeans.modules.javacore.scanning.ClassUpdater;
import org.netbeans.modules.javacore.scanning.JavaUpdater;

/**
 * Implementation of Resource interface.
 *
 * @author  Martin Matula
 */
public abstract class ResourceImpl extends SemiPersistentElement implements Resource {
    // this switch enable test JavaMetamodel without bridge and persistency behaviour
    public static boolean READ_ONLY = false;
    // java features flags
    public static final int HAS_GENERICS = 1;
    public static final int HAS_ENUMS = 2;
    public static final int HAS_ANNOTATIONTYPES = 4;
    public static final int HAS_ANNOTATION = 8;
    public static final int HAS_SYNTAX_ERROR = 16;
    // status indicating that the resource represents a classfile
    public static final int IS_FROM_CLASSFILE = 0x80000000;
    
    public static final String CLASSIFIERS_ATTR = "classifiers"; // NOI18N
    
    private static final ElementInfo DEFAULT_INFO = new ResourceInfo(null, ResourceInfo.RESOURCE_TYPE, null, null, null);
    
    private static final String lineSeparator = System.getProperty("line.separator"); // NOI18N
    private static final boolean lfOnly = "\n".equals(lineSeparator); // NOI18N
    private static final Pattern lineSepPattern = Pattern.compile("\r?\n"); // NOI18N
    
    private LightAttrList classifiers = null;
    private LightAttrList imports = null;
    private FileObject fileToDeleteOnRollback = null;

    private MultipartId packageIdentifier = null;

    private boolean elementsInited = false;

    private DiffList invertedDiffs = null;
    
    // source text info
    private String lastSourceText;
    private boolean isFromDoc;
    // cached dataobject corresponding to this resource
    private FileObject fobj;
    
    private DiffList extDiffs;

    // rollback related info
    private String rbText = null;
    private List rbList = null;
    
    private List errors = null;

    /** Creates a new instance of ResourceImpl */
    public ResourceImpl(StorableObject s) {
        super(s);
    }

    private static String lastPackageName = null;
    private static JavaPackage lastPackage = null;
    private static JavaPackageClass lastPackageProxy = null;

    public void _setPackageName(String name) {
        if (JMManager.INCONSISTENCY_DEBUG) System.err.println("ResourceImpl: Setting package name of resource " + getName() + " to " + name);
        JavaPackage oldPackage = null;
        try {
            oldPackage = (JavaPackage) ((NBMDRepositoryImpl) repository()).getHandler(((StorableObject) _getDelegate()).getImmediateComposite());
        } catch (StorageException e) {
            ErrorManager.getDefault().notify(e);
        }
        JavaPackageClassImpl pkgProxy = (JavaPackageClassImpl) ((JavaModelPackage) refImmediatePackage()).getJavaPackage();
        if (oldPackage != null) {
            if (name.equals(oldPackage.getName())) return;
            oldPackage.getResources().remove(this);
        }
        if (lastPackage == null || !name.equals(lastPackageName) || !pkgProxy.equals(lastPackageProxy)) {
            lastPackage = pkgProxy.resolveRealPackage(name);
            lastPackageName = name;
            lastPackageProxy = pkgProxy;
        }
        lastPackage.getResources().add(this);
    }

    public void setPackageName(String name) {
        String oldName = getPackageName();
        if (!name.equals(oldName)) {
            MultipartIdClass proxy = ((JavaModelPackage) refImmediatePackage()).getMultipartId();
            setPackageIdentifier(proxy.createMultipartId(name, null, null));
        }
    }

    private void setPkgNameAndUpdateIdx(String name) {
        if (isInitialized() && !isNew()) {
            getElementInfo().hardRefAST();
        }
        _setPackageName(name);
        for (Iterator it = getClassifiers().iterator(); it.hasNext();) {
            Object temp = it.next();
            if (temp instanceof JavaClassImpl) {
                JavaClassImpl cls = (JavaClassImpl) temp;
                String interName = name.length() == 0 ? "" : name + '.';
                cls.internalSetName(interName.concat(cls.getSimpleName()));
            }
        }
    }
    
    public Collection getReferences() {
        return Collections.EMPTY_LIST;
    }

    public boolean classifiersInited() {
        return classifiers != null;
    }
    
    public void reinitClassifiers() {
        assert classifiers!=null;
        classifiers.setInnerList(getPersistentClassifiers(), false);
    }
    
    public List getClassifiers() {
        if (checkModCount() && !(isSafeTrans() && isPersisted())) checkUpToDate(false,false,true);
        if (classifiers == null) {
            classifiers = createChildrenList(CLASSIFIERS_ATTR, (AttrListWrapper) super_getClassifiers(), null, CHANGED_CLASSIFIERS); // NOI18N
        }
        return classifiers;
    }

    public List getPersistentClassifiers() {
        AttrListWrapper list = (AttrListWrapper) super_getClassifiers();
        list.setAttrName(CLASSIFIERS_ATTR); // NOI18N
        return list;
    }

    protected abstract List super_getClassifiers();

    public List getNakedClassifiers() {
        try {
            return (List) ((StorableObject) _getDelegate()).getAttribute(CLASSIFIERS_ATTR); // NOI18N
        } catch (StorageException e) {
            throw (GeneralException) ErrorManager.getDefault().annotate(new GeneralException(e.getMessage()), e);
        }
    }
    
    public String getPackageName() {
        if (!isSafeTrans() && checkModCount()) checkUpToDate(false,false,true);
        JavaPackage pkg = (JavaPackage) _realImmediateComposite();
        return pkg == null ? null : pkg.getName();
    }

//    protected abstract void super_setPackageName(String name);

    /** Should be overriden by elements that have persistent attributes.
     * They should implement this method to compare values in newly passed AST info with
     * values of the persistent attributes and if the values do not match, they should be
     * updated and proper events should be fired.
     */
    protected void matchPersistent(ElementInfo info) {
        super.matchPersistent(info);
        if (!isPersisted()) {
            setPersisted(true);
        }
        processMembers(getClassifiers(), ((ResourceInfo) info).classes);
    }
    
    public boolean isPersisted() {
        return _getDelegate().getSlot1() != null;
    }
    
    public void setPersisted(boolean persisted) {
        _getDelegate().setSlot1(persisted ? "" : null);
    }

    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }

    /** The method has to make sure that the AST infos of children are also updated.
     */
    protected void matchElementInfo(ElementInfo newInfo) {
        Iterator classIt;
        ResourceInfo astInfo;
        int i=0;
        
        super.matchElementInfo(newInfo);
        astInfo=(ResourceInfo)newInfo;
        if (imports != null) {
            processMembers(getImports(),astInfo.imports);
        }
        resetASTElements();
    }

    protected void resetChildren() {
        super.resetChildren();
        if (childrenInited) {
            resetASTElements();
            initChildren(true);
        }
    }

    protected List getInitedChildren() {
        List list=new ArrayList();
        if (childrenInited) {
            list.addAll(getImports());
            list.addAll(getClassifiers());
        }
        if (elementsInited) {
           addIfNotNull(list, packageIdentifier);
        }
        return list;
    }
    
    public List getChildren() {
        List list = new ArrayList();
        addIfNotNull(list, getPackageIdentifier());
        list.addAll(getImports());
        list.addAll(getClassifiers());
        return list;
    }

    /**
     * Returns the value of attribute imports.
     * @return Value of imports attribute.
     */
    public List getImports() {
        checkUpToDate();
        if (!childrenInited) {
            initChildren();
        }
        return imports;
    }

    public void setName(String name) {
        // [TODO] change the name of the underlying file
        // [TODO] check for duplicates
        assert !name.startsWith("/") : "Resource name cannot start with /"; // NOI18N
        super_setName(name);
    }
    
    public void setTimestamp(long timestamp) {
        super_setTimestamp(timestamp);
    }
    
    /** Sets the resource timestamp along with an information if it is updated
     * from memory (i.e. from a file edited in the editor but not saved on the disk.
     * It is because in this case the timestamp is not sufficient for determining whether
     * a file is out of date.
     * @param timestamp Timestamp of the file on the disk.
     * @param isFromMemory Indicates whether a file is edited in memory and not saved yet.
     */
    public void setTimestamp(long timestamp, boolean isFromMemory) {
        setTimestamp(timestamp);
        setFromMemory(isFromMemory);
    }
    
    private void setFromMemory(boolean fromMemory) {
        _getDelegate().setSlot2(fromMemory ? "" : null);
    }
    
    /** Returns true if at the last parse this resource was updated from a document
     * modified in memory and not saved yet.
     * @return true if the resource was updated from a document edited in memory and not saved.
     */
    public boolean isFromMemory() {
        return _getDelegate().getSlot2() != null;
    }
    
    protected abstract void super_setTimestamp(long timestamp);

    protected abstract void super_setName(String name);

    protected void initChildren() {
        initChildren(false);
    }

    private void initChildren(boolean rebuild) {
        childrenInited = false;

        ClassInfo classes[] = ((ResourceInfo) getElementInfo()).classes;
        // create children
        if (getClassifiers().size() != classes.length) {
            fixMembers(getPersistentClassifiers(), classes);
            reinitClassifiers();
        } else {
            Iterator it = getClassifiers().iterator();
            for (int i = 0; i < classes.length; i++) {
                SemiPersistentElement element = (SemiPersistentElement) it.next();
                if (!element.infoIdenticalTo(classes[i])) {
                    element.setElementInfo(classes[i]);
                }
            }
        }

        imports = createChildrenList(imports, "imports", ((ResourceInfo) getElementInfo()).imports, CHANGED_IMPORTS, rebuild); // NOI18N

        childrenInited = true;

        if (elementsInited) {
            initASTElements();
        }

        fireImportsInited();
    }

    protected void initASTElements() {
        elementsInited = false;

        if (!childrenInited) {
            initChildren();
        }
        ResourceInfo info = (ResourceInfo) getElementInfo();
        packageIdentifier = (MultipartId) initOrCreate(packageIdentifier, info.getTypeAST(this));
        elementsInited = true;
    }
    
    protected void resetASTElements() {
        if (elementsInited) {
            if (packageIdentifier != null) {
                MultipartId temp = packageIdentifier;
                packageIdentifier = null;
                temp.refDelete();
            }
            elementsInited = false;
        }
    }

    public int getStatus() {
        _lock(false);
        try {
            if (getName().endsWith(".class")) { // NOI18N
                return IS_FROM_CLASSFILE;
            } else {
                MDRParser parser = getParser();
                int res = parser.getJavaFeatures();
                if (parser.hasSyntaxError()) {
                    res |= HAS_SYNTAX_ERROR;
                }
                return res;
            }
        } finally {
            _unlock();
        }
    }

    public List getErrors() {
        _lock(false);
        try {
            if (errors == null) {
                errors = new ErrorList();
            }
            return errors;
        } finally {
            _unlock();
        }
    }

    public void resetErrors() {
        if (errors != null) {
            _lock(false);
            try {
                errors = null;
            } finally {
                _unlock();
            }
        }
    }
    
    private ResourceInfo getResInfoFromClassFile() {
        Iterator iter = getPersistentClassifiers().iterator();
        FileObject resFile = JavaMetamodel.getManager().getFileObject(this);
        FileObject folder = resFile.getParent();
        ArrayList clsInfo = new ArrayList();
        ClassFileInfoUtil cfiu = new ClassFileInfoUtil();
        while (iter.hasNext()) {
            Object elem = iter.next();
            if (elem instanceof JavaClassImpl) {
                JavaClassImpl jc = (JavaClassImpl) elem;
                ClassInfo inf = cfiu.createClassInfo (folder, jc.getSimpleName(), -1);
                if (inf != null)
                    clsInfo.add(inf);
            } // if
        } // while
        return new ResourceInfo(null, ResourceInfo.RESOURCE_TYPE, this, (ClassInfo[]) clsInfo.toArray(new ClassInfo[clsInfo.size()]), null);
    }
    
    void initResource() {
        checkUpToDate(false, true, false);
    }

    protected void objectChanged(int mask) {
        if (disableChanges) return;
        if (!isChanged()) {
            ElementInfo info = getElementInfo();
            if (info.getASTree() == null && info != getDefaultInfo()) {
                throw new UnsupportedOperationException("This object is immutable."); // NOI18N
            }
            ((ExclusiveMutex) _getMdrStorage().getRepositoryMutex()).registerChange(this);
        }
        super.objectChanged(mask);
    }
    
    public void updateFromFileObject(FileObject fo, boolean force) {
        boolean fail = true;
        _lock(true);
        try {
            checkUpToDate(fo, force, false, false, false);
            fail = false;
        } finally {
            _unlock(fail);
        }
    }

    PositionBounds getFeaturePosition(FeatureImpl feature) {
        if (!isValid())
            return null;

        MDRParser parser = getParser();
        if (parser == null)
            return null;
        if (parser.mofidToBounds == null) {
            // cache position bounds
            MDRepository repository = repository();
            repository.beginTrans(false);
            try {
                parser.mofidToBounds = new HashMap();
                if (((JMManager) JavaMetamodel.getManager()).getTransactionMutex().getClassPath() == null) {
                    JavaMetamodel.getManager().setClassPath(this);
                }
                for (Iterator iter = getClassifiers().iterator(); iter.hasNext(); ) {
                    cachePositions(parser, (MetadataElement)iter.next());
                }
            } catch (InvalidObjectException e) {
                // ignore
            } finally {
                repository.endTrans(false);
            }
        }
        return parser.mofidToBounds == null ? null : (PositionBounds) parser.mofidToBounds.get(feature._getMofId());
    }
    
    private void cachePositions(MDRParser parser, MetadataElement f) {
        ASTree tree = f.getASTree();
        if (tree != null)
            parser.mofidToBounds.put(f._getMofId(), parser.createBounds(tree, tree, false));
        if (f instanceof JavaClassImpl) {
            Object[] features=((JavaClassImpl)f).getFeatures().toArray();
            for (int i=0;i<features.length;i++) {
                cachePositions(parser, (MetadataElement)features[i]);
            }
        }
    }
    
    public RuntimeException alreadyCheckingStackTrace;
    
    private void resetAST(MDRParser parser) {
        try {
            alreadyChecking = true;
            alreadyCheckingStackTrace = new RuntimeException();
            ClassIndex classIndex=ClassIndex.getIndex((JavaModelPackage)refOutermostPackage());
            Set ids=new HashSet(60);
            TokenIterator tokens;
            int idIndexes[];
            int token;
            int i=0;
            Iterator idIt;

            parser.getASTree();
            tokens=new TokenIterator(parser);
            while((token=tokens.getNextTokenType())!=0) {
                if (token==ParserTokens.IDENTIFIER) {
                    ids.add(tokens.getIdentifierText());
                }
            }
            idIndexes=new int[ids.size()];
            idIt=ids.iterator();
            while(idIt.hasNext()) {
                idIndexes[i++]=idIt.next().hashCode();
            }
            classIndex.setIdentifiers(this,idIndexes);
            updateMetadata(parser.enterMembers());
        } catch (IOException ex) {
            ErrorManager.getDefault().notify(ex);
        } finally {
            alreadyChecking = false;
        }
    }    

    public FileObject getFileObject() {
        FileObject fo = JMManager.getManager().getFileObject(this);
        if (fo != null) {
            if (fo.isVirtual()) {
                this.fobj = null;
                return null;
            }
        }
        if (fo == null) {
            fo = this.fobj;
        } else if (JMManager.PERF_DEBUG && !fo.equals(this.fobj) && this.fobj != null) {
            System.err.println("FileObject for resource " + getName() + " changed from " + this.fobj.toString() + " to " + fo.toString()); // NOI18N
        }
        if (fo != null && !fo.isValid()) fo = null;
        this.fobj = fo;
        return fo;
    }
    
    public MDRParser createMDRParser(FileObject fobj, boolean resetSourceText) {
        if (fobj == null) {
            fobj = getFileObject();
            if (fobj == null) {
                JMManager.getLog().log(ErrorManager.WARNING, "FileObject not found for resource: " + getName());
                return null;
            }
        }
        if (resetSourceText || lastSourceText == null) {
            if (!resetSourceText && isInitialized()) {
                System.err.println("Resource is initialized, but lastSourceText == null.");
                Thread.dumpStack();
            }
            MDRParser result = new MDRParser(this, fobj, null, false);
            lastSourceText = result.getSourceText();
            isFromDoc = result.isFromDocument();
            return result;
        } else {
            return new MDRParser(this, fobj, lastSourceText, isFromDoc);
        }
    }
    
    /** Method that checks whether this resource is up-to-date and does the scanning/parsing if needed.
     * @param initIfOutOfDate Forces parsing if the resource is not fully persisted yet or if it is out of date.
     *      This results in synchronizing of the persistent metadata with the source code in the resource.
     *      (Used by checkUpToDate on individual metadata elements.)
     * @param initialize Forces parsing to initialize this resource (create ResourceInfo). Used by {@link initResource}.
     * @param canUpdate Indicates whether the resource can be updated if out-of-date. This is unwanted when the resource
     *      and its children are already being used in a running transaction - updating them behind the scenes could have
     *      serious side-effects (InvalidObjectExceptions, ConcurrentModificationExceptions, etc.)
     */
    boolean checkUpToDate(boolean initIfOutOfDate, boolean initialize, boolean canUpdate) {
        FileObject fobj = JavaMetamodel.getManager().getFileObject(this);
        
        if (fobj==null) {
            JMManager.getLog().log(ErrorManager.WARNING, "FileObject not found for resource: " + getName());
            return true;
        }
        return checkUpToDate(fobj, false, initialize, initIfOutOfDate, canUpdate);
    }
    
    public boolean alreadyChecking = false;
    
    /** Method that checks whether this resource is up-to-date and does the scanning/parsing if needed.
     * @param dobj FileObject corresponding to this resource.
     * @param force Treats the resource as if it was out-of-date (i.e. as if its timestamp would not match timestamp
     *      of the corresponding file) - used for updating of files modified in the editor but not on the disk.
     * @param initialize Forces parsing to initialize this resource (create ResourceInfo). Used by {@link initResource}.
     * @param initIfOutOfDate Forces parsing if the resource is not fully persisted yet or if it is out of date.
     *      This results in synchronizing of the persistent metadata with the source code in the resource.
     *      (Used by checkUpToDate on individual metadata elements.)
     * @param canUpdate Indicates whether the resource can be updated if out-of-date. This is unwanted when the resource
     *      and its children are already being used in a running transaction - updating them behind the scenes could have
     *      serious side-effects (InvalidObjectExceptions, ConcurrentModificationExceptions, etc.)
     */
    private boolean checkUpToDate(FileObject fobj, boolean force, boolean initialize, boolean initIfOutOfDate, boolean canUpdate) {
        // we do not want to update resource when commit is in progress (it will be updated at the end)
        if (!alreadyChecking && !isChanged()) {
            canUpdate |= checkAndUpdateModCount();
            try {
                alreadyChecking = true;
                alreadyCheckingStackTrace = new RuntimeException();
                long timestamp;
                if (canUpdate) {
                    timestamp = fobj.lastModified().getTime();
                    
                    // code necessary when a file is parsed sooner that the classpath root is scanned
                    // to properly check whether a file needs to be updated
                    // (to cover the case when it was edited in the editor and on
                    // shutdown the changes were discarded)
                    if (!force && timestamp == getTimestamp()) {
                        force = isFromMemory() && !Util.isModified(fobj);
                    }
                    // ------
                    
                    // TODO: ideally we should parse the document if it is in the "modified" collection
                    // however since the parsing takes too long currently, we will not parse the file
                    // after every modification (because of the performance of the code completion)
                    // that's why this code is commented out for now
//                    force |= JavaMetamodel.getManager().removeModified(dobj);
                } else {
                    timestamp = getTimestamp();
                    if (force) {
                        // since canUpdate will be false, the resource will not be updated
                        // so I have to make sure that it stays in the needParsing set for the next
                        // transaction
                        JMManager.getTransactionMutex().addModified(fobj);
                    }
                }
                
                boolean isOutOfDate = force || getTimestamp() != timestamp;
                boolean shouldUpdate = canUpdate && isOutOfDate;
                boolean shouldParse = initialize || (initIfOutOfDate && !isPersisted()) || (shouldUpdate && (initIfOutOfDate || isInitialized()));
                if (shouldUpdate || shouldParse) {
                    if (getName().endsWith(".java")) { // NOI18N
                        directUpdate(fobj, shouldUpdate, shouldParse, timestamp);
                    } else {
                        // updating of class files does not need to be that robust - we will ignore canUpdate
                        // flag and pass isOutOfDate directly
                        directClassFileUpdate(fobj, isOutOfDate, shouldParse);
                    }
                    return false;
                } else {
                    if (lastSourceText == null && getName().endsWith(".java")) { // NOI18N
                        createMDRParser(fobj, true);
                    }
                }
            } finally {
                alreadyChecking = false;
            }
        }
        return true;
    }

    private void directClassFileUpdate(FileObject fobj, boolean isOutOfDate, boolean shouldParse) {
        if (isOutOfDate) {
            boolean changes = disableChanges;
            disableChanges = true;
            try {
                ClassUpdater.updateIndex(this,fobj);
            } finally {
                disableChanges = changes;
            }
        }
        if (shouldParse) {
            ResourceInfo resInfo=getResInfoFromClassFile();
            updateMetadata(resInfo);
        } else {
            setPersisted(false);
        }
    }

    private void directUpdate(FileObject fobj, boolean isOutOfDate, boolean shouldParse, long timestamp) {
        if (JMManager.INCONSISTENCY_DEBUG) {
            System.err.println("Direct update of: " + getName());
            System.err.println("isOutOfDate: " + isOutOfDate + " shouldParse: " + shouldParse);
            Thread.dumpStack();
        }
        
        MDRParser parser = createMDRParser(fobj, isOutOfDate);
        
        TokenIterator tokenIterator = null;
        Reader reader = null;
        
        boolean changes = disableChanges;
        disableChanges = true;
        try {
            if (shouldParse) {
                parser.getASTree();
                tokenIterator = new TokenIterator(parser);
            } else if (isOutOfDate) {
                reader = parser.getReader();
                tokenIterator = new TokenIterator(reader, parser.getSourceLevel());
            }

            if (isOutOfDate) {
                errors = null;
                int ids[];
                JavaModelPackage pck=(JavaModelPackage)refOutermostPackage();
                ClassIndex classIndex=ClassIndex.getIndex(pck);
                
                boolean isModified = Util.isModified(fobj);
                
                setTimestamp(timestamp, isModified);
                ids=new JavaUpdater(pck, null, null).computeIndex(this, tokenIterator);
                classIndex.setIdentifiers(this,ids);
                if (!shouldParse) {
                    setPersisted(false);
                }
            }
        } catch (IOException e) {
            ErrorManager.getDefault().notify(e);
        } finally {
            disableChanges = changes;
            try {
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException e) {
                ErrorManager.getDefault().notify(e);
            }
        }
        if (shouldParse) {
            errors = null;
            ResourceInfo resInfo=parser.enterMembers();

            if (resInfo==null) { // syntax error
                if (isInitialized()) {
                    JMManager.fireResourceParsed(_getMofId());
                    return;
                }
                Collection javaClasses=getPersistentClassifiers();
                ClassInfo classes[]=new ClassInfo[javaClasses.size()];
                int i=0;
                // the naked collection does not implement clear() method
                // we need to iterate through it and remove elements one by one
                for (Iterator it = javaClasses.iterator(); it.hasNext();) {
                    JavaClassImpl jcls=(JavaClassImpl)it.next();
                    int infoType=jcls.isInterface()?ClassInfo.INTERFACE_TYPE:ClassInfo.CLASS_TYPE;
                    ClassInfo clsInfo=new ClassInfo(null, infoType, jcls.getName(), jcls.getModifiers(), null, null, null, null, null);

                    classes[i++]=clsInfo;
                }
                resInfo=new ResourceInfo(null, ResourceInfo.RESOURCE_TYPE, this, classes, null);
            }
            updateMetadata(resInfo);
            JMManager.fireResourceParsed(_getMofId());
        }
    }
    
    private void updateMetadata(ResourceInfo info) {
        if (infoIdenticalTo(info)) return;
        updatePersistent(info);
        setElementInfo(info);
    }
    
    void setFileToDeleteOnRollback(FileObject file){
        this.fileToDeleteOnRollback = file;
    }

    public void rollbackChanges() {
        if (fileToDeleteOnRollback != null) {
            try {
                fileToDeleteOnRollback.delete();
            } catch (IOException ioe) {
                ErrorManager.getDefault().notify(ioe);
            }
        }

        rollback();

        if (invertedDiffs != null) {
            rollbackDiff();
            invertedDiffs = null;
            MDRParser parser = createMDRParser(null, true);
            try {
                alreadyChecking = true;
                alreadyCheckingStackTrace = new RuntimeException();
                ResourceInfo resInfo=parser.enterMembers();
                uninitialize();
                updateMetadata(resInfo);
            } finally {
                alreadyChecking = false;
            }
        }

    }

    protected void uninitialize() {
        super.uninitialize();
    }

    public void commitChanges() {
        errors = null;
        fileToDeleteOnRollback = null;
        if (!READ_ONLY) {
            String newSourceText = null;
            DiffList tmpdiff = new DiffList(getParser().isFromDocument());
            if (isNew()) {
                newSourceText = getSourceText();
            } else {
                getDiff(tmpdiff);
                if (isChanged(CHANGED_EXTERNAL)) {
                    DiffList dl = new DiffList(getParser().isFromDocument());
                    Iterator external = extDiffs.iterator();
                    DiffElement extDiff = (DiffElement) external.next();
                    for (Iterator it = tmpdiff.iterator(); it.hasNext();) {
                        DiffElement de = (DiffElement) it.next();
                        while (extDiff != null && extDiff.getStartOffset() < de.getStartOffset()) {
                            dl.add(extDiff);
                            extDiff = external.hasNext() ? (DiffElement) external.next() : null;
                        }
                        dl.add(de);
                    }

                    while (extDiff != null) {
                        dl.add(extDiff);
                        extDiff = external.hasNext() ? (DiffElement) external.next() : null;
                    }
                    
                    tmpdiff = dl;
                }
            }
            extDiffs = null;
            DataObject dobj = JavaMetamodel.getManager().getDataObject(this);
            final CloneableEditorSupport editor = Util.findCloneableEditorSupport(dobj);
            StyledDocument doc = null;
            if (editor != null) {
                doc = editor.getDocument();
                if (doc == null && getParser().isFromDocument()) {
                    try {
                        doc = editor.openDocument();
                    } catch (IOException ex) {
                        throw (RuntimeException) ErrorManager.getDefault().annotate(new RuntimeException(), ex);
                    }
                }
            }
            if (doc != null) {
                if (newSourceText == null) {
                    final DiffList diff = tmpdiff;
                    NbDocument.runAtomic(doc, new Runnable() {
                        public void run() {
                            applyDiff(diff, editor); // fail
                        }
                    });
                } else {
                    try {
                        PositionRef start = editor.createPositionRef(0, Position.Bias.Forward);
                        PositionRef end = editor.createPositionRef(doc.getLength(), Position.Bias.Backward);
                        new PositionBounds(start, end).setText(newSourceText);
                        setTimestamp(getTimestamp(), true);
                    } catch (Exception e) {
                        throw (RuntimeException) ErrorManager.getDefault().annotate(new RuntimeException(), e);
                    }
                }
            } else {
                if (newSourceText == null) {
                    applyDiff(tmpdiff, dobj); // fail
                } else {
                    FileObject primaryFile = dobj.getPrimaryFile();
                    OutputStream stream = null;
                    FileLock lock = null;
                    try {
                        lock = primaryFile.lock();
                        stream = primaryFile.getOutputStream(lock);
                        stream.write(newSourceText.getBytes());
                        stream.close();
                        setTimestamp(primaryFile.lastModified().getTime(), false);
                    }
                    catch (IOException ex) {
                        throw (RuntimeException) ErrorManager.getDefault().annotate(new RuntimeException(ex.toString()), ex);
                    }
                    finally {
                        if (lock != null)
                            lock.releaseLock();
                    }
                }
            }
        }
    }

    /** Method called by ExclusiveMutex to confirm that the file is still parsable after commit.
     */
    public void parseResource() {
        if (!READ_ONLY) {
            FileObject fobj = getFileObject();
            MDRParser parser = createMDRParser(fobj, true);
            if (parser.getASTree() == null) {
                throw new ConstraintViolationException(null, this, "Cannot parse resource " + getName()); // NOI18N
            }
            commit();
            try {
                resetAST(parser);
            } catch (Exception e) {
                ErrorManager.getDefault().notify(e);
            }
        }
    }
    
    public void commitConfirmed() {
        JavaMetamodel.getUndoManager().addItem(this, invertedDiffs);
        invertedDiffs = null;
        rbText = null;
        rbList = null;
        JMManager.fireResourceParsed(_getMofId());
    }

    public void applyDiff(DiffList diff) {
        DataObject dobj = JavaMetamodel.getManager().getDataObject(this);
        final CloneableEditorSupport editor = Util.findCloneableEditorSupport(dobj);
        if (diff.isFromDocument()) {
            if (editor!=null) {
                StyledDocument doc = null;
                try {
                    doc = editor.openDocument();
                } catch (IOException ex) {
                    ErrorManager.getDefault().notify(ex);
                }
                assert doc !=null : this.getName();
                final DiffList diffList = diff;
                NbDocument.runAtomic(doc, new Runnable() {
                    public void run() {
                        applyDiff(diffList, editor);
                    }
                });
            }
        } else {
            if (editor!=null) {
                StyledDocument doc = editor.getDocument();
                if (doc!=null) {
                    final DiffList diffList = diff;
                    NbDocument.runAtomic(doc, new Runnable() {
                        public void run() {
                            applyDiff(diffList, editor);
                        }
                    });
                } else {
                    applyDiff(diff, dobj);
                }
            }
        }
        JavaMetamodel.getUndoManager().addItem(this, invertedDiffs);
    }

    private void applyDiff(DiffList diffList, CloneableEditorSupport doc) {
        List diff = new ArrayList(diffList);
        DiffList tempDiffs = null;
        List list = new ArrayList(diff.size());
        try {
            HashMap positions = new HashMap(diff.size());
            int[] offsets = new int[diff.size() * 2];
            int i = 0;
            MDRParser parser = getParser();

            for (ListIterator it = diff.listIterator(); it.hasNext();) {
                DiffElement de = (DiffElement) it.next();
                if (de.getText().startsWith("\n")) {
                    int increment = 0;
                    String sourceText = parser.getSourceText();
                    int startOffset = de.getStartOffset();
                    if (sourceText.length() > startOffset) {
                        char ch = sourceText.charAt(startOffset);
                        if (ch == '\n') {
                            increment = 1;
                        } else if (ch == '\r') {
                            if (sourceText.substring(startOffset).startsWith("\r\n")) {
                                increment = 2;
                            }
                        }

                        if (increment > 0) {
                            int so = startOffset + increment, eo = de.getEndOffset();
                            String text = de.getText().substring(1);
                            if (so > eo) {
                                eo = so;
                                text += '\n';
                            }
                            de = new DiffElement(so, eo, text);
                            it.set(de);
                        }
                    }
                }
                offsets[i++] = de.getStartOffset();
                offsets[i++] = de.getEndOffset();
            }

            if (parser != null && offsets.length > 0 && !diffList.isFromDocument()) {
                offsets = parser.convertToDocumentOffsets(offsets);
            }
            
            for (i = 0; i < diff.size(); i++) {
                PositionRef start = doc.createPositionRef(offsets[i * 2], Position.Bias.Forward);
                PositionRef end = doc.createPositionRef(offsets[i * 2 + 1], Position.Bias.Backward);
                positions.put(diff.get(i), new PositionBounds(start, end));
            }

            tempDiffs = new DiffList(true);
            for (Iterator it = diff.iterator(); it.hasNext();) {
                DiffElement de = (DiffElement) it.next();
                PositionBounds pos = (PositionBounds) positions.get(de);
                int startOffset = pos.getBegin().getOffset();
                DiffElement diffElement = new DiffElement(startOffset, startOffset+de.getText().length(), pos.getText());
                String text = pos.getText();
                pos.setText(de.getText());
                list.add (new PosInfo(text, pos));
                tempDiffs.add(diffElement);
            }
        } catch (Exception e) {
            throw (ConstraintViolationException) ErrorManager.getDefault().annotate(
                new ConstraintViolationException(null, this, "Exception thrown when applying diff: " + e.getMessage()), e // NOI18N
            );
        }
        invertedDiffs = tempDiffs;
        rbList = list;
        // force timestamp change events (#56380)
        setTimestamp(getTimestamp(), true);
    }

    private void applyDiff(DiffList diff, DataObject dobj) {
        MDRParser parser = getParser();
        String orig = parser.getSourceText();
        int lastPrinted = 0;
        StringBuffer buf = new StringBuffer();

        invertedDiffs = null;
        DiffList tempDiffs = new DiffList(false);
        int shift = 0;
        int shiftedOffset = 0;

        for (Iterator it = diff.iterator(); it.hasNext();) {
            DiffElement de = (DiffElement) it.next();
            int startOffset = de.getStartOffset();
            int endOffset = de.getEndOffset();
            buf.append(orig.substring(lastPrinted, startOffset));
            String d = lineSepPattern.matcher(de.getText()).replaceAll(lineSeparator);
            buf.append(d);
            lastPrinted = endOffset;
            shiftedOffset = startOffset + shift;
            tempDiffs.add(new DiffElement(shiftedOffset, shiftedOffset + d.length(), orig.substring(startOffset, endOffset)));
            shift += startOffset + d.length() - endOffset;
        }

        buf.append(orig.substring(lastPrinted));
        FileObject primaryFile = dobj.getPrimaryFile();
        OutputStream stream = null;
        FileLock lock = null;
        try {
            lock = primaryFile.lock();
            stream = primaryFile.getOutputStream(lock);
            String encoding = Util.getFileEncoding(primaryFile);
            if (encoding == null) {
                stream.write(buf.toString().getBytes());
            } else {
                stream.write(buf.toString().getBytes(encoding));
            }
            stream.close();

            invertedDiffs = tempDiffs;
            rbText = orig;
            setTimestamp(primaryFile.lastModified().getTime(), false);
        }
        catch (IOException ex) {
            ex.printStackTrace();
            throw (ConstraintViolationException) ErrorManager.getDefault().annotate(
                new ConstraintViolationException(null, this, "Exception thrown when applying diff: " + ex.getMessage()), ex // NOI18N
            );
        }
        finally {
            if (lock != null)
                lock.releaseLock();
        }
    }

    private void rollbackDiff() {
        DataObject dobj = JavaMetamodel.getManager().getDataObject(this);
        if (rbList != null) {            
            final CloneableEditorSupport editor = Util.findCloneableEditorSupport(dobj);
            StyledDocument doc = editor.getDocument();
            final List list = rbList;
            rbList = null;
            NbDocument.runAtomic(doc, new Runnable() {
                public void run() {
                    for (Iterator iter = list.iterator(); iter.hasNext();) {
                        PosInfo info = (PosInfo) iter.next();
                        try {
                            info.pos.setText(info.text);
                        } catch (Exception ex) {
                            ErrorManager.getDefault().notify(ex);
                        }
                    }
                }
            });
            setTimestamp(getTimestamp(), true);
        } else {
            FileObject primaryFile = dobj.getPrimaryFile();
            OutputStream stream = null;
            FileLock lock = null;
            try {
                lock = primaryFile.lock();
                stream = primaryFile.getOutputStream(lock);
                stream.write(rbText.getBytes());
                stream.close();
                setTimestamp(primaryFile.lastModified().getTime(), false);
            }
            catch (IOException ex) {
                ErrorManager.getDefault().notify(ex);
            }
            finally {
                if (lock != null)
                    lock.releaseLock();
                rbText = null;
            }
        }
    }
    
    public void getDiff(List diff) {
        ResourceInfo astInfo = (ResourceInfo) getElementInfo();
        ASTProvider parser = getParser();

        // package statement
        ASTree tree = getASTree();
        String packageName = getPackageName();
        int endOffsetAfterRemovePackage = -1;
        if (isChanged(CHANGED_PACKAGE_NAME) && tree.getSubTrees()[0] == null) {
            if (packageName != null && packageName.length() > 0) {
                diff.add(new DiffElement(0, 0, "package " + getPackageName() + ";\n")); // NOI18N
            }
        } else if (isChanged(CHANGED_PACKAGE_NAME) && (packageName == null || packageName.length() == 0)) {
            // remove package statement if the resource is located in default
            // package
            int startPads = getStartOffset(parser, tree.getSubTrees()[0], false);
            // assume there is only package statement with no new line after.
            // (value will be rewritten if it is not true)
            int endOff = getEndOffset(parser, tree.getSubTrees()[0]);
            ASTree[] rscChildren = tree.getSubTrees();
            Token newLine = null;
            // find the first new line after package statement, if there exists
            if (rscChildren[1] != null) {
                // there is at least one import
                newLine = getFirstEOL(parser, rscChildren[1].getFirstToken());
            } else if (rscChildren[2] != null) {
                // there is no import, but at least one classifier
                newLine = getFirstEOL(parser, rscChildren[2].getFirstToken());
            }
            if (newLine != null) {
                // we knows about first new line after the package statement,
                // We will update the end offset character.
                endOff = newLine.getEndOffset();
            }
            diff.add(new DiffElement(startPads, endOff, ""));
            endOffsetAfterRemovePackage = endOff;
        } else if (tree.getSubTrees()[0] != null) {
            getChildDiff(diff, parser, tree.getSubTrees()[0].getSubTrees()[1], (MetadataElement) getPackageIdentifier(), CHANGED_PACKAGE_NAME);
        }

        // import statements
        int endOffset;
        int endOffset2 = -1;
        ASTree typeDecls = tree.getSubTrees()[2];
        if (typeDecls == null) {
            endOffset = parser.getSourceText().length();
        } else {
            if (isChanged(CHANGED_IMPORTS)) {
                // find the end of import section.
                if (tree.getSubTrees()[0] == null && tree.getSubTrees()[1] == null) {
                    // there is not any package or import statement, the
                    // compilation unit starts with 'class Anything...'.
                    // start at the beginning of file.
                    endOffset = 0;
                } else if (tree.getSubTrees()[1] == null) {
                    if (packageName==null || "".equals(packageName)) {
                        endOffset = endOffsetAfterRemovePackage;
                    } else {
                        // there is package statement but no import
                        int firstAfterPckg = tree.getSubTrees()[0].getLastToken()+1;
                        Token t1 = parser.getToken(firstAfterPckg);
                        Token[] pads = t1.getPadding();
                        endOffset = getStartOffset(parser, typeDecls, true);
                        // look for first EOL token and set the endOffset var.
                        for (int i = 0; i < pads.length; i++) {
                            if (pads[i].getType() == ParserTokens.EOL) {
                                endOffset = pads[i].getStartOffset();
                                break;
                            }
                        }
                        diff.add(new DiffElement(endOffset, endOffset, "\n\n")); // NOI18N
                    }
                } else {
                    // there are imports already.
                    int lastImp = tree.getSubTrees()[1].getLastToken();
                    Token t1 = parser.getToken(lastImp+1);
                    Token[] pads = t1.getPadding();
                    endOffset = parser.getToken(lastImp).getEndOffset();
                    // look for first EOL token and set the endOffset var.
                    for (int i = 0; i < pads.length; i++) {
                        if (pads[i].getType() == ParserTokens.EOL) {
                            endOffset = pads[i].getStartOffset();
                            if (getImports().isEmpty()) {
                                endOffset2 = pads[i].getEndOffset();
                            } else {
                                int inPos = parser.getText(pads[i]).lastIndexOf('\n');
                                endOffset2 = inPos != -1 ? endOffset + inPos : endOffset;
                            }
                            break;
                        }
                    }
                }
            } else {
                endOffset = getStartOffset(parser, typeDecls, true);
            }
        }
        int diff_size = diff.size();
        
        getImportsDiff(diff, parser, astInfo.imports, getImports(), endOffset, endOffset2, "\n", false, CHANGED_IMPORTS); // NOI18N

        // there was neither package nor import statement in source, put
        // new line after new imports when imports where changed.
        if (isChanged(CHANGED_IMPORTS) && tree.getSubTrees()[0] == null && tree.getSubTrees()[1] == null || endOffset == endOffsetAfterRemovePackage) {
            diff.add(new DiffElement(endOffset, endOffset, "\n")); // NOI18N
        }

        // classes
        endOffset = parser.getSourceText().length();
        getCollectionDiff(diff, parser, CHANGED_CLASSIFIERS, astInfo.classes, getClassifiers(), endOffset, "\n\n"); // NOI18N
    }

    public String getSourceText() {
        String origElem;
        if ((origElem = checkChange()) != null) {
            return origElem;
        }
        StringBuffer buf = new StringBuffer();

        String packageName = getPackageName();
        
        if (packageName != null && packageName.length() > 0) {
            buf.append("package "); // NOI18N
            buf.append(getPackageName());
            buf.append(";\n\n"); // NOI18N
        }
        
        for (Iterator importsIt = getImports().iterator(); importsIt.hasNext(); ) {
            ImportImpl imp = (ImportImpl) importsIt.next();
            buf.append(imp.getSourceText() + "\n"); // NOI18N
        }
        for (Iterator clazzIt = getClassifiers().iterator(); clazzIt.hasNext(); ) {
            JavaClassImpl javaClass = (JavaClassImpl) clazzIt.next();
            buf.append('\n' + javaClass.getSourceText() + '\n');
        }
        return buf.toString();
    }

    /**
     * Overrides default implementation from SemiPersistentElement.
     * It guarantees interruption of the getIndentation() calls' chain for
     * newly created elements.
     * See example below.
     *
     * Example:
     * Consider situation, when you have created new Resource 'Foo.java'.
     * You have created JavaClass 'public class Foo' and put it to the
     * Resource. In addition to, you have created Field 'int count;'. Now
     * when generator starts, it goes through the elements and generates them.
     * When getSourceText() method in Field is called, it asks for the
     * indentation. Default implementation in SemiPersistentElement.getIndentation()
     * asks for indentation of its parent, which is JavaClass. It calls the
     * same default implementation, which asks for the parents (in this case Resource)
     * implementation. It is empty string, so it stops the getIndentation()
     * calls' chain.
     *
     * @return  empty string, see comment above.
     */
    protected String getIndentation() {
        return "";
    }

    // ..........................................................................

    protected void _delete() {
        // --- delete from package -----------------------------------------
        JavaPackage parent = (JavaPackage) _realImmediateComposite();
        if (parent != null) {
            parent.getResources().remove(this);
        }
        // --- delete components -------------------------------------------
        // delete all classifiers
        for (Iterator it =  getPersistentClassifiers().iterator(); it.hasNext();) {
            RefObject classifier = (RefObject) it.next();
            it.remove();
            classifier.refDelete();
        }
        if (childrenInited) {
            deleteChildren(imports);
            if (elementsInited)
                deleteChild(packageIdentifier);
        }
        // --- delete from indexes ----------------------------------------
        ClassIndex.getIndex((JavaModelPackage) refOutermostPackage()).removeResource(this);
        ((ExclusiveMutex) _getMdrStorage().getRepositoryMutex()).unregisterChange(this);

        super._delete();
    }

    public void replaceChild(Element oldElement, Element newElement) {
        if (replaceObject(classifiersInited() ? getClassifiers() : getPersistentClassifiers(), oldElement, newElement)) return;
        
        if (childrenInited) {
            if (replaceObject(getImports(), oldElement, newElement)) return;
        }
        if (elementsInited && oldElement.equals(packageIdentifier)) {
            if (newElement != null) { // [PENDING] null value causes that a composite (package) is set
                setPackageIdentifier((MultipartId) newElement);
            } // [PENDING]
            return;
        }
        super.replaceChild(oldElement, newElement);
    }

    // ..........................................................................
    // a hack allowing the bridge to be informed about imports initialization
    // ..........................................................................

    // [TODO] synchronization

    private PropertyChangeSupport importsSupp;

    public boolean importsInited () {
        return childrenInited;
    }

    public void addImportsListener (PropertyChangeListener listener) {
        if (childrenInited) {
            listener.propertyChange (null);
        } else {
            synchronized (this) {
                if (importsSupp == null) {
                    importsSupp = new PropertyChangeSupport (this);
                }
            }
            importsSupp.addPropertyChangeListener(listener);
        }
    }

    private void fireImportsInited () {
        if (importsSupp != null)
            importsSupp.firePropertyChange (null, null, null);
        importsSupp = null;
    }

    // end of the hack ..........................................................

    protected ASTree getPartTree(ElementPartKind part) {
        if (ElementPartKindEnum.HEADER.equals(part))
            return getASTree().getSubTrees()[0];
        throw new IllegalArgumentException("Invalid part for this element: " + part); // NOI18N
    }

    public MultipartId getPackageIdentifier() {
        checkUpToDate();
        if (!elementsInited) {
            initASTElements();
        }
        return packageIdentifier;
    }

    public void setPackageIdentifier(MultipartId packageIdentifier) {
        objectChanged(CHANGED_PACKAGE_NAME);
        changeChild(getPackageIdentifier(), packageIdentifier);
        this.packageIdentifier = packageIdentifier;
//        doSetPackageIdentifier(packageIdentifier);
        setPkgNameAndUpdateIdx(packageIdentifier != null ?
            ((MetadataElement) packageIdentifier).getSourceText() : "");
    }

//    private void doSetPackageIdentifier(MultipartId packageIdentifier) {
//        objectChanged(CHANGED_PACKAGE_NAME);
//        changeChild(getPackageIdentifier(), packageIdentifier);
//        this.packageIdentifier = packageIdentifier;
//    }

    void childChanged(MetadataElement element) {
        if (element == getPackageIdentifier()) {
            setPkgNameAndUpdateIdx(((MetadataElement) packageIdentifier).getSourceText());
        }
    }

    public boolean containsIdentifier(String identifier) {
        return ClassIndex.containsIdentifier(this, identifier.hashCode());
    }

    /**
     * Tries to find first EOL in paddings of provided token. If there are not
     * any padding connected to token or there is not any EOL token in its
     * paddings, returns null. Otherwise returns EOL token.
     *
     * @param   parser  parser which is used for obtaining token by token id
     * @param   token   token id
     * @return  token representing first EOL token of parameters paddings
     *          or null if EOL token is not present
     */
    private static Token getFirstEOL(ASTProvider parser, int token) {
        Token[] pads = parser.getToken(token).getPadding();
        for (int i = 0; i < pads.length; i++) {
            if (pads[i].getType() == ParserTokens.EOL) {
                return pads[i];
            }
        }
        return null;
    }
    
    // ..........................................................................
    
    private static class PosInfo {
        String text;
        PositionBounds pos;
        
        PosInfo(String text, PositionBounds pos) {
            this.text = text;
            this.pos = pos;
        }
    }

    private class ErrorList extends AbstractList implements ErrConsumer {
        private List errors = null;
        private String name;
        
        public ErrorList() {
            name = getName().replace('/', File.separatorChar);
        }

        public Object get(int index) {
            initCheck();
            return errors.get(index);
        }

        private void initCheck() {
            synchronized (this) {
                if (errors != null) {
                    return;
                }
            }
            MDRParser provider;
            _getRepository().beginTrans(false);
            try {
                provider = getParser();
            } finally {
                _getRepository().endTrans();
            }
            synchronized (this) {
                if (errors == null && provider != null) {
                    try {
                        ECRequestDesc desc = new ECRequestDescImpl(name, provider, this);
                        Factory.getDefault().getErrorChecker(desc).parse();
                    } catch (CompilerException ex) {
                        ErrorManager.getDefault().log(ErrorManager.WARNING, "ErrorChecker: " + ex.getMessage()); // NOI18N
                    }
                }
                if (errors == null) {
                    errors = Collections.EMPTY_LIST;
                }
            }
        }

        public int size() {
            initCheck();
            return errors.size();
        }

        public void pushError(Object severity, String errorFileName, int line, int column, String message, String key, String[] args) {
            if (errorFileName.equals(name)) {
                if (errors == null) {
                    errors = new ArrayList();
                }
                errors.add(new ErrorInfoImpl(severity, line, column, message, key, args));
            }
        }
    }

    private static class ErrorInfoImpl implements ErrorInfo {
        private final String description;
        private final int line, column;
        private final ErrorType severity;
        private final String errorId;
        private final String[] args;

        private static List fieldNames = Arrays.asList((Object[]) new String[] {"description", "lineNumber", "column", "severity"}); // NOI18N
        private static List typeName = Arrays.asList((Object[]) new String[] {"JavaModel", "ErrorInfo"}); // NOI18N

        ErrorInfoImpl(Object severity, int line, int column, String message, String key, String[] args) {
            this.description = message;
            this.line = line;
            this.column = column;
            this.errorId = key;
            this.args = args;
            this.severity = severity == ErrConsumer.WARNING ? ErrorTypeEnum.WARNING : ErrorTypeEnum.ERROR;
        }

        public String getDescription() {
            return description;
        }

        public int getLineNumber() {
            return line;
        }

        public int getColumn() {
            return column;
        }

        public ErrorType getSeverity() {
            return severity;
        }

        public List refFieldNames() {
            return fieldNames;
        }

        public String getErrorId() {
            return errorId;
        }

        public List getArguments() {
            return Arrays.asList(args);
        }
        
        public Object refGetValue(String s) {
            if ("description".equals(s)) { // NOI18N
                return getDescription();
            } else if ("lineNumber".equals(s)) { // NOI18N
                return new Integer(getLineNumber());
            } else if ("column".equals(s)) { // NOI18N
                return new Integer(getColumn());
            } else if ("severity".equals(s)) { // NOI18N
                return getSeverity();
            } else if ("errorId".equals(s)) { // NOI18N
                return getErrorId();
            } else if ("arguments".equals(s)) { // NOI18N
                return getArguments();
            } else {
                throw new InvalidNameException(s);
            }
        }

        public List refTypeName() {
            return typeName;
        }
    }
    
    public static class DiffList extends AbstractList {
        private final ArrayList delegate = new ArrayList();
        private boolean fromDocument;
        
        public DiffList(boolean isFromDocument) {
            fromDocument = isFromDocument;
            
        }
        
        public int size() {
            return delegate.size();
        }
        
        public boolean isFromDocument() {
            return fromDocument;
        }
        
        public Object get(int index) {
            return delegate.get(index);
        }

        public void add(int index, Object object) {
            DiffElement element = (DiffElement) object;
            if (index == delegate.size()) {
                add(object);
                return;
            }
            DiffElement diff = (DiffElement) delegate.get(index);
            if (element.getEndOffset() > diff.getStartOffset()) {
                throw new IllegalArgumentException("Start offset of the diff element is lower than end offset of the previous diff element. (previous: " + diff + ", this: " + element + ")"); // NOI18N
            } else if (element.getEndOffset() == diff.getStartOffset()) {
                delegate.remove(index);
                element = new DiffElement(element.getStartOffset(), diff.getEndOffset(), element.getText() + diff.getText());
            }
            if (index > 0) {
                diff = (DiffElement) delegate.get(index - 1);
                if (diff.getEndOffset() > element.getStartOffset()) {
                    throw new IllegalArgumentException("Start offset of the diff element is lower than end offset of the previous diff element. (previous: " + diff + ", this: " + element + ")"); // NOI18N
                } else if (diff.getEndOffset() == element.getStartOffset()) {
                    delegate.set(index - 1, new DiffElement(diff.getStartOffset(), element.getEndOffset(), diff.getText() + element.getText()));
                    return;
                }
            }
            delegate.add(index, element);
        }
        
        public Object remove(int index) {
            return delegate.remove(index);
        }
        
        public Object set(int index, Object object) {
            Object result = remove(index);
            add(index, object);
            return result;
        }

        public boolean add(Object object) {
            DiffElement element = (DiffElement) object;
            if (!delegate.isEmpty()) {
                int lastIndex = delegate.size() - 1;
                DiffElement diff = (DiffElement) delegate.get(lastIndex);
                if (diff.getEndOffset() > element.getStartOffset()) {
                    throw new IllegalArgumentException("Start offset of the diff element is lower than end offset of the previous diff element. (previous: " + diff + ", this: " + element); // NOI18N
                } else if (diff.getEndOffset() == element.getStartOffset()) {
                    delegate.set(lastIndex, new DiffElement(diff.getStartOffset(), element.getEndOffset(), diff.getText() + element.getText()));
                    return true;
                }
            }
            return delegate.add(element);
        }
    }

    protected void hardRefParent(boolean enabled) {
    }

    protected void parentChanged() {
    }
    
    public Element duplicate(JavaModelPackage targetExtent) {
        throw new UnsupportedOperationException("The operation is intentionally unsupported at this element."); // NOI18N 
    }
    
    /** Returns true if a given offset points into a comment in a source file.
     */
    public boolean isComment(int offset) {
        MDRParser parser = getParser();
        Token token = parser.getTokenByOffset(offset);
        if (token == null) return false;
        return token.getType() == ParserTokens.COMMENT || token.getType() == ParserTokens.EOL_COMMENT;
    }
    
    public void addExtDiff(DiffElement diff) {
        objectChanged(CHANGED_EXTERNAL);
        if (extDiffs == null) {
            extDiffs = new DiffList(getParser().isFromDocument());
        }
        extDiffs.add(diff);
    }
    
    public Element getElementByOffset(int offset) {
        _lock(false);
        try {
            ElementFinder finder = new ElementFinder(this);

            return finder.getElementByOffset(offset);
        } finally {
            _unlock();
        }
    }

    public boolean addImport(Import newImport) {
        boolean fail = true;
        _lock(true);
        try {
            String name=newImport.getName();
            ListIterator it = getImports().listIterator();

            while (it.hasNext()) {
                Import imp =(Import)it.next();
                String impName=imp.getName();
                
                if (impName!=null && name.compareTo(impName)<0) {
                    it.previous();
                    break;
                }
            }   
            it.add(newImport);
            fail = false;
            return true;
        } finally {
            _unlock(fail);
        }
    }

    public List getMain() {
        _lock(false);
        try {
            if (getName().endsWith(".class") || containsIdentifier("main")) { //NOI18N
                List mainClasses=new ArrayList(1);
                for (Iterator i = getClassifiers().iterator(); i.hasNext();) {
                    findMainIn((JavaClass) i.next(), mainClasses);
                }
                return mainClasses;
            }
            return Collections.EMPTY_LIST;
        } finally {
            _unlock();
        }
    }

    private void findMainIn(final JavaClass clazz, final Collection mainClasses) {
        if (clazz.isInterface())
            return;
        final int correctMods = (Modifier.PUBLIC | Modifier.STATIC);
        Object[] features = clazz.getContents().toArray();
        boolean mainFound = false;
        
        for (int j = 0; j<features.length;j++) {
            ClassMember feature = (ClassMember)features[j];
            
            if (feature instanceof JavaClass)
                findMainIn((JavaClass)feature, mainClasses);
            if (mainFound)
                continue;
            // if it is not a method, continue with next feature
            if (!(feature instanceof Method))
                continue;

            Method m = (Method) feature;
            // check that method is named 'main' and has set public 
            // and static modifiers! Method has to also return
            // void type.
            if (!"main".equals(m.getName()) || // NOI18N
               ((m.getModifiers() & correctMods) != correctMods) ||
               (!"void".equals(m.getType().getName()))) // NOI18N
               continue;

            // check parameters - it has to be one of type String[]
            // or String...
            Iterator parIt=m.getParameters().iterator();
            if (parIt.hasNext()) {
                Parameter par = (Parameter) parIt.next();
                if (parIt.hasNext()) // more than one parameter
                    continue;
                String typeName = par.getType().getName();
                if (par.isVarArg() && ("java.lang.String".equals(typeName) || "String".equals(typeName))) { // NOI18N
                    // Main methods written with variable arguments parameter:
                    // public static main(String... args) {
                    // }
                    mainClasses.add(clazz);
                    mainFound = true;
                } else if ("java.lang.String[]".equals(typeName) || "String[]".equals(typeName)) { // NOI18N
                    // Main method written with array parameter:
                    // public static main(String[] args) {
                    // }
                    mainClasses.add(clazz);
                    mainFound = true;
                }
            } // end if parameters
        } // end features cycle
    }

    void setData() {
        imports = createChildrenList("imports",Collections.EMPTY_LIST, CHANGED_IMPORTS); // NOI18N
    }
}
