/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.ArrayList;
import java.util.Iterator;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.handlers.AttrListWrapper;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.parser.ASTProvider;
import org.netbeans.modules.javacore.parser.ElementInfo;
import org.netbeans.modules.javacore.parser.ParameterInfo;
import org.netbeans.modules.javacore.parser.TypeRef;
import org.openide.util.Utilities;
import java.util.List;
import org.netbeans.modules.javacore.parser.ArrayRef;

/**
 * Implementation of Parameter object instance interface.
 *
 * @author Vladimir Hudec
 * @author Pavel Flaska
 */
public abstract class ParameterImpl extends SemiPersistentElement implements Parameter {
    private static final ElementInfo DEFAULT_INFO = new ParameterInfo(null, ParameterInfo.PARAMETER_TYPE, null, false, null, false, null);
    
    private TypeReference typeName = null;
    private int dimCount = 0;
    private LightAttrList annotations = null;
    private boolean internalSetDim = false;
    
    protected ParameterImpl(StorableObject s) {
        super(s);
    }
    
    /** Should be overriden by elements that have persistent attributes.
     * They should implement this method to compare values in newly passed AST info with
     * values of the persistent attributes and if the values do not match, they should be
     * updated and proper events should be fired.
     */
    protected void matchPersistent(ElementInfo info) {
        super.matchPersistent(info);
        ParameterInfo pinfo = (ParameterInfo) info;
        
        if (pinfo.isFinal != isFinal()) {
            setFinal(pinfo.isFinal);
        }
        if (pinfo.isVarArg != isVarArg()) {
            setVarArg(pinfo.isVarArg);
        }
        
        if (!isPersisted()) {
            setPersisted(true);
            persist();
            setTypeRef(pinfo.type);
            persistChildren(getPersistentList("annotations", super_getAnnotations()), pinfo.annotations);
        } else {
            if (!Utilities.compareObjects(pinfo.type, getTypeRef())) {
                setType(resolveType(pinfo.type));
            }    
            processMembers(getAnnotations(), pinfo.annotations);
        }
    }
    
    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }
    
    protected void resetChildren() {
        super.resetChildren();
        if (typeName != null) {
            TypeReference temp = typeName;
            changeChild(typeName, null);
            typeName = null;
            temp.refDelete();
        }
        if (annotations != null) {
            annotations.setInnerList(getPersistentList("annotations", super_getAnnotations()));     // NOI18N
        }
        childrenInited = false;
    }
    
    /**
     * Returns the value of reference type.
     * @return Value of reference type.
     */
    public Type getType() {
        checkUpToDate();
        return resolveType(getTypeRef());
    }

    private void fireTypeNameChange(TypeReference typeReference) {
        Object oldValue = null;
        Object newValue = null;
        if (childrenInited && !disableChanges) {
            oldValue = getTypeName();
            newValue = typeReference;
        }
        fireAttrChange("typeName", oldValue, newValue); // NOI18N
    }
    
    /**
     * Sets the value of reference type. See {@link #getType} for description
     * on the reference.
     * @param newValue New value to be set.
     */
    public void setType(Type newValue) {
        TypeRef tr = typeToTypeRef(newValue);
        TypeReference typeReference = null;
        if (!disableChanges) {
            updateDimCount(tr);
            typeReference = (TypeReference) typeRefToTypeReference(tr, getDimCount());
        }
        fireTypeNameChange(typeReference);
        _setTypeName(typeReference, tr);
    }
    
    private void updateDimCount(TypeRef tr) {
        int dimCount = getDimCount();
        if (tr instanceof ArrayRef) {
            if (((ArrayRef) tr).dimCount < dimCount) {
                _setDimCount(((ArrayRef) tr).dimCount);
            }
        } else if (dimCount > 0) {
            _setDimCount(0);
        }
    }
    
    private void _setDimCount(int dimCount) {
        internalSetDim = true;
        try {
            setDimCount(dimCount);
        } finally {
            internalSetDim = false;
        }
    }
    
    public TypeReference getTypeName() {
        checkUpToDate();
        if (!childrenInited) {
            initChildren();
        }
        return typeName;
    }

    public void setTypeName(TypeReference typeName) {
        _setTypeName(typeName, typeReferenceToTypeRef(typeName, getDimCount()));
    }
    
    private void _setTypeName(TypeReference typeName, TypeRef typeRef) {
        if (!disableChanges) {
            objectChanged(CHANGED_TYPE);
            changeChild(getTypeName(), typeName);
            this.typeName = typeName;
        }
        setTypeRef(typeRef);
    }
    
    /**
     * Set the value of isFinal attribute. See {@link #isFinal} for description 
     * on the attribute.
     *
     * @param newValue New value to be set.
     */
    public void setFinal(boolean newValue) {
        objectChanged(CHANGED_IS_FINAL);
        super_setFinal(newValue);
    }
    
    public List/*<org.netbeans.jmi.javamodel.Annotation>*/ getAnnotations() {
        checkUpToDate();
        if (annotations == null) {
            annotations = createChildrenList("annotations", (AttrListWrapper) super_getAnnotations(), null, CHANGED_ANNOTATION);
        }
        return annotations;
    }

    protected abstract void super_setFinal(boolean newValue);

    /**
     * Set the value of isVarArg attribute. See {@link #isVarArg} for description 
     * on the attribute.
     *
     * @param newValue New value to be set.
     */
    public void setVarArg(boolean newValue) {
        objectChanged(CHANGED_IS_VARARG);
        super_setVarArg(newValue);
    }
    
    protected abstract void super_setVarArg(boolean newValue);
    
    protected ASTree getPartTree(ElementPartKind part) {
        if (ElementPartKindEnum.NAME.equals(part)) {
            return getASTree().getSubTrees()[VARIABLE_DECLARATOR_ID].getSubTrees()[0];
        }
        throw new IllegalArgumentException("Invalid part for this element: " + part); // NOI18N
    }
    
    public String getSourceText() {
        String origElem;
        if ((origElem = checkChange()) != null)
            return origElem;
        
        StringBuffer buf = new StringBuffer();
        for (Iterator annIt = getAnnotations().iterator(); annIt.hasNext(); ) {
            AnnotationImpl ann = (AnnotationImpl) annIt.next();
            buf.append(ann.getSourceText()).append(' ');
        }
        if (isFinal())
            buf.append("final "); // NOI18N
        buf.append(((MetadataElement) getTypeName()).getSourceText());
         if (isVarArg())
            buf.append("..."); // NOI18N
        buf.append(' ');
        buf.append(getName());
        appendDims(buf, getDimCount());
        return buf.toString();
    }
    
    protected void diffModifiers(List diffList, ASTree nextNode, ASTProvider parser) {
        String text = "";
        for (Iterator annIt = getAnnotations().iterator(); annIt.hasNext(); ) {
            AnnotationImpl ann = (AnnotationImpl) annIt.next();
            text += ann.getSourceText();
            if (annIt.hasNext()) {
                text += ' ';
            }
        }
        if (isFinal()) {
            if (text.length() > 0) {
                text += ' ';
            }
            text += "final"; // NOI18N
        }
        int nextToken = nextNode.getFirstToken();
        int startOffset, endOffset;
        int startToken;
        endOffset = parser.getToken(nextToken).getStartOffset();
        ASTree modifiers = getASTree().getSubTrees()[FINAL_OPT];
        if (modifiers != null) {
            startToken = modifiers.getFirstToken();
            startOffset = parser.getToken(startToken).getStartOffset();
            if (text.length() > 0) {
                int endToken = modifiers.getLastToken();
                endOffset = parser.getToken(endToken).getEndOffset();
            }
        } else {
            startOffset = endOffset;
            text += ' ';
        }
        diffList.add(new DiffElement(startOffset, endOffset, text));
    }
    
    public void getDiff(List diffList) {
        // parameter exist
        ASTProvider parser = getParser();
        ASTree[] children = getASTree().getSubTrees();
        if (isChanged(CHANGED_IS_FINAL) || isChanged(CHANGED_ANNOTATION)) {
            diffModifiers(diffList, children[TYPE], parser);
        } else if (children[FINAL_OPT] != null) {
            ParameterInfo astInfo = (ParameterInfo) getElementInfo();
            getCollectionDiff(diffList, parser, CHANGED_ANNOTATION, astInfo.annotations, getAnnotations(), parser.getToken(children[FINAL_OPT].getLastToken()).getEndOffset(), " "); // NOI18N
        }
        // type print
        getChildDiff(diffList, parser, children[TYPE], (MetadataElement) getTypeName(), CHANGED_TYPE);
        if (isChanged(CHANGED_IS_VARARG)) {
            replaceNode(diffList, parser, children[VARARG_OPT], isVarArg() ? "..." : "", parser.getToken(children[TYPE].getLastToken()).getEndOffset(), ""); // NOI18N
        }
        // name print
        ASTree[] varDeclaratorIdChildren = children[VARIABLE_DECLARATOR_ID].getSubTrees();
        if (isChanged(CHANGED_NAME)) {
            Token identifier = (Token) varDeclaratorIdChildren[0]; // identifier
            int startOffset = identifier.getStartOffset();
            int endOffset = identifier.getEndOffset();
            diffList.add(new DiffElement(startOffset, endOffset, getName()));
        }
        if (isChanged(CHANGED_DIM_COUNT)) {
            replaceNode(diffList, parser, varDeclaratorIdChildren[1], appendDims(new StringBuffer(), getDimCount()).toString(), getEndOffset(getParser(), varDeclaratorIdChildren[0]), "");
        }
    }
    
    private static final int FINAL_OPT = 0;
    private static final int TYPE = 1;
    private static final int VARARG_OPT = 2;
    private static final int VARIABLE_DECLARATOR_ID = 3;

    public void replaceChild(Element oldElement, Element newElement) {
        if (childrenInited) {
            if (oldElement.equals(typeName)) {
                setTypeName((TypeReference) newElement);
                return;
            }
            if (replaceObject(annotations, oldElement, newElement)) return;
        }
        super.replaceChild(oldElement, newElement);
    }

    public int getDimCount() {
        if (isChanged(CHANGED_DIM_COUNT)) {
            return dimCount;
        } else {
            ASTree tree = getASTree();
            if (tree != null) {
                ASTree dims = tree.getSubTrees()[VARIABLE_DECLARATOR_ID].getSubTrees()[1];
                if (dims != null) {
                    return (dims.getLastToken() - dims.getFirstToken() + 1) / 2;
                }
            }
            return 0;
        }
    }
    
    public void setDimCount(int dimCount) {
        objectChanged(CHANGED_DIM_COUNT);
        this.dimCount = dimCount;
        if (!internalSetDim) {
            // TODO: fire type change event
            setTypeRef(typeReferenceToTypeRef(getTypeName(), dimCount));
        }
    }
    
    public List getChildren() {
        List list = new ArrayList(1);
        list.addAll(getAnnotations());
        addIfNotNull(list, getTypeName());
        return list;
    }
    
    protected void initChildren() {
        childrenInited = false;
        ParameterInfo info = (ParameterInfo) getElementInfo();
        ASTree tree = info.getTypeAST(this);
        typeName = (TypeReference) initOrCreate(typeName, tree);
        annotations = createChildrenList(annotations, "annotations", (AttrListWrapper)super_getAnnotations(),// NOI18N
            ((ParameterInfo) getElementInfo()).annotations, CHANGED_ANNOTATION);
        childrenInited = true;
    }
    
    protected void setData(List annotations, TypeReference typeName, int dimCount) {
        changeChild(null, typeName);
        this.typeName = typeName;
        this.annotations = createChildrenList("annotations", (AttrListWrapper)super_getAnnotations(), annotations, CHANGED_ANNOTATION); // NOI18N
        this.dimCount = dimCount;
        setTypeRef(typeReferenceToTypeRef(typeName, dimCount));
        childrenInited = true;
    }
    
    protected void setTypeRef(TypeRef type) {
        _getDelegate().setSlot1(type);
    }
    
    public TypeRef getTypeRef() {
        return (TypeRef) _getDelegate().getSlot1();
    }
    
    public boolean isPersisted() {
        return _getDelegate().getSlot2() != null;
    }
    
    public void setPersisted(boolean persisted) {
        _getDelegate().setSlot2(persisted ? "" : null);
    }
    
    void childChanged(MetadataElement mpi) {
        super.childChanged(mpi);
        if (childrenInited) {
            if (mpi == typeName) {
                setTypeName((TypeReference) mpi);
            }
        }
    }
    
    public Element duplicate(JavaModelPackage targetExtent) {
        return targetExtent.getParameter().createParameter(
                getName(), 
                duplicateList(getAnnotations(), targetExtent),
                isFinal(),
                (TypeReference) duplicateElement(getTypeName(), targetExtent),
                getDimCount(),
                isVarArg()
            );
    }

    public void fixImports(Element scope, Element original) {
        Parameter par=(Parameter)original;

        fixImports(scope,getAnnotations(),par.getAnnotations());
        setTypeName(JavaModelUtil.resolveImportsForType(scope,par.getType()));
        setDimCount(0);
    }

    protected abstract List super_getAnnotations();
    
    protected void _delete() {
        // --- delete components -------------------------------------------
        // delete all annotations (if initialized)
        deleteChildren("annotations", (AttrListWrapper) super_getAnnotations());
        if (childrenInited) {
            deleteChild(typeName);
        }
        // --- delete links -----------------------------------------------
        // no links to delete
        // [TODO] should Throws association be notified?
        // --- call super -------------------------------------------------
        super._delete();
    }
}
