/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.EnumConstant;
import org.netbeans.jmi.javamodel.Expression;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.FieldGroup;
import org.netbeans.jmi.javamodel.ForStatement;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.LocalVarDeclaration;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.Statement;
import org.netbeans.jmi.javamodel.StatementBlock;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.modules.javacore.parser.MDRParser;

/**
 * Helper method for indentation of source text.
 *
 * @author Pavel Flaska
 */
public final class IndentUtil {
    
    /**
     * Returns a copy of the parameter <tt>s</tt>, with leading white space
     * omitted.
     *
     * @param s string will be copied without leading white space
     * @return  A copy of parameter <tt>s</tt> with leading white space removed,
     *          or parameter <tt>s</tt> if it has no leading white space.
     */
    static String leftTrim(String s) {
        int len = s.length();
        char[] val = s.toCharArray();
        int pos = 0;
	while ((pos < len) && (val[pos] <= ' '))
	    pos++;
        return pos == 0 ? s : s.substring(pos, len);
    }
    
    /**
     * Computes original indentation of the element. The computation mechanism
     * is based on count of space or tab characters. Method will find first 
     * non-whitespace character and then it goes back and saves the space and
     * tab characters.
     *
     * @param s java element string, for which original indentation is computed
     * @return  original indentation text (spaces and/or tabs)
     */
    static String getOriginalIndentation(String s, int[] origPos) {
        int len = s.length();
        char[] val = s.toCharArray();
        int pos = 0;
        StringBuffer result = new StringBuffer();
        
        while ((pos < len) && (val[pos] <= ' '))
            pos++;
        
        while (pos > 0 && (val[--pos] == ' ' || val[pos] == '\t'))
            if (val[pos] == ' ' || val[pos] == '\t')
                result.append(val[pos]);
        origPos[0] = pos;
        return result.reverse().toString();
    }
    
    /**
     * Takes java element string representation and tries to indent it correctly
     * in new place. Used especially when we want to move java model element
     * between two places. We want to preserve formatting of the element,
     * so we cut it from original source and put to the new place. (E.g. Element
     * inside comments should be preserved, element inside formatting too.)
     * But we have to care about its indentation as it can be on another
     * indent level on new place.
     *
     * @param  element  java model element
     * @param  src  text representation of java model element
     * @return indented  text representation of newly indented element
     */
    static String indentExistingElement(MetadataElement element, String src) {
        StringBuffer result = new StringBuffer(80);
        if (element instanceof EnumConstant ||
           ((element instanceof Field && element.refImmediateComposite() instanceof FieldGroup))) {
            // we do not care about enum constant, put it back as it was
            return src;
        } else if (element instanceof Statement || element instanceof Feature || element instanceof FieldGroup) {
            int[] oP = new int[1];
            String origInd = getOriginalIndentation(src, oP);
            String[] statementLines = src.split("\\n"); // NOI18N
            for (int i = 0; i < statementLines.length; i++) {
                String line = statementLines[i];
                if (leftTrim(line).length() == 0) {
                    result.append('\n');
                    continue;
                }
                if (line.endsWith("\r")) {  // NOI18N
                    line = line.substring(0, line.length()-1);
                }
                result.append(element.getIndentation());
                if (line.indexOf(origInd) > -1) {
                    result.append(line.substring(origInd.length()));
                } else {
                    result.append(line);
                }
                if (i < (statementLines.length-1))
                    result.append('\n');
            }
            return result.toString();
        }
        // we have done no indent changes, return the original text
        return src;
    }

    static String indentNewElement(MetadataElement e) {
        StringBuffer sb = new StringBuffer(1024);
        String sourceText = e.getRawText();
        MetadataElement parent = (MetadataElement) e.refImmediateComposite();
        if (e instanceof StatementBlock) {
            sb.append(e.getElementPrefix(MetadataElement.BLOCK_OPEN_CURLY));
            sb.append(sourceText);
        } else if (e instanceof Annotation ||
                   e instanceof EnumConstant) 
        {
            sb.append(sourceText);
        } else if (e instanceof Statement ||
                   e instanceof Feature ||
                   e instanceof FieldGroup) 
        {
            // for special cases like:
            // for statements: for (int i = 0; .... )
            // field declarations in group: int a, b = 5, ...;
            // skip the standard formatting.
            if ((e instanceof LocalVarDeclaration && parent instanceof ForStatement) ||
                (e instanceof Field && parent instanceof FieldGroup))
            {
                sb.append(sourceText);
            } else {
                sb.append('\n');
                sb.append(e.getIndentation());
                sb.append(sourceText);
            }
        } else {
            sb.append(sourceText);
        }
        return sb.toString();
    }
    
    /**
     * Returns an index of the first character of the element. It takes
     * element padding (strictly speaking padding of its first token)
     * and tries to separate them between two neighboring elements.
     * Consider following code:
     * <pre>
     *    System.out.println("Nothing to do."); // log comment
     *    // this code does nothing
     *    i = i;
     * </pre>
     * If the method will get statement <i><tt>i = i;</tt></i> as a parameter,
     * it returns an offset representing line separator following after
     * <i></tt>log&nbsp;comment</tt></i> comment. Second comment is considered
     * as a part of parameter.
     *
     * @param   el  method count start offset of this parameter
     * @return  start offset of the element
     */
    static int getElementStart(Element el) {
        MetadataElement e = (MetadataElement) el;
        // common result, takes element paddings.
        int result = e.getStartOffset(e.getParser(), e.getASTree(), true);
        if (e instanceof Resource) { 
            // start of source text
            result = 0;
            
        } else if (e instanceof Expression) {
            // start offset of the first token, do not count padding
            result = e.getStartOffset(e.getParser(), e.getASTree(), false);
            
        } else if (e instanceof Annotation) {
            // We do not consider leading whitespaces as a part of an annotation.
            // All the whitespaces and new lines are part of element, which owns annotation.
            result = e.getParser().getToken(e.getASTree().getFirstToken()).getStartOffset();
        }
        return result;
    }
    
    /**
     * Returns an index of the last character of the element. It takes
     * padding of token following after the last token of the element
     * and tries to separate them between two neighboring elements.
     * Consider following code:
     * <pre>
     *    System.out.println("Nothing to do."); // log comment
     *    // this code does nothing
     *    i = i;
     * </pre>
     * If the method will get statement 
     * <i><tt>System.out.println("Nothing&nbsp;to&nbsp;do.");</tt></i>
     * as a parameter, it returns an offset representing position exactly
     * before new line after comment <i><tt>// log comment</tt></i>.
     *
     * @param   el  method count end offset of this parameter
     * @return  end offset of the parameter
     */
    static int getElementEnd(Element el) {
        MetadataElement e = (MetadataElement) el;
        // common result, takes end offset of the last character of last token
        int result = e.getEndOffset(e.getParser(), e.getASTree());
        if (e instanceof Resource) {
            // end offset of the whole source text
            result = e.getParser().getSourceText().length();
        }
        return result;
    }
    
    /**
     * Prints to <tt>buf</tt> all whitespaces and comments connected to
     * <tt>token</tt>. Both heading and tailing are printed.
     *
     * @param token      whitespaces connected to token are printed
     * @param origToken  token id for obtaining original indentation
     * @param element    existing element with tree and parser
     * @param buf        buffer where data are printed
     */
    static final void printTokenWithAllAround(int token, int origToken, MetadataElement element, StringBuffer buf) {
        assert !element.isNew() : "Cannot call it for the new element!";
        MDRParser parser = element.getParser();
        Token t = parser.getToken(token-1);
        int begin = t.getEndOffset();
        t = parser.getToken(token+1);
        int end = t.getStartOffset();
        String s = parser.getSourceText().substring(begin, end);
        // temporary buffer, for original indentation and for the result
        StringBuffer tempInd = new StringBuffer();
        if (!getOriginalIndentation(origToken, element, tempInd)) {
            buf.append(s);
        } else {
            buf.append(reformat(s, element.getIndentation(), tempInd.toString()));
        }
    }
    
    /**
     * Prints tail of the element. Finds the next token after element, grabs
     * its padding and prints all paddings to first new line character to buffer.
     *
     * @param  element  print tail of this element
     * @param  buf      buffer to append tail to
     */
    static final void printTailGarbage(MetadataElement element, StringBuffer buf) {
        assert !element.isNew() : "Cannot call it for new elements!"; // NOI18N
        ASTree tree = element.getASTree();
        MDRParser parser = element.getParser();
        buf.append(getGarbage(tree.getLastToken()+1, parser, true));
    }
    
    /**
     * Prints head of the element. Finds the first token of the element, grabs
     * its padding and prints all paddings after first new line character to buffer.
     *
     * @param  element  print tail of this element
     * @param  buf      buffer to append tail to
     */
    static final void printHeadGarbage(MetadataElement element, StringBuffer buf) {
        assert !element.isNew() : "Cannot call it for new elements!"; // NOI18N
        ASTree tree = element.getASTree();
        MDRParser parser = element.getParser();
        buf.append(getGarbage(tree.getFirstToken(), parser, false));
    }
    
    static String getGarbage(int tokenId, MDRParser p, boolean tail) {
        StringBuffer buf = new StringBuffer(15);
        int startOffset = tokenId == 0? 0 : p.getToken(tokenId-1).getEndOffset(), 
            endOffset = p.getToken(tokenId).getStartOffset();
        Token[] pad = p.getToken(tokenId).getPadding();
        if (pad.length > 0) {
            for (int i = 0; i < pad.length; i++) {
                if (pad[i].getType() == ParserTokens.EOL) {
                    if (tail)
                        endOffset = pad[i].getStartOffset();
                    else
                        startOffset = pad[i].getStartOffset();
                    break;
                }
            }
        }
        return p.getSourceText().substring(startOffset, endOffset);
    }
    
    static String printTokenWithPads(int tokenId, MDRParser p) {
        String startPads = getGarbage(tokenId, p, false);
        String endPads = getGarbage(tokenId+1, p, true);
        Token token = p.getToken(tokenId);
        return startPads + p.getText(token) + endPads;
    }
    
    static final String reformatTokenWithPads(MetadataElement element, int tokenId) {
        MDRParser p = element.getParser();
        String startPads = getGarbage(tokenId, p, false);
        String endPads = getGarbage(tokenId+1, p, true);
        return indentExistingElement(element, startPads, tokenId) + p.getText(p.getToken(tokenId)) + endPads;
    }
    
    static final void reformatHeadGarbage(MetadataElement element, int tokenId, StringBuffer buf) {
        MDRParser p = element.getParser();
        String startPads = getGarbage(tokenId, p, false);
        buf.append(indentExistingElement(element, startPads, tokenId));
    }
    
    static void reformatHeadGarbage(MetadataElement element, StringBuffer buf) {
        StringBuffer sb = new StringBuffer(80);
        printHeadGarbage(element, sb);
        buf.append(indentExistingElement2(element, sb.toString()));
    }
    
    // optimize 
    static String indentExistingElement2(MetadataElement element, String src) {
        StringBuffer result = new StringBuffer(80);
        if (element instanceof EnumConstant ||
           ((element instanceof Field && element.refImmediateComposite() instanceof FieldGroup))) {
            // we do not care about enum constant, put it back as it was
            return src;
        } else if (element instanceof Statement || element instanceof Feature || element instanceof FieldGroup) {
            int[] oP = new int[1];
            String origInd = getOriginalIndentation(src, oP);
            String[] statementLines = src.split("\\n"); // NOI18N
            for (int i = 0; i < statementLines.length; i++) {
                String line = statementLines[i];
                if (line.endsWith("\r")) {  // NOI18N
                    line = line.substring(0, line.length()-1);
                }
                if (line.indexOf(origInd) > -1) {
                    result.append(element.getIndentation());
                    result.append(line.substring(origInd.length()));
                } else {
                    result.append(line);
                }
                if (i < statementLines.length-1)
                    result.append('\n');
            }
            return result.toString();
        }
        // we have done no indent changes, return the original text
        return src;
    }

    /**
     * Used for obtaining original indentation from element. Takes all
     * whitespaces and paddings before the <tt>tokenId</tt>, go through
     * them to the last new line character and puts all the whitespaces
     * after the last new line character to <tt>buf</bb>. If there is
     * not any new line character before <tt>tokenId</tt>, it returns
     * false (buf is not filled), otherwise true.
     *
     * @param   tokenId  token number used for computation
     * @param   element  already existing element (not newly created) with
     *                   containing parser
     * @param   buf      method fills this buffer with indentation
     * @return  indentation was obtained from source, parameter buf is filled
     *          with it
     */
    private static boolean getOriginalIndentation(int tokenId, MetadataElement element, StringBuffer buf) {
        assert !element.isNew() : "Cannot call it for the new element!";
        MDRParser p = element.getParser();
        Token[] pad = p.getToken(tokenId).getPadding();
        int startOffset = -1;
        if (pad.length > 0) {
            int endOffset = p.getToken(tokenId).getStartOffset();
            for (int i = 0; i < pad.length; i++) {
                if (pad[i].getType() == ParserTokens.EOL) {
                    startOffset = pad[i].getEndOffset();
                }
            }
            if (startOffset != -1) {
                buf.append(p.getSourceText().substring(startOffset, endOffset));
                return true;
            }
        }
        return false;
    }
    
    /**
     * Method re-indents element. It takes new indentation for <tt>element</tt>,
     * go through the source representation in <tt>src</tt> where it replaces
     * original indentation with new one obtained from <tt>element</tt>.
     * Original indentation is computed from <tt>tokenId</tt>
     *
     * @param element   method calls getIndentation() on it for obtaining
     *                  new indentation
     * @param  src      original source, which method re-indents
     * @param  tokenId  token number, original indentation is computed
     *                  from this token
     * @return  re-indented element
     */
    static String indentExistingElement(MetadataElement element, String src, int tokenId) {
        assert !element.isNew() : "Cannot call it for the new element!";
        if (element instanceof EnumConstant ||
           ((element instanceof Field && element.refImmediateComposite() instanceof FieldGroup))) {
            // we do not care about enum constant, put it back as it was
            return src;
        } else if (element instanceof Statement || element instanceof Feature || element instanceof FieldGroup) {
            // temporary buffer, for original indentation and for the result
            StringBuffer tempInd = new StringBuffer();
            if (!getOriginalIndentation(tokenId, element, tempInd)) {
                return src;
            }
            return reformat(src, element.getIndentation(), tempInd.toString());
        }
        // we have done no indent changes, return the original text
        return src;
    }
    
    private static String reformat(final String src, final String newInd, final String origInd) {
        StringBuffer result = new StringBuffer(80);
        String[] statementLines = src.split("\\n"); // NOI18N
        // go through all the lines in src, remove the origInd
        // from every line if present and put the new indentation
        // there instead
        for (int i = 0; i < statementLines.length; i++) {
            String line = statementLines[i];
            // windows specific
            if (line.endsWith("\r")) {  // NOI18N
                line = line.substring(0, line.length()-1);
            }
            // remove the original indentation from line and put
            // there new one
            if (line.indexOf(origInd) == 0) {
                result.append(newInd);
                result.append(line.substring(origInd.length()));
            } else {
                result.append(line);
            }
            // put the new line character after every line,
            // do not put it after last line
            if (i < statementLines.length-1)
                result.append('\n');
        }
        return result.toString();
    }

}
