/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.ArrayList;
import java.util.List;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.Expression;
import org.netbeans.jmi.javamodel.ForEachStatement;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.jmi.javamodel.Statement;
import org.netbeans.lib.java.parser.*;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.parser.ASTProvider;
import org.netbeans.modules.javacore.parser.ParameterInfo;

/**
 * Instance of elements represent enhanced for loop in java source, which is
 * specified in JSR201 and supported by JDK 1.5.
 *
 * <code>
 * Collection<String> c = ...;
 * for (String s : c)
 *    body;
 * </code>
 *
 * @see ForStatementImpl
 * 
 * @author  Pavel Flaska
 */
public abstract class ForEachStatementImpl extends StatementImpl implements ForEachStatement {

    // contains body (body can be represented by statement or statement block)
    private Statement body = null;
    // contains parameter, i.e. String s from the example above
    private Parameter parameter = null;
    // contains element which implements Iterable interface, i.e. c element
    // from the example above
    private Expression iterable = null;

    /** 
     * Creates a new handler for a specified storable.
     *
     * @param o  storable for which the handler is created.
     */
    public ForEachStatementImpl(StorableObject o) {
        super(o);
    }

    /**
     * Returns a statement representing body of the enhanced for loop.
     *
     * @return  statement representing the body (often StatementBlock instance)
     */
    public Statement getBody() {
        if (!childrenInited) {
            initChildren();
        }
        return body;
    }

    /**
     * Sets a new body for the enhanced for loop.
     *
     * @param  body  a new statement or a new statement block representing
     *               body
     */
    public void setBody(Statement body) {
        objectChanged(CHANGED_BODY);
        changeChild(getBody(), body);
        this.body = body;
    }

    /**
     * Returns a parameter representing object from the iterator.
     * From the example above it is 'String s' text.
     *
     * @return  a parameter object
     */
    public Parameter getParameter() {
        if (!childrenInited) {
            initChildren();
        }
        return parameter;
    }

    /**
     * Sets a new parameter for the enhanced for loop. 
     * From the example above it is replacement of 'String s' text.
     *
     * @param  a parameter object
     */
    public void setParameter(Parameter newValue) {
        objectChanged(CHANGED_PARAMETER);
        changeChild(getParameter(), parameter);
        this.parameter = parameter;
    }

    /**
     * Returns an iterable, i.e. class implementing iterable interface.
     * From the example above it is 'c' text in for statement.
     *
     * @return  iterable element
     */
    public Expression getIterable() {
        if (!childrenInited) {
            initChildren();
        }
        return iterable;
    }

    /**
     * Sets an iterable, i.e. class use for iterating the collection.
     * From the example above it is replacement of 'c' text in for statement.
     *
     * @param  an interable
     */
    public void setIterable(Expression newValue) {
        objectChanged(CHANGED_ITERABLE);
        changeChild(getIterable(), iterable);
        this.iterable = iterable;
    }

    public List getChildren() {
        List list = new ArrayList(3);
        list.add(getParameter());
        list.add(getIterable()); 
        list.add(getBody());
        return list;
    }

    // generate source-text representation of element
    String getRawText() {
        // create helping local variables (cast them to impl to allow
        // to call getSourceText() on them.
        StatementImpl body = (StatementImpl) getBody();
        ParameterImpl parameter = (ParameterImpl) getParameter();
        ExpressionImpl iterable = (ExpressionImpl) getIterable();
        
        StringBuffer buf = new StringBuffer();
        formatElementPart(FOR_KEYWORD, buf, isNew());
        formatElementPart(STMT_OPEN_BRACKET, buf);
        buf.append(parameter.getSourceText());
        formatElementPart(COND_EXPR_COLON, buf);
        buf.append(iterable.getSourceText());
        formatElementPart(STMT_CLOSE_BRACKET, buf);
        buf.append(body.getSourceText());
        
        return buf.toString();
    }

    // generate differencies from the changes
    public void getDiff(List diff) {
        ASTProvider parser = getParser();
        ASTree tree = getASTree();
        ASTree[] children = tree.getSubTrees();
        
        getChildDiff(diff, parser, children[0], (MetadataElement) getParameter(), CHANGED_PARAMETER);
        getChildDiff(diff, parser, children[1], (MetadataElement) getIterable(), CHANGED_BODY);
        getChildDiff(diff, parser, children[2], (MetadataElement) getBody(), CHANGED_BODY);
    }

    protected void _delete() {
        // --- delete components ----------------------------------------------
        if (childrenInited) {
            deleteChild(body);
            deleteChild(parameter);
            deleteChild(iterable);
        }
        // --- delete links ---------------------------------------------------
        // no links to delete
        // --- call super -----------------------------------------------------
        super._delete();
    }
    
    /**
     * Initialize elements (children) of this element.
     */
    protected void initChildren() {
        childrenInited = false;
        ASTree tree = getASTree();
        if (tree != null) {
            ASTree[] parts = tree.getSubTrees();
            // parameter init
            ASTree parameterAST = parts[0];
            ParameterInfo paramInfo=(ParameterInfo)getParser().getSemanticInfo(parameterAST, this);
            if (parameter == null) {
                parameter = (Parameter)createElement(paramInfo);
                changeChild(null, parameter);
            } else {
                ((SemiPersistentElement)parameter).updatePersistent(paramInfo);
                ((SemiPersistentElement)parameter).setElementInfo(paramInfo);
            }
            // iterable init
            iterable = (Expression) initOrCreate(iterable, parts[1]);
            // body init
            body = (Statement) initOrCreate(body, parts[2]);
        }
        childrenInited = true;
    }

    void setData(Statement body, Parameter parameter, Expression iterable) {
        // body
        changeChild(null, body);
        this.body = body;
        // parameter
        changeChild(null, parameter);
        this.parameter = parameter;
        // iterable
        changeChild(null, iterable);
        this.iterable = iterable;
    }
    
    public Element duplicate(JavaModelPackage targetExtent) {
        return targetExtent.getForEachStatement().createForEachStatement(
                (Statement) duplicateElement(getBody(), targetExtent),
                (Parameter) duplicateElement(getParameter(), targetExtent),
                (Expression) duplicateElement(getIterable(), targetExtent)
               );
    }

}
