/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.lang.reflect.Modifier;
import java.util.List;
import java.util.ListIterator;
import javax.jmi.reflect.ConstraintViolationException;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.ASTreeTypes;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.parser.*;
import org.openide.util.Utilities;


/** Implementation of Field object instance interface.
 *
 * @author Vladimir Hudec
 * @author Pavel Flaska
 */
public abstract class FieldImpl extends FeatureImpl implements Field {
    private static final ElementInfo DEFAULT_INFO = new FieldInfo(null, FieldInfo.FIELD_TYPE, null, 0, null, FieldInfo.SINGLE_FIELD_INDEX, null);

    protected String initialValueText;
    protected InitialValue initialValue;
    private boolean initValueInited = false;
    private boolean elementsInited = false;
    private TypeReference typeName = null;
    private int dimCount = 0;
    private boolean internalSetDim = false;

    /** Creates a new instance of FieldImpl */
    public FieldImpl(StorableObject s) {
        super(s);
    }

    public String toString() {
        return "field " + getName(); // NOI18N
    }

    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }

    public List getChildren() {
        List list = super.getChildren();
        addIfNotNull(list, getTypeName());
        if (!isChanged(CHANGED_INITIAL_VALUE) || initialValueText == null) {
            addIfNotNull(list, getInitialValue());
        }
        return list;
    }

    public void fixImports(Element scope, Element original) {
        Field field = (Field)original;
        MetadataElement newInitVal = (MetadataElement)getInitialValue();
        InitialValue oldInitVal = field.getInitialValue();
        
        if (newInitVal != null && oldInitVal != null) {
            newInitVal.fixImports(scope,oldInitVal);
        }
        setTypeName(JavaModelUtil.resolveImportsForType(scope,field.getType()));
        setDimCount(0);
        super.fixImports(scope,original);
    }

    protected List getInitedChildren() {
        List list = super.getInitedChildren();
        if (elementsInited) {
            addIfNotNull(list, typeName);
        }
        if (initValueInited && (!isChanged(CHANGED_INITIAL_VALUE) || initialValueText == null)) {
            addIfNotNull(list, initialValue);
        }
        return list;
    }

    protected void initASTElements() {
        elementsInited = false;
        if (!childrenInited) {
            initChildren();
        }
        FieldInfo info = (FieldInfo) getElementInfo();
        ASTree tree = info.getTypeAST(this);
        typeName = (TypeReference) initOrCreate(typeName, tree);
        elementsInited = true;
    }

    protected void initChildren() {
        childrenInited = false;
        super.initChildren();
        childrenInited = true;
        if (elementsInited) {
            initASTElements();
        }
        if (initValueInited) {
            JMManager.getTransactionMutex().addBFeatureToInitQueue(this);
        }
    }
    
    protected void resetChildren() {
        super.resetChildren();
        resetASTElements();
    }
    
    protected void setData(List annotations, int modifiers, String javadocText, JavaDoc javadoc, boolean isFinal, TypeReference typeName, int dimCount, InitialValue initialValue, String initialValueText) {
        super.setData(annotations, javadocText, javadoc);
        if (initialValueText == null) {
            changeChild(null, initialValue);
            this.initialValue = initialValue;
        } else {
            if (initialValue != null) {
                throw new ConstraintViolationException(null, null, "Cannot set both initialValue and initialValueText."); // NOI18N
            }
            this.initialValueText = initialValueText;
        }
        changeChild(null, typeName);
        this.typeName = typeName;
        this.dimCount = dimCount;
        setTypeRef(typeReferenceToTypeRef(typeName, dimCount));
        initValueInited = true;
        elementsInited = true;
    }
    
    public void initInitValue() {
        initValueInited = false;
        if (!elementsInited) {
            initASTElements();
        }
        FieldInfo initValInfo = (FieldInfo) getElementInfo();
        if (initValInfo != null) {
            initValInfo.doAttribution(this);
            initialValue = (InitialValue) initOrCreate(initialValue, extractInitialValue());
        }
        initValueInited = true;
    }
    
    protected void matchPersistent(ElementInfo info) {
        super.matchPersistent(info);
        boolean inFieldGroup = refImmediateComposite() instanceof FieldGroup;
        FieldInfo newInfo = (FieldInfo) info;
        
        if (newInfo.modifiers != getSourceModifiers()) {
            if (inFieldGroup) {
                super_setModifiers(newInfo.modifiers);
            } else {
                setModifiers(newInfo.modifiers);
            }
        }
        
        if (!isPersisted()) {
            setPersisted(true);
            persist();
            setTypeRef(newInfo.type);
            persistChildren(getPersistentList("annotations", super_getAnnotations()), ((FeatureInfo) info).annotations);
        } else {
            if (!Utilities.compareObjects(newInfo.type, getTypeRef())) {
                if (refImmediateComposite() instanceof FieldGroup) {
                    setTypeRef(newInfo.type);
                } else {
                    setType(resolveType(newInfo.type));
                }
            }
            processMembers(getAnnotations(), newInfo.annotations);
        }
    }
    
    protected void matchModifiers(ElementInfo info) {
    }

    /** The method has to make sure that the AST infos of children are also updated.
     */
    protected void matchElementInfo(ElementInfo newInfo) {
        super.matchElementInfo(newInfo);
        resetASTElements();
    }

    protected void resetASTElements() {
        if (elementsInited) {
            if (initialValue != null) {
                InitialValue temp = initialValue;
                initialValue = null;
                changeChild(temp, null);
                temp.refDelete();
            }

            if (typeName != null) {
                TypeReference temp = typeName;
                typeName = null;
                changeChild(temp, null);
                temp.refDelete();
            }
            initValueInited = false;
            elementsInited = false;
        }
    }

    // Interfaces inherited from Variable, StructuralElement

    /**
     * Returns the value of reference type.
     * @return Value of reference type.
     */
    public Type getType() {
        checkUpToDate();
        MetadataElement parent = (MetadataElement) refImmediateComposite();
        if (parent instanceof FieldGroup && parent.isChanged()) {
            FieldGroup group = (FieldGroup) parent;
            Type t=group.getType();
            int dims=getDimCount();
            
            if (dims>0) {
                int i;
                ArrayClass arrClass=((JavaModelPackage)t.refImmediatePackage()).getArray();
                
                for(i=0;i<dims;i++) {
                    t=arrClass.resolveArray(t);
                }
            }
            return t;
        } else {
            return resolveType(getTypeRef());
        }
    }

    private void fireTypeNameChange(TypeReference typeReference) {
        Object oldValue = null;
        Object newValue = null;
        if (elementsInited && !disableChanges) {
            oldValue = getTypeName();
            newValue = typeReference;
        }
        fireAttrChange("typeName", oldValue, newValue); // NOI18N
    }
    
    /**
     * Sets the value of reference type. See {@link #getType} for description
     * on the reference.
     * @param newValue New value to be set.
     */
    public void setType(Type newValue) {
        TypeRef tr = typeToTypeRef(newValue);
        TypeReference typeReference = null;
        if (!disableChanges) {
            updateDimCount(tr);
            typeReference = (TypeReference) typeRefToTypeReference(tr, getDimCount());
        }
        fireTypeNameChange(typeReference);
        _setTypeName(typeReference, tr);
    }

    private void updateDimCount(TypeRef tr) {
        int dimCount = getDimCount();
        if (tr instanceof ArrayRef) {
            if (((ArrayRef) tr).dimCount < dimCount) {
                _setDimCount(((ArrayRef) tr).dimCount);
            }
        } else if (dimCount > 0) {
            _setDimCount(0);
        }
    }
    
    private void _setDimCount(int dimCount) {
        internalSetDim = true;
        try {
            setDimCount(dimCount);
        } finally {
            internalSetDim = false;
        }
    }

    public TypeReference getTypeName() {
        checkUpToDate();
        if (!elementsInited) {
            initASTElements();
        }
        MetadataElement parent = (MetadataElement) refImmediateComposite();
        if (parent instanceof FieldGroup) {
            FieldGroup group = (FieldGroup) parent;
            return group.getTypeName();
        }
        return typeName;
    }
    
    public void setTypeName(TypeReference typeName) {
        _setTypeName(typeName, typeReferenceToTypeRef(typeName, getDimCount()));
    }

    public void _setTypeName(TypeReference typeName, TypeRef typeRef) {
        if (!disableChanges) {
            objectChanged(CHANGED_TYPE);
            if (typeName != this.typeName) {
                splitGroup();
            }
            changeChild(getTypeName(), typeName);
            this.typeName = typeName;
        }
        setTypeRef(typeRef);
    }

    public void setModifiers(int newValue) {
        objectChanged(CHANGED_MODIFIERS);
        splitGroup();
        super_setModifiers(newValue);
    }

    public int getModifiers() {
        MetadataElement parent = (MetadataElement) refImmediateComposite();
        if (parent instanceof FieldGroup) {
            return ((FieldGroup) parent).getModifiers();
        }
        ClassDefinition cd = getDeclaringClass();
        int mods = super.getModifiers();
        if (cd instanceof JavaClass && ((JavaClass)cd).isInterface()) {            
            return mods | Modifier.STATIC | Modifier.FINAL;
        } else {
            return mods;
        }
    }
    
    public void setJavadocText(String newValue) {
        super.setJavadocText(newValue);
        splitGroup();
    }
    
    private void splitGroup() {
        MetadataElement parent = (MetadataElement) refImmediateComposite();
        if (parent instanceof FieldGroup) {
            if (!elementsInited) {
                initASTElements();
            }
            FieldGroupImpl group = (FieldGroupImpl) parent;
            int modifiers = group.getSourceModifiers();
            Type type = getType();
            JavaModelPackage pkg = (JavaModelPackage) refImmediatePackage();
            TypeReference typeName = group.getTypeName();
            TypeReference typeNameCopy1 = (TypeReference)typeName.duplicate();
            TypeReference typeNameCopy2 = (TypeReference)typeName.duplicate();
            int i = group.getFields().indexOf(this);
            ClassDefinition cls = (ClassDefinition) group.refImmediateComposite();
            int j = cls.getContents().indexOf(group);
            if (i == (group.getFields().size() - 1)) {
                j++;
            } else if (i != 0) {
                FieldGroup newGroup = pkg.getFieldGroup().createFieldGroup(null, null, group.getSourceModifiers(),
                        null, null, null, null);
                newGroup.setTypeName(typeNameCopy1);
                ListIterator fields = group.getFields().listIterator(i + 1);
                while (fields.hasNext()) {
                    Field field = (Field) fields.next();
                    fields.remove();
                    newGroup.getFields().add(field);
                }
                j++;
                cls.getContents().add(j, newGroup);
            }
            group.getFields().remove(this);
            cls.getContents().add(j, this);
            if (group.getFields().isEmpty()) {
                group.refDelete();
            }
            setTypeRef(typeToTypeRef(type));
            this.typeName = typeNameCopy2;
            // todo: call setTypeRef
            super_setModifiers(modifiers);
        }
    }

    /**
     * Returns the value of attribute isFinal.
     * @return Value of attribute isFinal.
     */
    public boolean isFinal(){
        return Modifier.isFinal(getModifiers());
    }
    /**
     * Sets the value of isFinal attribute. See {@link #isFinal} for description
     * on the attribute.
     * @param newValue New value to be set.
     */
    public void setFinal(boolean newValue){
        //objectChanged(); will be called from setModifiers()
        if (newValue) {
            setModifiers(getSourceModifiers() | Modifier.FINAL);
        } else {
            setModifiers(getSourceModifiers() & ~Modifier.FINAL);
        }
    }

    /**
     * Returns the value of attribute initialValue.
     * @return Value of attribute initialValue.
     */
    public InitialValue getInitialValue() {
        checkUpToDate();
        if (isChanged(CHANGED_INITIAL_VALUE) && initialValueText != null) {
            throw new ConstraintViolationException(null, null, "Cannot ask for initial value after the initial value text was changed."); // NOI18N
        }
        if (!initValueInited) {
            initInitValue();
        }
        return initialValue;
    }
    /**
     * Sets the value of initialValue attribute. See {@link #getInitialValue}
     * for description on the attribute.
     * @param newValue New value to be set.
     */
    public void setInitialValue(InitialValue newValue){
        if (!initValueInited) {
            initInitValue();
        }
        changeChild(initialValue, newValue);
        initialValue = newValue;
        initialValueText = null;
        objectChanged(CHANGED_INITIAL_VALUE);
    }
    /**
     * Returns the value of attribute initialValueText.
     * @return Value of attribute initialValueText.
     */
    public java.lang.String getInitialValueText(){
        if (isChanged(CHANGED_INITIAL_VALUE)) {
            if (initialValue != null) {
                throw new ConstraintViolationException(null, null, "Cannot ask for initial value text after the initial value was changed."); // NOI18N
            }
            return initialValueText;
        } else {
            return extractInitialValueText();
        }
    }

    private ASTree extractVariableDeclarator() {
        ASTree fieldTree = getASTree();
        if (fieldTree != null && fieldTree.getType() != ASTreeTypes.VARIABLE_DECLARATOR) {
            return fieldTree.getSubTrees()[2];
        }
        return fieldTree;
    }
    
    private ASTree extractInitialValue() {
        return extractVariableDeclarator().getSubTrees()[2];
    }

    private String extractInitialValueText() {
        ASTProvider parser = getParser();
        if (parser == null)
            return null;
        ASTree initValue = extractInitialValue();
        if (initValue == null)
            return null;
        return parser.getText(initValue);
    }

    /**
     * Sets the value of initialValueText attribute. See {@link #getInitialValueText}
     * for description on the attribute.
     * @param newValue New value to be set.
     */
    public void setInitialValueText(java.lang.String newValue){
        if (!elementsInited) {
            initASTElements();
        }
        if (initValueInited && initialValue != null) {
            changeChild(initialValue, null);
        }
        objectChanged(CHANGED_INITIAL_VALUE);
        initialValueText = newValue;
        initialValue = null;
        initValueInited = true;
    }

    // .........................................................................
    // printing and formatting fuctionality
    // .........................................................................
    String getRawText() {
        if (!initValueInited) {
            initInitValue();
        }
        // we have element which is new or changed and moved.
        StringBuffer buf = new StringBuffer();
        if (!(refImmediateComposite() instanceof FieldGroup)) {
            if (isNew()) generateNewJavaDoc(buf);
            generateNewModifiers(buf);
            if (getTypeName() != null) {
                buf.append(((MetadataElement) getTypeName()).getSourceText());
            } else {
                buf.append(getType().getName());
            }
            buf.append(' '); // NOI18N
        }
        buf.append(getName());
        appendDims(buf, getDimCount());
        if (initialValueText != null && initialValueText.trim().length() != 0) {
            formatElementPart(FIELD_EQUALS, buf);
            buf.append(initialValueText);
        } else {
            if (initialValue != null) {
                formatElementPart(FIELD_EQUALS, buf);
                buf.append(((MetadataElement) initialValue).getSourceText());
            }
        }
        if (!(refImmediateComposite() instanceof FieldGroup)) {
            if (isNew() || getASTree().getType() == ASTreeTypes.VARIABLE_DECLARATOR) {
                // for the new elements and fields comming from field group, put
                // the semicolon only.
                buf.append(";"); // NOI18N
            } else {
                buf.append(IndentUtil.reformatTokenWithPads(this, getASTree().getLastToken()));
            }
        }
        return buf.toString();
    }
    
    public void getDiff(List diffList) {
        // element was present in the original source code
        ASTProvider parser = getParser();
        ASTree tree = getASTree();
        ASTree[] children = tree.getSubTrees();

        // javadoc print
        replaceJavaDoc(diffList);
        // todo (#pf): currently, we ignore changes of type on fields in groups
        if (tree.getType() != ASTreeTypes.VARIABLE_DECLARATOR) {
            // modifier print
            if (isChanged(CHANGED_MODIFIERS) || isChanged(CHANGED_ANNOTATION)) {
                diffModifiers(diffList, children[TYPE], parser);
            } else if (children[0] != null) {
                FieldInfo astInfo=(FieldInfo)getElementInfo();
                
                getCollectionDiff(diffList, parser, CHANGED_ANNOTATION, astInfo.annotations, getAnnotations(), parser.getToken(children[0].getLastToken()).getEndOffset(), " "); // NOI18N
            }
            // type print
            getChildDiff(diffList, parser, children[TYPE], (MetadataElement) getTypeName(), CHANGED_TYPE);
        }
        // name print
        ASTree[] varDeclarator;
        if (tree.getType() == ASTreeTypes.VARIABLE_DECLARATOR) {
            varDeclarator = tree.getSubTrees();
        } else {
            varDeclarator = children[VARIABLE_DECLARATORS].getSubTrees();
        }
        if (isChanged(CHANGED_NAME)) {
            Token identifier = (Token) varDeclarator[0];
            int startOffset = identifier.getStartOffset();
            int endOffset = identifier.getEndOffset();
            diffList.add(new DiffElement(startOffset, endOffset, getName()));
        }
        if (isChanged(CHANGED_DIM_COUNT)) {
            replaceNode(diffList, parser, varDeclarator[1], appendDims(new StringBuffer(), getDimCount()).toString(), getEndOffset(getParser(), varDeclarator[0]), "");
        }
        // field initializer
        if (isChanged(CHANGED_INITIAL_VALUE)) {
            int startOffset = 0, endOffset = 0;
            String text = initialValue == null ? initialValueText :
                ((MetadataElement) getInitialValue()).getSourceText();
            // there was an initial value in original source code.
            if (varDeclarator[2] != null) {
                Token startToken;
                if (text == null || text.trim().length() == 0) {
                    startToken = parser.getToken((varDeclarator[1] == null ? varDeclarator[0] : varDeclarator[1]).getLastToken());
                    startOffset = startToken.getEndOffset();
                } else {
                    startToken = parser.getToken(varDeclarator[2].getFirstToken());
                    startOffset = startToken.getStartOffset();
                }
                Token endToken = parser.getToken(varDeclarator[2].getLastToken());
                endOffset = endToken.getEndOffset();
            }
            // there was not a initial value in original source
            else {
                // is the field part of some FieldGroup? (i.e. its root node
                // is of VARIABLE_DECLARATOR type.
                Token lastTokenOfDecl = parser.getToken(tree.getLastToken());
                if (tree.getType() == ASTreeTypes.VARIABLE_DECLARATOR)
                    startOffset = lastTokenOfDecl.getEndOffset(); // it is part of group
                else
                    startOffset = lastTokenOfDecl.getStartOffset(); // it is single field declaration
                // we are inserting new text, offsets contain the same value
                endOffset = startOffset;
                if (text != null) {
                    StringBuffer buf = new StringBuffer();
                    formatElementPart(FIELD_EQUALS, buf);
                    text = buf + text;
                }
            }
            diffList.add(new DiffElement(startOffset, endOffset, text == null ? "" : text));
        } else if (initialValue != null && ((MetadataElement) initialValue).isChanged()) {
            ((MetadataElement) initialValue).getDiff(diffList);
        }
    }

    /**
     */
    protected ASTree getPartStartTree(ElementPartKind part) {
        if (ElementPartKindEnum.HEADER.equals(part)) {
            return getASTree();
        }
        return super.getPartStartTree(part);
    }

    /**
     */
    protected ASTree getPartEndTree(ElementPartKind part) {
        if (ElementPartKindEnum.HEADER.equals(part)) {
            ASTree[] headerParts = getASTree().getSubTrees();
            for (int i = 2; true; i--) {
                ASTree result = headerParts[i];
                if (result != null) {
                    return result;
                }
            }
        }
        return super.getPartEndTree(part);
    }

    /**
     */
    protected ASTree getPartTree(ElementPartKind part) {
        // name
        if (ElementPartKindEnum.NAME.equals(part)) {
            if (getASTree().getType() == ASTreeTypes.FIELD_DECLARATION) {
                ASTree[] children = getASTree().getSubTrees();
                return children[2].getSubTrees()[0];
            } else {
                return getASTree();
            }
        }
        throw new IllegalArgumentException("Invalid part for this element: " + part); // NOI18N
    }

    // useful constants
    private static final int TYPE = 1;
    private static final int VARIABLE_DECLARATORS = 2;

    public void replaceChild(Element oldElement, Element newElement) {
        if (elementsInited && oldElement.equals(typeName)) {
            setTypeName((TypeReference) newElement);
            return;
        }
        if (initValueInited && oldElement.equals(initialValue)) {
            setInitialValue((InitialValue) newElement);
            return;
        }
        super.replaceChild(oldElement, newElement);
    }

    public int getDimCount() {
        if (isChanged(CHANGED_DIM_COUNT)) {
            return dimCount;
        } else {
            ASTree tree = extractVariableDeclarator();
            if (tree != null) {
                ASTree dims = tree.getSubTrees()[1];
                if (dims != null) {
                    return (dims.getLastToken() - dims.getFirstToken() + 1) / 2;
                }
            }
            return 0;
        }
    }

    public void setDimCount(int dimCount) {
        objectChanged(CHANGED_DIM_COUNT);
        this.dimCount = dimCount;
        if (!internalSetDim) {
            // TODO: fire type change event
            setTypeRef(typeReferenceToTypeRef(getTypeName(), dimCount));
        }
    }

    void childChanged(MetadataElement mpi) {
        super.childChanged(mpi);
        if (elementsInited) {
            if (mpi == typeName) {
                setTypeName((TypeReference) mpi);
            }
        }
    }
    
    public Element duplicate(JavaModelPackage targetExtent) {
        InitialValue initVal;
        String initValText;
        
        if (isChanged(CHANGED_INITIAL_VALUE) && initialValueText != null) {
            initVal = null;
            initValText = initialValueText;
        } else {
            initVal = (InitialValue) duplicateElement(getInitialValue(), targetExtent);
            initValText = null;
        }
        
        return targetExtent.getField().createField(
            getName(),
            duplicateList(getAnnotations(), targetExtent),
            getModifiers(),
            null,
            (JavaDoc) duplicateElement(getJavadoc(), targetExtent),
            isFinal(),
            (TypeReference) duplicateElement(getTypeName(), targetExtent),
            getDimCount(),
            initVal,
            initValText
        );
    }

    protected void _delete() {
        if (elementsInited) {
            deleteChild(typeName);
        }
        if (initValueInited) {
            deleteChild(initialValue);
        }
        super._delete();
    }    
}
