/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.jmiimpl.javamodel;

import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.handlers.AttrListWrapper;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.parser.ASTProvider;
import org.netbeans.modules.javacore.parser.ElementInfo;
import org.netbeans.modules.javacore.parser.FeatureInfo;
import org.netbeans.modules.javacore.parser.JavaDocParser;
import org.openide.util.Utilities;
import javax.jmi.model.MofClass;
import javax.jmi.reflect.ConstraintViolationException;
import javax.jmi.reflect.RefFeatured;
import javax.jmi.reflect.RefObject;
import java.lang.reflect.Modifier;
import java.util.*;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.modules.javacore.parser.MDRParser;
import org.netbeans.modules.javacore.parser.TypeRef;
import org.openide.text.PositionBounds;


/** Implementation of Feature object instance interface.
 *
 * @author Vladimir Hudec
 * @author Pavel Flaska
 */
public abstract class FeatureImpl extends SemiPersistentElement {
    
    public static final int DEPRECATED = 0x80000000;
    
    public static final String TYPE_PARAMETERS_ATTR = "typeParameters"; // NOI18N
    
    protected String javadocText = null;
    protected JavaDoc javadoc = null;
    protected LightAttrList annotations = null;
    
    /** Creates a new instance of FeatureImpl */
    public FeatureImpl(StorableObject s) {
        super(s);
    }
    
    protected void initChildren() {
      annotations = createChildrenList(annotations, "annotations", (AttrListWrapper)super_getAnnotations(), // NOI18N
            ((FeatureInfo) getElementInfo()).annotations, CHANGED_ANNOTATION);
    }
    
    protected void resetChildren() {
        super.resetChildren();
        if (childrenInited) {
            if (annotations != null) {
                annotations.setInnerList(getPersistentList("annotations", super_getAnnotations()));     // NOI18N
            }
        }
        resetJavaDoc();
    }

    public List getPersistentAnnotations() {
        AttrListWrapper list = (AttrListWrapper) super_getAnnotations();
        list.setAttrName("annotations"); // NOI18N
        return list;
    }
    
    /**
     * Returns the value of attribute annotations.
     * @return Value of annotations attribute.
     */
    public List getAnnotations() {
        checkUpToDate();
        if (annotations == null) {
            annotations = createChildrenList("annotations", (AttrListWrapper) super_getAnnotations(), null, CHANGED_ANNOTATION);
        }
        return annotations;
    }

    protected abstract void super_setModifiers(int modifiers);
    protected abstract int super_getModifiers();

    /**
     * Returns the value of attribute modifiers.
     * @return Value of attribute modifiers.
     */
    public int getModifiers() {
        int mods = super_getModifiers();
        ClassDefinition cd = getDeclaringClass();
        if (cd instanceof JavaClass) {
            JavaClass jc = (JavaClass)cd;
            if ((jc.getModifiers() & Modifier.STRICT) != 0) {
                mods = mods | Modifier.STRICT;
            }
            if (jc.isInterface()) {
                mods = mods | Modifier.PUBLIC;
                if (this instanceof Method) {
                    mods |= Modifier.ABSTRACT;
                }
            } else {
                if (this instanceof Method && ((jc.getModifiers() & Modifier.FINAL) != 0)) {
                    mods |= Modifier.FINAL;
                }
            }
        }
        return ~DEPRECATED & mods;
    }

    public int getSourceModifiers() {
        int mods = super_getModifiers();
        return ~DEPRECATED & mods;
    }
    
    /**
     * Sets the value of modifiers attribute. See {@link #getModifiers} for description
     * on the attribute.
     * @param newValue New value to be set.
     */
    public void setModifiers(int newValue) {
        boolean isDeprecated = isDeprecated();
        objectChanged(CHANGED_MODIFIERS);
        super_setModifiers((newValue & ~DEPRECATED) | (isDeprecated ? DEPRECATED : 0));
    }

    public boolean isDeprecated() {
        int mods = super_getModifiers();
        return (mods & DEPRECATED) != 0;
    }
    
    public void setDeprecated(boolean deprecated) {
        throw new UnsupportedOperationException();
    }
    
    /**
     * Returns the value of attribute javadocText.
     *
     * IT'S POSSIBLE TO USE SIMPLE IMPLEMENTATION JavaDocParser.surroundWithJavaDocStars()
     *
     * @return Value of attribute javadocText.
     */
    public java.lang.String getJavadocText() {
        checkUpToDate();
        if (javadoc != null) {
            List tags = javadoc.getTags();
            JavaDocParser.JavaDocTag[] javaDocTags = null;
            if (tags != null) {
                javaDocTags = new JavaDocParser.JavaDocTag[tags.size()];
                int i = 0;
                for (Iterator it = tags.iterator(); it.hasNext(); ) {
                    TagValue tagValue = (TagValue) it.next();
                    javaDocTags[i++] = new JavaDocParser.JavaDocTag(tagValue.getDefinition().getName(), tagValue.getValue());
                }
            }
            JavaDocParser parser = new JavaDocParser(javadoc.getText(), javaDocTags);
            return parser.getRawText(true); // reorder tags
        } else if (javadocText!=null) {
            return javadocText;
        } else {
            MDRParser parser = getParser();
            if (parser != null) {
                javadocText = parser.getJavaDoc(getASTree());
                if (javadocText != null) { // there is javadoc
                    if (javadocText.startsWith("\n")) { // NOI18N
                        javadocText = javadocText.substring(1);
                    }
                    if (javadocText.startsWith("\r\n")) { // to be sure that it works on Win too // NOI18N
                        javadocText = javadocText.substring(2);
                    }
                }
                return javadocText;
            }
        }
        return null;
    }
    /**
     * Sets the value of javadocText attribute. See {@link #getJavadocText} for
     * description on the attribute.
     * @param newValue New value to be set.
     */
    public void setJavadocText(java.lang.String newValue) {
        objectChanged(CHANGED_JAVADOC);
        if (javadoc !=null) {
            changeChild(javadoc, null);
            javadoc = null;
        }
        javadocText = newValue;
        if ((getParser() != null) && Utilities.compareObjects(getParser().getJavaDoc(getASTree()), newValue)) {
            resetChange(CHANGED_JAVADOC);
        }
    }

    /**
     * Returns the value of attribute javadoc.
     * @return Value of attribute javadoc.
     */
    public JavaDoc getJavadoc() {
        checkUpToDate();
        if (javadoc != null)
            return javadoc;
        String javaDocText = getJavadocText();
        if (javaDocText == null)
            return null;

        JavaModelPackage pkg = (JavaModelPackage) refImmediatePackage();
        TagValueClassImpl tagValueClass = (TagValueClassImpl) pkg.getTagValue();
        JavaDocClassImpl javaDocClass = (JavaDocClassImpl) pkg.getJavaDoc();

        JavaDocParser parser = new JavaDocParser(javaDocText);
        JavaDocParser.JavaDocTag[] tags = parser.getTags();

        List tagsList = new ArrayList();
        String text = null;

        if (tags != null && tags.length > 0) {
            for (int i = 0; i < tags.length; i++) {
                JavaDocParser.JavaDocTag tag = tags[i];
                if (tag.isText()) {
                    text = tag.getValue();
                    continue;
                }
                String tagName = tag.getName();
                String tagText = tag.getValue();
                tagsList.add(tagValueClass.create(tagName, tagText));
            }
        }

        JavaDocImpl javaDoc = (JavaDocImpl) javaDocClass.createJavaDoc(text, tagsList);

        changeChild(javadoc, javaDoc);
        return javadoc = javaDoc;
    }

    /**
     * Sets the value of javadoc attribute. See {@link #getJavadoc} for description
     * on the attribute.
     * @param newValue New value to be set.
     */
    public void setJavadoc(JavaDoc newValue) {
        objectChanged(CHANGED_JAVADOC);
        changeChild(javadoc, newValue);
        javadocText = null;
        javadoc = newValue;
    }

    /** Helper class
     */
    public ClassDefinition getDeclaringClass() {
        RefFeatured obj = refImmediateComposite();

        if (obj instanceof ClassDefinition) {
            return (ClassDefinition) obj;
        } else if (this instanceof Field && obj instanceof FieldGroup) {
            return ((FieldGroup) obj).getDeclaringClass();
        }
        return null;
    }

    public PositionBounds getPosition(boolean inclDoc) {
        ResourceImpl resource = (ResourceImpl)getResource();
        PositionBounds result = null;
        if (!inclDoc && (result = resource.getFeaturePosition(this)) != null)
            return result;
        else {
            MDRepository repository = repository();
            repository.beginTrans(false);
            try {
                if (JMManager.getTransactionMutex().getClassPath() == null) {
                    JavaModel.setClassPath(resource);
                }
                return super.getPosition(false);
            } finally {
                repository.endTrans(false);
            }    
        }
    }
    
    /** Helper method
     */
    protected RefObject getNameAttr(String name) {
        try {
            return ((MofClass) refMetaObject()).lookupElementExtended(name);
        }
        catch (javax.jmi.model.NameNotFoundException e) {
            return null;
        }
    }

    /**
     * Adds the new diff item to the list if the javaDoc is changed in element,
     * otherwise it does nothing.
     * todo (#pf): currently, we use only javadocText. 
     *
     * @param  diff  list of collected differences where the new element is added
     */
    protected void replaceJavaDoc(List diff) {
        if (isChanged(CHANGED_JAVADOC)) {
            ASTree tree = getASTree();
            Token docToken = getParser().getComment(tree);
            // todo (#pf): now we only handle javaDoc text, we ignore
            // settings from model -- (For the time being it is enough 
            // to provide information for compatability bridge.)
            String newContent = JavaDocParser.surroundWithJavaDocStars(javadocText);
            if (docToken.getType() == ParserTokens.DOC_COMMENT && javadocText != null) {
                // replace the original javaDoc if exists already
                diff.add(new DiffElement(docToken.getStartOffset(),
                                         docToken.getEndOffset(),
                                         indentJavaDoc(newContent).trim()
                                        )
                        );
                // We cannot use general replaceNode method because it
                // doesn't work on padding token. (Method getFirstToken()
                // and getLastToken() cannot return token indexes because they
                // aren't in the token list.)
                //
                // replaceNode(diff, 
                //            getParser(),
                //            docToken,
                //            , 
                //            docToken.getStartOffset(),
                //            null);
            } else {
                // we have to reset original javadoc
                if (javadocText == null) {
                    int startOffset = docToken.getStartOffset();
                    int endOffset = docToken.getEndOffset();
                    if (getParser().getSourceText().length() > endOffset) {
                        int temp = getParser().getSourceText().indexOf('\n', endOffset);
                        if (temp > -1) {
                            endOffset = temp + 1;
                        }
                    }
                    diff.add(new DiffElement(startOffset, endOffset, ""));
                } else {
                    // create new javaDoc in case it didn't exist yet
                    // we will look for the last new line before the element. The
                    // comment will be put before this new line. If the new line
                    // does not exist before the element, we will put the javadoc
                    // comment directly before the first char of the element and
                    // we will add new line after the comment.
                    int startOffset = getLastNewLineOffset(docToken) + 1;
                    String indentedDoc = getIndentation().concat(indentJavaDoc(newContent).trim()).concat("\n");
                    if (startOffset <= 0) {
                        startOffset = docToken.getStartOffset();
                        indentedDoc = '\n' + indentedDoc + getIndentation();
                    }
                    diff.add(new DiffElement(startOffset,
                                             startOffset,
                                             indentedDoc
                                            )
                            );
                }
            }
        }
    }

    /**
     * Looks for the last new line before the token in its paddings.
     * 
     * @param   token  token to look for the new line before
     * @return  offset of the last new line in the token's paddings
     */
    private int getLastNewLineOffset(Token token) {
        Token[] pad = token.getPadding();
        // previously, empty array was returned - with gjast.jar, it is possible
        // to get null value. We have to check it.
        if (pad == null) return -1; 
        for (int i = pad.length; i > 0; i--) {
            if (pad[i-1].getType() == ParserTokens.EOL) {
                String value = getParser().getText(pad[i-1]);
                return pad[i-1].getStartOffset() + value.lastIndexOf('\n');
            }
        }
        return -1;
    }
    
    String getModifiersText() {
        StringBuffer sb = new StringBuffer();
        int mod = getSourceModifiers();
        int len;
        
        if ((mod & Modifier.PUBLIC) != 0)       sb.append("public "); // NOI18N
        if ((mod & Modifier.PROTECTED) != 0)    sb.append("protected "); // NOI18N
        if ((mod & Modifier.PRIVATE) != 0)      sb.append("private "); // NOI18N
        
        /* Canonical order */
        if ((mod & Modifier.ABSTRACT) != 0)     sb.append("abstract "); // NOI18N
        if ((mod & Modifier.STATIC) != 0)       sb.append("static "); // NOI18N
        if ((mod & Modifier.FINAL) != 0)        sb.append("final "); // NOI18N
        if ((mod & Modifier.TRANSIENT) != 0)    sb.append("transient "); // NOI18N
        if ((mod & Modifier.VOLATILE) != 0)     sb.append("volatile "); // NOI18N
        if ((mod & Modifier.SYNCHRONIZED) != 0) sb.append("synchronized "); // NOI18N
        if ((mod & Modifier.NATIVE) != 0)       sb.append("native "); // NOI18N
        if ((mod & Modifier.STRICT) != 0)       sb.append("strictfp "); // NOI18N
        
        if ((len = sb.length()) > 0)    /* trim trailing space */
            return sb.toString().substring(0, len-1);
        return "";
    }

    protected void diffModifiers(List diffList, ASTree nextNode, ASTProvider parser) {
        String text = "";
        for (Iterator annIt = getAnnotations().iterator(); annIt.hasNext(); ) {
            AnnotationImpl ann = (AnnotationImpl) annIt.next();
            text += ann.getSourceText() + '\n';
            text += getIndentation();
        }
        String modText = getModifiersText();
        if (modText.length() > 0) {
            text += modText;
        }
        int nextToken = nextNode.getFirstToken();
        int startOffset, endOffset;
        int startToken;
        endOffset = parser.getToken(nextToken).getStartOffset();
        ASTree modifiers = getASTree().getSubTrees()[0];
        if (modifiers != null) {
            startToken = modifiers.getFirstToken();
            startOffset = parser.getToken(startToken).getStartOffset();
            if (text.length() > 0) {
                int endToken = modifiers.getLastToken();
                endOffset = parser.getToken(endToken).getEndOffset();
            }
        } else {
            startOffset = endOffset;
            if (modText.length() > 0) {
                text += ' ';
            }
        }
        diffList.add(new DiffElement(startOffset, endOffset, text));
    }
    
    /**
     * Generates element's javadoc and appends it to the buffer.
     *
     * @param  buf  buffer to append javadoc to
     * @return true, if there is JavaDoc, otherwise false
     */
    public boolean generateNewJavaDoc(StringBuffer buf) {
        String javadoc = getJavadocText();
        if (javadoc != null) {
            buf.append(indentJavaDoc(JavaDocParser.surroundWithJavaDocStars(javadoc)));
            buf.append('\n').append(getIndentation());
            return true;
        }
        else
            return false;
    }

    /**
     * Generates element's modifiers and append them to the buffer.
     * Moreover, it appends also annotation for this feature.
     *
     * @param  buf  append modifiers to the buffer
     * @return      true if there is any modifier, otherwise false
     */
    public boolean generateNewModifiers(StringBuffer buf) {
        boolean nju = isNew();
        if (!nju) IndentUtil.reformatHeadGarbage(this, buf);
        for (Iterator annIt = getAnnotations().iterator(); annIt.hasNext(); ) {
            AnnotationImpl ann = (AnnotationImpl) annIt.next();
            buf.append(ann.getSourceText()).append('\n').append(getIndentation());
        }
        String modString = getModifiersText();
        if (modString.length() > 0) {
            buf.append(modString);
            buf.append(' ');
            return true;
        }
        else
            return false;
    }

    /**
     * Find referenced resources using ClassIndex.findResourcesForIdentifier().
     * Modifiers are considered to reduce number of resources
     */
    Resource[] findReferencedResources() {
        return findReferencedResources(false);
    }
    
    /**
     * Find referenced resources using ClassIndex.findResourcesForIdentifier().
     * Modifiers are considered to reduce number of resources
     */
    Resource[] findReferencedResources(boolean includeLibraries) {
        int modifiers=getModifiers();
        String name;
        Resource[] res;

        if (Modifier.isPrivate(modifiers) || isTransient()) {
            return new Resource[] {getResource()};
        }
        if (this instanceof JavaClass) {
            name=((JavaClass)this).getSimpleName();
        } else if (this instanceof Constructor) {
            name=((JavaClass)getDeclaringClass()).getSimpleName();
        } else {
            name=getName();
        }
        res=ClassIndex.findResourcesForIdentifier(name, includeLibraries);
        
        if (Modifier.isPublic(modifiers) || Modifier.isProtected(modifiers)) {
            if (this instanceof Field) {
                Element cd = getDeclaringClass();
                boolean isPrivate = false;
                boolean isPackagePrivate = false;
                while (cd != null && !(cd instanceof Resource)) {
                    if (cd instanceof JavaClass) {
                        int m = ((JavaClass) cd).getModifiers();
                        if (Modifier.isPrivate(m)) {
                            isPrivate = true;
                            break;
                        } else if (!isPackagePrivate && !Modifier.isPublic(m) && !Modifier.isProtected(m)) {
                            isPackagePrivate = true;
                        }
                    }
                    cd = (Element) cd.refImmediateComposite();
                }
                if (isPrivate) {
                    return new Resource[] {getResource()};
                } else if (isPackagePrivate) {
                    return filterResourcesFromThisPackage(res);
                }
            }
            return res;
        }
        return  filterResourcesFromThisPackage(res);
    }
    
    Resource[] filterResourcesFromThisPackage(Resource[] res) {
        List filteredResources=new ArrayList(res.length);
        String packageName=getResource().getPackageName();
        for (int i=0;i<res.length;i++) {
            Resource r=res[i];

            if (r.getPackageName().equals(packageName))
                filteredResources.add(r);
        }
        return (Resource[])filteredResources.toArray(new Resource[filteredResources.size()]);
    }

    /**
     * Returns indented JavaDoc
     *
     * @return  indented Javadoc
     */
    protected String indentJavaDoc(String javadoc) {
        if (javadoc == null)
            return "";
        StringTokenizer t = new StringTokenizer(javadoc, "\n", true); // NOI18N
        StringBuffer b = new StringBuffer();
        String i = getIndentation();
        while (t.hasMoreTokens()) {
            String line = t.nextToken();
            b.append(line);
            if (line.equals("\n")) // NOI18N
                b.append(i);
        }
        return b.toString();
    }
    
    protected String getIndentation() {
        return ((MetadataElement) refImmediateComposite()).getIndentation().concat(INDENTATION);
    }

    public List getChildren() {
        List list = new ArrayList();
        list.addAll(getAnnotations());
        return list;
    }
    
    public void fixImports(Element scope, Element original) {
        fixImports(scope,getAnnotations(),((Feature)original).getAnnotations());
    }

    protected List getInitedChildren() {
        List result = new ArrayList();
        if (childrenInited) {
            result.addAll(getAnnotations());
        }
        return result;
    }

    // [TODO] remove this method after it is properly implemented in Field, Method and Constructor
    protected ASTree getPartTree(ElementPartKind part) {
        if (ElementPartKindEnum.HEADER.equals(part)) {
            return getASTree();
        }
        throw new IllegalArgumentException("Invalid part for this element: " + part.toString()); // NOI18N
    }

    public void replaceChild(Element oldElement, Element newElement) {
        super.replaceChild(oldElement, newElement);
        if (childrenInited) {
            if (replaceObject(getAnnotations(), oldElement, newElement)) return;
        }
        if (javadoc != null) {
            if (javadoc.equals(oldElement)) {
                changeChild(oldElement, newElement);
                javadoc.refDelete();
                javadoc = (JavaDoc) newElement;
                return;
            }
        }
    }

    protected void setData(List annotations, String javadocText, JavaDoc javadoc) {
        this.annotations = createChildrenList( "annotations", (AttrListWrapper) super_getAnnotations(), annotations, CHANGED_ANNOTATION); // NOI18N
        if (javadocText == null) {
            changeChild(null, javadoc);
            this.javadoc = javadoc;
        } else {
            if (javadoc != null) {
                throw new ConstraintViolationException(this, getNameAttr("javadoc"), "Cannot set both javadocText and javadoc."); // NOI18N
            }
            this.javadocText = javadocText;
        }
        childrenInited = true;
    }
    
    protected void setTypeRef(TypeRef type) {
        _getDelegate().setSlot1(type);
    }
    
    public TypeRef getTypeRef() {
        return (TypeRef) _getDelegate().getSlot1();
    }

    protected void matchPersistent(ElementInfo info) {
        super.matchPersistent(info);
        matchModifiers(info);
    }
    
    protected void matchElementInfo(ElementInfo newInfo) {
        super.matchElementInfo(newInfo);
        resetJavaDoc();
    }
    
    protected void matchModifiers(ElementInfo info) {
        if (((FeatureInfo) info).modifiers != getSourceModifiers()) {
            setModifiers(((FeatureInfo) info).modifiers);
        }
    }
    
    protected abstract List super_getAnnotations();
    protected abstract String super_getJavadocText();
    protected abstract void super_setJavadocText(String text);

    public boolean isPersisted() {
        return super_getJavadocText() != null;
    }
    
    public void setPersisted(boolean persisted) {
        super_setJavadocText(persisted ? "" : null);
    }

    private void resetJavaDoc() {
        javadocText = null;
        if (javadoc != null) {
            changeChild(javadoc, null);
            javadoc.refDelete();
            javadoc = null;
        }
    }
    
    protected void _delete() {
        // --- delete components -------------------------------------------
        // delete all annotations (if initialized)
        deleteChildren("annotations", (AttrListWrapper) super_getAnnotations());
        resetJavaDoc();
        // --- delete links -----------------------------------------------
        // no links to delete
        // [TODO] should Throws association be notified?
        // --- call super -------------------------------------------------
        super._delete();
    }

    protected void reformatComment(final StringBuffer buf) {
        Token[] t = getParser().getToken(getASTree().getFirstToken()).getPadding();
        if (t.length > 0) {
            String doc = IndentUtil.indentExistingElement(this, 
                    getParser().getSourceText().substring(t[0].getStartOffset(), t[t.length-1].getEndOffset()));
            if (doc.length() > 0) {
                doc = doc.substring(1) + '\n';
                buf.append(doc);
            }
        }
    }
}
