/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.lang.reflect.Modifier;
import java.util.List;
import javax.jmi.reflect.ConstraintViolationException;
import javax.jmi.reflect.RefFeatured;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.ElementPartKind;
import org.netbeans.jmi.javamodel.ElementPartKindEnum;
import org.netbeans.jmi.javamodel.EnumConstant;
import org.netbeans.jmi.javamodel.InitialValue;
import org.netbeans.jmi.javamodel.JavaDoc;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.TypeReference;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.parser.ASTProvider;
import org.netbeans.modules.javacore.parser.ClassInfo;
import org.netbeans.modules.javacore.parser.ElementInfo;
import org.netbeans.modules.javacore.parser.FeatureInfo;


/** Implements object representing enum constant.
 * 
 * @author Martin Matula
 */
public abstract class EnumConstantImpl extends FeatureImpl implements EnumConstant {
    private static final ElementInfo DEFAULT_INFO = new FeatureInfo(null, FeatureInfo.ENUM_CONSTANT_TYPE, null, 0, null);

    RefFeatured parent; // used for hard-referencing parent
    
    // attribute values
    ClassDefinition body;
    InitialValue initialValue;
    String initialValueText;
    
    private boolean initValueInited = false;


    /** Creates a new instance of EnumConstantImpl */
    public EnumConstantImpl(StorableObject s) {
        super(s);
    }

    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }
    
    public void setModifiers(int modifiers) {
        throw new ConstraintViolationException(null, null, "Cannot set modifiers of enum constant."); // NOI18N
    }
    
    public int getModifiers() {
        return Modifier.FINAL | Modifier.PUBLIC | Modifier.STATIC;
    }
    
    public boolean isFinal() {
        return true;
    }
    
    public void setFinal(boolean isFinal) {
        setModifiers(0);
    }

    /**
     * Returns the value of reference type.
     * @return Value of reference type.
     */
    public Type getType() {
        return (Type) refImmediateComposite();
    }

    /**
     * Sets the value of reference type. See {@link #getType} for description
     * on the reference.
     * @param newValue New value to be set.
     */
    public void setType(Type newValue) {
        throw new ConstraintViolationException(null, null, "Cannot change enum constant type."); // NOI18N
    }

    /**
     * Find referenced resources using ClassIndex.findResourcesForIdentifier().
     * Modifiers are considered to reduce number of resources
     */
    Resource[] findReferencedResources() {
        return ClassIndex.findResourcesForIdentifier(getName());
    }
    
    protected List getInitedChildren() {
        List list = super.getInitedChildren();
        if (childrenInited) {
            if (initValueInited) {
                if (!isChanged(CHANGED_INITIAL_VALUE) || initialValueText == null) {
                    addIfNotNull(list, initialValue);
                }
                addIfNotNull(list, body);
            }
        }
        return list;
    }

    public List getChildren() {
        List list = super.getChildren();
        if (isChanged(CHANGED_INITIAL_VALUE) && initialValueText != null) {
            throw new IllegalStateException("Cannot get children when the initial value text has been changed."); // NOI18N
        }
        addIfNotNull(list, getInitialValue());
        addIfNotNull(list, getBody());
        return list;
    }

    protected void initChildren() {
        childrenInited = false;
        super.initChildren();
        childrenInited = true;
        if (initValueInited) {
            JMManager.getTransactionMutex().addBFeatureToInitQueue(this);
        }
    }
    
    public void initInitValue() {
        initValueInited = false;
        if (!childrenInited) {
            initChildren();
        }
        FeatureInfo initValInfo = (FeatureInfo) getElementInfo();
        if (initValInfo != null) {
            initValInfo.doAttribution(this);
            ASTree tree = getASTree();
            if (tree != null) {
                ASTree[] parts = tree.getSubTrees();
                initialValue = (InitialValue) initOrCreate(initialValue, parts[PARAMETERS]);
                if (parts[2] == null) {
                    body = null;
                } else {
                    ClassInfo constantInfo = (ClassInfo) getParser().getSemanticInfo(parts[CLASS_DEFINITION], this);
                    // do we have already body of constant?
                    if (body != null) {
                        // body was removed, remove it from model
                        if (constantInfo == null) {
                            changeChild(body, null);
                            body.refDelete();
                            body = null;
                        } else {
                            // replace old constant body with the new one
                            ((ClassDefinitionImpl) body).setElementInfo(constantInfo);
                        }
                    } else if (constantInfo != null) {
                        // create new body - ClassDefinition
                        SemiPersistentElement s = (SemiPersistentElement) JavaModelUtil.getDeclaringFeature(this);
                        body = (ClassDefinition) s.createElement(constantInfo);
                        changeChild(null, body);
                    }
                }
            }
        }
        initValueInited = true;
    }

    private static final int NAME = 0;
    private static final int PARAMETERS = 1;
    private static final int CLASS_DEFINITION = 2;

    protected void matchModifiers(ElementInfo info) {
        // do not match modifiers
    }
    
    /** The method has to make sure that the AST infos of children are also updated.
     */
    protected void matchElementInfo(ElementInfo newInfo) {
        super.matchElementInfo(newInfo);
        resetInitValue();
    }
    
    protected void resetChildren() {
        super.resetChildren();
        resetInitValue();
        childrenInited = false;
    }

    private void resetInitValue() {
        if (initialValue != null) {
            InitialValue temp = initialValue;
            initialValue = null;
            changeChild(temp, null);
            temp.refDelete();
        }

        if (body != null) {
            ClassDefinition temp = body;
            body = null;
            changeChild(temp, null);
            temp.refDelete();
        }
        initValueInited = false;
    }

    public void replaceChild(Element oldElement, Element newElement) {
        if (initValueInited) {
            if (oldElement.equals(initialValue)) {
                setInitialValue((InitialValue) newElement);
                return;
            }
            if (oldElement.equals(body)) {
                setBody((ClassDefinition) newElement);
                return;
            }
        }
        super.replaceChild(oldElement, newElement);
    }

    protected void setData(List annotations, String javadocText, JavaDoc javadoc, InitialValue initialValue, String initialValueText, ClassDefinition body) {
        super.setData(annotations, javadocText, javadoc);
        if (initialValueText == null) {
            changeChild(null, initialValue);
            this.initialValue = initialValue;
        } else {
            if (initialValue != null) {
                throw new ConstraintViolationException(null, null, "Cannot set both initialValue and initialValueText."); // NOI18N
            }
            this.initialValueText = initialValueText;
        }
        changeChild(null, body);
        this.body = body;
        initValueInited = true;
        childrenInited = true;
    }

    String getRawText() {
        // we have element which is new or changed and moved.
        StringBuffer buf = new StringBuffer();
        generateNewJavaDoc(buf);
        buf.append(getName());
        if (initialValueText != null) {
            buf.append('(').append(initialValueText).append(')');
        } else if (initialValue != null) {
            buf.append('(').append(((MetadataElement) initialValue).getSourceText()).append(')');
        }
        if (body != null) {
            buf.append(' ').append(((MetadataElement) body).getSourceText());
        }
        return buf.toString();
    }

    public void getDiff(List diffList) {
        // element was present in the original source code
        ASTProvider parser = getParser();
        ASTree tree = getASTree();
        ASTree[] children = tree.getSubTrees();

        // javadoc print
        replaceJavaDoc(diffList);
        if (isChanged(CHANGED_NAME)) {
            replaceNode(diffList, parser, children[NAME], getName(), 0, null);
        }
        if (isChanged(CHANGED_INITIAL_VALUE)) {
            int startOffset = 0, endOffset = 0;
            String text = initialValue == null ? initialValueText :
                ((MetadataElement) getInitialValue()).getSourceText();
            Token lpar = parser.getToken(children[0].getLastToken() + 1);
            // there was an initial value in original source code.
            if (lpar.getType() == ParserTokens.L_PAR) {
                startOffset = lpar.getStartOffset();
                Token endToken = parser.getToken(children[1] != null ? children[1].getLastToken() + 1 : children[0].getLastToken() + 2);
                endOffset = endToken.getEndOffset();
            }
            // there was no initial value in original source
            else {
                startOffset = parser.getToken(children[0].getLastToken()).getEndOffset();
                // we are inserting new text, offsets contain the same value
                endOffset = startOffset;
            }
            diffList.add(new DiffElement(startOffset, endOffset, text == null ? "" : "(" + text + ")")); // NOI18N
        } else if (initialValue != null && ((MetadataElement) initialValue).isChanged()) {
            ((MetadataElement) initialValue).getDiff(diffList);
        }
        int startToken = children[PARAMETERS] == null ? children[0].getLastToken() : children[PARAMETERS].getLastToken() + 1;
        getChildDiff(diffList, parser, children[CLASS_DEFINITION], (MetadataElement) getBody(), CHANGED_ENUM_CONST_BODY, parser.getToken(startToken).getEndOffset(), " "); // NOI18N
    }

    public void setBody(ClassDefinition newValue) {
        if (!initValueInited) {
            initInitValue();
        }
        objectChanged(CHANGED_ENUM_CONST_BODY);
        changeChild(body, newValue);
        body = newValue;
    }

    public ClassDefinition getBody() {
        checkUpToDate();
        if (!initValueInited) {
            initInitValue();
        }
        return body;
    }

    public InitialValue getInitialValue() {
        checkUpToDate();
        if (isChanged(CHANGED_INITIAL_VALUE) && initialValueText != null) {
            throw new ConstraintViolationException(null, null, "Cannot ask for initial value after the initial value text was changed."); // NOI18N
        }
        if (!initValueInited) {
            initInitValue();
        }
        return initialValue;
    }

    public void setInitialValue(InitialValue newValue) {
        if (!initValueInited) {
            initInitValue();
        }
        changeChild(initialValue, newValue);
        initialValue = newValue;
        initialValueText = null;
        objectChanged(CHANGED_INITIAL_VALUE);
    }
    
    public java.lang.String getInitialValueText(){
        checkUpToDate();
        if (isChanged(CHANGED_INITIAL_VALUE)) {
            if (initialValue != null) {
                throw new ConstraintViolationException(null, null, "Cannot ask for initial value text after the initial value was changed."); // NOI18N
            }
            return initialValueText;
        } else {
            return extractInitialValueText();
        }
    }

    public void setInitialValueText(java.lang.String newValue){
        if (!initValueInited) {
            initInitValue();
        }
        if (initialValue != null) {
            changeChild(initialValue, null);
        }
        objectChanged(CHANGED_INITIAL_VALUE);
        initialValueText = newValue;
        initialValue = null;
    }

    private String extractInitialValueText() {
        ASTProvider parser = getParser();
        if (parser == null)
            return null;
        ASTree initValue = getASTree().getSubTrees()[PARAMETERS];
        if (initValue == null)
            return null;
        return parser.getText(initValue);
    }

    public int getDimCount() {
        return 0;
    }

    public void setDimCount(int dimCount) {
        throw new ConstraintViolationException(null, null, "Cannot change enum constant dimCount."); // NOI18N
    }

    public TypeReference getTypeName() {
        return null;
    }

    public void setTypeName(TypeReference typeName) {
        throw new ConstraintViolationException(null, null, "Cannot change enum constant type name."); // NOI18N
    }
    
    public Element duplicate(JavaModelPackage targetExtent) {
        InitialValue initVal;
        String initValText;
        
        if (isChanged(CHANGED_INITIAL_VALUE) && this.initialValueText != null) {
            initVal = null;
            initValText = this.initialValueText;
        } else {
            initVal = (InitialValue) duplicateElement(getInitialValue(), targetExtent);
            initValText = null;
        }
        
        return targetExtent.getEnumConstant().createEnumConstant(
                getName(),
                duplicateList(getAnnotations(), targetExtent),
                getModifiers(),
                null,
                (JavaDoc) duplicateElement(getJavadoc(), targetExtent),
                isFinal(),
                (TypeReference) duplicateElement(getTypeName(), targetExtent),
                getDimCount(),
                initVal,
                initValText,
                (ClassDefinition) duplicateElement(getBody(), targetExtent)
               );
    }

    protected void _delete() {
        if (initValueInited) {
            deleteChild(initialValue);
            deleteChild(body);
        }
        super._delete();
    }
    
    protected ASTree getPartTree(ElementPartKind part) {
        if (ElementPartKindEnum.NAME.equals(part)) {
            return getASTree().getSubTrees()[0];
        }
        return super.getPartTree(part);
    }
    
}
