/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.jmiimpl.javamodel;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.jmi.reflect.RefObject;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.Attribute;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.ElementReference;
import org.netbeans.jmi.javamodel.InitialValue;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.StringLiteral;
import org.netbeans.jmi.javamodel.StringLiteralClass;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.parser.AnnotationInfo;
import org.netbeans.modules.javacore.parser.AnnotationValueInfo;
import org.netbeans.modules.javacore.parser.ElementInfo;
import org.openide.util.Utilities;


/**
 *
 * @author  Pavel Flaska
 */
public abstract class AttributeValueImpl extends SemiPersistentElement implements AttributeValue, ElementReference {
    private static final ElementInfo DEFAULT_INFO = new AnnotationValueInfo(null, AnnotationValueInfo.ANNOTATIONVALUE_STRING, null, null);
    
    private String attributeValueText;
    private InitialValue attributeValue;
    
    
    /** Creates a new instance of AnnotationValueImpl */
    public AttributeValueImpl(StorableObject o) {
        super(o);
    }

    public InitialValue getValue() {
        if (!childrenInited) {
            initChildren();
        }
        return attributeValue;
    }
    
    public void setValue(InitialValue newValue) {
        objectChanged(CHANGED_INITIAL_VALUE);
        
        StringLiteral oldStrLit = attributeValue instanceof StringLiteral ? (StringLiteral)attributeValue : null;
        StringLiteral newStrLit = newValue instanceof StringLiteral ? (StringLiteral)newValue : null;
        String oldVal = oldStrLit != null ? oldStrLit.getValue() : null;
        String newVal = newStrLit != null ? newStrLit.getValue() : null;
        if (!Utilities.compareObjects(oldVal, newVal)) {
            _getDelegate().setSlot2(newVal);
            fireAttrChange("value", oldStrLit, newStrLit); // NOI18N
        }
        
        changeChild(getValue(), newValue);
        attributeValue = newValue;
    }
        
    public Attribute getDefinition() {
        String name=getName();
        
        if (name!=null) {
            return (Attribute)getElement();
        } else {
            Annotation ann=(Annotation)refImmediateComposite();
            AnnotationType annType=ann.getType();
            
            if (annType!=null) {
                int valIndex=ann.getAttributeValues().indexOf(this);
                Object[] features=annType.getContents().toArray();
                int attrIndex=0;
                
                for (int i=0;i<features.length;i++) {
                    Object obj = features[i];
                    if (obj instanceof Attribute) {
                        if (valIndex==attrIndex)
                            return (Attribute)obj;
                        attrIndex++;
                    }
                }
            }
        }
        return null;
    }
    
    public void setDefinition(Attribute newValue) {
        throw new UnsupportedOperationException("setDefinition()"); // NOI18N
    }
    
    public List getChildren() {
        List list = new ArrayList(1);
        addIfNotNull(list, getValue());
        return list;
    }
    
    ////////////////////////////////////////////////////////////////////////////
    public String getSourceText() {
        String origElem;
        if ((origElem = checkChange()) != null)
            return origElem;
        
        StringBuffer buf = new StringBuffer();
        String name = getName();
        if (name != null && name.length() > 0) {
            buf.append(name);
            buf.append(formatElementPart(FIELD_EQUALS));
        }
        buf.append(((MetadataElement) getValue()).getSourceText());
        return buf.toString();
    }
    
    public void getDiff(List diff) {
        if (isChanged(CHANGED_NAME)) {
            replaceNode(diff, getParser(), getASTree().getSubTrees()[0], getName(), 0, null);
        }
        if (isChanged(CHANGED_INITIAL_VALUE)) {
            getChildDiff(diff, getParser(), getASTree().getSubTrees()[1], (MetadataElement) attributeValue, CHANGED_INITIAL_VALUE);
        } else if (isChanged(CHANGED_CHILDREN)) {
            if (attributeValue != null) {
                ((MetadataElement) attributeValue).getDiff(diff);
            }
        }
    }
    
    void setData(InitialValue attributeValue, String attributeValueText) {
        changeChild(null, attributeValue);
        this.attributeValue = attributeValue;
        this.attributeValueText = attributeValueText;
    }
    
    protected void initChildren() {
        if (childrenInited) {
            JMManager.getTransactionMutex().addBFeatureToInitQueue(this);
        } else {
            initInitValue();
        }
    }
    
    public void initInitValue() {
        childrenInited = false;
        RefObject parent = (RefObject) refImmediateComposite();
        while (parent != null && !(parent instanceof AnnotationImpl)) {
            parent = (RefObject) parent.refImmediateComposite();
        }
        if (parent != null) {
            AnnotationImpl annParent = (AnnotationImpl) parent;
            AnnotationInfo initValInfo = (AnnotationInfo) annParent.getElementInfo();
            if (initValInfo != null) {
                AnnotationValueInfo info = (AnnotationValueInfo) getElementInfo();
                ASTree tree = getASTree();
                if (tree != null) {
                    initValInfo.doAttribution(annParent);
                }
                if (info.infoType == AnnotationValueInfo.ANNOTATIONVALUE_ANNOTATION) {
                    InitialValue oldInit = attributeValue;
                    attributeValue = (Annotation) createElement((AnnotationInfo)info.value);
                    changeChild(oldInit,attributeValue);
                    if (oldInit != null) {
                        oldInit.refDelete();
                    }
                } else {
                    attributeValue = (InitialValue) initOrCreate(attributeValue, extractAttributeValue());
                }
            }
        }
        childrenInited = true;
    }
    
    private ASTree extractAttributeValue() {
        return getASTree().getSubTrees()[1];
    }
    
    public Element duplicate(JavaModelPackage targetExtent) {
        return targetExtent.getAttributeValue().createAttributeValue(
                getName(), (InitialValue) duplicateElement(getValue(), targetExtent));
    }

    public NamedElement getElement() {
        String name=getName();
        
        if (name!=null) {
            AnnotationType annType=((Annotation)refImmediateComposite()).getType();
        
            if (annType!=null) {
                Iterator iter = annType.getFeatures().iterator();
                while (iter.hasNext()) {
                    Object obj = iter.next ();
                    if (obj instanceof Attribute) {
                        Attribute attrib = (Attribute) obj;
                        if (name.equals(attrib.getName()))
                            return attrib;
                    }
                }
            }
        }
        return null;
    }

    protected ElementInfo getDefaultInfo() {
        return DEFAULT_INFO;
    }

    protected void matchPersistent(ElementInfo info) {
        super.matchPersistent(info);
        AnnotationValueInfo ainfo = (AnnotationValueInfo)info;
        if (!isPersisted()) {
            setPersisted(true);
            persist();
            //  persist elements
            String aValue = null;
            if (ainfo.infoType == AnnotationValueInfo.ANNOTATIONVALUE_STRING) {
                aValue = (String) ainfo.value;
                if (aValue != null && aValue.length() >= 2 && aValue.startsWith("\"") && aValue.endsWith("\"")) { // NOI18N
                    aValue = aValue.substring(1, aValue.length() - 1);
                } else {
                    aValue = null;
                }
            }
            _getDelegate().setSlot2(aValue);
        } else {
            // match elements
            String oldValue = (String) _getDelegate().getSlot2();
            String newValue = ainfo.infoType == AnnotationValueInfo.ANNOTATIONVALUE_STRING ? (String)ainfo.value : null;
            if (newValue != null) {
                if (newValue.length() >= 2 && newValue.startsWith("\"") && newValue.endsWith("\"")) { // NOI18N
                    newValue = newValue.substring(1, newValue.length() - 1);
                } else {
                    newValue = null;
                }
            }
            if (!Utilities.compareObjects(oldValue, newValue)) {
                _getDelegate().setSlot2(newValue);
                StringLiteralClass proxy = ((JavaModelPackage)refImmediatePackage()).getStringLiteral();
                StringLiteral oldV = oldValue != null ? proxy.createStringLiteral(oldValue) : null;
                StringLiteral newV = newValue != null ? proxy.createStringLiteral(newValue) : null;
                fireAttrChange("value", oldV, newV); // NOI18N
            }
        }
    }

    protected void matchElementInfo(ElementInfo newInfo) {
        super.matchElementInfo(newInfo);
        resetChildren();
    }

    protected void resetChildren() {
        if (childrenInited) {
            InitialValue temp = attributeValue;
            attributeValue = null;
            changeChild(temp, null);
            temp.refDelete();
            childrenInited = false;
        }        
    }

    public void replaceChild(Element oldElement, Element newElement) {
        if (childrenInited && oldElement.equals(attributeValue)) {
            setValue((InitialValue) newElement);
            return;
        }
        super.replaceChild(oldElement, newElement);
    }
    
    public boolean isPersisted() {
        return _getDelegate().getSlot1() != null;
    }
    
    public void setPersisted(boolean persisted) {
        _getDelegate().setSlot1(persisted ? "" : null);
    }

    protected void _delete() {
        if (childrenInited) {
            deleteChild(attributeValue);
        }
        super._delete();
    }
    
}
