/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.classpath;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.classpath.GlobalPathRegistry;
import org.netbeans.api.java.classpath.GlobalPathRegistryEvent;
import org.netbeans.api.java.classpath.GlobalPathRegistryListener;
import org.netbeans.api.java.queries.SourceForBinaryQuery;
import org.netbeans.spi.java.classpath.ClassPathImplementation;
import org.netbeans.spi.java.classpath.PathResourceImplementation;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.URLMapper;
import org.openide.util.Utilities;
import org.openide.util.WeakListeners;

public class MergedClassPathImplementation implements ClassPathImplementation {

    public static final String PROP_UNRESOLVED_ROOTS = "unresolvedRoots";       //NOI18N

    private Set roots;
    private PropertyChangeSupport support;
    private GlobalPathRegistry reg;
    private final ArrayList/*<PathResourceImplementation>*/ cachedResources;
    private List unresolvedRoots;
    private List missingRoots;
    private ClassPathMap resourceMap;
    private GlobalPathRegistryListener gprListener;
    private PropertyChangeListener pcListener = new PropertyChangeListener() {
        public void propertyChange(PropertyChangeEvent event) {
            assert event != null : "event == null";                     //NOI18N
            synchronized (MergedClassPathImplementation.this) {
                roots = null;
            }
            if (ClassPath.PROP_ENTRIES.equals(event.getPropertyName())) {
                MergedClassPathImplementation.this.updateEntries((ClassPath)event.getSource());
            }
        }
    };
    
    private WeakHashMap/*<ClassPath, WeakReference<SFBQListener>>*/ sfbResultListeners = new WeakHashMap(100);
    
    private static MergedClassPathImplementation instance;

    private MergedClassPathImplementation () {
        this.support = new PropertyChangeSupport(this);
        this.cachedResources = new ArrayList ();
        this.missingRoots = new ArrayList();
        this.reg = GlobalPathRegistry.getDefault();
        this.gprListener = new GlobalPathRegistryListener() {
            public void pathsAdded(GlobalPathRegistryEvent event) {
                assert event != null : "event == null"; // NOI18N
                synchronized (MergedClassPathImplementation.this) {
                    roots = null;
                }
                MergedClassPathImplementation.this.updateEntries(event);
                MergedClassPathImplementation.this.firePropertyChange (PROP_UNRESOLVED_ROOTS);
            }

            public void pathsRemoved(GlobalPathRegistryEvent event) {
            }
        };
        this.reg.addGlobalPathRegistryListener ((GlobalPathRegistryListener)
                WeakListeners.create(GlobalPathRegistryListener.class,this.gprListener,this.reg));
        // XXX Shouldn't we also check for original contents of GPR?
        assert this.reg != null : "GloabalPathRegistry.getDefault()==null";              //NOI18N
    }


    public synchronized List/*<PathResourceImplementation>*/ getResources() {
        return Collections.unmodifiableList((List) this.cachedResources.clone());
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        assert listener != null : "gprListener == null";                               //NOI18N
        this.support.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        assert listener != null : "gprListener == null";                               //NOI18N
        this.support.removePropertyChangeListener(listener);
    }


    public void addClassPaths (ClassPath[] cps) {
        assert cps != null : "addClassPath called with null";                         //NOI18N
        for (int i = 0; i < cps.length; i++) {
            if (cps[i]!=null) {
                this.addClassPath(cps[i]);
            }
        }
        synchronized (this) {
            roots = null;
        }
        this.firePropertyChange(PROP_UNRESOLVED_ROOTS);
    }
    
    public void addRoot(URL url) {
        synchronized (this) {
            if (this.unresolvedRoots == null) {
                this.initEntries();
            }
            PathResourceImplementation root = ClassPathSupport.createResource(url);
            unresolvedRoots.add(root);
            roots = null;
        }
        this.firePropertyChange(PROP_UNRESOLVED_ROOTS);
    }

    public synchronized PathResourceImplementation[] getUnresolvedRoots () {
        if (this.unresolvedRoots == null) {
            this.initEntries();
        }
        return (PathResourceImplementation[]) this.unresolvedRoots.toArray (new PathResourceImplementation[this.unresolvedRoots.size()]);
    }

    public void classPathRootResolved (PathResourceImplementation impl) {
        synchronized (this) {
            if (this.unresolvedRoots.remove(impl)) {
                this.cachedResources.add(impl);
            }
            roots = null;
        }
        this.firePropertyChange (PROP_RESOURCES);
    }

    public synchronized void removeRoot(PathResourceImplementation impl) {
        this.unresolvedRoots.remove(impl);
        this.missingRoots.add(impl.getRoots()[0]);
        roots = null;
    }
    
    public boolean removeMissingRoot(URL url) {
        boolean result = false;
        synchronized (this) {
            while (missingRoots.remove(url)) {
                PathResourceImplementation resource = ClassPathSupport.createResource(url);
                assert resource != null : "ClassPathSupport.createResource() returned null";    //NOI18N
                unresolvedRoots.add(resource);
                result = true;
                roots = null;
            }
        }
        if (result) {
            firePropertyChange(PROP_UNRESOLVED_ROOTS);            
        }
        return result;
    }
    
    public boolean addMissingRoot(URL url) {
        boolean result = false;
        synchronized (this) {
            PathResourceImplementation resource = ClassPathSupport.createResource(url);
            while (this.cachedResources.remove(resource)) {
                missingRoots.add(url);
                result = true;
                roots = null;
            }
        }
        if (result) {
            this.firePropertyChange(PROP_RESOURCES);
        }
        return result;
    }
    
    public boolean updateRoot(URL url) {
        boolean result = false;
        synchronized (this) {
            PathResourceImplementation resource = ClassPathSupport.createResource(url);
            while (this.cachedResources.remove(resource)) {
                unresolvedRoots.add(resource);
                result = true;
                roots = null;
            }
        }
        if (result) {
            firePropertyChange(PROP_RESOURCES);
            firePropertyChange(PROP_UNRESOLVED_ROOTS);
        }
        return result;
    }

    public void addUnresolvedRoots (List/*<PathResourceImplementation>*/ unresolvedRoots) {
        synchronized (this) {
            this.unresolvedRoots.addAll (unresolvedRoots);
            roots = null;
        }
        this.firePropertyChange(PROP_UNRESOLVED_ROOTS);
    }

    /**
     * Slower version of classPathRootResolved, use classPathRootResolved(PathResourceImplementation)
     * where possible
     * @param url
     */
    public void classPathRootResolved (URL url) {
        synchronized (this) {
            for (Iterator it = unresolvedRoots.iterator(); it.hasNext();) {
                PathResourceImplementation resource = (PathResourceImplementation) it.next ();
                if (resource.getRoots()[0].equals(url)) {
                    it.remove();
                    this.cachedResources.add(resource);
                    break;
                }
            }
            roots = null;
        }
        this.firePropertyChange (PROP_RESOURCES);
    }

    private synchronized void initEntries () {
        Set classPaths = new HashSet ();
        roots = null;
        this.unresolvedRoots = new ArrayList ();
        this.resourceMap = new ClassPathMap();
        classPaths.addAll(this.reg.getPaths(ClassPath.SOURCE));
        classPaths.addAll(this.reg.getPaths(ClassPath.COMPILE));
        classPaths.addAll(this.reg.getPaths(ClassPath.BOOT));
        for(Iterator it = classPaths.iterator();it.hasNext();) {
            ClassPath cp = (ClassPath) it.next ();
            this.addClassPath (cp);
        }
    }

    private synchronized void updateEntries (GlobalPathRegistryEvent event) {
        if (this.cachedResources == null)
            return;
        for (Iterator it = event.getChangedPaths().iterator(); it.hasNext();) {
            ClassPath cp = (ClassPath) it.next();
            this.addClassPath (cp);
        }
    }

    private void updateEntries (ClassPath cp) {
        
        boolean fire = false;
        List oldResources;
        synchronized (this) {
            oldResources = (List) this.resourceMap.remove (cp);
            assert oldResources != null : "Change in unknown classpath"; // NOI18N
        }
        List newResources = addClassPathResources (cp);
        synchronized (this) {
            Collection toRemove = new HashSet (oldResources);
            toRemove.removeAll (newResources);
            newResources.removeAll (oldResources);  //To Add
            for (Iterator it = toRemove.iterator(); it.hasNext();) {
                PathResourceImplementation resource = (PathResourceImplementation) it.next();
                oldResources.remove(resource);
                if (!this.cachedResources.remove(resource)) {
                    if (!this.unresolvedRoots.remove (resource)) {
                        missingRoots.remove(resource.getRoots()[0]);
                    }
                }
                else {
                    fire = true;
                }
            }
            for (Iterator it = newResources.iterator(); it.hasNext();) {
                PathResourceImplementation resource = (PathResourceImplementation) it.next();
                oldResources.add (resource);
            }
            this.resourceMap.put (cp,oldResources);
            this.unresolvedRoots.addAll(newResources);
        }
        if (fire) {
            this.firePropertyChange(PROP_RESOURCES);
        }
        this.firePropertyChange(PROP_UNRESOLVED_ROOTS);
    }

    private void addClassPath (ClassPath cp) {
        synchronized (this) {
            if (this.resourceMap == null) {
                initEntries();
            }
            if (this.resourceMap.containsKey(cp)) {
                return;
            }
        }
        List c = addClassPathResources(cp);
        synchronized (this) {
            if (!this.resourceMap.containsKey(cp)) {
                this.resourceMap.put(cp,c);
                this.unresolvedRoots.addAll(c);
                cp.addPropertyChangeListener((PropertyChangeListener)WeakListeners.create(PropertyChangeListener.class,this.pcListener,cp));
            }
        }
    }
    
    private void firePropertyChange (String propName) {
        this.support.firePropertyChange(propName,null,null);
    }
    
    private class SFBQListener implements ChangeListener, PropertyChangeListener {
        private final WeakReference cp;
        private Map/*<URL,WeakReference<SourceForBinaryQuery.Result>>*/ results;
        
        public SFBQListener(ClassPath cp) {
            this.cp = new WeakReference(cp);
	    // The listener attached just to have strong reference
	    // from cp to SFBQListener, it is otherwise only weakly reachable
	    // and we need similar lifecycyle as the cp instance.
	    cp.addPropertyChangeListener(this);
        }
        
	public void propertyChange(PropertyChangeEvent event) {
            // ignore
        }

        public SourceForBinaryQuery.Result getResult(URL url) {
            if (results == null) {
                results = new HashMap();
            }
            SourceForBinaryQuery.Result result;
            Reference ref = (Reference) results.get(url);
            if (ref == null || (result = (SourceForBinaryQuery.Result) ref.get ()) == null) {
                result = SourceForBinaryQuery.findSourceRoots(url);
                results.put(url, new WeakReference(result));
                result.addChangeListener(WeakListeners.change(this, result));
            }
            return result;
        }

        public void stateChanged(ChangeEvent e) {
            //System.err.println("SourceForBinaryQuery.Result changed");
            ClassPath classPath = (ClassPath) cp.get();
            if (classPath != null) {
                updateEntries(classPath);
            }
        }
    }

    private List addClassPathResources (final ClassPath cp) {
        List entries = null;
        SFBQListener listener = null;
        synchronized (this) {
            WeakReference wr = (WeakReference) sfbResultListeners.get(cp);
            listener = (SFBQListener) (wr == null ? null : wr.get());
            if (listener == null) {
                listener = new SFBQListener(cp);
                sfbResultListeners.put(cp, new WeakReference(listener));
            }
        }
        entries = cp.entries();
        List c = new ArrayList();
        for (Iterator et = entries.iterator(); et.hasNext();) {
            ClassPath.Entry entry = (ClassPath.Entry)et.next();
            URL url = entry.getURL();
            assert url != null : "ClassPath.Entry.getURL() returned null";      //NOI18N
            addResources(url, listener, c);
        }
        return c;
    }

    private static void addResources (URL url, SFBQListener sfbResultListener, List resourceList) {
        PathResourceImplementation resource = ClassPathSupport.createResource(url);
        assert resource != null : "ClassPathSupport.createResource() returned null";    //NOI18N
        boolean needsBinary = true;
        SourceForBinaryQuery.Result result = sfbResultListener.getResult(url);
        FileObject[] sources = result.getRoots();
        List/*<FileObject>*/ sourcesL = Arrays.asList(sources);
        if (sourcesL.contains(null)) {
            // Diagnostic; cf. e.g. #58402.
            ErrorManager.getDefault().log(ErrorManager.WARNING, "Warning: " + result.getClass().getName() + " illegally returned a null element from getResult(URL): " + sourcesL);
            return;
        }
        List tempResult = new ArrayList(sources.length);
        for (int i=0; i< sources.length; i++) {
            try {
                URL surl = sources[i].getURL();
                if ("file".equals(surl.getProtocol())) needsBinary = false; // NOI18N
                PathResourceImplementation sresource = ClassPathSupport.createResource(surl);
                assert sresource != null : "ClassPathSupport.createResource() returned null";    //NOI18N
                tempResult.add(sresource);
            } catch (FileStateInvalidException e) {
                ErrorManager.getDefault().notify(e);
            }
        }
        if (needsBinary) resourceList.add(resource);
        resourceList.addAll(tempResult);
    }


    public synchronized static MergedClassPathImplementation getDefault () {
        if (instance == null ) {
            instance = new MergedClassPathImplementation();
        }
        return instance;
    }


    private class ClassPathMap {

        private List data;

        public ClassPathMap () {
            this.data = new ArrayList ();
        }

        public void put (Object key, Object value ) {
            synchronized (MergedClassPathImplementation.this) {
                WeakPair wp = new WeakPair (key, value);
                data.add (wp);
            }
        }

        public Object remove (Object key) {
            if (key == null) {
                return null;
            }
            synchronized (MergedClassPathImplementation.this) {
                for (Iterator it = this.data.iterator(); it.hasNext();) {
                    WeakPair pair = (WeakPair) it.next ();
                    Object wpk = pair.getKey();
                    if (key.equals(wpk)) {
                        it.remove();
                        return pair.getValue();
                    }
                }
            }
            return null;
        }

        public boolean containsKey (Object key) {
            if (key == null) {
                return false;
            }
            Iterator it;
            synchronized (MergedClassPathImplementation.this) {
                it = new ArrayList(data).iterator();
            }
            while (it.hasNext()) {
                WeakPair pair = (WeakPair) it.next();
                Object pk = pair.getKey();
                if (key.equals(pk)) {
                    return true;
                }
            }
            return false;
        }

        private void cleanUp (WeakPair toClean) {
            boolean fire = false;
            synchronized (MergedClassPathImplementation.this) {
                for (Iterator it = this.data.iterator(); it.hasNext();) {
                    WeakPair pair = (WeakPair) it.next ();
                    if (pair == toClean) {
                        it.remove();
                        for (Iterator  resIt= ((Collection)pair.getValue()).iterator(); resIt.hasNext();) {
                            PathResourceImplementation resource = (PathResourceImplementation) resIt.next();
                            if (!MergedClassPathImplementation.this.cachedResources.remove(resource)) {
                                if (!MergedClassPathImplementation.this.unresolvedRoots.remove(resource)) {
                                    missingRoots.remove(resource.getRoots()[0]);
                                }
                            } else {
                                fire = true;
                                roots = null;
                            }
                        }
                        break;
                    }
                }
            }
            if (fire) {
                firePropertyChange(PROP_RESOURCES);
            }
        }

        private class WeakPair extends WeakReference implements Runnable {

            private Object value;

            public WeakPair (Object key, Object value) {
                super (key, Utilities.activeReferenceQueue());
                this.value = value;
            }

            public Object getKey () {
                return get ();
            }

            public Object getValue () {
                return value;
            }

            public void run() {
                cleanUp(this);
            }
        }

    }
    
    public Set getRoots() {
        List _cachedResources;
        synchronized (this) {
            if (this.roots != null) {
                return this.roots;
            }
            _cachedResources = this.getResources(); //Save copy
        }                
        Set _roots = new HashSet();
        for (Iterator it = _cachedResources.iterator(); it.hasNext();) {
            PathResourceImplementation res = (PathResourceImplementation) it.next();
            URL[] urls = res.getRoots();
            for (int i = 0; i < urls.length; i++) {
                FileObject obj = URLMapper.findFileObject(urls[i]);
                if (obj != null) {
                    _roots.add(obj);
                }
            }
        }
        synchronized (this) {
            if (this.roots == null) {
                this.roots = Collections.unmodifiableSet(_roots);
            }
            return roots;
        }
        
    }
    
    public boolean contains(FileObject fo) {
        return findOwnerRoot(fo) != null;
    }
    
    public FileObject findOwnerRoot(FileObject fo) {
        Set roots = getRoots();
        for (FileObject f = fo; f != null; f = f.getParent()) {
            if (roots.contains(f)) {
                return f;
            }
        }
        return null;
    }
}
