/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.api;

import java.util.List;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.mdr.MDRManager;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.filesystems.FileObject;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.api.mdr.MDRepository;

/** Provides an entry point to the Java metadata.
 */
public final class JavaModel {
    private JavaModel() {
    }
    
    /** Sets active classpath for the current transaction. The real active classpath
     * is constructed from the passed classpath as follows:
     * <ul>
     *     <li>For every binary root its source roots are retrieved.</li>
     *     <li>If a given source root is immutable (a zip file) or does not exist,
     * the binary root is added to the active classpath, otherwise the source root is added.</li>
     *     <li>Some binary roots may point to the same source root (JDK jars -&gt; src.zip), which
     * could result in duplicate entries on the active classpath - these are filtered out.</li>
     * </ul>
     * 
     * @param cp Classpath to be set as active.
     */
    public static void setClassPath(ClassPath cp) {
        JavaMetamodel.getManager().setClassPath(cp);
    }
    
    /** Sets a list of classpaths as being an active classpath. The real active classpath
     * is constructed by merging all classpath roots of the classpath in the passed list
     * and applying the same rules as in {@link #setClassPath(ClassPath)}.
     *
     * @param cps List of classpaths.
     */
    public static void setClassPath(List/*<ClassPath>*/ cps) {
        JavaMetamodel.getManager().setClassPath(cps);
    }
    
    /** Sets the default classpath for a FileObject corresponding to the given
     * Resource as the active classpath. The real classpath is obtained from the default
     * classpath according to the same rules as described in {@link #setClassPath(ClassPath)}.
     *
     * @param rsc Resource from which to obtain the default classpath.
     */
    public static void setClassPath(Resource rsc) {
        JavaMetamodel.getManager().setClassPath(rsc);
    }
    
    /** Sets the default classpath for the passed FileObject as the active classpath.
     * The real classpath is obtained from the default classpath according to the same rules
     * as described in {@link #setClassPath(ClassPath)}.
     *
     * @param fo FileObject from which to obtain the default classpath.
     */
    public static void setClassPath(FileObject fo) {
        JavaMetamodel.getManager().setClassPath(fo);
    }
    
    /** Method that navigates from a FileObject to the corresponding Resource.
     *
     * @param fo FileObject for which the Resource should be found.
     * @return Resource corresponding to the passed FileObject.
     */
    public static Resource getResource(FileObject fo) {
        return JavaMetamodel.getManager().getResource(fo);
    }

    /** Method that finds a Resource corresponding to a file identified by
     * a classpath root and the file's relative name in this root.
     *
     * @param cpRoot FileObject representing a classpath root the desired file belongs to.
     * @param name Relative name of the file in the classpath root.
     * @return Resource corresponding to the file identified by the method parameters.
     */    
    public static Resource getResource(FileObject cpRoot, String name) {
        return JavaMetamodel.getManager().getResource(cpRoot, name);
    }
    
    /** Navigates from a Resource to the corresponding FileObject.
     *
     * @param r Resource for which the FileObject should be found.
     * @return FileObject corresponding to the specified resource.
     */
    public static FileObject getFileObject(Resource r) {
        return JavaMetamodel.getManager().getFileObject(r);
    }
    
    /** Tries to find an extent for a given classpath root. If the extent does not exist, returns null.
     *
     * @param cpRoot Classpath root.
     * @return Extent containing metadata for a given classpath root.
     */
    public static JavaModelPackage getJavaExtent(FileObject cpRoot) {
        return JavaMetamodel.getManager().getJavaExtent(cpRoot);
    }

    /** Returns the default extent. It can be used e.g. for accessing global operations
     * available from factory objects - e.g. resolving types by their names:<p/>
     * <pre>    JavaModel.getDefaultExtent().getType().resolve("java.lang.String");</pre>
     *
     * @return Default extent.
     */
    public static JavaModelPackage getDefaultExtent() {
        return JavaMetamodel.getManager().getDefaultExtent();
    }
    
    /** Returns metadata repository containing all Java metadata managed by JavaCore module.
     * It can be used for invoking repository-wide operations such as starting/ending transactions:<p/>
     * <pre>    JavaModel.getJavaRepository().beginTrans(false);
     *    try {
     *        // do something
     *    } finally {
     *        JavaModel.getJavaRepository().endTrans();
     *    }
     * </pre>
     *
     * @return Metadata repository.
     */
    public static MDRepository getJavaRepository() {
        return JavaMetamodel.getDefaultRepository();
    }
}
