/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

/*
 * RepositoryUpdater.java
 *
 * Created on 17. leden 2004, 14:17
 */

package org.netbeans.modules.javacore;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.ResourceClass;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.GeneralException;
import org.openide.ErrorManager;
import org.openide.filesystems.*;
import org.openide.filesystems.FileStateInvalidException;


/**
 *
 * @author  Jan Becicka
 */
public class RepositoryUpdater implements FileChangeListener {
    private static final boolean DEBUG = false;
    
    private static RepositoryUpdater updater = null;
    private final Set nonArchiveExts;
    private final JMManager manager = (JMManager) JavaMetamodel.getManager();
    private final Set fileObjectsToSave = Collections.synchronizedSet(new HashSet());

    
    /** Creates a new instance of RepositoryUpdater */
    private RepositoryUpdater() {
        Set exts = new HashSet(Arrays.asList(new String[] {
            "JAVA",
            "CLASS",
            "LOG",
            "TXT",
            "XML",
            "HTML",
            "FORM",
            "PROPERTIES",
        }));
        nonArchiveExts = Collections.unmodifiableSet(exts);
        //--- This will be replaced with master = MasterFileSystem.getDefault()
        Util.addFileSystemsListener(this);
    }
    
    public static RepositoryUpdater getDefault() {
        if (updater == null) {
            updater = new RepositoryUpdater();
        }
        return updater;
    }
    
    public void fileAttributeChanged(FileAttributeEvent fe) {
        //fileChanged(fe);
    }
    
    public void fileChanged(FileEvent fe) {
        try {
            if (changesDisabled)
                return ;
            if (DEBUG) System.out.println("file changed: " + fe.getFile().getPath()); // NOI18N
            if (!isJavaFile(fe)) {
                try {
                    URL rootURL = getRootURL(fe);
                    if (rootURL != null) {
                        manager.getMergedClassPathImpl().updateRoot(rootURL);
                    }
                } catch (Exception e) {
                    JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                }
                return;
            }
            updateResource(fe.getFile());
        } catch (Exception e) {
            ErrorManager.getDefault().notify(e);
        }
    }
    
    // this method never returns a URL which does not end by slash
    private URL getRootURL(FileEvent fe) throws FileStateInvalidException, MalformedURLException {
        return getRootURL(fe.getFile());
    }
    
    private URL getRootURL(FileObject fo) throws FileStateInvalidException, MalformedURLException {
        URL result;
        if (fo.isFolder()) {
            result = fo.getURL();
        } else if (mayBeArchiveFile(fo) && FileUtil.isArchiveFile(fo)) {
            result = FileUtil.getArchiveRoot(fo.getURL());
        } else {
            return null;
        }
        assert result.toExternalForm().endsWith("/") : // NOI18N
            "Bogus URL: " + result + " returned for FileObject: " + fo.getName() // NOI18N
            + " (isArchiveFile = " + FileUtil.isArchiveFile(fo) + ", isFolder = " + fo.isFolder() + ", isValid = " + fo.isValid() + ")"; // NOI18N
        return result;
    }

    private boolean changesDisabled = false;
    public void setListenOnChanges(boolean listen) {
        changesDisabled = !listen;
    }
    
    public void addFileObjectToSave(FileObject file) {
        fileObjectsToSave.add(file); 
    }
    
    static void updateTimeStamp(FileObject fo) {
        if (fo != null) {
            ResourceImpl r = (ResourceImpl) JavaMetamodel.getManager().getResource(fo);
            if (r != null) {
                Date lm = fo.lastModified();
                assert lm != null : "FileObject.lastModified() returned null for " + fo; // NOI18N
                r.setTimestamp(lm.getTime(), false);
            }
        }
    }
    
    private void updateResource(final FileObject fo) {
        boolean isSave = fileObjectsToSave.remove(fo);
        boolean synchronous = Thread.currentThread() == manager.getTransactionMutex().getThread();
        if (isSave) {
            if (synchronous) {
                updateTimeStamp(fo);
            } else {
                manager.getTransactionMutex().addUpdateTS(fo);
            }
        } else {            
            if (synchronous) {
                createOrUpdateResource(fo);
            } else {
                manager.getTransactionMutex().addModifiedRW(fo);
            }
        }
    }
    
    public void fileDataCreated(FileEvent fe) {
        FileObject fo = fe.getFile();
        try {
            if (DEBUG) System.out.println("file created: " + fo.getPath()); // NOI18N
            
            if (!Util.isJavaFile(fo)) {
                fileCreated(fo);
                return;
            }
            
            updateResource(fo);
        } catch (Exception e) {
            ErrorManager.getDefault().notify(e);
        }
    }

    private void fileCreated(FileObject fo) {
        try {
            URL rootURL = getRootURL(fo);
            if (rootURL != null) {
                manager.getMergedClassPathImpl().removeMissingRoot(rootURL);
            }
        } catch (Exception e) {
            JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
        }
    }
    
    void createOrUpdateResource(FileObject fo) {
        if (!fo.isValid())
            return ;
        FileObject cpRoot = manager.getMergedClassPathImpl().findOwnerRoot(fo);
        if (cpRoot == null) return;
        boolean failed = true;
        JavaModel.getJavaRepository().beginTrans(true);
        try {
            String name = manager.getResourceName(fo);
            if (name == null) {
                failed = false;
                return;
            }            
            JavaModelPackage modelPckg = manager.resolveJavaExtent(cpRoot);
            if (modelPckg==null) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, new GeneralException(fo.getPath() + " was not found in any extent. There is no resource to update.")); //NOI18N
                failed = false;
                return;
            }
            ResourceClass resClass=modelPckg.getResource();
            Resource r = resClass.resolveResource(name, false);
            if (r == null || !name.equals(r.getName())) {
                //resource does not exist - create it
                
                if (name.endsWith(".java")) { // NOI18N
                    String clsName = name.substring(0, name.length() - ".java".length()) + ".class"; // NOI18N
                    r = resClass.resolveResource(clsName, false);
                    if (r != null) {
                        r.refDelete();
                    }
                } else if (name.endsWith(".class")) { // NOI18N
                    String jName = name.substring(0, name.length() - ".class".length()) + ".java";   //NOI18N
                    if (resClass.resolveResource(jName, false) != null) {
                        failed = false;
                        return;
                    }
                }
                r = ((ResourceClassImpl) resClass).resolveResource(name, true, false);
            }
            ((ResourceImpl)r).updateFromFileObject(fo, true);
            failed = false;
        } finally {
            JavaModel.getJavaRepository().endTrans(failed);
        }
    }
    
    public void fileDeleted(FileEvent fe) {
        try {
            if (DEBUG) System.out.println("file deleted: " + fe.getFile().getPath()); // NOI18N
            if (!Util.isJavaFile(fe.getFile(), true)) {
                try {
                    URL rootURL = getRootURL(fe);
                    if (rootURL != null) {
                        manager.getMergedClassPathImpl().addMissingRoot(rootURL);
                    }
                } catch (Exception e) {
                    JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                }
                return;
            }
            manager.getTransactionMutex().addDeleted(fe.getFile());
        } catch (Exception e) {
            ErrorManager.getDefault().notify(e);
        }
        }

    private boolean isJavaFile(FileEvent fe) {
        return Util.isJavaFile(fe.getFile());
    }

    public void fileFolderCreated(FileEvent fe) {
        folderCreated(fe.getFile());
    }
    
    public void folderCreated(FileObject fo) {
        FileObject cpRoot = Util.getCPRoot(fo);
        if (cpRoot == null) {
            try {
                URL rootURL = getRootURL(fo);
                if (rootURL != null) {
                    if (manager.getMergedClassPathImpl().removeMissingRoot(rootURL)) {
                        return;
                    }
                }
            } catch (Exception e) {
                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
            }
            FileObject[] children = fo.getChildren();
            for (int i = 0; i < children.length; i++) {
                if (children[i].isFolder()) {
                    folderCreated(children[i]);
                } else {
                    fileCreated(children[i]);
                }
            }
            return;
        }
        Enumeration en = fo.getChildren(true);
        while (en.hasMoreElements()) {
            FileObject f = (FileObject) en.nextElement();
            if (Util.isJavaFile(f)) {
                updateResource(f);
            }
        }
    }
    
    public void fileRenamed(FileRenameEvent fe) {
        try {
            if (isJavaFile(fe)) {
                javaFileRenamed(fe);
            } else {
                try {
                    URL rootURL = getRootURL(fe);
                    if (rootURL != null) {
                        if (manager.getMergedClassPathImpl().removeMissingRoot(rootURL)) {
                            return;
                        }
                    }
                } catch (Exception e) {
                    JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                }
                if (fe.getFile().isFolder()) {
                    folderRenamed(fe);
                }
            }
        } catch (Exception e) {
            ErrorManager.getDefault().notify(e);
        }
    }
    
    private void javaFileRenamed(FileRenameEvent fe) {
        FileObject oldFo = fe.getFile();
        boolean failed = true;
        JavaModel.getJavaRepository().beginTrans(true);
        try {
            String oldName = fe.getName();
            FileObject cpRoot = Util.getCPRoot(oldFo);
            if (cpRoot == null) {
                failed = false;
                return; // ignore fileobjects that are not visible from our merged classpath
            }
            JavaModelPackage model = manager.getJavaExtent(cpRoot);
            if (model != null) {
                String path = manager.getResourceName(oldFo.getParent());
                String lookForName = oldName + '.' + fe.getExt(); // file name with extension
                if (path.length() > 0) {
                    path += '/';
                    // put the path to the string if the file is located in package
                    lookForName = path + lookForName;
                }
                Resource oldRes = model.getResource().resolveResource(lookForName, false);
                if (oldRes != null) {
                    oldRes.setName(path + oldFo.getNameExt());
                } else {
                    ((ResourceClassImpl) model.getResource()).resolveResource(path + oldFo.getNameExt(), true, false);
                }
            }

            failed = false;
        } finally {
            JavaModel.getJavaRepository().endTrans(failed);
        }
    }
    
    private void folderRenamed(FileRenameEvent fe) {
        FileObject oldFo = fe.getFile();
        boolean failed = true;
        
        JavaModel.getJavaRepository().beginTrans(true);
        try {
            Enumeration children = oldFo.getChildren(true);
            FileObject cpRoot = Util.getCPRoot(oldFo);
            if (cpRoot == null || cpRoot.equals(oldFo)) {
                // ignore rename of cp root
                // ignore folders that are not visible from our merged classpath
                failed = false;
                return;
            }
            String dirName = JMManager.getResourceName(cpRoot, oldFo);
            while (children.hasMoreElements()) {
                FileObject f = (FileObject) children.nextElement();
                if ("java".equals(f.getExt()) && !f.isVirtual()) { // NOI18N
                    String newName = JMManager.getResourceName(cpRoot, f);
                    String oldName = replaceStart(newName, dirName, fe.getName());
                    oldName = replaceEnd(dirName, fe.getName());
                    oldName = replaceStart(newName, dirName, oldName);
                    if (DEBUG)
                        System.out.println("Folder renamed: Resource " + oldName + " renamed to " + newName); // NOI18N
                    manager.getResource(cpRoot, oldName).setName(newName);
                }
            }
            failed = false;
        } finally {
            JavaModel.getJavaRepository().endTrans(failed);
        }
    }
    
    /**
     * example: where = "examples/colorpicker/ColorPreview.java"
     *          what  = "examples/colorpicker"
     *          withWhat = "examples/xxx
     * returns "examples/xxx/ColorPreview.java"
     */
    
    private static final String replaceStart(String where, String what, String withWhat) {
        return withWhat.concat(where.substring(what.length()));
    }
    
    /**
     * example: where = "examples/colorpicker"
     *          what  = "xxx"
     * returns "examples/xxx"
     */
    
    private static final String replaceEnd(String where, String what) {
        int i = where.lastIndexOf('/');
        if (i>0) { 
            return where.substring(0,i+1).concat(what);
        } else {
            return what;
        }
    }

    private boolean mayBeArchiveFile(FileObject fileObject) {
        return !nonArchiveExts.contains(fileObject.getExt().toUpperCase(Locale.US));
    }
}
