/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.WeakReference;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.jmi.reflect.InvalidObjectException;
import javax.jmi.reflect.RefBaseObject;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.queries.SourceForBinaryQuery;
import org.netbeans.api.java.queries.SourceLevelQuery;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.jmi.javamodel.Array;
import org.netbeans.jmi.javamodel.CallableFeature;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.Codebase;
import org.netbeans.jmi.javamodel.CodebaseClass;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.ElementPartKind;
import org.netbeans.jmi.javamodel.GenericElement;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.JavaPackage;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.jmi.javamodel.ParameterizedType;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.TypeParameter;
import org.netbeans.jmi.javamodel.TypedElement;
import org.netbeans.jmi.javamodel.UnresolvedClass;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.btreeimpl.btreestorage.BtreeFactory;
import org.netbeans.mdr.storagemodel.MdrStorage;
import org.netbeans.modules.javacore.classpath.FilterClassPathImplementation;
import org.netbeans.modules.javacore.classpath.MergedClassPathImplementation;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.GuardedQuery;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.ParsingListener;
import org.netbeans.modules.javacore.internalapi.ProgressSupport;
import org.netbeans.modules.javacore.jmiimpl.javamodel.JavaPackageClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MetadataElement;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ParameterizedTypeImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.netbeans.modules.javacore.parser.ArrayRef;
import org.netbeans.modules.javacore.parser.NameRef;
import org.netbeans.modules.javacore.parser.PrimitiveTypeRef;
import org.netbeans.modules.javacore.parser.TypeParamRef;
import org.netbeans.modules.javacore.parser.WildCardRef;
import org.netbeans.modules.javacore.scanning.FileScanner;
import org.netbeans.spi.java.classpath.ClassPathFactory;
import org.netbeans.spi.java.classpath.PathResourceImplementation;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.URLMapper;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.modules.InstalledFileLocator;
import org.openide.text.PositionBounds;
import org.openide.util.RequestProcessor;

/**
 *
 * @author Martin Matula
 */
public class JMManager extends JavaMetamodel implements PropertyChangeListener {
    private static final String NAME_MAIN_EXTENT = "JavaCoreMain"; // NOI18N
    private static final String NAME_MAIN_STORAGE = "Main"; // NOI18N

    private static final String PREFIX_EXTENT_CODEBASE = "codebase:"; // NOI18N
    private static final String PACKAGE_CODEBASE = "JavaModel"; // NOI18N
    private static final int THREAD_PRIORITY = Thread.MIN_PRIORITY;

    private static final RequestProcessor EVENT_RP = new RequestProcessor("Parsing Event Queue"); // NOI18N
    private static final RequestProcessor SCANNING_RP = new RequestProcessor("Scanning Queue"); // NOI18N
    private static final RequestPoster RESOLVE_POSTER = new RequestPoster(SCANNING_RP);
    private static final RequestPoster CLEANUP_POSTER = new RequestPoster(SCANNING_RP);

    // constants populated from system properties
    public static final boolean PERF_DEBUG = Boolean.getBoolean("perf.refactoring");
    public static final boolean PERF_TEST = Boolean.getBoolean("perf.refactoring.test");
    public static final boolean INCONSISTENCY_DEBUG = Boolean.getBoolean("javacore.debug.inconsistency");
    public static final boolean SCAN_DEBUG = Boolean.getBoolean("javacore.debug.scanning");
    public static final boolean TRANS_DEBUG = Boolean.getBoolean("javacore.debug.transactions");
    private static final boolean NO_SCAN = Boolean.getBoolean("netbeans.javacore.noscan");
    private static final int MDR_CACHE_SIZE = Integer.getInteger("org.netbeans.javacore.MDRCache.size", 256).intValue(); // NOI18N
    
    private static final boolean DEBUG = false;
    private static final String[] CP_TYPES = {ClassPath.BOOT, ClassPath.SOURCE, ClassPath.COMPILE};
    private static PropertyChangeListener indentListener;
    
    /**
     * Says a number of bytes read from input file. These bytes are used
     * for MD5 computation.
     */
    public static final int BYTES4MD5 = 8192;
    
    /**
     * Contains path to the structure with preparsed storages.
     */
    private static final String PREPARSED_LOCATION;
    private static final HashSet PREPARSED_CACHE;
    
    static {
        String s = System.getProperty("prebuilt.storage.dir"); // NOI18N
        if (s == null) {
            File f = InstalledFileLocator.getDefault().locate("./mdrstorage", "org.netbeans.modules.javacore", false); // NOI18N
            if (f != null) {
                PREPARSED_LOCATION = f.getAbsolutePath() + "/"; // NOI18N
            } else {
                PREPARSED_LOCATION = "./"; // NOI18N
            }
        } else {
            if (s.endsWith("/")) { // NOI18N
                PREPARSED_LOCATION = s;
            } else {
                PREPARSED_LOCATION = s + '/';
            }
        }
        PREPARSED_CACHE = new HashSet();
        File f = new File(PREPARSED_LOCATION);
        File[] files = f.listFiles();
        if (files != null) {
            for (int i = 0; i < files.length; i++) {
                String name = files[i].getName();
                if (name.endsWith(".btd")) {
                    int pos = name.lastIndexOf("-");
                    if (pos >= 0) {
                        PREPARSED_CACHE.add(name.substring(0, pos));
                    }
                }
            }
        }
    }

    private static final Map MDR_CACHE = new CacheClass();
    
    private static class CacheClass implements Map {
        private final Object inner[] = new Object[MDR_CACHE_SIZE];
        private int size, cursor;
        
        public CacheClass () {}
        
        public Set keySet() {
            throw new UnsupportedOperationException();
        }

        public Set entrySet() {
            throw new UnsupportedOperationException();
        }

        public void putAll(Map t) {
            throw new UnsupportedOperationException();
        }

        public boolean isEmpty() {
            return size == 0;
        }

        public boolean containsKey(Object key) {
            throw new UnsupportedOperationException();
        }

        public boolean containsValue(Object value) {
            throw new UnsupportedOperationException();
        }

        public Collection values() {
            throw new UnsupportedOperationException();
        }

        public Object put(Object key, Object value) {
            if (inner[cursor] != value) {
                cursor++;
                if (size < MDR_CACHE_SIZE) {
                    size++;
                }
                if (cursor >= MDR_CACHE_SIZE) {
                    cursor = 0;
                }
                inner[cursor] = value;
            }
            return null;
        }

        public void clear() {
            Arrays.fill(inner, null);
            size = cursor = 0;
        }

        public int size() {
            return size;
        }

        public Object get(Object key) {
            throw new UnsupportedOperationException();
        }

        public Object remove(Object key) {
            throw new UnsupportedOperationException();
        }
    }


    private JavaModelPackage mainExtent;
    // uri->storageId map for mounted partitions
    // uri => folderURL[sourceLevel]?
    // folderURL url ending with /
    // sourceLevel => digit.digit
    private final HashMap sids = new HashMap();
    // JavaModelPackage -> FileObject (CPRoot)
    private final HashMap filesystems = new HashMap();
    private final HashMap extents = new HashMap();

    private final MergedClassPathImplementation cpImpl;
    private final ClassPath classPath;

    private static ProgressSupport progressSupport = null;
    
    private Cache cpForFileObjectCache = new Cache(5);
    private HashMap cpForFileObjectMap = new HashMap(10);
    
    private static ErrorManager javamodelErr = ErrorManager.getDefault().getInstance("org.netbeans.javacore"); // NOI18N
    
    private Set extentsToScan = new HashSet();
    
    private boolean cancelScan = false;

    /**
     * Used from computation of src.zip and rt.jar digest. This is used
     * for setting filenames of pre-parsed storages.
     */
    private static final MessageDigest md5digest;
    private static final MessageDigest digest;
    
    private static ThreadLocal documentLocks;
    private static IndentationSettingsProvider settingsProvider;
    
    private static ModifiedDOProvider modifiedDOProvider;
    
    public static synchronized void setDocumentLocksCounter(ThreadLocal documentLocks) {
        JMManager.documentLocks = documentLocks;
    }
    
    static synchronized ThreadLocal getDocumentLocksCounter() {
        return documentLocks;
    }

    static {          
        try {              
            digest = MessageDigest.getInstance("SHA"); // NOI18N
            md5digest = MessageDigest.getInstance("MD5"); // NOI18N
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Fatal error: " + e); // NOI18N
        }
    }
    
    public JMManager(MDRepository repository) {
        super(repository, Codebase.class);
        cpImpl = MergedClassPathImplementation.getDefault();
        cpImpl.addPropertyChangeListener(this);
        classPath = ClassPathFactory.createClassPath (cpImpl);
        classPath.addPropertyChangeListener (this);
    }

    private void init() {
        if (mainExtent == null) {
            repository.beginTrans(true);
            boolean fail = true;
            try {
                int retry = 0;
                String mainExtentSid = null;
                String storageFileName = getFileName(NAME_MAIN_STORAGE);
                while (mainExtent == null && retry < 2) {
                    try {
                        if (DEBUG) System.err.println("Initializing repository."); // NOI18N
                        if (retry == 0) {
                            mainExtentSid = mountStorage(NAME_MAIN_STORAGE, storageFileName); // NOI18N
                        } else {
                            mainExtentSid = rebootStorage(NAME_MAIN_STORAGE, storageFileName, mainExtentSid); // NOI18N
                        }
                        JavaModelPackage main = (JavaModelPackage) repository.getExtent(NAME_MAIN_EXTENT);
                        if (main == null) {
                            NBMDRepositoryImpl nbRep = (NBMDRepositoryImpl) repository;
                            main = (JavaModelPackage) nbRep.createExtent(NAME_MAIN_EXTENT, findRootPackage(PACKAGE_CODEBASE), null, mainExtentSid);
                        }
                        mainExtent = main;
                    } catch (Exception e) {
                        retry++;
                        if (retry >= 2) {
                            JMManager.getLog().notify(e);
                            // TODO: fatal error - stop NetBeans
                        } else {
                            JMManager.getLog().log(ErrorManager.WARNING, "Rebooting Main storage because of a fatal error during storage mount."); // NOI18N
                            JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                        }
                    }
                }
                fail = false;
            } finally {
                repository.endTrans(fail);
            }
            RepositoryUpdater.getDefault();
        }
    }
    
    /**
     * Returns the instance of Error Manager used for logging model problems
     * and debug messages.
     *
     * @return  instance of error manager
     */
    public static ErrorManager getLog() {
        return javamodelErr;
    }
    
    private String getDirName() {
        NBMDRepositoryImpl nbRep = (NBMDRepositoryImpl) repository;
        String fileName = nbRep.getParameter(BtreeFactory.STORAGE_FILE_NAME);
        int li = Math.max(fileName.lastIndexOf('/'), fileName.lastIndexOf('\\'));
        String folderName = getVersion();
        if (li > 0) {
            folderName = fileName.substring(0, li + 1) + folderName;
        }
        return folderName;
    }
    
    public String getFileName(String name) {
        String folderName = getDirName();
        new File(folderName).mkdirs();
        return folderName.concat("/").concat(name); // NOI18N
    }
    
    private String mountStorage(String name, String fileName) {
        HashMap properties = new HashMap();
        NBMDRepositoryImpl nbRep = (NBMDRepositoryImpl) repository;
        properties.put(BtreeFactory.STORAGE_FILE_NAME, fileName);
        properties.put(BtreeFactory.CACHE_INSTANCE, MDR_CACHE); // NOI18N
        String sid;
        try {
            if (DEBUG) System.out.println("mounting: " + fileName); // NOI18N
            sid = nbRep.mountStorage(nbRep.getParameter("storage"), properties); // NOI18N
            MdrStorage.ValuesObject values = nbRep.getMdrStorage().storageValues(sid);
            values.store(NameRef.class.getName());
            values.store(TypeParamRef.class.getName());
            values.store(PrimitiveTypeRef.class.getName());
            values.store(WildCardRef.class.getName());
            values.store(ArrayRef.class.getName());
            
            return sid;
        } catch (Exception e) {
            throw (RuntimeException) ErrorManager.getDefault().annotate(new RuntimeException(), e);
        }
    }
    
    private String rebootStorage(String name, String fileName, String sid) {
        NBMDRepositoryImpl nbRep = (NBMDRepositoryImpl) repository;
        nbRep.disableEvents();
        nbRep.unmountStorage(sid);
        nbRep.enableEvents();
        new File(fileName + ".btx").delete(); // NOI18N
        new File(fileName + ".btd").delete(); // NOI18N
        new File(fileName + ".btb").delete(); // NOI18N
        return mountStorage(name, fileName);
    }

    public ClassPath getClassPath() {
        ClassPath result = ((ExclusiveMutex) ((NBMDRepositoryImpl) repository).getTransactionMutex()).getClassPath();

        // [TODO] remove the following line once all the usages of this method are fixed
        if (result == null) {
            result = FilterClassPathImplementation.createClassPath(Collections.singletonList(getMergedClassPath()), false);
        }
        
        return result;
    }

    public ClassPath getMergedClassPath() {
        return classPath;
    }
    
    MergedClassPathImplementation getMergedClassPathImpl() {
        return cpImpl;
    }
    
    public boolean mergedCPContains(FileObject fo) {
        return cpImpl.contains(fo);
    }
    
    public boolean mergedCPContainsRoot(FileObject fo) {
        return cpImpl.getRoots().contains(fo);
    }

    /** Sets classpath for the duration of the current transaction
     *
     * @param cp Classpath to be used during the current transaction.
     */
    public void setClassPath(List/*<ClassPath>*/ cps) {
        setClassPath(cps, false);
    }
    
    public void setClassPath(List/*<ClassPath>*/ cps, boolean preferSource) {
        if (cps == null || cps.isEmpty()) return;
        ((ExclusiveMutex) ((NBMDRepositoryImpl) repository).getTransactionMutex()).setClassPath(cps, preferSource);
    }
    
    public void setClassPath(ClassPath cp) {
        setClassPath(Collections.singletonList(cp));
    }
    
    public void setClassPath(Resource rsc) {
        setClassPath(getFileObject(rsc));
    }
    
    public void setClassPath(Resource rsc, boolean preferSource) {
        setClassPath(getFileObject(rsc), preferSource);
    }
    
    private class CPList extends AbstractList implements Cache.CachedElement {
        private final FileObject fo;
        private final WeakReference inner[] = new WeakReference[CP_TYPES.length];
        
        public CPList(FileObject fo) {
            this.fo = fo;
            for (int i = 0; i < CP_TYPES.length; i++) {
                ClassPath cp = ClassPath.getClassPath(fo, CP_TYPES[i]);
                inner[i] = cp == null ? null : new WeakReference(cp);
            }
            cpForFileObjectMap.put(fo, this);
        }
        
        public Object get(int index) {
            return inner[index] == null ? null : ((WeakReference) inner[index]).get();
        }
        
        public int size() {
            return 3;
        }
        
        public boolean isValid() {
            return true;
        }
        
        public void release() {
            cpForFileObjectMap.remove(fo);
        }
    }

    public void setClassPath(FileObject fo) {
        setClassPath(fo, false);
    }

    // does not need to be thread safe/synchronized, since it should always be invoked from a transaction
    public void setClassPath(FileObject fo, boolean preferSource) {
        if (fo == null) return;
        List cps = (List) cpForFileObjectMap.get(fo);
        if (cps == null) {
            cps = new CPList(fo);
        }
        cpForFileObjectCache.put(cps);
        setClassPath(cps, preferSource);
    }

    public JavaModelPackage getDefaultExtent() {
        init();
        return mainExtent;
    }
    
    public void setSafeTrans(boolean isSafeTrans) {
        getTransactionMutex().setSafeTrans(isSafeTrans);
    }

    public static ExclusiveMutex getTransactionMutex() {
        return (ExclusiveMutex) ((NBMDRepositoryImpl) getDefaultRepository()).getTransactionMutex();
    }

    public JavaModelPackage resolveJavaExtent(FileObject cpRoot) {
        repository.beginTrans(false);
        try {
            synchronized (this) {
                JavaModelPackage extent = getJavaExtent(cpRoot);
                if (extent == null) {
                    ClassPath cp = ClassPath.getClassPath(cpRoot, ClassPath.SOURCE);
                    if (cp!=null) {
                        resolveCPRoot(cpRoot, false);
                        extent = (JavaModelPackage) extents.get(cpRoot);
                        if (FileOwnerQuery.getOwner(cpRoot) != null) {
                            try {
                                URL cpRootUrl = cpRoot.getURL();
                                cpImpl.addClassPaths(new ClassPath[] {cp, ClassPath.getClassPath(cpRoot, ClassPath.COMPILE), ClassPath.getClassPath(cpRoot, ClassPath.BOOT)});
                            } catch (FileStateInvalidException e) {
                                // ignore
                            }
                        }
                    }
                }
                return extent;
            }
        } finally {
            repository.endTrans();
        }
    }

    public synchronized JavaModelPackage getJavaExtent(FileObject cpRoot) {
        if (cpImpl.getRoots().contains(cpRoot)) {
            return (JavaModelPackage) extents.get(cpRoot);
        } else {
            return null;
        }
    }
    
    public Resource getResource(FileObject fo) {
        return getResource(fo, true);
    }

    public Resource getResource(FileObject fo, boolean force) {
        if (fo.isVirtual() || !fo.isValid()) {
            //simply ignore virtual and invalid files
            return null;
        }
        repository.beginTrans(false);
        try {
            FileObject cpRoot = this.cpImpl.findOwnerRoot(fo);
            if (cpRoot==null && force) {
                String ext=fo.getExt();
                ClassPath cp=null;
                if ("java".equals(ext))         // NOI18N
                    cp = ClassPath.getClassPath(fo, ClassPath.SOURCE);
                else if ("class".equals(ext))   // NOI18N
                    cp = ClassPath.getClassPath(fo, ClassPath.EXECUTE);
               if (cp != null) {
                    cpRoot = cp.findOwnerRoot(fo);
                }
            }
            Resource result = null;
            if (cpRoot != null) {
                String resourceName = getResourceName (cpRoot, fo);
                result = getResource(cpRoot, resourceName);
            }
            return result;
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ex);
        } finally {
            repository.endTrans();
        }
        return null;
    }

    public JavaPackage getPackage(FileObject fo) {
        repository.beginTrans(false);
        try {
            FileObject cpRoot = this.cpImpl.findOwnerRoot (fo);
            if (cpRoot == null) return null;

            JavaModelPackage extent = getJavaExtent(cpRoot);
            if (extent == null) return null;
            
            String packageName = getResourceName (cpRoot, fo);
            packageName = packageName.replace('/', '.');
            
            return ((JavaPackageClassImpl) extent.getJavaPackage()).findRealPackage(packageName);
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ex);
        } finally {
            repository.endTrans();
        }
        return null;
    }

    public void registerExtChange(ExternalChange ch) {
        getTransactionMutex().registerExtChange(ch);
    }
    
    public void registerUndoElement(ExternalChange ch) {
        getTransactionMutex().registerUndoElement(ch);
    }
    
    public Resource getResource(FileObject cpRoot, String name) {
        repository.beginTrans(false);
        try {
            if (cpRoot != null && cpRoot.isValid()) {
                JavaModelPackage pck=resolveJavaExtent(cpRoot);
                if (pck == null) {
                    return null;
                }
                ResourceClassImpl resClass=(ResourceClassImpl)pck.getResource();
                ResourceImpl result = (ResourceImpl) resClass.resolveResource(name, false);
                
                // ignore classfiles if there are corresponding java files in the same directory
                // i.e. pass resource corresponding to the java file
                if (result == null && name.endsWith(".class")) { // NOI18N
                    String srcName = name.substring(0, name.length() - ".class".length()) + ".java"; // NOI18N
                    if (cpRoot.getFileObject(srcName) != null) {
                        name = srcName;
                        result = (ResourceImpl) resClass.resolveResource(name, false);
                    }
                }
                // -----------------------------------
                
                if (result == null) {
                    result = (ResourceImpl) resClass.resolveResource(name, true, false);
                    if (result != null) {
                        if (!result.isValid()) return null;
                        FileObject fobj = getFileObject(result);
                        if (fobj != null) {
                            result.updateFromFileObject(fobj, false);
                        } else {
                            result.refDelete();
                            return null;
                        }
                    }
                }
                return result;
            } else {
                return null;
            }
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ex);
        } finally {
            repository.endTrans();
        }

        return null;
    }

    public FileObject getFileObject(Resource res) {
        if (res == null) return null;
        repository.beginTrans(false);
        try {
            if (!res.isValid()) {
                return null;
            }
            FileObject cpRoot = getCPRoot((JavaModelPackage) res.refImmediatePackage());
            if (cpRoot != null) {
                FileObject result = cpRoot.getFileObject(res.getName());
                return result;
            }
            return null;
        } catch (InvalidObjectException e) {
            return null;
        } finally {
            repository.endTrans();
        }
    }

    public synchronized FileObject getCPRoot(JavaModelPackage javaModelPackage) {
        return (FileObject) filesystems.get(javaModelPackage);
    }

    public DataObject getDataObject(Resource res) {
        FileObject fo = getFileObject(res);
        DataObject result = null;
        if (fo != null) {
            if (fo.isVirtual()) {
                return null;
            }
            try {
                result = DataObject.find(fo);
            } catch (DataObjectNotFoundException e) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
                result = null;
            }
        }
        if (result != null && !result.isValid()) result = null;
        return result;
    }

    public boolean waitScanFinished() {
        if (NO_SCAN) return false;
        boolean wait=false;
        while (cpImpl.getUnresolvedRoots().length > 0) {
            wait=true;
            try {
                boolean isStartupInProgress = JavaCoreModule.isStartupInProgress();
                ProgressDisplayer disp = ProgressDisplayer.getVisibleProgressDisplayer();
                if (SCAN_DEBUG) {
                    System.err.println("JMManager.waitScanFinished(), isStartupInProgress = " + isStartupInProgress);
                    System.err.println("JMManager.waitScanFinished(), getVisibleProgressDisplayer() = " + disp);
                }
                if (!isStartupInProgress && (disp == null)) {
                    SCANNING_RP.post(new Runnable() {
                        public void run() {
                            refreshCodebases();
                        }
                    }, 0, THREAD_PRIORITY);
                }
                Thread.sleep(100);
            } catch (InterruptedException e) {
            }
        }
        return wait;
    }
    
    public boolean isScanInProgress() {
        return ProgressDisplayer.getVisibleProgressDisplayer()!=null;
    }
    
    // ..........................................................................
    
    public boolean isElementGuarded(Element element) {
        PositionBounds pos = getElementPosition(element);
        if (pos == null) {
            return false;
        }
        RefBaseObject refObj = element;
        while ((refObj instanceof Element) && !(refObj instanceof Resource)) {
            refObj = ((Element)refObj).refImmediateComposite();
        }
        if (!(refObj instanceof Resource)) {
            return false;
        }
        ResourceImpl resource = (ResourceImpl)refObj;
        return GuardedQuery.isSectionGuarded(resource, pos);
    }
    
    // ..........................................................................
    
    public PositionBounds getElementPosition(Element element) {
        return getElementPosition(element, false);
    }
    
    /**
     * @param inclDoc    if false, then start offset of first token of startTree
     *                   is beginning, otherwise it starts from its javadoc
     *                   comment.
     */
    public PositionBounds getElementPosition(Element element, boolean inclDoc) {
        if (element instanceof ParameterizedType)
            element=((ParameterizedType)element).getDefinition();
        else if (element instanceof ParameterizedTypeImpl.Wrapper)
            element=(Element)((ParameterizedTypeImpl.Wrapper)element).getWrappedObject();
        if (element instanceof MetadataElement) {
            return ((MetadataElement) element).getPosition(inclDoc);
        } else {
            return null;
        }
    }

    public PositionBounds getElementPartPosition(Element element, ElementPartKind part, int position) {
        if (element instanceof ParameterizedType)
            element=((ParameterizedType)element).getDefinition();
        else if (element instanceof ParameterizedTypeImpl.Wrapper)
            element=(Element)((ParameterizedTypeImpl.Wrapper)element).getWrappedObject();
        if (element instanceof MetadataElement) {
            return ((MetadataElement) element).getPartPosition(part, position);
        } else {
            return null;
        }
    }

    public Element getElementByOffset(FileObject file,int offset) {
        repository.beginTrans(false);
        try {
            Resource cr = getResource(file);
            if (cr == null) {
                return null;
            }
            return cr.getElementByOffset(offset);
        } finally {
            repository.endTrans();
        }
    }

    // ---- helper methods --------------------------------------------------------------------
    // THESE METHODS SHOULD NOT BE CALLED FROM OTHER MODULES

    private static String getCBExtentName(String uri) {
        return PREFIX_EXTENT_CODEBASE + uri;
    }
    
    void startResolution() {
        if (SCAN_DEBUG) System.err.println("JMManager: startResolution1");
        RESOLVE_POSTER.post(new Runnable () {
            public void run() {
                if (SCAN_DEBUG) System.err.println("JMManager: startResolution2 - runnable");
                try {
                    resolveCodebases();
                } catch (Throwable e) {
                    ErrorManager.getDefault().notify(e);
                }
                if (SCAN_DEBUG) System.err.println("JMManager: startResolution3 - runnable");
            }
        });
        if (SCAN_DEBUG) System.err.println("JMManager: startResolution4");
    }
    
    /** Resolves all extents referenced from project's classpath.
     * This method has to be called within a writable transaction.
     */
    protected void resolveCodebases() {
        boolean fail = true;
        //fix of #49020
        try {
            repository.beginTrans(true);
            repository.endTrans();
        } catch (Throwable t) {
            ErrorManager.getDefault().notify(t);
        }
        //end of fix
        repository.beginTrans(true);
        try {
            ArrayList resources = new ArrayList(Arrays.asList((Object[]) cpImpl.getUnresolvedRoots()));
//            resources.addAll(cpImpl.getResources());  Ufff, why to resolve something what is already resolved?
            this.getProgressSupport().fireProgressListenerStart(0,resources.size());
            for (Iterator it = resources.iterator(); it.hasNext() && !cancelScan;) {
                try {
                    this.getProgressSupport().fireProgressListenerStep();
                } catch (Throwable t) {
                    JMManager.getLog().notify(t);
                }
                PathResourceImplementation resource = (PathResourceImplementation) it.next();
                URL url = resource.getRoots()[0];
                FileObject root = URLMapper.findFileObject(url);
                if (root != null) {
                    URL rootURL;
                    try {
                        rootURL = root.getURL();
                    } catch (FileStateInvalidException e) {
                        JMManager.getLog().notify(e);
                        rootURL = null;
                    }
                    if (!url.equals(rootURL)) {
                        JMManager.getLog().notify(new RuntimeException("ClassPath returned the invalid (non-normalized) URL from getRoots(): " + url + ". This classpath element will not be scanned.")); // NOI18N
                        cpImpl.removeRoot(resource);
                        continue;
                    }
                    resolveCPRoot(root, true);
                } else {
                    //Todo: Root does not exist physically, needs more handling
                    //Todo: some listenig is needed otherwise will cause invalid nonparseble objects, FileScanner(URL) ???
                    cpImpl.removeRoot(resource);
                }
            }
            fail = false;
        } catch (RuntimeException e) {
            ErrorManager.getDefault().notify(e);
        } catch (AssertionError e) {
            ErrorManager.getDefault().notify(e);
        } finally {
            repository.endTrans(fail);
            this.getProgressSupport().fireProgressListenerStop();
        }
    }
    
    private void cleanUpCodebases () {
        List/*<PathResourceImplementation>*/ rootsToReresolve = new ArrayList ();
        getDefaultRepository().beginTrans(true);
        boolean fail = true;
        try {
            synchronized (this) {
                Set backup = new HashSet(sids.keySet());
                Set existingUnresolved = new HashSet();
                FileObject[] roots = this.classPath.getRoots();
                for (int i = 0; i < roots.length; i++) {
                    try {
                        String uriBase = roots[i].getURL().toString();
                        if (!removeURI(backup, uriBase)) {
                            existingUnresolved.add(uriBase);
                        }
                    } catch (FileStateInvalidException e) {
                        JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                    }
                }
                PathResourceImplementation unresolved[] = this.cpImpl.getUnresolvedRoots();
                for (int i = 0; i < unresolved.length; i++) {
                    PathResourceImplementation resource = unresolved[i];
                    URL url = resource.getRoots()[0];
                    FileObject root = URLMapper.findFileObject(url);
                    if (root != null) {
                        try {
                            String uriBase = root.getURL().toString();
                            if (!removeURI(backup, uriBase)) {
                                existingUnresolved.add(url.toString());
                            }
                        } catch (FileStateInvalidException e) {
                            JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                        }
                    }
                }

                for (Iterator it = backup.iterator(); it.hasNext();) {
                    String uri = (String) it.next ();
                    disposeCodebase(uri);
                    //The change of platform (boot classpath) may
                    //affect the source and build folders of project
                    int index = uri.lastIndexOf('/');   //NOI18N
                    if (index < uri.length()-1) {
                        String url = uri.substring(0,index+1);
                        if (existingUnresolved.remove(url)) {
                            try {
                                rootsToReresolve.add (ClassPathSupport.createResource(new URL (url)));
                            } catch (MalformedURLException mfe) {
                                ErrorManager.getDefault().notify(mfe);
                            }
                        }
                    }
                }
            }
            fail = false;
        } finally {
            getDefaultRepository().endTrans(fail);
        }
        if (rootsToReresolve.size() > 0) {
            cpImpl.addUnresolvedRoots (rootsToReresolve);
        }
    }
    private boolean removeURI(Set/*<String>*/ uris, String uriBase) {
        // #72006: cleanUpCodebases should not call getRootURI since it calls SourceLevelQuery
        // and that can trigger reloading the project we are supposed to be collecting!
        boolean removed = false; // try to remove >1 if possible, since we don't know which is right
        if (uris.remove(uriBase)) {
            removed = true;
        }
        if (uris.remove(uriBase + "1.4")) { // NOI18N
            removed = true;
        }
        if (uris.remove(uriBase + "1.5")) { // NOI18N
            removed = true;
        }
        if (removed) {
            return true;
        }
        // Didn't find it that way; maybe 1.3 or 1.6 or something else?
        String prefix = uriBase + "1."; // NOI18N
        Iterator it = uris.iterator();
        while (it.hasNext()) {
            String uri = (String) it.next();
            if (uri.startsWith(prefix)) {
                // Guess that it is a source level.
                it.remove();
                return true;
            }
        }
        return false;
    }

    private void resolveCPRoot(FileObject root, boolean scanIfNeeded) {
        if (root == null) return;
        boolean fail = true;
        if (!scanIfNeeded) getDefaultRepository().beginTrans(true);
        try {
            JavaModelPackage extent;
            int state = 0;
            synchronized (this) {
                String uri = getRootURI(root);
                String extentName = getCBExtentName(uri);
                extent = (JavaModelPackage) repository.getExtent(extentName);
                if (extent == null) {
                    state++;
                    init();
                    int retry = 0;
                    String sid = null;
                    String storageName = getValidName(uri);
                    String storageFileName = getFileName(storageName);
                    String simpleName = getCodebaseSimpleName(root);
                    while (extent == null && retry < 2) {
                        try {
                            boolean preparsedUsed = false;
                            if (retry == 0) {
                                if (simpleName != null) {
                                    preparsedUsed = usePreparsed(simpleName, root, getFileName(storageName));
                                    if (preparsedUsed) {
                                        System.err.println("Using '" + uri + "' pre-parsed database!");
                                    }
                                }
                                sid = mountStorage(storageName, storageFileName);
                            } else {
                                sid = rebootStorage(storageName, storageFileName, sid);
                            }
                            sids.put(uri, sid);
                            if (!preparsedUsed) {
                                extent = (JavaModelPackage) repository.getExtent(extentName);
                            }
                            if (extent == null && simpleName != null) {
                                extent = (JavaModelPackage) repository.getExtent(getCBExtentName(simpleName)); // NOI18N
                                if (extent != null) {
                                    preparsedUsed = true;
                                }
                            }
                            NBMDRepositoryImpl nbRep = (NBMDRepositoryImpl) repository;
                            if (extent == null) {
                                state += 2;
                                extent = (JavaModelPackage) nbRep.createExtent(extentName, findRootPackage(PACKAGE_CODEBASE), null, sid);
                                ClassIndex.addIndex(extent, storageFileName);
                            } else {
                                // check consistency of the extent
                                Collection/*<Codebase>*/ c = extent.getCodebase().refAllOfClass();
                                
                                if (preparsedUsed) {
                                    // we used preparsed database, copied from
                                    // install location to user dir. We have to
                                    // set codebase name correctly as different
                                    // users have their jdk installed in different
                                    // locations.
                                    if (c.size() != 1) {
                                        // storage is corrupted -> throw exception to make sure the storage is recreated
                                        throw new RuntimeException("Not exactly one codebase in extent!"); // NOI18N
                                    }
                                    Codebase cb = (Codebase) c.iterator().next();
                                    cb.setName(uri);
                                    cb.setTimestamp(FileUtil.getArchiveFile(root).lastModified().getTime());
                                    nbRep.renameExtent(extent, extentName);
                                } else {
                                    // the following call may through an exception in case the storage is inconsistent
                                    // in that case the storage will be automatically rebooted (#67158)
                                    c.isEmpty();
                                }
                                // walk through all the packages in the extent and add them to the proxy packages
                                ((JavaPackageClassImpl) extent.getJavaPackage()).addAllRealPackages();
                                if (!ClassIndex.loadIndex(storageFileName, extent)) {
                                    state++;
                                }
                                if (preparsedUsed) {
                                    ClassIndex.getIndex(extent).setTimestamp();
                                }
                            }
                        } catch (Exception e) {
                            sids.remove(uri);
                            retry++;
                            if (retry >= 2) {
                                JMManager.getLog().notify(e);
                            } else {
                                JMManager.getLog().log(ErrorManager.WARNING, "Rebooting storage for " + root.getURL() + " because of a fatal error during storage mount."); // NOI18N
                                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
                            }
                        }
                    }
                    extents.put(root, extent);
                    filesystems.put(extent, root);
                } else {
                    CodebaseClass cbClass = extent.getCodebase();
                    if (cbClass.refAllOfClass().isEmpty()) {
                        state++;
                    } else {
                        Codebase cb = (Codebase) cbClass.refAllOfClass().iterator().next();
                        long timeStamp = cb.getTimestamp();
                        long fileTimeStamp;
                        FileObject f = FileUtil.getArchiveFile(root);
                        if (f != null) {
                            fileTimeStamp = f.lastModified().getTime();
                        } else {
                            fileTimeStamp = root.lastModified().getTime();
                        }
                        if (timeStamp != 0 && fileTimeStamp != timeStamp) {
                            state++;
                        }
                    }
                }
            }
            if (scanIfNeeded) {
                Resource[] resourcesToParse;
                if (state != 0 || extentsToScan.remove(extent)) { // new or mounted codebase
                    resourcesToParse = scanFiles(root, extent, state > 1);
    //                ClassIndex.saveIndex(extent);
                } else {
                    resourcesToParse = new Resource[0];
                }
                // resolve the root only if the classpath scan did not fail (i.e. returned a non-null result)
                if (resourcesToParse != null) {
                    this.cpImpl.classPathRootResolved(root.getURL());
                    if (resourcesToParse.length > 0) {
                        for (int i = 0; i < resourcesToParse.length; i++) {
                            setClassPath(resourcesToParse[i]);
                            traverseResource(resourcesToParse[i]);
                            setClassPath(resourcesToParse[i], true);
                            traverseResource(resourcesToParse[i]);
                        }
                    }
                }
            } else {
                extentsToScan.add(extent);
            }
            fail = false;
        } catch (FileStateInvalidException e) {
            ErrorManager.getDefault().notify(e);
        } finally {
            if (!scanIfNeeded) getDefaultRepository().endTrans(fail);
        }
    }
    
    private void traverseResource(Resource resource) {
        for (Iterator it = resource.getClassifiers().iterator(); it.hasNext();) {
            traverseClass((JavaClass) it.next());
        }
    }
    
    private void traverseGenericElement(GenericElement element) {
        for (Iterator it = element.getTypeParameters().iterator(); it.hasNext();) {
            ((TypeParameter) it.next()).getSuperClass();
        }
    }
    
    private void traverseCallableFeature(CallableFeature element) {
        for (Iterator it = element.getParameters().iterator(); it.hasNext();) {
            ((Parameter) it.next()).getType();
        }
    }
    
    private void traverseClass(JavaClass cls) {
        traverseGenericElement(cls);
        JavaClass superClass = cls.getSuperClass();
        if (superClass != null) traverseClass(superClass);
        for (Iterator it = cls.getInterfaces().iterator(); it.hasNext();) {
            traverseClass((JavaClass) it.next());
        }
        Object[] features=cls.getContents().toArray();
        for (int i=0;i<features.length;i++) {
            Object temp = features[i];
            if (temp instanceof JavaClass) {
                traverseClass((JavaClass) temp);
            } else if (temp instanceof GenericElement) {
                traverseGenericElement((GenericElement) temp);
                if (temp instanceof CallableFeature) {
                    traverseCallableFeature((CallableFeature) temp);
                }
            } else if (temp instanceof TypedElement) {
                ((TypedElement) temp).getType();
            }
        }
    }
    
    // always called from within a transaction - no need for explicit transactions
    Resource[] scanFiles(FileObject root, JavaModelPackage extent, boolean isInitial) {
        CodebaseClass cbClass = extent.getCodebase();
        if (cbClass.refAllOfClass().isEmpty()) {
            cbClass.createCodebase(getRootURI(root), 0, false, null);            
        }
        Codebase cb = (Codebase) cbClass.refAllOfClass().iterator().next();
        try {
            URL url = root.getURL();
            String uri = null;
            long startT = 0, endT;
            if (PERF_DEBUG || PERF_TEST) {
                startT = 0;
                startT = System.currentTimeMillis();
                uri = url.toString();
            }
            String sourceLevel = SourceLevelQuery.getSourceLevel(root);
            Resource[] resources = new FileScanner(url,sourceLevel,cb).scan();
            if (PERF_DEBUG || PERF_TEST) {
                endT = System.currentTimeMillis();
                System.out.println("PERF: '" + uri + "' scanning takes " + (endT - startT) + " ms."); // NOI18N
                
                if (PERF_TEST) {
                    try {
                        Class c = Class.forName("org.netbeans.performance.test.utilities.LoggingScanClasspath",true,Thread.currentThread().getContextClassLoader()); // NOI18N
                        java.lang.reflect.Method m = c.getMethod("reportScanOfFile", new Class[] {String.class, Long.class}); // NOI18N
                        Object o = m.invoke(c.newInstance(), new Object[] {new String(uri), new Long(endT - startT)});
                    } catch (Exception e) {
                        e.printStackTrace(System.err);
                    }
                }
            }
            return resources;
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ex);
        }
        return null;
    }

    private String getSourceLevel(FileObject root) {
        String result = SourceLevelQuery.getSourceLevel(root);
        return result == null ? "" : result;
    }
    
    public String getRootURI(FileObject root) {
        try {
            return root.getURL().toString() + getSourceLevel(root);
        } catch (FileStateInvalidException e) {
            throw (RuntimeException) ErrorManager.getDefault().annotate(new RuntimeException(), e);
        }
    }

    /** Unmounts codebase - always called from within a transaction - no need for explicit transaction */
    private synchronized void disposeCodebase(String uri) {
        JavaModelPackage extent = (JavaModelPackage) repository.getExtent(getCBExtentName(uri));
        if (extent != null) {
            ((JavaPackageClassImpl) extent.getJavaPackage()).removeAllRealPackages();
            FileObject root = (FileObject) filesystems.remove(extent);
            ((NBMDRepositoryImpl) repository).unmountStorage((String) sids.remove(uri));
            JavaModelPackage oldExtent = (JavaModelPackage) extents.remove(root);
            if (!extent.equals(oldExtent)) {
                extents.put(root, oldExtent);
            }
            ClassIndex.removeIndex(extent);
        }
//        System.out.println("unmounted: " + uri);
    }
    
    private static String computeHash(String val) {
        int len = val.length();
        byte[] data=new byte[len*2];
        int offset=0;
        for (int i = 0; i < len; i++) {
            char ch = val.charAt(i);
            data[offset++]=(byte)(ch>>8);
            data[offset++]=(byte)ch;
        }
        byte[] hashCode = digest.digest(data);
        int len2 = hashCode.length;
        StringBuffer sb = new StringBuffer(len2 * 3);
        for (int i = 0; i < len2; i++) {
            String number = Integer.toHexString((int) hashCode[i] - Byte.MIN_VALUE);
            if (number.length() == 1) {
                sb.append('0');
            }
            sb.append(number);
        }
        return sb.toString();
    }

     public void resetExtents() {
        cleanUpCodebases();
        init();
        
        getDefaultRepository().beginTrans(true);
        boolean fail = true;
        try {
            synchronized (this) {
                Set sidSet = new HashSet(sids.keySet());
                for (Iterator it = sidSet.iterator(); it.hasNext();) {
                    String uri = (String) it.next();
                    disposeCodebase(uri);
                }
                Iterator itr = cpImpl.getRoots().iterator();
                while (itr.hasNext()) {
                    Object obj = itr.next();
                    FileObject resource = (FileObject)obj;
                    try {
                        cpImpl.updateRoot(resource.getURL());
                    } catch (FileStateInvalidException e) {
                        
                    }
                }
                resolveCodebases();
                fail = false;
            }
        } finally {
            getDefaultRepository().endTrans(fail);
            System.err.println("rescanning " + fail);
            if (!fail)
                startRescan();
        }
    }
     
    /** Returns a valid name for a storage file */
    public static String getValidName(String uri) {
        String hashCode = computeHash(uri);
        StringBuffer result = new StringBuffer(uri.length());
        for (int i = 0; i < uri.length(); i++) {
            char z = uri.charAt(i);
            if ((z >= 'A' && z <= 'Z') || (z >= 'a' && z <= 'z') || (z >= '0' && z <= '9')) {
                result.append(z);
            }
        }
        int len = result.length();
        return 'f' + hashCode + (len > 18 ? result.substring(len - 18) : result.toString());
    }


    private synchronized boolean shouldOpenProgress() {
        return ProgressDisplayer.getVisibleProgressDisplayer()==null;
    }

    public void propertyChange(PropertyChangeEvent evt) {
        String propName = evt.getPropertyName();
        if (MergedClassPathImplementation.PROP_UNRESOLVED_ROOTS.equals(propName)) {
            if (JavaCoreModule.isStartupInProgress()) {
                return;
            }
            if (SCAN_DEBUG) System.err.println("JMManager: globalpath changed");
            // refreshCodebases has to be called in a different thread
            // to prevent deadlocks (since refreshCodebases calls isProgressOpened which is synchronized)
            SCANNING_RP.post(new Runnable() {
                public void run() {
                    refreshCodebases();
                }
            }, 0, THREAD_PRIORITY);
        }
        else if (ClassPath.PROP_ROOTS.equals(propName)) {
            if (JavaCoreModule.isStartupInProgress()) {
                return;
            }
            if (SCAN_DEBUG) System.err.println("JMManager: posting cleanup");
            CLEANUP_POSTER.post(new Runnable() {
                public void run() {
                    try {
                        if (SCAN_DEBUG) System.err.println("JMManager: cleanup running");
                        cleanUpCodebases();
                    } catch (Throwable e) {
                        ErrorManager.getDefault().notify(e);
                    }
                }
            });
        }
    }
    
    private void refreshCodebases() {
        if (NO_SCAN) return;
        if (shouldOpenProgress()) {
            if (SCAN_DEBUG) System.err.println("JMManager: refreshCodebases1");
            if (cpImpl.getUnresolvedRoots().length > 0) {
                    ProgressDisplayer.showProgress(JMManager.this);
            }
        }
        if (SCAN_DEBUG) System.err.println("JMManager: refreshCodebases2");
    }
    
    /**
     * Computes the resource name of file within the classpath root
     * @param cpRoot the classpath root
     * @param file resource
     * @return String, never returns null
     */
    public static String getResourceName (FileObject cpRoot, FileObject file) {
        assert cpRoot != null && file!= null;
        return FileUtil.getRelativePath(cpRoot, file);
    }


    /**
     * Computes the resource name of file within the global merged classpath
     * @param fo resource
     * @return String, the resource name, or null when the fo is not valid resource on merged class path.
     */
    public String getResourceName (FileObject fo) {
        repository.beginTrans(false);
        try {
            FileObject cpRoot = this.cpImpl.findOwnerRoot (fo);
            if (cpRoot == null) {
                ClassPath cp = ClassPath.getClassPath(fo, ClassPath.SOURCE);
                if (cp!=null) {
                    cpRoot = cp.findOwnerRoot(fo);
                }
            }
            String result = null;
            if (cpRoot != null) {
                result = getResourceName(cpRoot,fo);
            }
            return cpRoot == null ? null : getResourceName(cpRoot, fo);
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ex);
        } finally {
            repository.endTrans();
        }
        return null;
    }

    void removeListener() {
        classPath.removePropertyChangeListener(this);
    }
    
    public ProgressSupport getProgressSupport() {
        if (progressSupport == null) {
            progressSupport = new ProgressSupport();
        }
        return progressSupport;
    }

    public void addModified(FileObject fo) {
        getTransactionMutex().addModified(fo);
    }
    
    public boolean isModified(FileObject fo) {
        return getTransactionMutex().isModified(fo);
    }

    public static void fireResourceParsed(final MOFID mofId) {
        /** processor dedicated for parser events. */
        EVENT_RP.post(new Runnable() {
            public void run() {
                ArrayList listenersCopy;
                synchronized (listeners) {
                    if (listeners.isEmpty()) return;
                    listenersCopy = (ArrayList) listeners.clone();
                }
                Resource rsc = (Resource) ((NBMDRepositoryImpl) JavaMetamodel.getDefaultRepository()).getByMofId(mofId);
                if (rsc == null) return;
                for (Iterator it = listenersCopy.iterator(); it.hasNext();) {
                    ParsingListener listener = (ParsingListener) it.next();
                    listener.resourceParsed(rsc);
                }
            }
        });
    }
    
    public void startupFinished() {
        if (cpImpl.getUnresolvedRoots().length > 0) {
            SCANNING_RP.post(new Runnable() {
                public void run() {
                    refreshCodebases();
                }
            });
        } else {
            RequestProcessor.getDefault().post(new Runnable() {
                public void run() {
                    init();
                }
            }, 1500);
        }
    }

    public ClassDefinition getSourceElementIfExists(ClassDefinition cls) {
        if (cls == null) return null;
        
        repository.beginTrans(false);
        try {
            Resource rsc=cls.getResource();

            if (rsc!=null) {
                if (rsc.getName().endsWith(".java")) {
                    return cls;
                }
            } else {
                return cls;
            }
            try {
                ClassDefinition realJcls=cls instanceof ParameterizedType ? ((ParameterizedType)cls).getDefinition(): cls;

                if (!(realJcls instanceof UnresolvedClass) && !(realJcls instanceof Array)) {
                    FileObject fo = getCPRoot((JavaModelPackage)realJcls.refImmediatePackage());
                    FileObject[] fos = SourceForBinaryQuery.findSourceRoots(fo.getURL()).getRoots();
                    String name=realJcls.getName();

                    for (int i = 0; i < fos.length; i++) {
                        FileObject fileObject = fos[i];
                        JavaModelPackage javaExtent = getJavaExtent(fileObject);
                        ClassIndex ci = ClassIndex.getIndex(javaExtent);

                        if (ci != null) {
                            JavaClass jc = ci.getClassByFqn(name);

                            if (jc != null && cls instanceof ParameterizedType) {
                                ParameterizedType parType = (ParameterizedType)cls;
                                ClassDefinition outerCls = parType.getDeclaringClass();
                                List pars = parType.getParameters();

                                jc = javaExtent.getParameterizedType().resolveParameterizedType(jc, pars, outerCls instanceof ParameterizedType ? (ParameterizedType)outerCls : null);
                                if (parType instanceof ParameterizedTypeImpl) {
                                    ParameterizedTypeImpl parTypeImpl = (ParameterizedTypeImpl)parType;
                                    ParameterizedTypeImpl newTypeImpl = (ParameterizedTypeImpl)jc;
                                    int size = pars.size();
                                    
                                    for (int j=0;j<size;j++) {
                                        newTypeImpl.setWildCardStatus(j,parTypeImpl.getWildCardStatus(j));
                                    }
                                }
                            }
                            if (jc != null) {
                                return jc;
                            }
                        }
                    }
                }
            } catch (Exception e) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
            }
            return cls;
        } finally {
            repository.endTrans();
        }
    }
    
    /**
     * Initializes indentation. Currently, it reads editor options and sets
     * our formatting engine to respect settings. Used for indentation,
     * bracket and parenthesis formatting.
     */
    public static void initIndentation() {
        if (MetadataElement.getIndentSpace() != -1) {
            return;
        }
        IndentationSettingsProvider settingsProvider = getIndentationSettingsProvider();
        if (settingsProvider == null) {
            setIndentSpace(4, true);
            formatCurlies(false);
            formatParenthesis(false);
            return;
        }
        Boolean tabs = (Boolean) settingsProvider.getPropertyValue("expandTabs"); // NO18N
        Integer spaces = (Integer) settingsProvider.getPropertyValue("spacesPerTab"); // NO18N
        if (spaces == null || tabs == null) {
            setIndentSpace(4, true);
        } else {
            setIndentSpace(spaces.intValue(), tabs.booleanValue());
        }
        Boolean newlineBrace = (Boolean) settingsProvider.getPropertyValue("javaFormatNewlineBeforeBrace"); // NO18N
        if (newlineBrace == null) {
            formatCurlies(false);
        } else {
            formatCurlies(newlineBrace.booleanValue());
        }
        Boolean parenthesisSpace = (Boolean) settingsProvider.getPropertyValue("javaFormatSpaceBeforeParenthesis"); // NO18N
        if (parenthesisSpace == null) {
            formatParenthesis(false);
        } else {
            formatParenthesis(parenthesisSpace.booleanValue());
        }
    }
    
    /**
     * Sets the correct formatting for all brackets. If the parameter is true,
     * it sets newline before brace. Otherwise brace is on the same line
     * separated only by the space.
     *
     * @param  breakBefore  use true if you want to have bracket on the new line
     */
    private static void formatCurlies(boolean breakBefore) {
        int[] curlies = {
            MetadataElement.CLASS_OPEN_CURLY,
            MetadataElement.BODY_OPEN_CURLY,
            MetadataElement.ARRAY_OPEN_CURLY,
            MetadataElement.BLOCK_OPEN_CURLY
        };
        String before = breakBefore ? "ni" : "s"; // NOI18N
        for (int i = 0; i < curlies.length; i++) {
            MetadataElement.setFormattingFor(curlies[i], before, MetadataElement.getFormattingFor(curlies[i])[2]);
        }
    }

    /**
     * Sets the correct formatting for the parenthesis. When parameter is true,
     * space is added before every opening parenthesis generated by our
     * code-generator. Otherwise, parenthesis is immediately after identifier.
     *
     * @param  parenthesisSpace true if you want space before opening parethesis
     */
    private static void formatParenthesis(boolean parenthesisSpace) {
        int[] parenthesis = { 
            MetadataElement.PAR_OPEN_BRACKET,
        };
        String before = parenthesisSpace ? "s" : ""; // NOI18N
        for (int i = 0; i < parenthesis.length; i++) {
            MetadataElement.setFormattingFor(parenthesis[i], before, MetadataElement.getFormattingFor(parenthesis[i])[2]);
        }
    }
    
    /**
     * Sets the default indentation of the element.
     *
     * @param  spaces      number of spaces used for indentation
     * @param  expandTabs  if this parameter is true, tab character is used
     *                     instead of spaces
     */
    public static void setIndentSpace(int spaces, boolean expandTabs) {
        MetadataElement.setIndentSpace(spaces, expandTabs);
    }
    
    /**
     * Copies the file from in location to out location.
     *
     * @param   in     path to file with filename without extension, which
     *                 will be copied (read)
     * @param   out    path to new location of the file with filename 
     *                 without extension which will be written
     * @param   force  if false, it checks 'out' file for existence. If exists,
     *                 method does nothing
     *
     * @return  true  if it was copied successfuly, false if error or force
     *                flag does not allow to copy it (it exists already)
     */
    public static boolean copyStorage(String in, String out, boolean force) {
        long startTime = 0;
        if (PERF_DEBUG) startTime = System.currentTimeMillis();

        String[] ext = new String[] { "btd", "btx", "cdx" }; // NOI18N

        // fix for #58064
        for (int i = 0; i < ext.length; i++) {
            File f = new File(in + '.' + ext[i]);
            if (!f.exists()) {
                // one of the source files is missing -> cannot copy
                return false;
            }
        }
        
        if (!force) {
            boolean doCopy = false;
            for (int i = 0; i < ext.length; i++) {
                File f = new File(out + '.' + ext[i]);
                if (!f.exists()) {
                    doCopy = true;
                    break;
                }
            }
            if (!doCopy) return false;
        }
        for (int i = 0; i < ext.length; i++) {
            FileInputStream fis = null;
            FileOutputStream fos = null;
            try {
                fis = new FileInputStream(new File(in + '.' + ext[i]));
                fos = new FileOutputStream(new File(out + '.' + ext[i]));
                byte[] buf = new byte[65535];
                int j = 0;
                while ((j = fis.read(buf)) != -1) {
                    fos.write(buf, 0, j);
                }
            } catch (Exception e) {
                System.out.println("copyStorage failed: " + e.getMessage()); // NOI18N
                e.printStackTrace();
                return false;
            } finally {
                try {
                    if (fis != null)
                        fis.close();
                    if (fos != null)
                        fos.close();
                } catch (IOException e) {
                    System.out.println("copyStorage failed: " + e.getMessage()); // NOI18N
                    e.printStackTrace();
                    return false;
                }
            }
        }
        if (PERF_DEBUG) System.err.println("PERF: Copy of " + in + " took " + // NOI18N
                        (System.currentTimeMillis() - startTime) + " ms."); // NOI18N
        return true;
    }

    /**
     * Checks, if the preparsed storage exists. If so, it checks existence
     * of storage in userdir. If it does not exists, it copies the file
     * to userdir location with correct name. Oterwise, it does nothing.
     *
     * @param   uri    URI to check
     * @param   root   archive root
     * @param   target location and file name
     *
     * @return  true, if preparsed storage exists and is used. (Either it
     *          is copied or not.)
     */
    private static boolean usePreparsed(String simpleName, FileObject root, String storageName) {
        try {
            if (simpleName != null) {
                FileObject archiveRoot = FileUtil.getArchiveFile(root);
                String toCopy = PREPARSED_LOCATION + simpleName + "-" + computePreparsedHash(FileUtil.toFile(archiveRoot), BYTES4MD5); // NOI18N
                return copyStorage(toCopy, storageName, false);
            }
        } catch (IOException e) {
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, e.getMessage());
        }
        return false;
    }
    
    private static String getCodebaseSimpleName(FileObject root) {
        try {
            String uri = root.getURL().toString();
            if (uri.endsWith("!/")) { // NOI18N
                String simpleName = uri.substring(0, uri.length() - 2);
                int pos = simpleName.lastIndexOf('/');
                if (pos >= 0) simpleName = simpleName.substring(pos + 1);
                pos = simpleName.lastIndexOf('.');
                if (pos >= 0) simpleName = simpleName.substring(0, pos);
                if (PREPARSED_CACHE.contains(simpleName)) {
                    return simpleName;
                }
            }
        } catch (FileStateInvalidException e) {
            ErrorManager.getDefault().notify(e);
        }
        return null;
    }
    
    /**
     * Computes hash for provided file. Reads len bytes from the file
     * and computes MD5 hash. 
     *
     * @param file      file which will be read
     * @param len       says how much bytes will be read from file for
     *                  computation
     * @throws java.io.IOException  file was not found or cannot be read
     * @return  computed hash in string representation
     */
    public static String computePreparsedHash(File file, int len) throws IOException {
        StringBuffer sb = new StringBuffer(32);
        InputStream stream = new BufferedInputStream(new FileInputStream(file));
        try {
            byte[] input = new byte[len];
            stream.read(input);
            byte[] hash = md5digest.digest(input);
            for (int i = 0; i < hash.length; i++) {
                String s = Integer.toHexString(hash[i] - Byte.MIN_VALUE);
                if (s.length() == 1) sb.append('0');
                sb.append(s);
            }
        } finally {
            stream.close();
        }
        return sb.toString();
    }
    
    public static synchronized void setIndentationSettingsProvider(IndentationSettingsProvider p) {
        if (p == null) {
            settingsProvider.removePropertyChangeListener(indentListener);
            indentListener = null;
        } else if (settingsProvider != null && indentListener != null) {
            // both orig and new one are non-null and already listening -> reattach listening
            settingsProvider.removePropertyChangeListener(indentListener);
            p.addPropertyChangeListener(indentListener);
        }
        settingsProvider = p;
    }
    
    private static synchronized IndentationSettingsProvider getIndentationSettingsProvider() {
        if (settingsProvider != null && indentListener == null) { // not listening to prop chgs yet
            indentListener = new PropertyChangeListener() {
                // listens on changes of properties which are important for
                // our formatting engine.
                public void propertyChange(PropertyChangeEvent evt) {
                    if ("spacesPerTab".equals(evt.getPropertyName())) { // NOI18N
                        setIndentSpace(((Integer) evt.getNewValue()).intValue(), MetadataElement.isExpandTab());
                    } else if ("expandTabs".equals(evt.getPropertyName())) { // NOI18N
                        setIndentSpace(MetadataElement.getIndentSpace(), ((Boolean) evt.getNewValue()).booleanValue());
                    } else if ("javaFormatNewlineBeforeBrace".equals(evt.getPropertyName())) { // NOI18N
                        formatCurlies(((Boolean) evt.getNewValue()).booleanValue());
                    } else if ("javaFormatSpaceBeforeParenthesis".equals(evt.getPropertyName())) { // NOI18N
                        formatParenthesis(((Boolean) evt.getNewValue()).booleanValue());
                    }
                }
            };
            settingsProvider.addPropertyChangeListener(indentListener);
        }
        return settingsProvider;
    }
    
    private static String defaultEncoding;
    
    public static String getDefaultEncoding() {
        return defaultEncoding;
    }
    
    public static void setDefaultEncoding(String enc) {
        defaultEncoding = enc;
    }
    
    public void startRescan() {
        ((RescanAction) RescanAction.get(RescanAction.class)).performAction();
    }

    void cancelScanning() {
        cancelScan = true;
    }
    
    public static abstract class ModifiedDOProvider {
        private ModifiedDOProvider delegate;
        private boolean assigned;
        
        public final DataObject getModifiedDataObject(FileObject fo) {
            DataObject result = getModifiedDOImpl(fo);
            if (result == null && delegate != null) {
                result = delegate.getModifiedDataObject(fo);
            }
            return result;
        }
        
        protected abstract DataObject getModifiedDOImpl(FileObject fo);

        public static synchronized void setModifiedDOProvider(ModifiedDOProvider provider) {
            assert !provider.assigned : "Cannot set the same ModifiedDOProvider twice."; // NOI18N
            provider.delegate = modifiedDOProvider;
            modifiedDOProvider = provider;
            provider.assigned = true;
        }
    
        public static synchronized ModifiedDOProvider getModifiedDOProvider() {
            return modifiedDOProvider;
        }
    }
}
