/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.lib.java.parser.ASTree;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.lib.java.parser.Token;
import org.netbeans.modules.javacore.internalapi.GuardedResolver;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.netbeans.modules.javacore.parser.MDRParser;

/**
 *
 * @author Jan Becicka
 */
public final class FileGuardedResolver implements GuardedResolver {
    
    private static final String GEN_BEGIN = "//GEN-BEGIN"; // NOI18N
    private static final String GEN_END = "//GEN-END"; // NOI18N
    private static final String GEN_HEADER = "//GEN-HEADER"; // NOI18N
    private static final String GEN_HEADEREND = "//GEN-HEADEREND"; // NOI18N
    private static final String GEN_LINE = "//GEN-LINE"; // NOI18N
    private static final String GEN_FIRST = "//GEN-FIRST"; // NOI18N
    private static final String GEN_LAST = "//GEN-LAST"; // NOI18N
    
    private static final int GUARDED_BEGIN = 0;
    private static final int GUARDED_END = 1;
    private static final int GUARDED_LINE = 2;
    private static final int GUARDED_NONE = -1;
    
    private static FileGuardedResolver instance = null;
    
    private FileGuardedResolver() {
    }
    
    public static synchronized FileGuardedResolver getDefault() {
        if (instance == null) {
            instance = new FileGuardedResolver();
        }
        return instance;
    }
    
    public final boolean isSectionGuarded(org.netbeans.jmi.javamodel.Resource resource, org.openide.text.PositionBounds bounds) {
        MDRParser parser = ((ResourceImpl) resource).getParser();
        if (parser.guardedBlocksBorders == null) {
            parser.guardedBlocksBorders = findGuardedBlocks(resource);
        }
        return isGuarded(parser.guardedBlocksBorders, bounds.getBegin().getOffset()) ||
                isGuarded(parser.guardedBlocksBorders, bounds.getEnd().getOffset());
    }
    
    private int[] findGuardedBlocks(Resource resource) {
        MDRParser provider = ((ResourceImpl)resource).getParser();
        int lastTokenIndex = 0;
        ASTree tree = provider.getASTree();
        if (tree != null) {
            lastTokenIndex = tree.getLastToken();
        }
        List borders = new ArrayList();
        int lineStart = 0;
        for (int x = 0; x <= lastTokenIndex; x++) {
            Token token = provider.getToken(x);
            Token[] pads = token.getPadding();
            if (pads == null)
                continue;
            for (int y = 0; y < pads.length; y++) {
                int padType = pads[y].getType();
                if (padType == ParserTokens.EOL) {
                    lineStart = pads[y].getEndOffset();
                } else if (padType == ParserTokens.EOL_COMMENT) {
                    String text = provider.getText(pads[y]);
                    int gid = guardedId(text);
                    switch (gid) {
                        case GUARDED_BEGIN:
                            borders.add(new Integer(lineStart));
                            break;
                        case GUARDED_END:
                            borders.add(new Integer(pads[y].getEndOffset()));
                            break;
                        case GUARDED_LINE:
                            borders.add(new Integer(lineStart));
                            borders.add(new Integer(pads[y].getEndOffset()));
                    } // switch
                } // if
            } // for
        } // for
        int[] result = new int[borders.size()];
        Iterator iter = borders.iterator();
        for (int x = 0; iter.hasNext(); x++) {
            result[x] = ((Integer)iter.next()).intValue();
        }
        return result.length > 0 ? provider.getDocumentOffsets(result) : result;
    }
    
    private int guardedId(String text) {
        if (text == null)
            return GUARDED_NONE;
        if (text.indexOf(GEN_BEGIN) > -1 || text.indexOf(GEN_HEADER) > -1)
            return GUARDED_BEGIN;
        if (text.indexOf(GEN_END) > -1 || text.indexOf(GEN_HEADEREND) > -1)
            return GUARDED_END;
        if (text.indexOf(GEN_LINE) > -1 || text.indexOf(GEN_FIRST) > -1 || text.indexOf(GEN_LAST) > -1)
            return GUARDED_LINE;
        return GUARDED_NONE;
    }
    
    private boolean isGuarded(int[] borders, int start, int end) {
        if (borders.length == 0) {
            return false;
        }
        int index = 0;
        if (start < borders[0]) {
            return end < borders[0];
        }
        while (index < borders.length && borders[index] < start)
            index++;
        if (index == borders.length)
            return false;
        if (borders[index] == start)
            return true;
        if (index % 2 == 0) {
            return end < borders[index];
        } else {
            return true;
        }
    }
    
    private boolean isGuarded(int[] borders, int offset) {
        if (borders.length == 0 || offset < borders[0]) {
            return false;
        }
        for (int x = 0; x < borders.length / 2; x++) {
            if (offset >= borders[2*x] && offset <= borders[2*x+1])
                return true;
        }
        return false;
    }
    
}
