/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore;

/** Private SPI for providing alternative backend behind the class index.
 *
 * @author Martin Matula
 */
public interface ClassIndexStorage {
    /** Checks whether there are any IDs registered for a given FQN.
     * @param fqn Fully qualified name to be checked.
     * @return true if the given fully qualified name exists (i.e. if {@link #getIDsForFQN}
     * returns a non-null value).
     */
    boolean existsFQN(String fqn);
    
    /** Returns IDs registered for a given fully qualified or simple name.
     * @param name Name (fully qualified or simple).
     * @param isSimpleName Indicates whether the first parameter is a fully qualified or simple name (true = simple name).
     * @return IDs for a given name or null if there are
     * no IDs for a given name. This method never returns an empty array.
     */
    long[] getIDsForName(String name, boolean isSimpleName);
    
    /** Sets IDs to be registered for a given name.
     * @param name Simple or fully qualified name.
     * @param isSimpleName Indicates whether the first parameter is a fully qualified or simple name (true = simple name).
     * @param classIDs IDs for a given name or null (if no IDs should be registered
     * for a given name).
     */
    void setIDsForName(String name, boolean isSimpleName, long[] classIDs);

    /** Retrieves IDs registered for names with a given prefix
     * and returns the IDs registered for the first name with a given prefix
     * found in form of QueryItem. QueryItem.getNext() can be used to move forward
     * in the result set. Each QueryItem represents an array of IDs registered for a single name.<p/>
     * In case of a fully qualified name prefix (isSimpleName == false) the semantics of "FQN has fqnPrefix"
     * in the context of this method means the following needs to be true:<br/>
     * <pre>    fqn.startsWith(fqnPrefix) && fqn.lastIndexOf('.') == fqnPrefix.lastIndexOf('.')
     * </pre>
     * @param prefix Name prefix for which the IDs should be returned.
     * @param isSimpleName Indicates whether a prefix is a simple name prefix or a FQN prefix.
     * @return QueryItem representing the IDs registered for the first simple name found (having the specified prefix)
     */
    QueryItem getFirstForNamePrefix(String prefix, boolean isSimpleName);

    /** Returns integers (identifier hash codes) registered for a given ID.
     * @param resourceId ID to return the hash codes for.
     * @return Identifier hash codes registered for a given ID (naturally ordered) or null (if no
     * identifier hash codes are registered for a given ID).
     */
    int[] getIdentifiers(long resourceId);
    
    /** Sets identifier hash codes to be registered for a given ID.
     * @param resourceId ID to register the identifier hash codes for.
     * @param hashCodes Identifier hash codes for a given ID or null if no IDs should be registered.
     * <strong>The elements in the array must be ordered using ascending natural ordering.</strong>
     */
    void setIdentifiers(long resourceId, int[] hashCodes);
    
    /** Returns and removes identifier hash codes for a given ID.
     * @param resourceId ID to remove the identifier hash codes registration for.
     * @return Removed identifier hash codes (naturally ordered).
     */
    int[] removeIdentifiers(long resourceId);
    
    /** Returns all IDs for which a given identifier hash code is registered.
     * @param hashCode Identifier hash code to search the IDs for.
     * @return Array of all IDs for which a given hash code is registered. (Never returns null.)
     */
    long[] getIDsForIdentifier(int hashCode);

    /** Initializes this storage. Can be used for deserializing the internal storage structures.
     * @return true if the initialization succeeded. Otherwise returns false (e.g. if the persisted
     * parts of the storage cannot be deserialized or are corrupted in some way or the index storage
     * does not exist yet).
     */
    boolean mount();
    
    /** Called when the corresponding class index is released by the IDE. Can be used for a cleanup tasks
     * and for serializing the index content.
     */
    void unmount();

    /** Simple represenation of a result set.
     */
    public interface QueryItem {
        /** Returns name the IDs represented by this QueryItem are registered.
         * @return Name.
         */
        String getName();
        
        /** Returns IDs represented by this QueryItem.
         * @return IDs
         */
        long[] getIDs();
        
        /** Sets IDs registered for a name this QueryItem corresponds to.
         * The method may throw ConcurrentModificationException if the underlying data structures were concurrently modified.
         * @param ids IDs to be registered or null if no IDs should be registered for a given name.
         */
        void setIDs(long[] ids);
        
        /** Returns next item in the result set.
         * The method may throw ConcurrentModificationException if the underlying data structures were concurrently modified.
         * @return Next item in the result set.
         */
        QueryItem getNext();
    }
}