/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore;

import java.io.*;
import java.util.*;
import java.util.Map.Entry;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.queries.SourceLevelQuery;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.lib.java.parser.Factory;
import org.netbeans.lib.java.parser.JScanner;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.JavaClassImpl;
import org.netbeans.mdr.util.IOUtils;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.JarFileSystem;

/**
 *
 * @author  Tomas Hurka, Martin Matula
 */
public class ClassIndex {
    private static final boolean DEBUG = false;

    private static final Map codebaseIndexes = Collections.synchronizedMap(new HashMap());
    private static boolean shutdownListenerRegistered = false;
    private static final ShutdownL shutdownListener = new ShutdownL();
    
    private final ClassIndexStorage storage;
    private final String storageId;
    private final NBMDRepositoryImpl rep;
    
    private long lastSaved;
    
    private HashMap changeLogSN;
    private HashMap changeLogFQN;
    private HashMap changeLogI;

    /** Creates a new instance of ClassIndex */
    private ClassIndex(String file, String storageId) {
        this.storageId = storageId;
        rep = (NBMDRepositoryImpl) JavaMetamodel.getDefaultRepository();
        storage = new CISImpl(file);
        
        // register shutdown listener
        if (!shutdownListenerRegistered) {
            synchronized (shutdownListener) {
                if (!shutdownListenerRegistered) {
                    ((NBMDRepositoryImpl) JavaMetamodel.getDefaultRepository()).addShutdownListener(shutdownListener);
                    shutdownListenerRegistered = true;
                }
            }
        }
    }
    
    static void commit() {
        for (Iterator it = codebaseIndexes.values().iterator(); it.hasNext();) {
            ((ClassIndex) it.next()).commitChanges();
        }
    }
    
    static void rollback() {
        for (Iterator it = codebaseIndexes.values().iterator(); it.hasNext();) {
            ((ClassIndex) it.next()).rollbackChanges();
        }
    }
    
    private void commitChanges() {
        changeLogSN = null;
        changeLogFQN = null;
        changeLogI = null;
    }
    
    private void rollbackChanges() {
        if (changeLogSN != null) {
            for (Iterator it = changeLogSN.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                it.remove();
                long mofId = ((Long) entry.getKey()).longValue();
                if (entry.getValue() == null) {
                    JavaClass cls = (JavaClass) rep.getByMofId(makeMofId(mofId));
                    if (cls != null) {
                        addClass(cls, cls.getName(), cls.getSimpleName());
                    }
                } else {
                    removeFromIndex(true, (String) entry.getValue(), mofId);
                    // list of fqn changes is superset of this list so I don't have to add the class
                    // to index here - it will be done when iterating through changeLogFQN below
                }
            }
        }
        
        if (changeLogFQN != null) {
            for (Iterator it = changeLogFQN.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                it.remove();
                long mofId = ((Long) entry.getKey()).longValue();
                JavaClass cls = (JavaClass) rep.getByMofId(makeMofId(mofId));
                if (entry.getValue() != null) {
                    removeFromIndex(false, (String) entry.getValue(), mofId);
                }
                if (cls != null) {
                    addClass(cls, cls.getName(), cls.getSimpleName());
                }
            }
        }
        
        if (changeLogI != null) {
            for (Iterator it = changeLogI.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                it.remove();
                storage.setIdentifiers(((Long) entry.getKey()).longValue(), (int[]) entry.getValue());
            }
        }
    }
    
    public static ClassIndex getIndex(JavaModelPackage mofPackage) {
        return (ClassIndex)codebaseIndexes.get(mofPackage);
    }
    
    private static void saveAllIndexes() {
        MDRepository rep = JavaMetamodel.getDefaultRepository();
        
        rep.beginTrans(false);
        try {
            synchronized (codebaseIndexes) {
                Iterator indexIt = codebaseIndexes.values().iterator();

                while (indexIt.hasNext()) {
                    ClassIndex index = (ClassIndex) indexIt.next();
                    index.storage.unmount();
                }
            }
        } finally {
            rep.endTrans();
        }
    }
    
    // must be called from a synchronized block
    static void removeIndex(JavaModelPackage mofPackage) {
        ClassIndex index = (ClassIndex) codebaseIndexes.remove(mofPackage);
        index.storage.unmount();
    }
    
    // must be called from a synchronized block
    static ClassIndex addIndex(JavaModelPackage mofPackage, String filename) {
        ClassIndex index = new ClassIndex(filename, ((BaseObjectHandler) mofPackage)._getDelegate().getMofId().getStorageID());
        codebaseIndexes.put(mofPackage, index);
        return index;
    }
    
    // must be called from a synchronized block
    static boolean loadIndex(String filename, JavaModelPackage mofPackage) {
        ClassIndex index = addIndex(mofPackage, filename);
        return index.storage.mount();
    }

    public void renameClass(JavaClass jcls, String newName) {
        long mofId = getMofId(jcls);
        String name = jcls.getName();
        String simpleName = jcls.getSimpleName();
        String newSimpleName = JavaClassImpl.getSimpleName(newName);

        if (JMManager.INCONSISTENCY_DEBUG) System.err.println("ClassIndex: Renaming class: " + name + " to: " + newName + " MOFID: " + jcls.refMofId());
        //Thread.dumpStack();
        
        removeFromIndex(false, name, mofId);
        
        Long objMofId = new Long(mofId);
        logFQNChange(objMofId, newName);
        if (!simpleName.equals(newSimpleName)) {
            removeFromIndex(true, simpleName, mofId);
            logSNChange(objMofId, newSimpleName);
        }
        addClass(jcls, newName, newSimpleName);
    }
    
    private static HashMap logChange(HashMap map, Long mofId, Object value) {
        if (map == null) {
            map = new HashMap();
        }
        map.put(mofId, value);
        return map;
    }
    
    private void logFQNChange(Long mofId, String name) {
        changeLogFQN = logChange(changeLogFQN, mofId, name);
    }
    
    private void logSNChange(Long mofId, String name) {
        changeLogSN = logChange(changeLogSN, mofId, name);
    }
    
    public void addClass(JavaClass jcls,String fqn,String simpleName) {
        if (JMManager.INCONSISTENCY_DEBUG) System.err.println("ClassIndex: Adding class " + fqn + " MOFID: " + jcls.refMofId());
        long mofId = getMofId(jcls);

        addToIndex(true, simpleName, mofId);
        addToIndex(false, fqn, mofId);
    }
    
    public long getTimestamp() {
        return lastSaved;
    }
    
    public void setTimestamp() {
        lastSaved = System.currentTimeMillis();
    }

    // may return invalid classes
    public Set getClassesByFqn(String fqn) {
        Set result = new HashSet();
        long[] classes = storage.getIDsForName(fqn, false);
        if (classes != null) {
            JavaClass cls = null;
            int j = 0;
            for (int i = 0; i < classes.length; i++) {
                JavaClass temp = (JavaClass) rep.getByMofId(makeMofId(classes[i]));
                if (temp == null) {
                    logFQNChange(new Long(classes[i]), null);
                } else {
                    classes[j] = classes[i];
                    j++;
                    result.add(temp);
                }
            }
            if (j == 0) {
                storage.setIDsForName(fqn, false, null);
            } else if (j < classes.length) {
                long[] newValue = new long[j];
                System.arraycopy(classes, 0, newValue, 0, j);
                storage.setIDsForName(fqn, false, newValue);
            }
        }
        return result;
    }

    // never returns an invalid class
    public JavaClass getClassByFqn(String fqn) {
        JavaClass cls = null;
        JavaClass oldCls;
        do {
            oldCls = cls;
            cls = tryToGetClassByFqn(fqn);
        } while (cls != null && !cls.isValid() && oldCls != cls);
        if (cls != null && oldCls == cls) {
            try {
                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, new RuntimeException("Class was not cleaned from index: " + cls.getName())); // NOI18N
            } catch (javax.jmi.reflect.InvalidObjectException e) {
                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, e);
            }
            cls = null;
        }
        return cls;
    }
    
    private JavaClass tryToGetClassByFqn(String fqn) {
        long[] classes = storage.getIDsForName(fqn, false);
        if (classes != null) {
            JavaClass result = null;
            JavaClass sureResult = null;
            int j = 0;
            for (int i = 0; i < classes.length; i++) {
                JavaClass temp = (JavaClass) rep.getByMofId(makeMofId(classes[i]));
                if (temp == null) {
                    logFQNChange(new Long(classes[i]), null);
                } else {
                    classes[j] = classes[i];
                    j++;
                    result = temp;
                    // [TODO] the following code can be uncommented if necessary
                    // (it tries to prefer classes that are in the correct folder)
                    // now it is commented out for performance reasons
//                    Resource res = (Resource) result.refImmediateComposite();
//                    if (res != null) {
//                        if (res.getName().replace('/', '.').startsWith(res.getPackageName())) {
//                            sureResult = result;
//                        }
//                    }
                }
            }
            if (j == 0) {
                storage.setIDsForName(fqn, false, null);
            } else if (j < classes.length) {
                long[] newValue = new long[j];
                System.arraycopy(classes, 0, newValue, 0, j);
                storage.setIDsForName(fqn, false, newValue);
            }
            return sureResult == null ? result : sureResult;
        }
        return null;
    }
    
    // may return invalid classes
    public LazyImmutableList getClassesByFQNPrefix(String fqnPrefix) {
        return new LazyImmutableList(new ClassesByFQNPrefixIterator(storage.getFirstForNamePrefix(fqnPrefix, false)));
    }
    
    public LazyImmutableList getClassesByFQNPrefix(String fqnPrefix, boolean caseSensitive) {
        if (caseSensitive) return getClassesByFQNPrefix(fqnPrefix);
        
        int dot = fqnPrefix.lastIndexOf('.');
        String packagePart = fqnPrefix.substring(0, dot + 1);
        return new LazyImmutableList(new ClassesByFQNPrefixCIIterator(storage.getFirstForNamePrefix(packagePart, false), fqnPrefix));
    }
    
    /**
     * Looks for the classes in class index by prefix. Finding mechanism is case
     * sensitive. May return invalid classes.
     *
     * @param  snPrefix  simple name leading characters of class which will 
     *                   be looked for
     * @return   list of matching JavaClasses instances ordered by their names
     */
    public LazyImmutableList getClassesBySNPrefix(String snPrefix) {
        return getClassesBySNPrefix(snPrefix, true);
    }
    
    /**
     * Looks for the classes in class index by prefix. Finding can be either 
     * case-sensitive or case-insensitive. May return invalid classes (i.e. classes for which cls.isValid() == false).
     *
     * @param  snPrefix  simple name leading characters of class which
     *                     will be looked for
     * @return   list of matching JavaClasses instances ordered by names
     */
    public LazyImmutableList getClassesBySNPrefix(String snPrefix, boolean caseSensitive) {
        ClassIndexStorage.QueryItem query;
        boolean camelCase;
        if (snPrefix.length() > 1 && snPrefix.equals(snPrefix.toUpperCase())) {
            query = storage.getFirstForNamePrefix(snPrefix.substring(0,1), true);
            camelCase = true;
        } else {
            query = storage.getFirstForNamePrefix(snPrefix.toUpperCase(), true);
            camelCase = false;
        }
        return new LazyImmutableList(new ClassesBySNPrefixIterator(query, snPrefix, caseSensitive, camelCase));
    }
    
    private abstract class ClassesByPrefixIterator extends LazyImmutableList.LazyIterator {
        private int maxSize = -1;
        private int currentSize = 0;
        
        protected ClassIndexStorage.QueryItem currentItem;
        
        private Object next;
        private boolean hasNext = true;
        private Iterator inner = Collections.EMPTY_LIST.iterator();
        
        ClassesByPrefixIterator(ClassIndexStorage.QueryItem item) {
            this.currentItem = item;
        }
        
        public final Object next() {
            if (next == null) {
                findNext();
            }
            Object result = next;
            next = null;
            return result;
        }
        
        public final boolean hasNext() {
            findNext();
            return hasNext;
        }
        
        protected final int maxEstimatedSize() {
            if (maxSize == -1) {
                computeSize();
            }
            return maxSize;
        }
        
        private void computeSize() {
            maxSize = currentSize;
            ClassIndexStorage.QueryItem item = currentItem;
            while (item != null) {
                maxSize += item.getIDs().length;
                item = item.getNext();
            }
        }
        
        private void findNext() {
            if (!hasNext) {
                throw new NoSuchElementException();
            }
            while (next == null && hasNext == true) {
                if (inner.hasNext()) {
                    next = inner.next();
                } else {
                    while ((currentItem != null) && (!accept(currentItem))) {
                        currentItem = currentItem.getNext();
                    }
                    if (currentItem != null) {
                        inner = getClassesByPrefix().iterator();
                    } else {
                        hasNext=false;
                    }
                }
            }
        }
        
        private Collection getClassesByPrefix() {
            Collection result = new ArrayList();
            
            long[] value = currentItem.getIDs();
            currentSize += value.length;
            
            int j = 0;
            for (int i = 0; i < value.length; i++) {
                JavaClass cls = (JavaClass) rep.getByMofId(makeMofId(value[i]));
                j += evaluateItem(value, i, j, cls, result);
            }
            if (j == 0) {
                currentItem.setIDs(null);
            } else if (j < value.length) {
                long[] newValue = new long[j];
                System.arraycopy(value, 0, newValue, 0, j);
                currentItem.setIDs(newValue);
            }
            
            currentItem = currentItem.getNext();
            return result;
        }
        
        protected boolean accept(ClassIndexStorage.QueryItem item) {
            return true;
        }
        
        protected abstract int evaluateItem(long[] value, int i, int j, JavaClass cls, Collection result);
    }
    
    private class ClassesBySNPrefixIterator extends ClassesByPrefixIterator {
        private final String prefix;
        private final boolean matchName;
        private final boolean camelCase;
        
        ClassesBySNPrefixIterator(ClassIndexStorage.QueryItem item, String prefix, boolean caseSensitive, boolean camelCase) {
            super(item);
            this.prefix = prefix;
            this.matchName = caseSensitive && prefix.length() > 0;
            this.camelCase = camelCase;
        }
        
        protected boolean accept(ClassIndexStorage.QueryItem item) {
            return !camelCase || item.getName().startsWith(prefix) || matchIndexEntry(item.getName());
        }
        
        private boolean matchIndexEntry(String indexEntryName) {
            int index = -1;
            for (int i = 0; i < prefix.length(); i++) {
                index = indexEntryName.indexOf(prefix.charAt(i), index + 1);
                if (index < 0) {
                    return false;
                }
            }
            return true;
        }
        
        private boolean matchName(String simpleName) {
            if (!matchName && simpleName.toUpperCase().startsWith(prefix)) return true;
            
            int sni, pi;
            for (pi = 0, sni = 0; sni < simpleName.length() && pi < prefix.length(); sni++) {
                char ch = simpleName.charAt(sni);
                if (Character.isUpperCase(ch)) {
                    if (ch != prefix.charAt(pi++)) {
                        return false;
                    }
                }
            }
            return pi == prefix.length();
        }
    
        protected int evaluateItem(long[] value, int i, int j, JavaClass cls, Collection result) {
            if (cls == null) {
                logSNChange(new Long(value[i]), null);
                return 0;
            } else {
                value[j] = value[i];
                if ((!matchName && !camelCase) || (matchName && cls.getSimpleName().startsWith(prefix)) || (camelCase && matchName(cls.getSimpleName()))) {
                    if (((JavaClassImpl) cls).hasComposite()) {
                        result.add(cls);
                    } else {
                        JMManager.getTransactionMutex().invalidateAtCommit(cls);
                    }
                }
                return 1;
            }
        }
    }
    
    private class ClassesByFQNPrefixIterator extends ClassesByPrefixIterator {
        ClassesByFQNPrefixIterator(ClassIndexStorage.QueryItem item) {
            super(item);
        }
        
        protected int evaluateItem(long[] value, int i, int j, JavaClass cls, Collection result) {
            if (cls == null) {
                logFQNChange(new Long(value[i]), null);
                return 0;
            } else {
                if (((JavaClassImpl) cls).hasComposite()) {
                    result.add(cls);
                } else {
                    JMManager.getTransactionMutex().invalidateAtCommit(cls);
                }
                value[j] = value[i];
                return 1;
            }
        }
    }
    
    private class ClassesByFQNPrefixCIIterator extends ClassesByFQNPrefixIterator {
        private final String namePrefix;
        
        ClassesByFQNPrefixCIIterator(ClassIndexStorage.QueryItem item, String namePrefix) {
            super(item);
            this.namePrefix = namePrefix.toUpperCase();
        }
        
        protected boolean accept(ClassIndexStorage.QueryItem item) {
            return (namePrefix.length() == 0) || item.getName().toUpperCase().startsWith(namePrefix);
        }
    }
    
    public boolean doesnotExist(String fqn) {
        return !storage.existsFQN(fqn);
    }
    
    public boolean hasClass(String fqn) {
        return getClassByFqn(fqn) != null;
    }
    
    /**
     * Looks for the classes in class index. Finding mechanism is case
     * sensitive. May return invalid classes.
     *
     * @param  simpleName  simple (short) name of class which will be looked for
     * @return   collection of matching JavaClasses instances
     */
    public Collection getClassesBySimpleName(String simpleName) {
        return getClassesBySimpleName(simpleName, true);
    }
    
    /**
     * Looks for the classes in class index. Finding can be either 
     * case-sensitive or case-insensitive. May return invalid classes.
     *
     * @param  simpleName  simple (short) name of class which will be looked for
     * @param  caseSensitive  use false when you want to ignore case
     * @return   collection of matching JavaClasses instances
     */
    public Collection getClassesBySimpleName(String simpleName, boolean caseSensitive) {
        String simpleUpper = simpleName.toUpperCase();
        long[] value = (long[]) storage.getIDsForName(simpleUpper, true);
        
        List res = new ArrayList();
        
        if (value != null) {
            int j = 0;
            for (int i = 0; i < value.length; i++) {
                JavaClass cls = (JavaClass) rep.getByMofId(makeMofId(value[i]));
                if (cls == null) {
                    logSNChange(new Long(value[i]), null);
                } else {
                    value[j] = value[i];
                    j++;
                    if (!caseSensitive || simpleName.equals(cls.getSimpleName())) {
                        res.add(cls);
                    }
                }
            }
            if (j == 0) {
                storage.setIDsForName(simpleUpper, true, null);
            } else if (j < value.length) {
                long[] newValue = new long[j];
                System.arraycopy(value, 0, newValue, 0, j);
                storage.setIDsForName(simpleUpper, true, newValue);
            }
        }
        return res;
    }
    
    public static boolean hasClass(String fqn,ClassPath classPath) {
        FileObject[] roots = classPath.getRoots();
        // iteruj pres rooty a hledej classu podle FQN
        for (int i = 0; i < roots.length; i++) {
            JavaModelPackage mofPackage = JavaMetamodel.getManager().getJavaExtent(roots[i]);
            // test for mofPackage != null is needed when any other thread 
            // mounts the new filesystem and we do not know about it yet.
            // It is in list of all filesystem roots, but we do not have created
            // package for it yet. In such a case, the index does not contain
            // class and we will return false.
            if (mofPackage!=null && ClassIndex.getIndex(mofPackage).hasClass(fqn))
                return true;
        }
        return false;
    }

    public static JavaClass getClassByFqn(String fqn, ClassPath classPath) {
        FileObject[] roots = classPath.getRoots();
        for (int i = 0; i < roots.length; i++) {
            JavaModelPackage mofPackage = org.netbeans.modules.javacore.internalapi.JavaMetamodel.getManager().getJavaExtent(roots[i]);
            if (mofPackage != null) {
                ClassIndex index = ClassIndex.getIndex(mofPackage);
                if (index != null) {
                    JavaClass cls = index.getClassByFqn(fqn);
                    if (cls != null) {
                        return cls;
                    }
                }
            }
        }
        return null;
    }

    public void setIdentifiers(Resource rsc,int identifiers[]) {
        Arrays.sort(identifiers);
        storage.setIdentifiers(getMofId(rsc), identifiers);
    }
    
    private long getMofId(Object obj) {
        return ((BaseObjectHandler) obj)._getMofId().getSerialNumber();
    }
    
    private MOFID makeMofId(long mofId) {
        return new MOFID(mofId, storageId);
    }
    
    public void removeResource(Resource rsc) {
        long mofId = getMofId(rsc);
        int[] o = storage.removeIdentifiers(mofId);
        changeLogI = logChange(changeLogI, new Long(mofId), o);
    }

    void updateIdentifiersInResource(Resource rsc,String sourceText) {
        FileObject file=JavaMetamodel.getManager().getFileObject(rsc);
        String sourceLevel=SourceLevelQuery.getSourceLevel(file);
        JScanner scanner=Factory.getDefault().getScanner(new StringReader(sourceText),sourceLevel);
        int token;
        int identifiers[] = storage.getIdentifiers(getMofId(rsc));
        List newIds=new ArrayList(10);
        int newIdSize;
        
        try {
            while((token=scanner.yylex())!=0) {
                if (token==ParserTokens.IDENTIFIER) {
                    String text=scanner.yytext();
                    int hash=text.hashCode();

                    if (Arrays.binarySearch(identifiers,hash)<0) {
                        newIds.add(text);
                    }
                }
            }
        }
        catch (IOException ex) {
            ErrorManager.getDefault().notify(ex);
        }
        newIdSize=newIds.size();
        if (newIdSize>0) {
            int ids[]=new int[identifiers.length+newIdSize];
            String newIdArr[]=(String[])newIds.toArray(new String[newIdSize]);
            int i;
            
            for (i=0;i<newIdSize;i++) {
                ids[i]=newIdArr[i].hashCode();
            }
            System.arraycopy(identifiers, 0, ids, newIdSize, identifiers.length);
            setIdentifiers(rsc,ids);
        }
    }
    
    // made public because java/j2seproject/J2SEProjectUtil depends on it
    public Collection findResourcesForIdent(String identifier) {
        long[] ids = storage.getIDsForIdentifier(identifier.hashCode());
        Collection res = new ArrayList(ids.length);
        for (int i = 0; i < ids.length; i++) {
            Object obj = rep.getByMofId(makeMofId(ids[i]));
            if (obj == null) {
                storage.removeIdentifiers(ids[i]);
            } else {
                res.add(obj);
            }
        }
        return res;
    }
        
    public static Resource[] findResourcesForIdentifier(String identifier,boolean includeLibraries) {
        FileObject[] roots;
        List cp = ((JMManager) JavaMetamodel.getManager()).getTransactionMutex().getSearchScope();
        if (cp == null) {
            roots = JavaMetamodel.getManager().getClassPath().getRoots();
        } else {
            if (cp.size() == 1) {
                roots = ((ClassPath) cp.get(0)).getRoots();
            } else {
                roots = ClassPathSupport.createProxyClassPath((ClassPath[]) cp.toArray(new ClassPath[cp.size()])).getRoots();
            }
        }
        // iterate through roots and find class by its FQN
        Collection resources=new ArrayList();
        HashSet visitedRoots = new HashSet((int) (roots.length / .7));
        org.netbeans.modules.javacore.internalapi.JavaMetamodel manager=org.netbeans.modules.javacore.internalapi.JavaMetamodel.getManager();
        boolean isJavaDocTag = identifier.startsWith("@"); // NOI18N
        
        for (int i = 0; i < roots.length; i++) {
            FileObject root = roots[i];
            if (!visitedRoots.add(root)) continue;
            try {
                if (includeLibraries || !(root.getFileSystem() instanceof JarFileSystem)) {
                    JavaModelPackage mofPackage = manager.getJavaExtent(root);
                    // [TODO]: temporary solution to enable IsValueFor association implementation

                    if (mofPackage != null) {
                        if (isJavaDocTag) {
                            ResourceClass resourceClass = mofPackage.getResource();
                            resources.addAll(resourceClass.refAllOfClass());
                        }
                        else {
                            resources.addAll(ClassIndex.getIndex(mofPackage).findResourcesForIdent(identifier));
                        }
                    }
                }
            } catch (FileStateInvalidException ex) {
                ErrorManager.getDefault().notify(ex);
            }
        }
        return (Resource[])resources.toArray(new Resource[resources.size()]);
    }

    public static Resource[] findResourcesForIdentifier(String identifier) {
        return findResourcesForIdentifier(identifier,false);
    }

    public static void updateIdentifiers(Resource rsc,String sourceText) {
        JavaModelPackage mofPackage=(JavaModelPackage)rsc.refOutermostPackage();
        ClassIndex.getIndex(mofPackage).updateIdentifiersInResource(rsc, sourceText);
    }
    
    public static boolean containsIdentifier(Resource rsc, int identifier) {
        JavaModelPackage mofPackage=(JavaModelPackage)rsc.refOutermostPackage();
        ClassIndex index = ClassIndex.getIndex(mofPackage);
        int[] idents = (int[]) index.storage.getIdentifiers(index.getMofId(rsc));

        if (idents==null) {
            JMManager.getLog().log("ids is null for "+rsc.getName());
            return false;
        }
        return Arrays.binarySearch(idents, identifier) >= 0;
    }

    private void addToIndex(boolean isSimpleName, String name, long value) {
        if (isSimpleName) {
            name = name.toUpperCase();
        }
        long[] temp = storage.getIDsForName(name, isSimpleName);
        long[] newValue;
        int len;
        if (temp == null) {
            len = 0;
            newValue = new long[len + 1];
        } else {
            long[] oldValue = temp;
            len = oldValue.length;
            newValue = new long[len + 1];
            for (int i = 0; i < len; i++) {
                if (oldValue[i] == value) {
                    newValue = oldValue;
                    break;
                }
                newValue[i] = oldValue[i];
            }
        }
        if (newValue != temp) {
            newValue[len] = value;
            storage.setIDsForName(name, isSimpleName, newValue);
        }
    }

    private void removeFromIndex(boolean isSimpleName, String name, long value) {
        if (isSimpleName) {
            name = name.toUpperCase();
        }
        Object temp = storage.getIDsForName(name, isSimpleName);
        if (temp != null) {
            long[] oldValue = (long[]) temp;
            int len = oldValue.length;
            long[] newValue = new long[len - 1];
            int i = 0, j = 0;
            for (; i < len && j < len - 1; i++) {
                if (oldValue[i] != value) {
                    newValue[j++] = oldValue[i];
                }
            }

            if (i < len && oldValue[i] != value) {
                if (j < len - 1) {
                    newValue[j] = oldValue[i];
                } else {
                    newValue = oldValue;
                }
            }

            if (oldValue != newValue) {
                if (newValue.length == 0) {
                    newValue = null;
                }
                storage.setIDsForName(name, isSimpleName, newValue);
            }
        }
    }
    
    // Shutdown listener ........................................................
    
    private static class ShutdownL implements NBMDRepositoryImpl.ShutdownListener {
        
        public void shutdown() {
            // RequestProcessor.getDefault().post(this);
            saveAllIndexes();
        }
        
        public void stepFinished() {
        }
    }

    private class CISImpl implements ClassIndexStorage {
        private static final int VERSION = 2;
        private static final String INDEX_SUFFIX=".cdx"; // NOI18N
        
        private final SortedMap fqnMap;
        private final SortedMap simpleMap;
        private Map identifiersMap;
        private final String filename;
        
        private transient boolean isDirty;

        CISImpl(String filename) {
            this.filename = filename;
            fqnMap = new TreeMap();
            simpleMap = new TreeMap();
            identifiersMap = new HashMap();
            isDirty = true;
        }
        
        private QueryItem createNextCIIImpl(Iterator it, int length) throws ConcurrentModificationException {
            while (it.hasNext()) {
                Map.Entry entry = (Map.Entry) it.next();
                if (((String) entry.getKey()).lastIndexOf('.') < length) {
                    return new CIIImpl(it, entry, length);
                }
            }
            return null;
        }
        
        public QueryItem createNextCIISNImpl(Iterator it) throws ConcurrentModificationException {
            if (it.hasNext()) {
                Map.Entry entry = (Map.Entry) it.next();
                return new CIISNImpl(it, entry);
            }
            return null;
        }
        
        private abstract class AbstractCIIImpl implements QueryItem {
            private Iterator it;
            private final Map.Entry entry;
            private QueryItem next ;
            
            AbstractCIIImpl(Iterator it, Map.Entry entry) {
                this.it = it;
                this.entry = entry;
            }

            public final void setIDs(long[] ids) throws ConcurrentModificationException {
                isDirty=true;
                entry.setValue(ids);
                if (ids == null) {
                    if (it == null) {
                        QueryItem item = getNext();
                        while (item != null) item = item.getNext();
                        remove();
                    } else {
                        it.remove();
                    }
                }
            }

            public final String getName() {
                return (String) entry.getKey();
            }

            public final long[] getIDs() {
                return (long[]) entry.getValue();
            }
            
            public final QueryItem getNext() throws ConcurrentModificationException {
                if (it != null) {
                    next = createNext(it);
                    it = null;
                }
                return next;
            }
            
            protected abstract QueryItem createNext(Iterator it) throws ConcurrentModificationException;
            protected abstract void remove();
        }
        
        private class CIIImpl extends AbstractCIIImpl {
            private final int length;

            CIIImpl(Iterator it, Map.Entry entry, int length) {
                super(it, entry);
                this.length = length;
            }

            protected QueryItem createNext(Iterator it) throws ConcurrentModificationException {
                return createNextCIIImpl(it, length);
            }
            
            protected void remove() {
                setIDsForName(getName(), false, null);
            }
        }

        private class CIISNImpl extends AbstractCIIImpl {
            CIISNImpl(Iterator it, Map.Entry entry) {
                super(it, entry);
            }

            protected QueryItem createNext(Iterator it) throws ConcurrentModificationException {
                return createNextCIISNImpl(it);
            }
            
            protected void remove() {
                setIDsForName(getName(), true, null);
            }
        }

        public QueryItem getFirstForNamePrefix(String prefix, boolean isSimpleName) {
            SortedMap map;
            if (prefix == null || prefix.length() == 0) {
                map = isSimpleName ? simpleMap : fqnMap;
            } else {
                map = (isSimpleName ? simpleMap : fqnMap).subMap(prefix, prefix + '\uffff');
            }
            if (isSimpleName) {
                return createNextCIISNImpl(map.entrySet().iterator());
            } else {
                int length = prefix.length();
                return createNextCIIImpl(map.entrySet().iterator(), length);
            }
        }
        
        public boolean existsFQN(String fqn) {
            return fqnMap.get(fqn) != null;
        }
        
        public void setIdentifiers(long resourceId, int[] hashCodes) {
            assert hashCodes != null;
            isDirty=true;
            identifiersMap.put(new Long(resourceId), hashCodes);
        }
        
        public boolean mount() {
            long t = 0;
            try {
                if (JMManager.PERF_DEBUG) {
                    t = System.currentTimeMillis();
                }
                
                File indexFile = new File(filename.concat(INDEX_SUFFIX));
                InputStream indexStream = new BufferedInputStream(new FileInputStream(indexFile), 65536);
                
                try {
                    read(indexStream);
                } finally {
                    indexStream.close();
                }                
                isDirty=false;
                return true;
            } catch (Exception ex) {
                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, ex);
                return false;
            } finally {
                if (JMManager.PERF_DEBUG) {
                    System.err.println("loading index " + filename + " took: " + (System.currentTimeMillis() - t) + "ms");
                }
            }
        }
        
        public void unmount() {
            if (!isDirty) return;
            
            if (JMManager.PERF_DEBUG) {
                System.err.println("saving index " + filename);
            }
            
            rep.beginTrans(false);
            try {
                FileOutputStream fos=new FileOutputStream(filename.concat(INDEX_SUFFIX));
                BufferedOutputStream indexStream=new BufferedOutputStream(fos, 65536);
                try {
                    write(indexStream);
                } finally {
                    indexStream.close();
                }
                isDirty=false;
            } catch (Exception ex) {
                JMManager.getLog().notify(ErrorManager.INFORMATIONAL, ex);
            } finally {
                rep.endTrans();
            }
        }

        private void read(InputStream indexStream) throws IOException {
            if (VERSION != IOUtils.readInt(indexStream)) return;
            if (!storageId.equals(IOUtils.readString(indexStream))) return;
            readMap(indexStream, fqnMap);
            readMap(indexStream, simpleMap);

            int count = IOUtils.readInt(indexStream);
            identifiersMap = new HashMap(count * 4 / 3 + 1);
            for (int i = 0; i < count; i++) {
                Long key = new Long(IOUtils.readLong(indexStream));
                int count2 = IOUtils.readInt(indexStream);
                int[] value = new int[count2];
                for (int j = 0; j < count2; j++) {
                    value[j] = IOUtils.readInt(indexStream);
                }
                identifiersMap.put(key, value);
            }
            lastSaved = IOUtils.readLong(indexStream);
        }

        private void write(OutputStream indexStream) throws IOException {
            if (JMManager.PERF_DEBUG) System.err.println("fqnMap: " + fqnMap.size() + " records; identifiersMap: " + identifiersMap.size() + "records");
            IOUtils.writeInt(indexStream, VERSION);
            IOUtils.writeString(indexStream, storageId);
            writeMap(indexStream, fqnMap);
            writeMap(indexStream, simpleMap);

            IOUtils.writeInt(indexStream, identifiersMap.size());
            for (Iterator it = identifiersMap.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                IOUtils.writeLong(indexStream, ((Long) entry.getKey()).longValue());

                int[] array = (int[]) entry.getValue();
                IOUtils.writeInt(indexStream, array.length);
                for (int i = 0; i < array.length; i++) {
                    IOUtils.writeInt(indexStream, array[i]);
                }
            }

            IOUtils.writeLong(indexStream, System.currentTimeMillis());
        }

        private void readMap(InputStream indexStream, Map map) throws IOException {
            map.putAll(new StreamBasedSortedMap(indexStream));
        }

        private void writeMap(OutputStream indexStream, Map map) throws IOException {
            IOUtils.writeInt(indexStream, map.size());
            for (Iterator it = map.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                IOUtils.writeString(indexStream, (String) entry.getKey());

                long[] array = (long[]) entry.getValue();
                IOUtils.writeInt(indexStream, array.length);
                for (int i = 0; i < array.length; i++) {
                    IOUtils.writeLong(indexStream, array[i]);
                }
            }
        }

        public long[] getIDsForName(String name, boolean isSimpleName) {
            return (long[]) (isSimpleName ? simpleMap : fqnMap).get(name);
        }

        public long[] getIDsForIdentifier(int hashCode) {
            Iterator it = identifiersMap.entrySet().iterator();
            Collection result = new ArrayList(50);
            while(it.hasNext()) {
                Map.Entry en = (Map.Entry) it.next();
                int ids[] = (int[]) en.getValue();
                if (ids == null) {
                    isDirty = true;
                    it.remove();
                } else if (Arrays.binarySearch(ids, hashCode) >= 0) {
                    result.add(en.getKey());
                }
            }
            long[] res = new long[result.size()];
            it = result.iterator();
            for (int i = 0; it.hasNext(); i++) {
                res[i] = ((Long) it.next()).longValue();
            }
            return res;
        }

        public void setIDsForName(String name, boolean isSimpleName, long[] classIDs) {
            isDirty=true;
            Map map = isSimpleName ? simpleMap : fqnMap;
            if (classIDs == null || classIDs.length == 0) {
                map.remove(name);
            } else {
                map.put(name, classIDs);
            }
        }

        public int[] getIdentifiers(long resourceId) {
            return (int[]) identifiersMap.get(new Long(resourceId));
        }

        public int[] removeIdentifiers(long resourceId) {
            isDirty=true;
            return (int[]) identifiersMap.remove(new Long(resourceId));
        }
    }
    
    /**
     * Special implementation of SortedMap based on archived stream.
     * This class is used to initialize TreeMap from sorted data when deserializing ClassIndex
     */
    private class StreamBasedSortedMap implements SortedMap {
        private int size;
        InputStream stream;
        
        private StreamBasedSortedMap(InputStream indexStream) throws IOException {
            stream = indexStream;
            size = IOUtils.readInt(stream);
        }
        
        public Comparator comparator() {
            return null;
        }

        public SortedMap subMap(Object fromKey, Object toKey) {
            throw new UnsupportedOperationException();
        }

        public SortedMap headMap(Object toKey) {
            throw new UnsupportedOperationException();
        }

        public SortedMap tailMap(Object fromKey) {
            throw new UnsupportedOperationException();
        }

        public Object firstKey() {
            throw new UnsupportedOperationException();
        }

        public Object lastKey() {
            throw new UnsupportedOperationException();
        }

        public int size() {
            return size;
        }

        public boolean isEmpty() {
            return size == 0;
        }

        public boolean containsKey(Object key) {
            throw new UnsupportedOperationException();
        }

        public boolean containsValue(Object value) {
            throw new UnsupportedOperationException();
        }

        public Object get(Object key) {
            throw new UnsupportedOperationException();
        }

        public Object put(Object key, Object value) {
            throw new UnsupportedOperationException();
        }

        public Object remove(Object key) {
            throw new UnsupportedOperationException();
        }

        public void putAll(Map t) {
            throw new UnsupportedOperationException();
        }

        public void clear() {
            throw new UnsupportedOperationException();
        }

        public Set keySet() {
            throw new UnsupportedOperationException();
        }

        public Collection values() {
            throw new UnsupportedOperationException();
        }

        public Set entrySet() {
            return new EntrySet();
        }
        
        private class EntrySet extends AbstractSet {
            public Iterator iterator() {
                return new EntrySetIterator();
            }

            public int size() {
                return size;
            }
            
            private class EntrySetIterator implements Iterator, Map.Entry {
                private int index;
                private String key;
                private long[] value;
                                
                public boolean hasNext() {
                    return index<size;
                }

                public Object next() {
                    try {
                        key = IOUtils.readString(stream);
                        int count = IOUtils.readInt(stream);
                        value = new long[count];
                        for (int j = 0; j < count; j++) {
                            value[j] = IOUtils.readLong(stream);
                        }
                        index++;
                    } catch (IOException ex) {
                        ex.printStackTrace();
                        index = size;
                    }
                    return this;
                }

                public void remove() {
                    throw new UnsupportedOperationException();
                }

                public Object getKey() {
                    return key;
                }

                public Object getValue() {
                    return value;
                }

                public Object setValue(Object value) {
                    throw new UnsupportedOperationException();
                }
                
            }
        }
    }
}
