/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.hints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.editor.hints.spi.ChangeInfo;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;

/**
 *
 * @author Tomas Hurka
 */
class CreateMethodJavaHint extends CreateElementJavaHint {

    private ClassDefinition javaClass;
    private static final Integer zero=new Integer(0);
    
    CreateMethodJavaHint(NamedElement symbol,ClassDefinition cls) {
        super(symbol);
        javaClass=cls;
        if (cls instanceof ParameterizedType) {
            javaClass=((ParameterizedType)cls).getDefinition();
        }
        String signature=name+'('+getParameters()+')';
        hintText=NbBundle.getMessage(CreateMethodJavaHint.class, "MSG_CreateMethod", signature, cls.getName());
    }
    
    public ChangeInfo implement() {
        boolean fail = true;
        Method newMethod;
        boolean bothInOneFile;
        JavaModel.getJavaRepository().beginTrans(true);
        try {
            JavaModel.setClassPath(unresolvedSymbol.getResource());
            Resource classRes = javaClass.getResource();
            Resource unresRes = unresolvedSymbol.getResource();
            bothInOneFile = classRes.equals(unresRes);
            Type fieldType=DeclarationInfo.computeType(unresolvedSymbol);
            int modifiers=computeAccessModifiers(javaClass, unresRes);
            
            modifiers|=computeStaticModifier(unresolvedSymbol);
            newMethod=createMethod(modifiers,fieldType);
            fail = false;
        } finally {
            JavaModel.getJavaRepository().endTrans(fail);
        }
        ChangeInfo ch = PositionUtils.selectFirstStatement(newMethod);
        if (!bothInOneFile) addModified();
        
        return ch;
    }
    
    private Method createMethod(int modifiers,Type type) {
        JavaModelPackage pac=(JavaModelPackage)javaClass.refImmediatePackage();
        TypeReference typeRef=JavaModelUtil.resolveImportsForType(javaClass,type);
        List pars=createParams();
        String bodyText=createBody(type);
        Method method = pac.getMethod().createMethod(name,null,modifiers,null,null,null,bodyText,null,pars,null,typeRef,0);
        List features=javaClass.getContents(); 
        int index = PositionUtils.getProposedMethodIndex(features);

        features.add(index,method);
        return method;
    }

    private String createBody(Type type) {
        String initVal;
        
        if (type instanceof PrimitiveType) {
            PrimitiveTypeKind primitiveType = ((PrimitiveType) type).getKind();
            if (PrimitiveTypeKindEnum.BOOLEAN.equals(primitiveType)) {
                initVal = "false"; // NOI18N
            } else if (PrimitiveTypeKindEnum.CHAR.equals(primitiveType)) {
                initVal = "'\\0'"; // NOI18N
            } else if (PrimitiveTypeKindEnum.VOID.equals(primitiveType)) {
                return "throw new UnsupportedOperationException(\"Not yet implemented\");"; // NOI18N
            } else {
                initVal = "0"; // NOI18N
            }
        } else if (type instanceof ClassDefinition) {
            initVal = "null"; // NOI18N
        } else {
            throw new IllegalArgumentException("Type "+type.getClass()); // NOI18N
        }
        return "return ".concat(initVal).concat(";"); // NOI18N
    }

    private List createParams() {
        JavaModelPackage pac=(JavaModelPackage)javaClass.refImmediatePackage();
        ParameterClass parProxy=pac.getParameter();
        Invocation inv=(Invocation)unresolvedSymbol;
        Expression parExprs[]=(Expression[])inv.getParameters().toArray(new Expression[0]);
        List parTypes=new ArrayList(parExprs.length);
        Map names=new HashMap();
        int i;
        
        for (i=0;i<parExprs.length;i++) {
            Type pType=parExprs[i].getType();
            TypeReference typeRef;
            String name;
            Integer count;
            Parameter par;
            
            if (pType==null) {
                pType=pac.getType().resolve("java.lang.Object"); // NOI18N
            } else {
                pType=getNamedType(pType);
            }
            typeRef=JavaModelUtil.resolveImportsForType(javaClass,pType);
            name=getName(parExprs[i],pType);
            count=(Integer)names.get(name);
            if (count!=null) {
                names.put(name,new Integer(count.intValue()+1));
                name=name.concat(count.toString());
            } else {
                names.put(name,zero);
            }
            par=parProxy.createParameter(name,null,false,typeRef,0,false);
            parTypes.add(par);
        }
        return parTypes;
    }

    private static String getName(final Expression parExpr,final Type pType) {
        if (parExpr instanceof VariableAccess) {
            VariableAccess var=(VariableAccess)parExpr;

            return var.getName();
        }
        return getName(pType);
    }

    private static String getName(final Type pType) {
        if (pType instanceof PrimitiveType) {
            return pType.getName().substring(0,1);
        } else if (pType instanceof JavaClass) {
            String name=((JavaClass)pType).getSimpleName();
            
            String loweredName = name.substring(0,1).toLowerCase().concat(name.substring(1));
            return Utilities.isJavaIdentifier(loweredName) ? loweredName : 'a' + name;
        } else if (pType instanceof Array) {
            return getName(((Array)pType).getType());
        } else {
            throw new IllegalArgumentException("Uknow type "+pType.getClass()); // NOI18N
        }
    }

    private String getParameters() {
        Invocation inv=(Invocation)unresolvedSymbol;
        Expression parExprs[]=(Expression[])inv.getParameters().toArray(new Expression[0]);
        
        if (parExprs.length>0) {
            StringBuffer text=new StringBuffer();
            int i;

            for (i=0;i<parExprs.length;i++) {
                Type pType=parExprs[i].getType();
                String typeName;

                if (pType!=null) {
                    typeName=getNamedType(pType).getName();
                } else {
                    typeName="java.lang.Object"; // NOI18N
                }
                text.append(typeName).append(',');
            }
            return text.substring(0,text.length()-1); 
        }
        return "";
    }

    private static Type getNamedType(Type pType) {
        if (pType instanceof ClassDefinition && pType.getName()==null)  {   // anonymous class
            ClassDefinition anonClass=(ClassDefinition)pType; 
            JavaClass[] ifcs=(JavaClass[])anonClass.getInterfaces().toArray(new JavaClass[1]);

            if (ifcs[0]!=null) {
                pType=ifcs[0];
            } else {
                pType=anonClass.getSuperClass();
            }
        }
        return pType;
    }

}
