/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import java.awt.Dialog;
import java.awt.event.ActionEvent;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import javax.swing.JDialog;
import javax.swing.JMenuItem;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.editor.BaseAction;
import org.netbeans.editor.BaseDocument;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.JavaPackage;
import org.netbeans.modules.editor.MainMenuAction;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.awt.StatusDisplayer;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;
public class JavaFixAllImports extends BaseAction {
    static final long serialVersionUID = 6020950800832542269L;
    private static final int HAS_SYNTAX_ERROR = 16; // [PENDING] constant from ResourceImpl
    private JMIUtils jmiUtils;

    public JavaFixAllImports() {
        super(JavaKit.fixImportsAction);
        putValue(SHORT_DESCRIPTION, NbBundle.getBundle(JavaKit.class).getString("fix-imports")); // NOI18N
        putValue("helpID", JavaFixAllImports.class.getName()); // NOI18N
    }
    
    private Dialog createProgressDialog(FixImportsProgressPanel progressPanel) {
        String title = NbBundle.getMessage(FixImportsProgressPanel.class, "FixImportsProgressTitle_Lbl"); // NOI18N
        DialogDescriptor desc = new DialogDescriptor(progressPanel, title, true, new Object[]{}, null, 0, null, null);
        desc.setClosingOptions(new Object[]{});
        Dialog dialog = DialogDisplayer.getDefault().createDialog(desc);
        ((JDialog) dialog).setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        dialog.validate();
        return dialog;
    }
    
    public void actionPerformed(final ActionEvent evt, final JTextComponent t) {
        Runnable run = new Runnable() {
            public void run() {
                if (t == null)
                    return;
                final JTextComponent target = t;
                final FixDuplicateImportStmts duplImportsPanel = new FixDuplicateImportStmts();
                final DialogDescriptor descr = new DialogDescriptor(duplImportsPanel,
                    NbBundle.getMessage(JavaFixAllImports.class, "FixDuplicateImports_dlgTitle"));
                descr.setOptionType(DialogDescriptor.OK_CANCEL_OPTION);
                final Dialog duplImportsDialog  = org.openide.DialogDisplayer.getDefault().createDialog(descr);
                final FixImportsProgressPanel progress = new FixImportsProgressPanel();
                final Dialog progressDialog = createProgressDialog(progress);
                
                RequestProcessor.getDefault().post(new Runnable() {
                    public void run() {
                        addAllNeededImports(target.getText(), (BaseDocument)target.getDocument(), target, progress, progressDialog, duplImportsPanel, duplImportsDialog, descr);
                    }
                });
            }
        };
        JavaMetamodel.getManager().invokeAfterScanFinished(run, NbBundle.getMessage(JavaFixAllImports.class, "fix-imports"));
    }
    
    public void addAllNeededImports(String txt, BaseDocument doc, JTextComponent target, final FixImportsProgressPanel progress,
            final Dialog progressDialog, FixDuplicateImportStmts duplImportsPanel, Dialog duplImportsDialog, DialogDescriptor descr) {
        Object sdp = doc.getProperty(Document.StreamDescriptionProperty);
        FileObject fo = null;
        if (sdp instanceof FileObject) {
            fo = (FileObject)sdp;
        } else if (sdp instanceof DataObject) {
            fo = ((DataObject)sdp).getPrimaryFile();
        } else {
            return;
        }
        
        /*
        mainWindow = WindowManager.getDefault().getMainWindow();
        editCursor = target.getCursor();
        mainCursor = mainWindow.getCursor();
        Cursor waitCursor = Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR);
        target.setCursor(waitCursor);
        mainWindow.setCursor(waitCursor);
         */
        
        ArrayList importsToAdd = null, importsToRemove = null;
        HashSet unresolved = null;
        HashMap ambigs = new HashMap();
        HashSet usedImports = new HashSet();
        Resource resource = null;
        boolean failed = true;
        boolean diagPassed = true;
        boolean hasSyntaxError = false;

        jmiUtils = JMIUtils.get(doc);
        jmiUtils.beginTrans(true);
        try {
            resource = JavaModel.getResource(fo);
            if (resource == null){
                org.netbeans.editor.Utilities.annotateLoggable(new NullPointerException("Resource is null for FileObject:"+fo)); //NOI18N
                failed = false;
                return;
            }
            hasSyntaxError = (resource.getStatus() & HAS_SYNTAX_ERROR) > 0;

            if (!hasSyntaxError) {
                StatusDisplayer.getDefault().setStatusText(NbBundle.getMessage(JavaFixAllImports.class, "FixingImports_Lbl")); // NOI18N
                HashSet clsNames = new HashSet();
                HashSet resolvedClsNames = new HashSet();
                HashSet javadocTags = new HashSet();
                findPotentialClassNames(resource, clsNames, resolvedClsNames, javadocTags, progress, progressDialog);
                HashMap importedPackages = new HashMap();
                HashMap importedClasses = new HashMap();

                for (Iterator iter = resource.getImports().iterator(); iter.hasNext();) {
                    Import imp = (Import) iter.next();
                    if (imp.isStatic())
                        continue;
                    String id = idToName(imp.getIdentifier());
                    if (imp.getImportedNamespace() instanceof JavaPackage) {
                        importedPackages.put(id, imp);
                    } else {
                        if (imp.isOnDemand()) {
                            id = id + ".*"; // NOI18N
                        }
                        importedClasses.put(id, imp);
                    }
                }

                importsToAdd = new ArrayList();
                importsToRemove = new ArrayList();
                unresolved = new HashSet();
            
                Import imp;
                FileObject[] cpRoots = JavaMetamodel.getManager().getClassPath().getRoots();
                ClassIndex[] cis = new ClassIndex[cpRoots.length];
                for (int i = 0; i < cpRoots.length; i++) {
                    cis[i] = ClassIndex.getIndex(JavaModel.getJavaExtent(cpRoots[i]));
                }

                // detect imports corresponding to resolved types
                for (Iterator iter = resolvedClsNames.iterator(); iter.hasNext();) {
                    JavaClass jc = (JavaClass)iter.next();
                    imp = (Import)importedClasses.get(jc.getName());
                    if (imp != null) {
                        usedImports.add(imp);
                    } else {
                        Object comp = jc.refImmediateComposite();
                        if (comp instanceof Resource) {
                            imp = (Import)importedPackages.get(((Resource)comp).getPackageName());
                            if (imp != null) {
                                usedImports.add(imp);
                            }
                        } else if (comp instanceof JavaClass) {
                            imp = (Import)importedClasses.get(((JavaClass)comp).getName() + ".*"); // NOI18N
                            if (imp != null) {
                                usedImports.add(imp);
                            }
                        }
                    }
                } // for

                Iterator classifiers = resource.getClassifiers().iterator();
                JavaClass from = classifiers.hasNext() ? (JavaClass)classifiers.next() : null;

                boolean finished = false;
                boolean processingJavadocTags = false;
                Iterator iter = clsNames.iterator();
                
                Outer:
                while (!finished) {
                    if (!iter.hasNext()) {
                        if (processingJavadocTags) {
                            finished = true;
                            break;
                        } else {
                            iter = javadocTags.iterator();
                            processingJavadocTags = true;
                            continue;
                        }
                    }
                    String simpleName = (String)iter.next();

                    Collection ret = new ArrayList();
                    Set fullNames = new HashSet();
                    for (int i = 0; i < cis.length; i++) {
                        if (cis[i] == null)
                            continue;
                        Collection res = cis[i].getClassesBySimpleName(simpleName);
                        if (res != null) {
                            for (Iterator tmpIt = res.iterator(); tmpIt.hasNext();) {
                                JavaClass javaClass = (JavaClass) tmpIt.next();
                                String fqn = javaClass.getName();
                                if (!fullNames.contains(fqn) && jmiUtils.isAccessible(javaClass, from)) {
                                    ret.add(javaClass);
                                    fullNames.add(fqn);
                                }
                            }
                        }
                    }
                    if (ret.size() == 0) {
                        if (!processingJavadocTags) {
                            unresolved.add(simpleName);
                        }
                        continue;
                    }

                    for (Iterator iter2 = ret.iterator(); iter2.hasNext();) {
                        JavaClass cls = (JavaClass)iter2.next();
                        Object comp = cls.refImmediateComposite();
                        String pkg = null;
                        if (comp instanceof Resource) {
                            pkg = ((Resource)comp).getPackageName();
                        }
                        if (pkg != null) {
                            if (pkg.equals("java.lang")) { // NOI18N
                                continue Outer;
                            }
                            imp = (Import) importedPackages.get(pkg);
                            if (imp != null) { // NOI18N
                                usedImports.add(imp);
                                continue Outer;
                            }
                        }

                        imp = (Import) importedClasses.get(cls.getName());
                        if (imp != null) {
                            usedImports.add(imp);
                            continue Outer;
                        }
                        if (comp instanceof JavaClass) {
                            imp = (Import)importedClasses.get(((JavaClass)comp).getName() + ".*"); // NOI18N
                            if (imp != null) {
                                usedImports.add(imp);
                                continue Outer;
                            }
                        }
                    }

                    if (!processingJavadocTags) {
                        int sz = ret.size();
                        if (sz > 1)
                            ambigs.put(simpleName, ret);
                        else if (sz > 0)
                            importsToAdd.add(((JavaClass)ret.iterator().next()).getName());
                    }
                } // for

                /*
                target.setCursor(editCursor);
                mainWindow.setCursor(mainCursor);
                 */
                failed = false;
            } else {
                failed = false;
            }
        } finally {
            SwingUtilities.invokeLater(
                new Runnable() {
                    public void run() {
                        progress.stop();
                        // #54585 - delaying AWT thread for 50ms to avoid focus problems
                        try {
                            Thread.currentThread().sleep(50);
                        } catch (InterruptedException e) {
                            // ignored
                        }
                        progressDialog.setVisible(false);
                        progressDialog.dispose();
                    }
                }
            );
            jmiUtils.endTrans(failed);
        }
                
        String[] selections = null;
        boolean ambigsWasCalled = false;
        if (ambigs != null && ambigs.size() > 0) {
            selections = userResolvesAmbiguities(ambigs, duplImportsPanel, duplImportsDialog, descr);
            ambigsWasCalled = true;
        }

        if (!hasSyntaxError) {
            jmiUtils.beginTrans(true);
            try {
                failed = true;
                if (ambigsWasCalled) {
                    if (selections != null) {
                        for (int i = 0; i < selections.length; i++)
                            importsToAdd.add(selections[i]);
                    } else {
                        // user Cancelled the "Add Imports" Action
                        diagPassed = false;
                    }
                }
                if (diagPassed) {
                    changeImports(importsToAdd, importsToRemove, usedImports, resource);
                }
                failed = false;
            } finally {
                /*
                target.setCursor(editCursor);
                mainWindow.setCursor(mainCursor);
                 */
                jmiUtils.endTrans(failed);
            }
        }

        if (hasSyntaxError) {
            String msg = NbBundle.getMessage(JavaFixAllImports.class, "SourceContainsSyntaxErrors_Lbl"); // NOI18N
            StatusDisplayer.getDefault().setStatusText(msg);
        } else if (diagPassed && unresolved.size() > 0) {
            showInformational(unresolved);
            StatusDisplayer.getDefault().setStatusText(""); // NOI18N
        } else if (diagPassed) {
            String msg = null;
            if (importsToAdd.size() > 0 || importsToRemove.size() > 0) {
                msg = NbBundle.getMessage(JavaFixAllImports.class, "AllImportsWereFixed_Lbl"); // NOI18N
            } else {
                msg = NbBundle.getMessage(JavaFixAllImports.class, "NoImportsToFix_Lbl"); // NOI18N
            }
            StatusDisplayer.getDefault().setStatusText(msg);
        } else {
            StatusDisplayer.getDefault().setStatusText(""); // NOI18N
        }
    }
    
    // Present to the user all the ambiguous imports and get their repsonse back
    private static String[] userResolvesAmbiguities(HashMap ambigs, final FixDuplicateImportStmts duplImportsPanel, final Dialog duplImportsDialog, final DialogDescriptor descr) {
        int numberOfAmbigs = ambigs.size();
        
        // For now, we simply use the first element as the default. We could be smarter in the future
        int cnt = 0;
        final String[] simpleNames = new String[numberOfAmbigs];
        final String[][] choices = new String[numberOfAmbigs][];
        final String[] defaults = new String[numberOfAmbigs];
        int [] defaultWeights = new int[numberOfAmbigs];

        for (Iterator ambigIter = ambigs.keySet().iterator(); ambigIter.hasNext(); cnt++) {
            String simpleName = (String)ambigIter.next();
            List l = (List)ambigs.get(simpleName);
            
            simpleNames[cnt] = simpleName;
            String[] curChoice = choices[cnt] = new String[l.size()];
            defaultWeights[cnt] = -1;

            for (int k = 0; k < l.size(); k++) {
                JavaClass cls = (JavaClass)l.get(k);
                String fullName = cls.getName();
                curChoice[k] = fullName;
                int weight = JMIUtils.getDefaultSelectionWeight(cls);
                if (weight > defaultWeights[cnt]) {
                    defaults[cnt] = fullName;
                    defaultWeights[cnt] = weight;
                }
            }
            Arrays.sort(curChoice);
        }
        
        final Object[] result = new Object[1];
        
        try {
            SwingUtilities.invokeAndWait(new Runnable () {
                public void run() {
                    duplImportsPanel.initPanel(simpleNames, choices, defaults);
                    duplImportsDialog.pack();
                    duplImportsDialog.setVisible(true);
                    Object o = descr.getValue();
                    if (o == DialogDescriptor.OK_OPTION) {
                        result[0] = duplImportsPanel.getSelections();
                    } else { // Cancel pressed
                        result[0] = null;
                    }
                    duplImportsDialog.setVisible(false);
                    duplImportsDialog.dispose();
                }
            });
        } catch (InterruptedException e) {
        } catch (InvocationTargetException e) {
        }
        return (String[])result[0];
    }

    private static void showInformational(HashSet unresolved) {
        StringBuffer sb = new StringBuffer(NbBundle.getMessage(JavaFixAllImports.class, "NoImportsFound_Lbl"));
        sb.append('\n');
        for (Iterator iter = unresolved.iterator(); iter.hasNext();) {
            sb.append("     - ");    // NOI18N
            sb.append((String)iter.next());
            sb.append('\n');    // NOI18N
        }
        
        NotifyDescriptor d = new NotifyDescriptor.Message(sb.toString());
        DialogDisplayer.getDefault().notify(d);
    }
    
    private static void changeImports(ArrayList importsToAdd, List importsToRemove, Set usedImports, Resource resource) {
        ListIterator iter;
        Object[] importsArray = importsToAdd.toArray();
        Arrays.sort(importsArray,
        new Comparator() {
            public int compare(Object o1, Object o2) {
                return ((String)o1).compareTo((String)o2);
            }
        });
        String thisPkgName = resource.getPackageName();
        List imports = resource.getImports();
        for (iter = imports.listIterator(); iter.hasNext();) {
            Import imp = (Import)iter.next();
            if (imp.isStatic())
                continue;
            
            NamedElement impNs = imp.getImportedNamespace();
            if (impNs instanceof UnresolvedClass) {
                // do not remove imports of unresolved classes and packages (#63366)
                continue;
            }
            
            if (!usedImports.contains(imp)) {
                importsToRemove.add(imp);
                iter.remove();
                imp.refDelete();
            } else {
                String pkgName = null;
                if (!imp.isOnDemand()) {
                    JavaClass jc = (JavaClass) impNs;
                    if (!jc.isInner()) {
                        Resource res = jc.getResource();
                        if (res != null)
                            pkgName = res.getPackageName();
                    }
                } else {
                    Object obj = impNs;
                    if (obj instanceof JavaPackage) {
                        pkgName = ((JavaPackage) obj).getName();
                    }
                }
                if (pkgName != null && (pkgName.equals("java.lang") || pkgName.equals(thisPkgName))) { // NOI18N
                    importsToRemove.add(imp);
                    iter.remove();
                    imp.refDelete();
                }
            }
        }
        
        if (importsArray.length == 0)
            return;
        ImportClass proxy = ((JavaModelPackage)resource.refOutermostPackage()).getImport();
        Import imp = null;
        iter = imports.listIterator(imports.size());
        
        String currName = null;
        if (iter.hasPrevious()) {
            currName = ((Import)iter.previous()).getName();
            iter.next();
        }
        for (int i = importsArray.length - 1; i >=0; i--) {
            String impName = (String)importsArray[i];
            imp = proxy.createImport(impName, null, false, false);
            while (currName != null && (currName.compareTo(impName) > 0)) {
                iter.previous();
                if (iter.hasPrevious()) {
                    currName = ((Import)iter.previous()).getName();
                    iter.next();
                } else {
                    currName = null;
                }
            }
            iter.add(imp);
            iter.previous();
        }
    }
    
    // ...........................................................................
    
    private static void idToName(StringBuffer buffer, MultipartId id) {
        MultipartId parent = id.getParent();
        if (parent != null) {
            idToName(buffer, parent);
            buffer.append('.');
        }
        buffer.append(id.getName());
    }
    
    private static String idToName(MultipartId id) {
        StringBuffer buffer = new StringBuffer();
        idToName(buffer, id);
        return buffer.toString();
    }
    
    private void findPotentialClassNames(Set set, Set resolved, Set javadocTags, Element elem, Set checkedElements, int level, final FixImportsProgressPanel progress) {
        Iterator iterator;
        level++;
        if (elem instanceof ArrayReference) {
            elem = ((ArrayReference)elem).getParent();
        }
        if (elem instanceof MultipartId) {
            List typeArgs = new ArrayList();
            MultipartId id = (MultipartId)elem;
            typeArgs.addAll(id.getTypeArguments());
            while (id.getParent() != null) {
                id = id.getParent();
                typeArgs.addAll(id.getTypeArguments());
            }
            NamedElement namedElem = id.getElement();
            if (namedElem instanceof UnresolvedClass) {
                set.add(idToName(id));
            } else if (namedElem instanceof JavaClass) {
                resolved.add(namedElem);
            }
            iterator = typeArgs.iterator();
        } else {
            iterator = elem.getChildren().iterator();
        }
        
        // search for javadoc tags that can contain references to types
        if (elem instanceof ClassMember) {
            JavaDoc javaDoc = ((ClassMember) elem).getJavadoc();
            if (javaDoc != null) {
                List values = new ArrayList();
                String javaDocText = javaDoc.getText();
                if (javaDocText != null) {
                    addTagValues(values, javaDocText);
                }
                for (Iterator iter = javaDoc.getTags().iterator(); iter.hasNext(); ) {
                    TagValue tag = (TagValue) iter.next();
                    String tagName = tag.getDefinition().getName();
                    String tagValue = tag.getValue();
                    addTagValues(values, tagValue);
                    if (!"@see".equals(tagName) && !"@link".equals(tagName) && !"@linkplane".equals(tagName) && // NOI18N
                            !"@exception".equals(tagName) && !"@throws".equals(tagName) && !"@value".equals(tagName)) { // NOI18N
                        continue;
                    }
                    if (tagValue != null) {
                        values.add(tagValue);
                    }
                }
                for (Iterator iter = values.iterator(); iter.hasNext();) {
                    String tagValue = (String)iter.next();
                    tagValue = tagValue.trim();
                    int length = tagValue.length();
                    for (int x = 0; x < length; x++) {
                        if (Character.isWhitespace(tagValue.charAt(x))) {
                            tagValue = tagValue.substring(0, x);
                            break;
                        }
                    }
                    if (tagValue.length() == 0) {
                        continue;
                    }
                    
                    int index = tagValue.indexOf('#');
                    int end = index > -1 ? index : tagValue.length();
                    index = tagValue.indexOf('.');
                    if (index > -1 && index < end)
                        end = index;
                    if (end > 0) {
                        tagValue = tagValue.substring(0, end);
                        if (org.openide.util.Utilities.isJavaIdentifier(tagValue)) {
                            javadocTags.add(tagValue);
                        }
                    }
                } // for
            } // if
        } // if
        
        for (; iterator.hasNext();) {
            Element usedElem = (Element)iterator.next();
            if (!checkedElements.contains(usedElem)) {
                checkedElements.add(usedElem);
                findPotentialClassNames(set, resolved, javadocTags, usedElem, checkedElements, level, progress);
            }
        }
        if (level == 2) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    progress.step();
                }
            });
        }
    }
    
    private void addTagValues(List values, String text) {
        int length = text.length();
        int index = 0;
        while (index >= 0 && index < length) {
            index = text.indexOf("@link", index); // NOI18N
            if (index > -1) {
                index += 5;
                if (index < length && Character.isWhitespace(text.charAt(index))) {
                    // do nothing
                } else if (index + 5 < length && "plane".equals(text.substring(index, index + 5)) && // NOI18N
                        Character.isWhitespace(text.charAt(index + 5))) {
                    index+= 5;
                } else {
                    continue;
                }
                int i = text.indexOf('}', index);
                if (i > -1) {
                    values.add(text.substring(index, i));
                }
            }
        } // while
    }
    
    private void findPotentialClassNames(Resource resource, Set unresolved, Set resolved, Set javadocTags, final FixImportsProgressPanel progress, final Dialog progressDialog) {
        HashSet checkedElements = new HashSet();
        Iterator iter = resource.getClassifiers().iterator();
        int count = 0;
        while (iter.hasNext()) {
            Element elem = (Element) iter.next();
            count += elem.getChildren().size();
        }
        final int num = count + 1;
        SwingUtilities.invokeLater(
            new Runnable() {
                public void run() {
                    progress.start(num);
                    progressDialog.setVisible(true);
                }
            }
        );
        iter = resource.getClassifiers().iterator();
        while (iter.hasNext()) {
            findPotentialClassNames(unresolved, resolved, javadocTags, (Element)iter.next(), checkedElements, 0, progress);
        }
    }

    /** Fix Imports action in Source main menu, wrapper for JavaFixAllImports action
     */ 
    public static final class MainMenuWrapper extends MainMenuAction {
        
        private JMenuItem menuItem;

        public MainMenuWrapper () {
            super();
            menuItem = new JMenuItem(getMenuItemText());
            setMenu();
        }
        
        protected String getMenuItemText () {
            return NbBundle.getBundle(MainMenuWrapper.class).getString(
                "fix_imports_main_menu_item"); //NOI18N
        }

        public JMenuItem getMenuPresenter () {
            return menuItem;
        }

        protected String getActionName () {
            return JavaKit.fixImportsAction;
        }
        
    } // end of MainMenuWrapper
    
}
