/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.java;

import java.awt.Container;
import java.awt.Cursor;
import java.awt.Point;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.lang.reflect.Modifier;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.TreeSet;
import javax.jmi.reflect.RefClass;
import javax.jmi.reflect.RefException;
import javax.jmi.reflect.RefFeatured;
import javax.jmi.reflect.RefObject;
import javax.jmi.reflect.RefPackage;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;
import javax.swing.JEditorPane;
import javax.swing.JViewport;

import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.editor.*;
import org.netbeans.editor.ext.CompletionQuery;
import org.netbeans.editor.ext.ExtSettingsDefaults;
import org.netbeans.editor.ext.ExtSettingsNames;
import org.netbeans.editor.ext.java.JCExpression;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ParameterizedTypeImpl;
import org.openide.awt.StatusDisplayer;
import org.openide.cookies.EditorCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.text.PositionBounds;
import org.openide.windows.TopComponent;
import org.openide.util.RequestProcessor;
import org.openide.util.NbBundle;

public class JMIUtils {

    private static MDRepository repository = JavaModel.getJavaRepository();
    private static JavaMetamodel javaMetamodel = JavaMetamodel.getManager();

    private static boolean caseSensitive = true;
    private static boolean naturalSort = true;
    private static boolean showDeprecated = true;
    private static SettingsChangeListener settingsListener = new SettingsListener();

    static final boolean debugCompletionFailures
    = Boolean.getBoolean("org.netbeans.editor.debug.completion.failures"); // NOI18N

    private static final Comparator NATURAL_MEMBER_NAME_COMPARATOR = new NaturalMemberNameComparator(true);
    private static final Comparator INSENSITIVE_NATURAL_MEMBER_NAME_COMPARATOR = new NaturalMemberNameComparator(false);

    private static final boolean[][] assignVals = new boolean[][] {
        new boolean[] { true, false, false, false, false, false, false, false, false}, // boolean
        new boolean[] { false, true, false, true, true, true, true, true, false}, // byte
        new boolean[] { false, false, true, true, true, true, true, false, false}, // char
        new boolean[] { false, false, false, true, false, false, false, false, false}, // double
        new boolean[] { false, false, false, true, true, false, false, false, false}, // float
        new boolean[] { false, false, false, true, true, true, true, false, false}, // int
        new boolean[] { false, false, false, true, true, false, true, false, false}, // long
        new boolean[] { false, false, false, true, true, true, true, true, false}, // short
        new boolean[] { false, false, false, false, false, false, false, false, true} // void
    };

    static {
        refreshSettings();
        Settings.addSettingsChangeListener(settingsListener);
    }

    private BaseDocument doc;
    private FileObject fileObject = null;
    private boolean is15Enabled = true;
    private Resource resource;

    // ..........................................................................

    public static synchronized JMIUtils get(BaseDocument doc) {
        JMIUtils utils = (JMIUtils)doc.getProperty(JMIUtils.class);
        if (utils == null) {
            utils = new JMIUtils(doc);
            doc.putProperty(JMIUtils.class, utils);
        }
        return utils;
    }

    public static boolean isCaseSensitive() {
        return caseSensitive;
    }

    public void beginTrans(boolean write) {
        repository.beginTrans(write);
        resource=null;
        if (fileObject != null) {
            try {
                // needs to be done in try..catch to make sure the transaction is closed
                // if this operation fails
                JavaModel.setClassPath(fileObject);
            } catch (RuntimeException e) {
                repository.endTrans(write); // fail if b==true (i.e. write transaction)
                throw e;
            } catch (Error e) {
                repository.endTrans(write); // fail if b==true (i.e. write transaction)
                throw e;
            }
        } // else -> global class path should be used
    }

    public void endTrans(boolean fail) {
        repository.endTrans(fail);
    }

    public Type resolveType(String typeName) {
        if (!"null".equals(typeName)) { // NOI18N
            return JavaModel.getDefaultExtent().getType().resolve(typeName);
        }
        return null;
    }

    public Type resolveArray(Type type) {
        return ((JavaModelPackage)type.refImmediatePackage()).getArray().resolveArray(type);
    }

    public Type resolveParameterizedType(JavaClass def, List params) {
        JavaModelPackage pck = (JavaModelPackage)def.refImmediatePackage();
        return pck.getParameterizedType().resolveParameterizedType(def, params, null);
    }

    public boolean isAssignable(Type from, Type to) {
        if (from == null || to == null)
            return false;
        if (from.equals(to))
            return true;
        if (from instanceof Array && to instanceof Array)
            return isAssignable(((Array)from).getType(), ((Array)to).getType());
        if (to instanceof PrimitiveType && (is15Enabled || from instanceof PrimitiveType))
            return assignVals[getPrimitiveTypeIdx(from)][getPrimitiveTypeIdx(to)];
        if (to.getName().equals("java.lang.Object") && (is15Enabled || !(from instanceof PrimitiveType))) // NOI18N
            return true;
        if (to instanceof JavaClass && is15Enabled && from instanceof PrimitiveType)
            return isAssignable(getObjectType((PrimitiveType)from), to);
        if (from instanceof JavaClass && to instanceof JavaClass)
            return isSubTypeOf((JavaClass)from, (JavaClass)to);
        return false;
    }

    public boolean isSubTypeOf(JavaClass thisCls, JavaClass otherCls) {
        if (otherCls instanceof TypeParameter) {
            if (!isSubTypeOf(thisCls, otherCls.getSuperClass()))
                return false;
            for (Iterator it = otherCls.getInterfaces().iterator(); it.hasNext();) {
                if (!isSubTypeOf(thisCls, (JavaClass)it.next()))
                    return false;
            }
            return true;
        }
        if (thisCls.isSubTypeOf(otherCls)) {
            List thisParams = thisCls instanceof ParameterizedType ? ((ParameterizedType)thisCls).getParameters() : new ArrayList();
            List otherParams = otherCls instanceof ParameterizedType ? ((ParameterizedType)otherCls).getParameters() : new ArrayList();
            int thisParamsCount = thisParams.size();
            int otherParamsCount = otherParams.size();
            if (thisParamsCount == 0 || otherParamsCount == 0)
                return true;
            if (thisParamsCount != otherParamsCount)
                return false;
            for (Iterator it = thisParams.iterator(), itt = otherParams.iterator(); it.hasNext();) {
                if (!isAssignable((Type)it.next(), (Type)itt.next()))
                    return false;
            }
            return true;
        }
        return false;
    }

    public boolean isEqualType(Type thisCls, Type otherCls) {
        return isAssignable(thisCls, otherCls) && isAssignable(otherCls, thisCls);
    }

    public JavaPackage getExactPackage(String packageName) {
        return resolvePackage(packageName, true);
    }

    public String getPackageName(ClassDefinition jc) {
        if (jc instanceof UnresolvedClass) {
            String name = jc.getName();
            int index = name.lastIndexOf('.');
            return index < 0 ? "" : name.substring(0, index);
        }
        if (jc instanceof JavaClass && jc.isValid()) {
            Resource res = jc.getResource();
            if (res != null){
                String result = res.getPackageName();
                if (result != null) {
                    return result;
                }
            }
        }
        return "";
    }

    public JavaClass getExactClass(String name, String pkgName) {
        return getExactClass((pkgName != null && pkgName.length() != 0) ? (pkgName + "." + name) : name); // NOI18N
    }

    public JavaClass getExactClass(String classFullName) {
        Type cls = resolveType(classFullName);
        if (cls instanceof UnresolvedClass)
            return null;
        if (cls instanceof JavaClass)
            return (JavaClass)cls;
        return null;
    }

    public List findPackages(String name, boolean exactMatch, boolean subPackages, boolean createResultItems) {

        ArrayList ret = new ArrayList ();
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;

        if (exactMatch) {
            JavaPackage pkg = getExactPackage (name);
            if (pkg != null) {
                if (itemFactory != null)
                    ret.add(itemFactory.createPackageResultItem(pkg));
                else
                    ret.add(pkg);
            }
        } else {
            int index = name.lastIndexOf('.');
            String prefix = index > 0 ? name.substring(0, index) : "";
            JavaPackage pkg = resolvePackage(prefix, caseSensitive);
            if (pkg != null) {
                Collection subpackages = pkg.getSubPackages();
                for (Iterator it = subpackages.iterator(); it.hasNext();) {
                    JavaPackage subPackage = (JavaPackage) it.next();
                    if (startsWith(subPackage.getName(), name)) {
                        if (itemFactory != null)
                            ret.add(itemFactory.createPackageResultItem(subPackage));
                        else
                            ret.add(subPackage);
                    }
                }
            }
        } // else

        if (subPackages) {
            int size = ret.size ();
            for (int x = 0; x < size; x++) {
                JavaPackage sPkg = (JavaPackage) ret.get(x);
                addSubPackages(ret, sPkg, createResultItems);
            }
        }

        Collections.sort(ret, getNaturalMemberNameComparator());
        return ret;
    }

    /** Find classes by name and possibly in some package
    * @param pkg package where the classes should be searched for. It can be null
    * @param name begining of the name of the class. The package name must be omitted.
    * @param exactMatch whether the given name is the exact requested name
    *   of the class or not.
    * @param includeAnnotations
     * @return list of the matching classes
    */
    public List findClasses(JavaPackage pkg, String name, boolean exactMatch, boolean includeInnerClasses, boolean includeAnnotations, JavaClass context, boolean createResultItems, boolean isInstanceCreationCtx) {

        TreeSet ret = new TreeSet(getNaturalMemberNameComparator());
        ClassPath cp = JavaMetamodel.getManager().getClassPath();
        FileObject[] cpRoots = cp.getRoots();
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
        for (int i = 0; i < cpRoots.length; i++) {
            ClassIndex ci = ClassIndex.getIndex(JavaModel.getJavaExtent(cpRoots[i]));
            if (ci == null) continue;
            Collection col = null;
            if (pkg == null) {
                if (exactMatch) {
                    col = ci.getClassesBySimpleName(name);
                } else {
                    col = ci.getClassesBySNPrefix(name, caseSensitive);
                }
            } else {
                String clsName = (pkg.getName().length() > 0 ? pkg.getName() + '.' + name : name);
                if (exactMatch) {
                    JavaClass cls = ci.getClassByFqn(clsName);
                    if (cls != null) {
                        col = Collections.singleton(cls);
                    }
                } else {
                    col = ci.getClassesByFQNPrefix(clsName, caseSensitive);
                }
            }
            if (col != null) {
                for (Iterator it = col.iterator(); it.hasNext();) {
                    JavaClass javaClass = (JavaClass) it.next();
                    if ((includeAnnotations || !(javaClass instanceof AnnotationType))
                        && (showDeprecated || !javaClass.isDeprecated())
                        && (includeInnerClasses || !javaClass.isInner())
                        && isAccessible(javaClass, context)) {
                        if (itemFactory == null)
                            ret.add(javaClass);
                        else
                            ret.add(itemFactory.createClassResultItem(javaClass, pkg == null, isInstanceCreationCtx));
                    }
                }
            }
        }
        return new ArrayList(ret);
    }

    public List findAnnotations(JavaPackage pkg, String name, boolean exactMatch, JavaClass context, boolean createResultItems) {
        // TODO: More effective way to get annotation list should be created
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
        TreeSet ret = new TreeSet(getNaturalMemberNameComparator());
        ClassPath cp = JavaMetamodel.getManager().getClassPath();
        FileObject[] cpRoots = cp.getRoots();
        for (int i = 0; i < cpRoots.length; i++) {
            JavaModelPackage jmp = JavaModel.getJavaExtent(cpRoots[i]);
            if (jmp != null) {
                AnnotationTypeClass atc = jmp.getAnnotationType();
                for (Iterator it = atc.refAllOfType().iterator(); it.hasNext();) {
                    AnnotationType at = (AnnotationType) it.next();
                    if (!showDeprecated && at.isDeprecated())
                        continue;
                    if (exactMatch) {
                        if (pkg != null) {
                            name = (pkg.getName().length() >  0 ? pkg.getName() + '.' + name : name);
                            if ((caseSensitive ? at.getName().equals(name) : at.getName().equalsIgnoreCase(name)) &&
                                    isAccessible(at, context)) {
                                if (itemFactory != null)
                                    ret.add(itemFactory.createClassResultItem(at, false, false));
                                else
                                    ret.add(at);
                            }
                        } else {
                            if ((caseSensitive ? at.getSimpleName().equals(name) : at.getSimpleName().equalsIgnoreCase(name)) &&
                                    isAccessible(at, context)) {
                                if (itemFactory != null)
                                    ret.add(itemFactory.createClassResultItem(at, true, false));
                                else
                                    ret.add(at);
                            }
                        }
                    } else {
                        if (pkg != null) {
                            if ((caseSensitive ? getPackageName(at).equals(pkg.getName()) : getPackageName(at).equalsIgnoreCase(pkg.getName())) &&
                                    startsWith(at.getSimpleName(), name) && isAccessible(at, context)) {
                                if (itemFactory != null)
                                    ret.add(itemFactory.createClassResultItem(at, false, false));
                                else
                                    ret.add(at);
                            }
                        } else {
                            if (startsWith(at.getSimpleName(), name) && isAccessible(at, context)) {
                                if (itemFactory != null)
                                    ret.add(itemFactory.createClassResultItem(at, true, false));
                                else
                                    ret.add(at);
                            }
                        }
                    }
                }
            }
        }
        return new ArrayList(ret);
    }

    public List findFeatures(Type type, String name, boolean exactMatch,
                             boolean inspectOuterClasses, JavaClass context, boolean staticContext, boolean thisContext, JCExpression exp, boolean createResultItems, boolean preferSources, boolean isInstanceCreationCtx, boolean sorted) {
        return findFeatures(type, name, exactMatch, true, inspectOuterClasses, context, staticContext, thisContext, Feature.class, exp, createResultItems, preferSources, isInstanceCreationCtx, sorted);
    }

    /** Find fields by name in a given class.
    * @param type class which is searched for the fields.
    * @param name start of the name of the field
    * @param exactMatch whether the given name of the field is exact
    * @param staticContext whether search for the static fields only
    * @return list of the matching fields
    */
    public List findFields(Type type, String name, boolean exactMatch,
                           boolean inspectOuterClasses, JavaClass context, boolean staticContext, boolean thisContext, boolean createResultItems, boolean preferSources) {
        return findFeatures(type, name, exactMatch, true, inspectOuterClasses, context, staticContext, thisContext, Field.class, null, createResultItems, preferSources, false, true);
    }

    /** Find methods by name in a given class.
    * @param type class which is searched for the methods.
    * @param name start of the name of the method
    * @param exactMatch whether the given name of the method is exact
    * @param staticContext whether search for the static methods only
    * @return list of the matching methods
    */
    public List findMethods(Type type, String name, boolean exactMatch,
                            boolean inspectOuterClasses, JavaClass context, boolean staticContext, boolean thisContext, JCExpression exp, boolean createResultItems, boolean preferSources) {
        return findFeatures(type, name, exactMatch, true, inspectOuterClasses, context, staticContext, thisContext, Method.class, exp, createResultItems, preferSources, false, true);
    }

    public List findConstructors(Type type, JavaClass context, boolean thisContext) {
        return findFeatures(type, "", false, false, false, context, false, thisContext, Constructor.class, null, false, true, false, true);
    }

    public List findInnerClasses(Type type, String name,  boolean exactMatch,
                                 boolean inspectOuterClasses, JavaClass context, boolean thisContext, boolean createResultItems, boolean preferSources, boolean isInstanceCreationCtx) {
        return findFeatures(type, name, exactMatch, true, inspectOuterClasses, context, false, thisContext, JavaClass.class, null, createResultItems, preferSources, isInstanceCreationCtx, true);
    }

    public List findEnumConstants(JavaEnum en, String name, boolean exactMatch, boolean createResultItems, boolean preferSources) {
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
        if (preferSources)
            en = (JavaEnum)JMIUtils.getSourceElementIfExists(en);
        TreeSet ret = new TreeSet(getNaturalMemberNameComparator());
        for (Iterator it = en.getConstants().iterator(); it.hasNext();) {
            Field constant = (Field) it.next();
            if (!showDeprecated && constant.isDeprecated())
                continue;
            String constantName = constant.getName();
            if (exactMatch) {
                if (!constantName.equals(name)) continue;
            } else {
                if (!startsWith(constantName, name)) continue;
            }
            if (itemFactory != null)
                ret.add(itemFactory.createFieldResultItem(constant, null, en));
            else
                ret.add(constant);
        }
        return new ArrayList(ret);
    }

    public List findAnnotationMembers(AnnotationType type, String name, boolean exactMatch, boolean createResultItems, boolean preferSources) {
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
        if (preferSources)
            type = (AnnotationType)JMIUtils.getSourceElementIfExists(type);
        List ret = new ArrayList();
        for (Iterator it = type.getContents().iterator(); it.hasNext();) {
            Object o = it.next();
            if (o instanceof Attribute) {
                Attribute attr = (Attribute)o;
                String attrName = attr.getName();
                if (exactMatch) {
                    if (!attrName.equals(name)) continue;
                } else {
                    if (!startsWith(attrName, name)) continue;
                }
                if (itemFactory != null)
                    ret.add(itemFactory.createAttributeResultItem(attr));
                else
                    ret.add(attr);
            }
        }
        return ret;
    }

    public Resource getResource() {
        if (resource == null || !resource.isValid()) {
            DataObject dob = NbEditorUtilities.getDataObject(doc);
            resource = dob != null ? JavaModel.getResource(dob.getPrimaryFile()) : null;
        }
        return resource;
    }


    public JavaClass getImportedClass(String simpleName, JavaClass topClass, JavaClass context, Resource res){

        JavaClass ret = null;

        if (res != null) {
            // class imports
            String simpleNamePrefix=".".concat(simpleName); // NOI18N
            Object[] imports=res.getImports().toArray();
            for (int i=0;i<imports.length;i++) {
                Import imp=(Import)imports[i];
                
                if (!imp.isOnDemand() && !imp.isStatic()) {
                    if (imp.getName().endsWith(simpleNamePrefix)) {
                        return (JavaClass)imp.getImportedNamespace();
                    }
                }
            }

            // pacakge content
            ret = getExactClass(simpleName, getPackageName(topClass));
            if (ret != null)
                return ret;

            // java.lang content
            List clsList = findClasses(getExactPackage("java.lang"), simpleName, true, false, true, context, false, false); // NOI18N
            if (clsList != null && clsList.size() > 0) {
                return (JavaClass)clsList.get(0);
            }

            // on demand imports
            for (int i=0;i<imports.length;i++) {
                Import imp=(Import)imports[i];
                if (imp.isOnDemand()) {
                    Element el = imp.getImportedNamespace();
                    if (el instanceof JavaPackage) {
                        clsList = findClasses((JavaPackage)el, simpleName, true, false, true, context, false, false);
                        if (clsList != null && clsList.size() > 0) {
                            return (JavaClass)clsList.get(0);
                        }
                    } else if (el instanceof JavaClass) {
                        clsList = findInnerClasses((JavaClass)el, simpleName, true, true, context, false, false, false, false);
                        if (clsList != null && clsList.size() > 0) {
                            return (JavaClass)clsList.get(0);
                        }
                    }
                }
            }
        }
        return ret;
    }

    public List getStaticallyImportedFields(String name, boolean exactMatch, JavaClass context, boolean isThisContext) {
        List ret = new ArrayList();
        Resource res = getResource();
        if (res != null) {
            for (Iterator it = res.getImports().iterator(); it.hasNext();) {
                Import imp = (Import)it.next();
                if (imp.isStatic()) {
                    for (Iterator itt = imp.getImportedElements().iterator(); itt.hasNext();) {
                        NamedElement el = (NamedElement) itt.next();
                        if (el instanceof Field && (((Field)el).getModifiers() & Modifier.STATIC) != 0) {
                            if (exactMatch) {
                                if (el.getName() != null && el.getName().equals(name)) {
                                    ClassDefinition decl = ((Field)el).getDeclaringClass();
                                    ClassDefinition sourceDecl = getSourceElementIfExists(decl);
                                    if (decl != sourceDecl)
                                        el = sourceDecl.getField(el.getName(), true);
                                    ret.add(el);
                                }
                            } else {
                                if (el.getName() != null && el.getName().startsWith(name)) {
                                    ClassDefinition decl = ((Field)el).getDeclaringClass();
                                    ClassDefinition sourceDecl = getSourceElementIfExists(decl);
                                    if (decl != sourceDecl)
                                        el = sourceDecl.getField(el.getName(), true);
                                    ret.add(el);
                                }
                            }
                        }
                        if (el instanceof JavaClass) {
                            ret.addAll(findFeatures(getSourceElementIfExists((JavaClass)el), name, exactMatch, true, false, context, true, isThisContext, Field.class, null, false, false, false, true));
                        }
                    }
                }
            }
        }
        return ret;
    }

    public List getStaticallyImportedMethods(String name, boolean exactMatch, JavaClass context, boolean isThisContext) {
        List ret = new ArrayList();
        Resource res = getResource();
        if (res != null) {
            for (Iterator it = res.getImports().iterator(); it.hasNext();) {
                Import imp = (Import)it.next();
                if (imp.isStatic()) {
                    for (Iterator itt = imp.getImportedElements().iterator(); itt.hasNext();) {
                        NamedElement el = (NamedElement) itt.next();
                        if (el instanceof Method && (((Method)el).getModifiers() & Modifier.STATIC) != 0) {
                            if (exactMatch) {
                                if (el.getName() != null && el.getName().equals(name)) {
                                    ClassDefinition decl = ((Method)el).getDeclaringClass();
                                    ClassDefinition sourceDecl = getSourceElementIfExists(decl);
                                    if (decl != sourceDecl) {
                                        List params = new ArrayList();
                                        for (Iterator i = ((Method)el).getParameters().iterator(); i.hasNext(); params.add(((Parameter)i.next()).getType()));
                                        el = sourceDecl.getMethod(el.getName(), params, true);
                                    }
                                    ret.add(el);
                                }
                            } else {
                                if (el.getName() != null && el.getName().startsWith(name)) {
                                    ClassDefinition decl = ((Method)el).getDeclaringClass();
                                    ClassDefinition sourceDecl = getSourceElementIfExists(decl);
                                    if (decl != sourceDecl) {
                                        List params = new ArrayList();
                                        for (Iterator i = ((Method)el).getParameters().iterator(); i.hasNext(); params.add(((Parameter)i.next()).getType()));
                                        el = sourceDecl.getMethod(el.getName(), params, true);
                                    }
                                    ret.add(el);
                                }
                            }
                        }
                        if (el instanceof JavaClass) {
                            ret.addAll(findFeatures(getSourceElementIfExists((JavaClass)el), name, exactMatch, true, false, context, true, isThisContext, Method.class, null, false, false, false, true));
                        }
                    }
                }
            }
        }
        return ret;
    }

    public List getImportedInnerClasses(String name, boolean exactMatch, JavaClass context, boolean isThisContext, boolean createResultItems, boolean isInstanceCreationCtx) {
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
        List ret = new ArrayList();
        Resource res = getResource();
        if (res != null) {
            for (Iterator it = res.getImports().iterator(); it.hasNext();) {
                Import imp = (Import)it.next();
                NamedElement el = imp.getImportedNamespace();
                if (el instanceof JavaClass) {
                    if (imp.isOnDemand()) {
                        ret.addAll(findInnerClasses((JavaClass)el, name, exactMatch, true, context, isThisContext, createResultItems, false, isInstanceCreationCtx));
                    } else if (((JavaClass)el).isInner()) {
                        if (exactMatch) {
                            if (((JavaClass)el).getSimpleName() != null && ((JavaClass)el).getSimpleName().equals(name)) {
                                if (itemFactory != null)
                                    ret.add(itemFactory.createClassResultItem((JavaClass)el, false, isInstanceCreationCtx));
                                else
                                    ret.add(el);
                            }
                        } else {
                            if (((JavaClass)el).getSimpleName() != null && ((JavaClass)el).getSimpleName().startsWith(name)) {
                                if (itemFactory != null)
                                    ret.add(itemFactory.createClassResultItem((JavaClass)el, false, isInstanceCreationCtx));
                                else
                                    ret.add(el);
                            }
                        }
                    }
                }
            }
        }
        return ret;
    }

    public Collection filterNames(Collection col, String namePrefix, boolean exactMatch) {
        Collection ret = new ArrayList();
        for (Iterator it = col.iterator(); it.hasNext();) {
            String s = (String) it.next();
            if (exactMatch) {
                if (s.equals(namePrefix))
                    ret.add(s);
            } else {
                if (startsWith(s, namePrefix))
                    ret.add(s);
            }
        }
        return ret;
    }

    public List findMatchingTypes(TypeParameter tp, JavaClass context, boolean createResultItems) {
        TreeSet ts = new TreeSet(getNaturalMemberNameComparator());
        JavaClass sup = tp.getSuperClass();
        if (sup != null) {
            if ("java.lang.Object".equals(sup.getName())) { //NOI18N
                if (tp.getInterfaces().size() == 0)
                    ts.addAll(findClasses(null, "", false, false, true, context, createResultItems, false));
                else
                    sup = null;
            } else {
                ts.add(sup);
                NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
                for (Iterator it = sup.findSubTypes(true).iterator(); it.hasNext();) {
                    JavaClass subType = (JavaClass) it.next();
                    if ((showDeprecated || !subType.isDeprecated())
                        && (subType.isInner() ? isAccessible(subType, null, context) : isTopLevelClassAccessible(subType, context))) {
                        if (itemFactory == null)
                            ts.add(subType);
                        else
                            ts.add(itemFactory.createClassResultItem(subType, true, false));
                    }

                }
            }
        }
        if (ts.size() != 0) {
            for (Iterator it = tp.getInterfaces().iterator(); it.hasNext();) {
                ts.retainAll(((JavaClass)it.next()).findSubTypes(true));
            }
        } else if (sup == null) {
            Iterator it = tp.getInterfaces().iterator();
            if (it.hasNext()) {
                ts.addAll(((JavaClass)it.next()).findSubTypes(true));
            }
            while(it.hasNext()) {
                ts.retainAll(((JavaClass)it.next()).findSubTypes(true));
            }
        }
        return new ArrayList(ts);
    }

    public static boolean openElement(final Element element) {
        repository.beginTrans(false);
        try {
            Resource resource = element.getResource();
            if (resource != null) {
                javaMetamodel.setClassPath(resource);
                DataObject dob = javaMetamodel.getDataObject(resource);
                if (dob != null) {
                    final EditorCookie.Observable ec = (EditorCookie.Observable)dob.getCookie(EditorCookie.Observable.class);
                    if (ec != null) {
                        StatusDisplayer.getDefault().setStatusText(NbBundle.getMessage(JMIUtils.class, "opening-element", element instanceof NamedElement ? ((NamedElement)element).getName() : "")); // NOI18N
                        Utilities.runInEventDispatchThread(new Runnable() {
                            public void run() {
                                JEditorPane[] panes = ec.getOpenedPanes();
                                if (panes != null && panes.length > 0) {
                                    selectElementInPane(panes[0], element, false);
                                } else {
                                    ec.addPropertyChangeListener(new PropertyChangeListener() {
                                        public void propertyChange(PropertyChangeEvent evt) {
                                            if (EditorCookie.Observable.PROP_OPENED_PANES.equals(evt.getPropertyName())) {
                                                final JEditorPane[] panes = ec.getOpenedPanes();
                                                if (panes != null && panes.length > 0) {
                                                    selectElementInPane(panes[0], element, true);
                                                }
                                                ec.removePropertyChangeListener(this);
                                            }
                                        }
                                    });
                                    ec.open();
                                }
                            }

                        });
                        return true;
                    }
                }
            }
        } finally {
            repository.endTrans(false);
        }
        return false;
    }

    public static ClassDefinition getSourceElementIfExists(ClassDefinition next) {
        return ((JMManager)JMIUtils.javaMetamodel).getSourceElementIfExists(next);
    }

    // TODO: temporary hack - a better support from javacore needed
    public static Feature getDefintion(Feature e) {
        if (e instanceof ParameterizedType)
            return getDefintion(((ParameterizedType)e).getDefinition());
        try {
            return getDefintion((Feature)((ParameterizedTypeImpl.Wrapper) e).getWrappedObject());
        } catch (Exception e1) {}
        return e;
    }
    
    public static String getTypeName(Type typ, boolean displayFQN, boolean substituteTypeParams) {
        if (typ instanceof Array)
            return getTypeName(((Array)typ).getType(), displayFQN, false) + "[]"; // NOI18N
        if (typ instanceof TypeParameter && substituteTypeParams)
            return getTypeName(((TypeParameter)typ).getSuperClass(), displayFQN, true);
        if (typ instanceof AnnotationType || (typ instanceof ParameterizedType && ((ParameterizedType)typ).getDefinition() instanceof AnnotationType))
            return "@" + (displayFQN ? ((JavaClass)typ).getName() : ((JavaClass)typ).getSimpleName()); // NOI18N
        if (typ instanceof JavaClass)
            return displayFQN ? ((JavaClass)typ).getName() : ((JavaClass)typ).getSimpleName();
        return typ != null ? typ.getName() : ""; //NOI18N
    }

    public boolean isAccessible(ClassDefinition cls, JavaClass from) {
        if (cls == null)
            return false;
        if (from == null)
            return true;
        while (cls instanceof Array) {
            Type typ = ((Array)cls).getType();
            if (typ instanceof ClassDefinition)
                cls = (ClassDefinition)typ;
            else
                return true;
        }
        return ((JavaClass)cls).isInner() ? isAccessible((JavaClass)cls, ((JavaClass)cls).getDeclaringClass(), from) : isTopLevelClassAccessible((JavaClass)cls, from);
    }


    // ..........................................................................

    private JMIUtils(BaseDocument doc){
        this.doc = doc;
        DataObject dob = NbEditorUtilities.getDataObject(doc);
        if (dob != null) {
            FileObject fo=dob.getPrimaryFile();
            ClassPath sourceCP = ClassPath.getClassPath(fo, ClassPath.SOURCE);
            if (sourceCP != null) {
                this.fileObject = sourceCP.findOwnerRoot(fo);
            }
        }
    }


    private List findFeatures(Type type, String name, boolean exactMatch, boolean inspectSuperTypes, boolean inspectOuterClasses, JavaClass context, boolean staticContext, boolean thisContext, Class featureType, JCExpression exp, boolean createResultItems, boolean preferSources, boolean isInstanceCreationCtx, boolean sorted) {

        if (type == null || type instanceof PrimitiveType) {
            return new ArrayList ();
        }

        Collection ts = null;
        if (sorted)
            ts = new TreeSet(getNaturalMemberNameComparator());
        else
            ts = new ArrayList();

        boolean camelCase = name.length() > 1 && name.equals(name.toUpperCase());        
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
        ClassDefinition typeCtx = (preferSources ? JMIUtils.getSourceElementIfExists((ClassDefinition) type) : (ClassDefinition)type);

        int outerInd = 0;
        ClassDefinition cls = thisContext ? type.equals(context) ? null : context : type instanceof ClassDefinition ? (ClassDefinition)type : null;
        boolean matchName = exactMatch || name.length()>0;
        boolean staticType = (type instanceof JavaClass) && Modifier.isStatic(((JavaClass)type).getModifiers());
        List outerList = (inspectOuterClasses && type instanceof JavaClass) ? getOuterClasses((JavaClass)type) : Collections.singletonList(type);
        for (Iterator it = outerList.iterator(); it.hasNext(); outerInd++) {
            for (Iterator itt = new ClassesIterator((Type)it.next(), inspectSuperTypes, preferSources); itt.hasNext();) {
                ClassDefinition tempClass = (ClassDefinition) itt.next();
                AccessibilityDescriptor accessDesc = new AccessibilityDescriptor(tempClass, cls, context);
                Object[] features = tempClass.getFeatures().toArray();
                int ctorsCount = 0;
                for (int i=0;i<features.length;i++) {
                    Feature feature = (Feature) features[i];
                    if (!(featureType.isInstance(feature))) continue;
                    if (featureType.isAssignableFrom(Constructor.class))
                        ctorsCount++;
                    if (matchName) {
                        String featureName = feature instanceof JavaClass ? ((JavaClass)feature).getSimpleName() : feature.getName();
                        if (featureName == null)
                            featureName = ""; // NOI18N
                        if (exactMatch) {
                            if (!featureName.equals(name)) continue;
                        } else {
                            if (!startsWith(featureName, name) && (!camelCase || !(feature instanceof JavaClass) || !matchesCamelCase(featureName, name))) continue;
                        }
                    }
                    int mods = feature.getModifiers();
                    if ((staticContext && !(feature instanceof JavaClass) && (mods & Modifier.STATIC) == 0) ||
                            ((outerInd > 0) && staticType) && ((mods & Modifier.STATIC) == 0) ||
                            (!showDeprecated && feature.isDeprecated()) ||
                            !accessDesc.isAccessible(mods)) {
                        continue;
                    }
                    if (itemFactory != null)
                        ts.add(itemFactory.createResultItem(feature, exp, typeCtx, isInstanceCreationCtx));
                    else
                        ts.add(feature);
                } // while
                // HACK!!! - adding the values() and valueOf(...) Enum's methods since the new JMI Java infrastructure
                // does not support them.
                if (tempClass instanceof JavaEnum && featureType.isAssignableFrom(Method.class)
                        && (thisContext || !(type instanceof JavaClass) || isAccessible((JavaClass)type, context))) {
                    if (exactMatch) {
                        if ("values".equals(name)) { // NOI18N
                            List tags = Collections.singletonList(new FakeTagValue(createTagDefinition("@return"), "an array containing the constants of this enum class, in the order they're declare")); // NOI18N
                            JavaDoc jd = new FakeJavaDoc(tags, "Returns an array containing the constants of this enum class, in the order they're declared. This method may be used to iterate over the constants as follows:<blockquote><pre>for(ClassName c : ClassName.values())<br>    System.out.println(c);</pre></blockquote>"); // NOI18N
                            Method mtd = new FakeMethod("values", tempClass, resolveArray(tempClass), Collections.EMPTY_LIST, jd); // NOI18N
                            if (itemFactory != null)
                                ts.add(itemFactory.createResultItem(mtd, exp, typeCtx, false));
                            else
                                ts.add(mtd);
                        } else if ("valueOf".equals(name)) { // NOI18N
                            List tags = new ArrayList(3);
                            tags.add(new FakeTagValue(createTagDefinition("@param"), "name the name of the constant to return")); // NOI18N
                            tags.add(new FakeTagValue(createTagDefinition("@return"), "the enum constant with the specified name")); // NOI18N
                            tags.add(new FakeTagValue(createTagDefinition("@throws"), "IllegalArgumentException if this enum class has no constant with the specified name")); // NOI18N
                            JavaDoc jd = new FakeJavaDoc(tags, "Returns the enum constant of this class with the specified name. The string must match exactly an identifier used to declare an enum constant in this type. (Extraneous whitespace characters are not permitted.)"); // NOI18N
                            Method mtd = new FakeMethod("valueOf", tempClass, tempClass, Collections.singletonList(new FakeParameter("name", resolveType("java.lang.String"))), jd); // NOI18N
                            if (itemFactory != null)
                                ts.add(itemFactory.createResultItem(mtd, exp, typeCtx, false));
                            else
                                ts.add(mtd);
                        }
                    } else {
                        if (startsWith("values", name)) { // NOI18N
                            List tags = Collections.singletonList(new FakeTagValue(createTagDefinition("@return"), "an array containing the constants of this enum class, in the order they're declare")); // NOI18N
                            JavaDoc jd = new FakeJavaDoc(tags, "Returns an array containing the constants of this enum class, in the order they're declared. This method may be used to iterate over the constants as follows:<blockquote><pre>for(ClassName c : ClassName.values())<br>    System.out.println(c);</pre></blockquote>"); // NOI18N
                            Method mtd = new FakeMethod("values", tempClass, resolveArray(tempClass), Collections.EMPTY_LIST, jd); // NOI18N
                            if (itemFactory != null)
                                ts.add(itemFactory.createResultItem(mtd, exp, typeCtx, false));
                            else
                                ts.add(mtd);
                        }
                        if (startsWith("valueOf", name)) { // NOI18N
                            List tags = new ArrayList(3);
                            tags.add(new FakeTagValue(createTagDefinition("@param"), "name the name of the constant to return")); // NOI18N
                            tags.add(new FakeTagValue(createTagDefinition("@return"), "the enum constant with the specified name")); // NOI18N
                            tags.add(new FakeTagValue(createTagDefinition("@throws"), "IllegalArgumentException if this enum class has no constant with the specified name")); // NOI18N
                            JavaDoc jd = new FakeJavaDoc(tags, "Returns the enum constant of this class with the specified name. The string must match exactly an identifier used to declare an enum constant in this type. (Extraneous whitespace characters are not permitted.)"); // NOI18N
                            Method mtd = new FakeMethod("valueOf", tempClass, tempClass, Collections.singletonList(new FakeParameter("name", resolveType("java.lang.String"))), jd); // NOI18N
                            if (itemFactory != null)
                                ts.add(itemFactory.createResultItem(mtd, exp, typeCtx, false));
                            else
                                ts.add(mtd);
                        }
                    }
                }
                if (ctorsCount == 0 && featureType.isAssignableFrom(Constructor.class)) {
                    ClassDefinition cd = tempClass instanceof ParameterizedType ? ((ParameterizedType)tempClass).getDefinition() : tempClass;
                    if (cd instanceof JavaClass) {
                        String cdName = ((JavaClass)cd).getSimpleName();
                        if (exactMatch) {
                            if (name.equals(cdName)) {
                                FakeConstructor ctr = new FakeConstructor(cd, null);
                                if (itemFactory != null)
                                    ts.add(itemFactory.createResultItem(ctr, exp, typeCtx, false));
                                else
                                    ts.add(ctr);
                            }
                        } else {
                            if (startsWith(cdName, name)) {
                                FakeConstructor ctr = new FakeConstructor(cd, null);
                                if (itemFactory != null)
                                    ts.add(itemFactory.createResultItem(ctr, exp, typeCtx, false));
                                else
                                    ts.add(ctr);
                            }
                        }
                    }
                }
            }
        }        
        return ts instanceof List ? (List)ts : new ArrayList(ts);
    }

    private static void selectElementInPane(final JEditorPane pane, final Element element, boolean delayProcessing) {
        final Cursor editCursor = pane.getCursor();
        pane.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        if (delayProcessing) {
            pane.addFocusListener(new FocusAdapter() {
                public void focusGained(FocusEvent e) {
                    RequestProcessor.getDefault().post(new Runnable() {
                        public void run() {
                            int caretPos = pane.getCaretPosition();
                            Container parent = pane.getParent();
                            Point viewPos = parent instanceof JViewport ? ((JViewport)parent).getViewPosition() : null;
                            PositionBounds bounds = null;
                            repository.beginTrans(false);
                            try {
                                javaMetamodel.setClassPath(element.getResource());
                                bounds = JavaMetamodel.getManager().getElementPosition(element);
                            } finally {
                                repository.endTrans(false);
                            }
                            if (bounds != null && pane.getCaretPosition() == caretPos &&
                                    (viewPos == null || viewPos.equals(((JViewport)parent).getViewPosition()))) {
                                pane.setCaretPosition(bounds.getBegin().getOffset());
                            }
                            JumpList.checkAddEntry(pane);
                            pane.setCursor(editCursor);
                            StatusDisplayer.getDefault().setStatusText(""); // NOI18N
                        }
                    });
                    pane.removeFocusListener(this);
                }
            });}
        else {
            RequestProcessor.getDefault().post(new Runnable() {
                public void run() {
                    int caretPos = pane.getCaretPosition();
                    Container parent = pane.getParent();
                    Point viewPos = parent instanceof JViewport ? ((JViewport)parent).getViewPosition() : null;
                    PositionBounds bounds = null;
                    repository.beginTrans(false);
                    try {
                        javaMetamodel.setClassPath(element.getResource());
                        bounds = JavaMetamodel.getManager().getElementPosition(element);
                    } finally {
                        repository.endTrans(false);
                    }
                    if (bounds != null && pane.getCaretPosition() == caretPos &&
                            (viewPos == null || viewPos.equals(((JViewport)parent).getViewPosition()))) {
                        pane.setCaretPosition(bounds.getBegin().getOffset());
                    }
                    JumpList.checkAddEntry(pane);
                    pane.setCursor(editCursor);
                    StatusDisplayer.getDefault().setStatusText(""); // NOI18N
                }
            });

            Container temp = pane;
            while (!(temp instanceof TopComponent)) {
                temp = temp.getParent();
            }
            ((TopComponent) temp).requestActive();
        }
    }

    private JavaPackage resolvePackage(String packageName, boolean caseSensitive) {
        JavaPackageClass pkgProxy = JavaModel.getDefaultExtent().getJavaPackage();
        if (caseSensitive) {
            return pkgProxy.resolvePackage(packageName);
        } else {
            // [TODO] implement case insensitive search
            return pkgProxy.resolvePackage(packageName);
        }
    }

    private TagDefinition createTagDefinition(String def) {
        JavaModelPackage pkg = JavaModel.getDefaultExtent();
        return pkg.getTagDefinition().createTagDefinition(def);
    }

    private void addSubPackages (List list, JavaPackage pkg, boolean createResultItems) {
        NbJavaJMICompletionQuery.JMIItemFactory itemFactory = createResultItems ? NbJavaJMICompletionQuery.getJMIItemFactory() : null;
        Iterator iter = pkg.getSubPackages ().iterator ();
        while (iter.hasNext ()) {
            JavaPackage p = (JavaPackage) iter.next ();
            if (itemFactory != null)
                list.add (itemFactory.createPackageResultItem(p));
            else
                list.add(p);
            addSubPackages (list, p, createResultItems);
        }
    }

    /** Get outer classes to search
    * the fields and methods there. The original class is added
    * as the first member of the resulting list.
    */
    private List getOuterClasses(JavaClass jc) {
        ArrayList outers = new ArrayList();
        outers.add(jc);
        while (jc.isInner()) {
            jc = (JavaClass) jc.getDeclaringClass ();
            if (jc == null)
                 break;
            outers.add (jc);
        }
        return outers;
    }

    private JavaClass getOutermostClass(JavaClass jc) {
        while (jc.isInner()) {
            JavaClass decl = (JavaClass)jc.getDeclaringClass();
            if (decl == null)
                return jc;
            jc = decl;
        }
        return jc;
    }

    private boolean isTopLevelClassAccessible(JavaClass cls, JavaClass from) {
        if (cls == null)
            return false;
        if (from == null)
            return true;
        if ((cls.getModifiers() & Modifier.PUBLIC) != 0)
            return true;
        String pkgName = getPackageName(cls);
        String fromPkgName = getPackageName(from);
        return pkgName.equals(fromPkgName);
    }

    private class AccessibilityDescriptor {
        int mprotected=-1, mpackprivate=-1, mprivate=-1;
        ClassDefinition decCls;
        ClassDefinition cls;
        JavaClass from;
        
        AccessibilityDescriptor(ClassDefinition decCls,ClassDefinition cls, JavaClass from) {
            if (from == null) {
                mprotected = mpackprivate = mprivate = 1;
                return;
            }
            if (cls instanceof TypeParameter)
                cls = decCls;
            if (cls != null && !from.equals(cls) && !(JMIUtils.this.isAccessible(cls, from))) {
                mprotected = mpackprivate = mprivate = 0;
                return;
            }
            this.decCls = decCls;
            this.cls = cls;
            this.from = from;
        }

        private void computeProtectedAccess() {
            JavaClass jc = from;
            while (jc != null) {
                if (isAssignable(jc, decCls) && (cls == null || isAssignable(cls, jc))) {
                    mprotected = 1;
                    return;
                }
                jc = (JavaClass)jc.getDeclaringClass();
            }
            mprotected = 0;
        }
        
        private void computePackagePrivateAccess() {
            String fromPkgName = getPackageName(from);
            String pkgName = getPackageName(decCls);
            boolean mpackpriv = fromPkgName.equals(pkgName);
            
            if (mpackpriv)
                mprotected = mpackprivate = 1;
            else
                mpackprivate = 0;
        }
        
        private void computePrivateAccess() {
            boolean mpriv = getOutermostClass(from).equals(decCls instanceof JavaClass
                    ? getOutermostClass(decCls instanceof ParameterizedType ? ((ParameterizedType)decCls).getDefinition() : (JavaClass)decCls)
                    : null);
            if (mpriv)
                mprotected = mpackprivate = mprivate = 1;
            else
                mprivate = 0;
        }
    
        boolean isAccessible(int modifiers) {
            if (Modifier.isPublic(modifiers)) {
                return true;
            }
            if (mprivate == -1)
                computePrivateAccess();
            if (mprivate == 1) {
                return true;
            }
            if (Modifier.isPrivate(modifiers)) {
                return false;
            }
            if (mpackprivate == -1)
                computePackagePrivateAccess();
            if (mpackprivate == 1) {
                return true;
            }
            if (!Modifier.isProtected(modifiers)) {     // pavkage private
                return false;
            }
            if (mprotected == -1)
                computeProtectedAccess();
            return mprotected == 1;
        }
    }
    
    private boolean isAccessible(Feature feature, ClassDefinition cls, JavaClass from) {
        ClassDefinition decCls = feature.getDeclaringClass();
        AccessibilityDescriptor desc = new AccessibilityDescriptor(decCls, cls, from);
        
        return desc.isAccessible(feature.getModifiers());
    }

    public static boolean startsWith(String theString, String prefix){
        return caseSensitive ? theString.startsWith(prefix) :
            theString.toLowerCase().startsWith(prefix.toLowerCase());
    }
    
    public static boolean matchesCamelCase(String simpleName, String prefix) {
        int sni;
        int pi;
        for (pi = 0, sni = 0; sni < simpleName.length() && pi < prefix.length(); sni++) {
            char ch = simpleName.charAt(sni);
            if (Character.isUpperCase(ch)) {
                if (ch != prefix.charAt(pi++)) {
                    return false;
                }
            }
        }
        return pi == prefix.length();
    }

    private byte getPrimitiveTypeIdx(Type type) {
        PrimitiveTypeKind kind = type instanceof PrimitiveType ? ((PrimitiveType)type).getKind() : null;
        if ("java.lang.Boolean".equals(type.getName()) || PrimitiveTypeKindEnum.BOOLEAN.equals(kind)) return 0; // NOI18N
        if ("java.lang.Byte".equals(type.getName()) || PrimitiveTypeKindEnum.BYTE.equals(kind)) return 1; // NOI18N
        if ("java.lang.Char".equals(type.getName()) || PrimitiveTypeKindEnum.CHAR.equals(kind)) return 2; // NOI18N
        if ("java.lang.Double".equals(type.getName()) || PrimitiveTypeKindEnum.DOUBLE.equals(kind)) return 3; // NOI18N
        if ("java.lang.Float".equals(type.getName()) || PrimitiveTypeKindEnum.FLOAT.equals(kind)) return 4; // NOI18N
        if ("java.lang.Integer".equals(type.getName()) || PrimitiveTypeKindEnum.INT.equals(kind)) return 5; // NOI18N
        if ("java.lang.Long".equals(type.getName()) || PrimitiveTypeKindEnum.LONG.equals(kind)) return 6; // NOI18N
        if ("java.lang.Short".equals(type.getName()) || PrimitiveTypeKindEnum.SHORT.equals(kind)) return 7; // NOI18N
        return 8;
    }

    private Type getObjectType(PrimitiveType type) {
        PrimitiveTypeKind kind = type.getKind();
        if (PrimitiveTypeKindEnum.BOOLEAN.equals(kind)) return resolveType("java.lang.Boolean"); // NOI18N
        if (PrimitiveTypeKindEnum.BYTE.equals(kind)) return resolveType("java.lang.Byte"); // NOI18N
        if (PrimitiveTypeKindEnum.CHAR.equals(kind)) return resolveType("java.lang.Char"); // NOI18N
        if (PrimitiveTypeKindEnum.DOUBLE.equals(kind)) return resolveType("java.lang.Double"); // NOI18N
        if (PrimitiveTypeKindEnum.FLOAT.equals(kind)) return resolveType("java.lang.Float"); // NOI18N
        if (PrimitiveTypeKindEnum.INT.equals(kind)) return resolveType("java.lang.Integer"); // NOI18N
        if (PrimitiveTypeKindEnum.LONG.equals(kind)) return resolveType("java.lang.Long"); // NOI18N
        if (PrimitiveTypeKindEnum.SHORT.equals(kind)) return resolveType("java.lang.Short"); // NOI18N
        return null;
    }

/*
    public Object findItemAtCaretPos(JTextComponent target){
        Completion completion = ExtUtilities.getCompletion(target);
        if (completion != null) {
            if (completion.isPaneVisible()) { // completion pane visible
                Object item = getAssociatedObject(completion.getSelectedValue());
                if (item != null) {
                    return item;
                }
            } else { // pane not visible
                try {
                    SyntaxSupport sup = Utilities.getSyntaxSupport(target);
                    NbJavaJMICompletionQuery query = (NbJavaJMICompletionQuery)completion.getQuery();

                    int dotPos = target.getCaret().getDot();
                    BaseDocument doc = (BaseDocument)target.getDocument();
                    int[] idFunBlk = NbEditorUtilities.getIdentifierAndMethodBlock(doc, dotPos);

                    if (idFunBlk == null) {
                        idFunBlk = new int[] { dotPos, dotPos };
                    }

                    for (int ind = idFunBlk.length - 1; ind >= 1; ind--) {
                        CompletionQuery.Result result = query.query(target, idFunBlk[ind], sup, true);
                        if (result != null && result.getData().size() > 0) {
                            Object itm = getAssociatedObject(result.getData().get(0));
                            if (result.getData().size() > 1 && (itm instanceof Constructor || itm instanceof Method)) {
                                // It is overloaded method, lets check for the right one
                                int endOfMethod = JCExtension.findEndOfMethod(target, idFunBlk[ind]);
                                if (endOfMethod > -1){
                                    CompletionQuery.Result resultx = query.query(target, endOfMethod, sup, true);
                                    if (resultx != null && resultx.getData().size() > 0) {
                                        return getAssociatedObject(resultx.getData().get(0));
                                    }
                                }
                            }
                            return itm;
                        }
                    }
                } catch (BadLocationException e) {
                }
*/
    public static Object findItemAtCaretPos(JTextComponent target){
        try {
            SyntaxSupport sup = Utilities.getSyntaxSupport(target);
            NbJavaJMICompletionQuery query = new NbJavaJMICompletionQuery(true);
            
            int dotPos = target.getCaret().getDot();
            BaseDocument doc = (BaseDocument)target.getDocument();
            int[] idFunBlk = NbEditorUtilities.getIdentifierAndMethodBlock(doc, dotPos);
            
            if (idFunBlk == null) {
                idFunBlk = new int[] { dotPos, dotPos };
            }
            
            if (idFunBlk.length == 3) {
                int endOfMethod = JCExtension.findEndOfMethod(target, idFunBlk[2]);
                if (endOfMethod > -1)
                    dotPos = endOfMethod;
            } else if (idFunBlk.length == 2) {
                dotPos = idFunBlk[1];
            }
            CompletionQuery.Result result = query.query(target, dotPos, sup, true);
            if (result != null && result.getData().size() > 0) {
                return getAssociatedObject(result.getData().get(0));
            }
        } catch (BadLocationException e) {
        }
        // Complete the messages
        return null;

    }

    private static Object getAssociatedObject(Object item) {
        if (item instanceof NbJMIResultItem){
            Object ret = ((NbJMIResultItem)item).getAssociatedObject();
            if (ret instanceof FakeConstructor)
                ret = ((FakeConstructor)ret).getDeclaringClass();
            if (ret != null)
                return ret;
        }
        return item;
    }

    private static void refreshSettings() {
        Class kitClass = JavaKit.class;
        caseSensitive = SettingsUtil.getBoolean(kitClass,
            ExtSettingsNames.COMPLETION_CASE_SENSITIVE,
            ExtSettingsDefaults.defaultCompletionCaseSensitive);
        naturalSort = SettingsUtil.getBoolean(kitClass,
            ExtSettingsNames.COMPLETION_NATURAL_SORT,
            ExtSettingsDefaults.defaultCompletionNaturalSort);
        showDeprecated = SettingsUtil.getBoolean(kitClass,
            ExtSettingsNames.SHOW_DEPRECATED_MEMBERS,
            ExtSettingsDefaults.defaultShowDeprecatedMembers);
    }

    public static int getDefaultSelectionWeight(JavaClass cls) {
        int weight = 500;
        String fullName = cls.getName();
        if (fullName.startsWith("java.lang.") || fullName.startsWith("java.util.")) // NOI18N
            weight += 100;
        else if (fullName.startsWith("org.omg.") || fullName.startsWith("org.apache.")) // NOI18N
            weight -= 100;
        else if (fullName.startsWith("com.sun.") || fullName.startsWith("com.ibm.") || fullName.startsWith("com.apple.")) // NOI18N
            weight -= 200;
        else if (fullName.startsWith("sun.") || fullName.startsWith("sunw.") || fullName.startsWith("netscape.")) // NOI18N
            weight -= 300;
        if (!cls.isInner())
            weight += 1000;
        return weight;
    }

    public static Comparator getNaturalMemberNameComparator() {
        return naturalSort ? INSENSITIVE_NATURAL_MEMBER_NAME_COMPARATOR : NATURAL_MEMBER_NAME_COMPARATOR;
    }

    private static final class SettingsListener implements SettingsChangeListener {
        public SettingsListener () {}
        
        public void settingsChange(SettingsChangeEvent evt) {
            refreshSettings();
        }
    }

    private final class ClassesIterator implements Iterator {

        private LinkedList toVisit = new LinkedList();
        private HashSet addedToVisit = new HashSet();
        private boolean inspectSuperTypes;
        private boolean preferSources;
        private JavaClass objectClass = (JavaClass)resolveType("java.lang.Object"); // NOI18N

        public ClassesIterator(Type type, boolean inspectSuperTypes, boolean preferSources) {
            if (type instanceof ClassDefinition) {
                toVisit.add(type);
                addedToVisit.add(type.getName());
            }
            this.inspectSuperTypes = inspectSuperTypes;
            this.preferSources = preferSources;
        }

        public boolean hasNext() {
            return !toVisit.isEmpty();
        }

        public Object next() {
            ClassDefinition next = (ClassDefinition)toVisit.removeFirst();
            if (next == null)
                throw new NoSuchElementException();
            if (inspectSuperTypes) {
                JavaClass superCls = next.getSuperClass();
                if (superCls == null)
                    superCls = objectClass;
                String name = superCls.getName();
                if (!addedToVisit.contains(name)) {
                    toVisit.add(superCls);
                    addedToVisit.add(name);
                }
                for (Iterator it = next.getInterfaces().iterator(); it.hasNext();) {
                    JavaClass iface = (JavaClass) it.next();
                    name = iface.getName();
                    if (!addedToVisit.contains(name)) {
                        toVisit.add(iface);
                        addedToVisit.add(name);
                    }
                }
            }
            if (preferSources)
                next = JMIUtils.getSourceElementIfExists(next);
            return next;
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    private static final class NaturalMemberNameComparator implements Comparator {

        private boolean sensitive;

        public NaturalMemberNameComparator(boolean sensitive) {
            this.sensitive = sensitive;
        }

        public int compare(Object o1, Object o2) {
            if (o1 == o2) {
                return 0;
            }
            int order = getOrderLevel(o1) - getOrderLevel(o2);
            if (order != 0)
                return order;
            if (o1 instanceof NbJMIResultItem) {
                if (o2 instanceof NbJMIResultItem) {
                    String o1Text = ((NbJMIResultItem) o1).getItemText();
                    String o2Text = ((NbJMIResultItem) o2).getItemText();
                    order = sensitive ?
                        o1Text.compareTo(o2Text) :
                        o1Text.compareToIgnoreCase(o2Text);
                    if (order != 0)
                        return order;
                    List param1 = o1 instanceof NbJMIResultItem.CallableFeatureResultItem ? ((NbJMIResultItem.CallableFeatureResultItem)o1).getParams() : Collections.EMPTY_LIST;
                    List param2 = o2 instanceof NbJMIResultItem.CallableFeatureResultItem ? ((NbJMIResultItem.CallableFeatureResultItem)o2).getParams() : Collections.EMPTY_LIST;

                    int commonCnt = Math.min(param1.size(), param2.size());
                    for (int i = 0; i < commonCnt; i++) {
                        order = sensitive ?
                                ((NbJMIResultItem.ParamStr)param1.get(i)).getTypeName().compareTo(((NbJMIResultItem.ParamStr)param2.get(i)).getTypeName()) :
                                ((NbJMIResultItem.ParamStr)param1.get(i)).getTypeName().compareToIgnoreCase(((NbJMIResultItem.ParamStr)param2.get(i)).getTypeName());
                        if (order != 0) {
                            return order;
                        }
                    }
                    order = param1.size() - param2.size();
                    if (order != 0)
                        return order;
                    //do not allow methods merge
                    int sameName = o1Text.compareTo(o2Text);
                    if (sameName != 0)
                        return sameName;
                    o1 = ((NbJMIResultItem)o1).getAssociatedObject();
                    o2 = ((NbJMIResultItem)o2).getAssociatedObject();
                } else {
                    o1 = ((NbJMIResultItem)o1).getAssociatedObject();
                }
            } else if (o2 instanceof NbJMIResultItem) {
                o2 = ((NbJMIResultItem)o2).getAssociatedObject();
            }
            if (o1 instanceof Feature && o2 instanceof Feature){
                Feature f1 = (Feature)o1;
                Feature f2 = (Feature)o2;
                String f1Name = f1 instanceof JavaClass ? ((JavaClass)f1).getSimpleName() : f1.getName();
                if (f1Name == null)
                    f1Name = ""; // NOI18N
                String f2Name = f2 instanceof JavaClass ? ((JavaClass)f2).getSimpleName() : f2.getName();
                if (f2Name == null)
                    f2Name = ""; // NOI18N
                order = sensitive ?
                    f1Name.compareTo(f2Name) :
                    f1Name.compareToIgnoreCase(f2Name);
                if (order != 0 )
                    return order;
                if (f1 instanceof JavaClass && f2 instanceof JavaClass) {
                    f1Name = f1.getName();
                    f2Name = f2.getName();
                    return sensitive ? f1Name.compareTo(f2Name) : f1Name.compareToIgnoreCase(f2Name);
                }

                List param1 = f1 instanceof CallableFeature ? ((CallableFeature)f1).getParameters() : Collections.EMPTY_LIST;
                Object[] params1=param1.toArray();
                List param2 = f2 instanceof CallableFeature ? ((CallableFeature)f2).getParameters() : Collections.EMPTY_LIST;
                Object[] params2=param2.toArray();
                
                int commonCnt = Math.min(params1.length, params2.length);
                for (int i = 0; i < commonCnt; i++) {

                    String param1Name = ((Parameter)params1[i]).getType().getName();
                    if (((Parameter)params1[i]).isVarArg())
                        param1Name += "..."; //NOI18N
                    String param2Name = ((Parameter)params2[i]).getType().getName();
                    if (((Parameter)params2[i]).isVarArg())
                        param2Name += "..."; //NOI18N
                    order = sensitive ?
                            param1Name.compareTo(param2Name) :
                            param1Name.compareToIgnoreCase(param2Name);

                    if (order != 0) {
                        return order;
                    }
                }
                order = params1.length - params2.length;
                if (order != 0)
                    return order;

                //do not allow methods merge
                int sameName = f1Name.compareTo(f2Name);
                if (sameName != 0)
                    return sameName;

                return order;
            }
            if (o1 instanceof NamedElement && o2 instanceof NamedElement) {
                return sensitive ? ((NamedElement)o1).getName().compareTo(((NamedElement)o2).getName()) :
                    ((NamedElement)o1).getName().compareToIgnoreCase(((NamedElement)o2).getName());
            }
            return 0;
        }

        private int getOrderLevel(Object o) {
            if (o instanceof NbJMIResultItem.FieldResultItem || o instanceof Field)
                return 10;
            if (o instanceof NbJMIResultItem.ConstructorResultItem || o instanceof Constructor)
                return 20;
            if (o instanceof NbJMIResultItem.MethodResultItem || o instanceof Method)
                return 30;
            if (o instanceof NbJMIResultItem.ClassResultItem || o instanceof JavaClass)
                return 40;
            return 50;
        }

    }

    private static class FakeMethod extends FakeCallableFeature implements Method {

        public FakeMethod(String name, ClassDefinition declClass, Type retType, List params, JavaDoc javaDoc) {
            super(name, declClass, retType, params, Modifier.PUBLIC | Modifier.STATIC, javaDoc);
        }

        public boolean signatureEquals(Method method) {
            return false;
        }

        public Collection findDependencies(boolean usages, boolean overridingMethods, boolean fromBaseClass) {
            return null;
        }

        public TypeReference getTypeName() {
            return null;
        }

        public void setTypeName(TypeReference newValue) {
        }

        public int getDimCount() {
            return 0;
        }

        public void setDimCount(int newValue) {
        }
    }

    static class FakeConstructor extends FakeCallableFeature implements Constructor {

        public FakeConstructor(ClassDefinition declClass, JavaDoc javaDoc) {
            super(null, declClass, declClass, Collections.EMPTY_LIST, Modifier.PUBLIC, javaDoc);
        }
    }

    private static class FakeCallableFeature extends FakeTypedBase implements CallableFeature {

        private ClassDefinition declClass;
        private int modifiers;
        private List params;
        private JavaDoc javaDoc;

        public FakeCallableFeature(String name, ClassDefinition declClass, Type retType, List params, int modifiers, JavaDoc javaDoc) {
            super(name, retType);
            this.declClass = declClass;
            this.params = params;
            this.modifiers = modifiers;
            this.javaDoc = javaDoc;
        }

        public List getParameters() {
            return params;
        }

        public List getExceptionNames() {
            return Collections.EMPTY_LIST;
        }

        public List getExceptions() {
            return Collections.EMPTY_LIST;
        }

        public StatementBlock getBody() {
            return null;
        }

        public void setBody(StatementBlock newValue) {
        }

        public String getBodyText() {
            return null;
        }

        public void setBodyText(String newValue) {
        }

        public boolean isDeprecated() {
            return false;
        }

        public void setDeprecated(boolean newValue) {
        }

        public ClassDefinition getDeclaringClass() {
            return declClass;
        }

        public int getModifiers() {
            return modifiers;
        }

        public void setModifiers(int newValue) {
        }

        public String getJavadocText() {
            return null;
        }

        public void setJavadocText(String newValue) {
        }

        public JavaDoc getJavadoc() {
            return javaDoc;
        }

        public void setJavadoc(JavaDoc newValue) {
        }

        public List getTypeParameters() {
            return Collections.EMPTY_LIST;
        }

    }

    private static class FakeParameter extends FakeTypedBase implements Parameter {

        public FakeParameter(String name, Type type) {
            super(name, type);
        }

        public boolean isVarArg() {
            return false;
        }

        public void setVarArg(boolean newValue) {
        }

        public boolean isFinal() {
            return false;
        }

        public void setFinal(boolean newValue) {
        }

        public TypeReference getTypeName() {
            return null;
        }

        public void setTypeName(TypeReference newValue) {
        }

        public int getDimCount() {
            return 0;
        }

        public void setDimCount(int newValue) {
        }
    }

    private static class FakeBase {
        public FakeBase () {}
        
        public Element duplicate() {
            return null;
        }

        public List getChildren() {
            return Collections.EMPTY_LIST;
        }

        public int getEndOffset() {
            return 0;
        }

        public int getPartEndOffset(ElementPartKind part) {
            return 0;
        }

        public int getPartStartOffset(ElementPartKind part) {
            return 0;
        }

        public Resource getResource() {
            return null;
        }

        public int getStartOffset() {
            return 0;
        }

        public boolean isValid() {
            return true;
        }

        public void replaceChild(Element oldChild, Element newChild) {
        }

        public RefClass refClass() {
            return null;
        }

        public void refDelete() {
        }

        public RefFeatured refImmediateComposite() {
            return null;
        }

        public boolean refIsInstanceOf(RefObject refObject, boolean b) {
            return false;
        }

        public RefFeatured refOutermostComposite() {
            return null;
        }

        public Object refGetValue(RefObject refObject) {
            return null;
        }

        public Object refGetValue(String s) {
            return null;
        }

        public Object refInvokeOperation(RefObject refObject, List list) throws RefException {
            return null;
        }

        public Object refInvokeOperation(String s, List list) throws RefException {
            return null;
        }

        public void refSetValue(RefObject refObject, Object o) {
        }

        public void refSetValue(String s, Object o) {
        }

        public RefPackage refImmediatePackage() {
            return null;
        }

        public RefObject refMetaObject() {
            return null;
        }

        public String refMofId() {
            return null;
        }

        public RefPackage refOutermostPackage() {
            return null;
        }

        public Collection refVerifyConstraints(boolean b) {
            return null;
        }
    }

    private static class FakeTypedBase extends FakeBase {

        private String name;
        private Type type;

        public FakeTypedBase(String name, Type type) {
            this.name = name;
            this.type = type;
        }

        public List getAnnotations() {
            return Collections.EMPTY_LIST;
        }

        public String getName() {
            return name;
        }

        public void setName(String newValue) {
        }

        public Collection getReferences() {
            return null;
        }

        public Type getType() {
            return type;
        }

        public void setType(Type newValue) {
        }

        public String refMofId() {
            return "JMIUtils-TypedBase-" + getName(); // NOI18N
        }
    }

    private static class FakeJavaDoc extends FakeBase implements JavaDoc {

        private List tags;
        private String text;

        public FakeJavaDoc(List tags, String text) {
            this.tags = tags;
            this.text = text;
        }

        public List getTags() {
            return tags;
        }

        public String getText() {
            return text;
        }

        public void setText(String newValue) {
        }
    }

    private static class FakeTagValue extends FakeBase implements TagValue {

        private TagDefinition def;;
        private String value;

        public FakeTagValue(TagDefinition def, String value) {
            this.def = def;
            this.value = value;
        }

        public TagDefinition getDefinition() {
            return def;
        }

        public String getValue() {
            return value;
        }

        public void setDefinition(TagDefinition newValue) {
        }

        public void setValue(String newValue) {
        }
    }

}
