/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.editor.ext.java;

import java.awt.Color;
import java.awt.Component;
import java.util.ArrayList;
import java.util.List;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.Formatter;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.ext.CompletionQuery;
import org.netbeans.editor.ext.ExtFormatter;

/**
 *
 * @author  Martin Roskanin
 */

public abstract class JCResultItem
implements CompletionQuery.ResultItem, CompletionQuery.ResultItemAssociatedObject {
    
    Object associatedObject;
    private static final Color KEYWORD_COLOR = Color.darkGray;
    private static final Color TYPE_COLOR = Color.black;
    
    
    public JCResultItem(Object associatedObject){
        this.associatedObject = associatedObject;
    }
    
    
    public abstract String getItemText();
    
    protected abstract Component getPaintComponent(boolean isSelected);
    
    public Object getAssociatedObject(){
        return associatedObject;
    }
    
    protected String getReplaceText() {
        return getItemText();
    }
    
    
    public boolean substituteCommonText(JTextComponent c, int offset, int len, int subLen) {
        // [PENDING] not enough info in parameters...
        // commonText
        // substituteExp
        return false;
    }
    
    public boolean substituteText(JTextComponent c, int offset, int len, boolean shift) {
        BaseDocument doc = (BaseDocument)c.getDocument();
        String text = getReplaceText();
        int selectionStartOffset = -1;
        int selectionEndOffset = -1;
        
        if (text != null) {
            // Update the text
            doc.atomicLock();
            try {
                String textToReplace = doc.getText(offset, len);
                if (text.equals(textToReplace)) return false;
                
                doc.remove(offset, len);
                doc.insertString(offset, text, null);
                if (selectionStartOffset >= 0) {
                    c.select(offset + selectionStartOffset,
                    offset + selectionEndOffset);
                }
            } catch (BadLocationException e) {
                // Can't update
            } finally {
                doc.atomicUnlock();
            }
        }
        
        return true;
    }
    
    public java.awt.Component getPaintComponent(javax.swing.JList list, boolean isSelected, boolean cellHasFocus) {
        Component ret;
        ret = getPaintComponent(isSelected);
        if (ret==null) return null;
        if (isSelected) {
            ret.setBackground(list.getSelectionBackground());
            ret.setForeground(list.getSelectionForeground());
        } else {
            ret.setBackground(list.getBackground());
            ret.setForeground(list.getForeground());
        }
        ret.getAccessibleContext().setAccessibleName(getItemText());
        ret.getAccessibleContext().setAccessibleDescription(getItemText());
        return ret;
    }
    
    protected Color getTypeColor(JCClass cls) {
        String s = cls.getName();
        return (JavaCompletion.isPrimitiveClassName(s))
        ? KEYWORD_COLOR : TYPE_COLOR;
    }
    
    protected static String getTypeName(JCType typ) {
        return typ.format(false);
    }
    
    
    
    public static class FieldResultItem extends JCResultItem{
        
        private String typeName;
        private Color typeColor;
        private String fldName;
        private int modifiers;
        private boolean isDeprecated;
        
        private static JCPaintComponent.FieldPaintComponent fieldComponent = null;
        
        public FieldResultItem(JCField fld){
            super(fld);
            this.fldName = fld.getName();
            this.modifiers = fld.getModifiers();
            this.typeName = getTypeName(fld.getType());
            this.typeColor = getTypeColor(fld.getType().getClazz());
            this.isDeprecated = JCUtilities.isDeprecated(fld);
        }

        public String getItemText() {
            return fldName;
        }
        
        protected JCPaintComponent.FieldPaintComponent createPaintComponent(){
            return new JCPaintComponent.FieldPaintComponent();
        }
        
        public java.awt.Component getPaintComponent(boolean isSelected) {
            if (fieldComponent == null){
                fieldComponent = createPaintComponent();
            }
            fieldComponent.setTypeName(typeName);
            fieldComponent.setName(fldName);
            fieldComponent.setTypeColor(typeColor);
            fieldComponent.setModifiers(modifiers);
            fieldComponent.setSelected(isSelected);
            fieldComponent.setDeprecated(isDeprecated);
            
            return fieldComponent;
        }
    }
    
    public static class MethodResultItem extends ConstructorResultItem{
        
        private static JCPaintComponent.MethodPaintComponent mtdComponent = null;
        private String typeName;
        private Color typeColor;
        private String mtdName;
       
       
        public MethodResultItem(JCMethod mtd, JCExpression substituteExp){
            super(mtd, substituteExp);
            typeName = getTypeName(mtd.getReturnType());
            mtdName = mtd.getName();
            typeColor = getTypeColor(mtd.getReturnType().getClazz());
        }

        public String getName(){
            return mtdName;
        }
        
        
        public String getItemText() {
            return getName();
        }

        protected JCPaintComponent.ConstructorPaintComponent createPaintComponent(){
            return new JCPaintComponent.MethodPaintComponent();
        }
        
        public Component getPaintComponent(boolean isSelected) {
            if (mtdComponent == null) {
                mtdComponent = (JCPaintComponent.MethodPaintComponent)createPaintComponent();
            }
            mtdComponent.setName(getName());
            mtdComponent.setModifiers(getModifiers());
            mtdComponent.setTypeName(typeName);
            mtdComponent.setTypeColor(typeColor);
            mtdComponent.setParams(getParams());
            mtdComponent.setExceptions(getExceptions());
            mtdComponent.setDeprecated(isDeprecated());
            mtdComponent.setSelected(isSelected);
            return mtdComponent;
        }
        
    }
    
    public static class ConstructorResultItem extends JCResultItem{
        
        private JCConstructor ctr;
        private JCExpression substituteExp;
        private List params = new ArrayList();
        private List excs = new ArrayList();
        private int modifiers;
        private boolean isDeprecated;
        private static JCPaintComponent.ConstructorPaintComponent ctrComponent = null;
        
        public ConstructorResultItem(JCConstructor ctr, JCExpression substituteExp){
            super(ctr);
            this.ctr = ctr;
            this.substituteExp = substituteExp;
            this.modifiers = ctr.getModifiers();
            isDeprecated = JCUtilities.isDeprecated(ctr);
            JCParameter prms[] = ctr.getParameters();
            for (int i=0; i<prms.length; i++) {
                JCParameter prm = (JCParameter) prms[i];
                JCType type = prm.getType();
                params.add(new ParamStr(type.format(true), type.format(false) , prm.getName(), getTypeColor(type.getClazz())));
            }
            JCClass excepts[] = ctr.getExceptions();
            for (int i=0; i<excepts.length; i++) {
                JCClass ex = (JCClass) excepts[i];
                excs.add(new ExcStr(ex.getName(), getTypeColor(ex)));
            }
            
        }
        
        public boolean isDeprecated(){
            return isDeprecated;
        }
        
        public int getModifiers(){
            return modifiers;
        }

        public String getName(){
            return ctr.getClazz().getName();
        }

        public List getParams(){
            return params;
        }
        
        public List getExceptions(){
            return excs;
        }
        
        public boolean substituteText(JTextComponent c, int offset, int len, boolean shift) {
            
            String text = null ;
            BaseDocument doc = (BaseDocument)c.getDocument();
            
            int selectionStartOffset = -1;
            int selectionEndOffset = -1;
            
            switch ((substituteExp != null) ? substituteExp.getExpID() : -1) {
            case JCExpression.METHOD:
                // no subst
                break;

            case JCExpression.METHOD_OPEN:
                JCParameter[] parms = ctr.getParameters();
                if (parms.length == 0) {
                    try {
                        int fnwpos = Utilities.getFirstNonWhiteFwd(doc, offset + len);
                        if (fnwpos > -1 && doc.getChars(fnwpos, 1)[0] == ')') { // NOI18N
                            text = doc.getText(offset + len, fnwpos + 1 - offset - len);
                            len = fnwpos + 1 - offset;
                        }
                    } catch (BadLocationException e) {
                    }
                    if (text == null)
                        text = ")"; // NOI18N
                } else { // one or more parameters
                    int ind = substituteExp.getParameterCount() - 1;
                    boolean addSpace = false;
                    boolean addClosingParen = false;
                    Formatter f = doc.getFormatter();
                    if (f instanceof ExtFormatter) {
                        Object o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.JAVA_FORMAT_SPACE_AFTER_COMMA);
                        if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                            addSpace = true;
                        }
                        o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.PAIR_CHARACTERS_COMPLETION);
                        if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                            addClosingParen = true;
                        }
                    }

                    try {
                        if (addSpace && (ind == 0 || (offset > 0
                                                      && Character.isWhitespace(doc.getText(offset - 1, 1).charAt(0))))
                           ) {
                            addSpace = false;
                        }
                    } catch (BadLocationException e) {
                    }

                    if (ind < parms.length) {
                        text = addSpace ? " " : ""; // NOI18N
                        selectionStartOffset = text.length();
                        text += parms[ind].getName();
                        selectionEndOffset = text.length();
                        if (addClosingParen && ind == parms.length - 1) {
                            String paramsText = null;
                            try {
                                int fnwpos = Utilities.getFirstNonWhiteFwd(doc, offset + len);
                                if (fnwpos > -1 && doc.getChars(fnwpos, 1)[0] == ')') { // NOI18N
                                    paramsText = doc.getText(offset + len, fnwpos + 1 - offset - len);
                                    text += paramsText;
                                    len = fnwpos + 1 - offset;
                                }
                            } catch (BadLocationException e) {
                            }
                            if (paramsText == null)
                                text += ')'; // NOI18N
                        }
                    }
                }
                break;

            default:
                text = getItemText();
                boolean addSpace = false;
                boolean addClosingParen = false;
                Formatter f = doc.getFormatter();
                if (f instanceof ExtFormatter) {
                    Object o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.JAVA_FORMAT_SPACE_BEFORE_PARENTHESIS);
                    if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                        addSpace = true;
                    }
                    o = ((ExtFormatter)f).getSettingValue(JavaSettingsNames.PAIR_CHARACTERS_COMPLETION);
                    if ((o instanceof Boolean) && ((Boolean)o).booleanValue()) {
                        addClosingParen = true;
                    }
                }

                String paramsText = null;
                try {
                    int fnwpos = Utilities.getFirstNonWhiteFwd(doc, offset + len);
                    if (fnwpos > -1 && doc.getChars(fnwpos, 1)[0] == '(') { // NOI18N
                        paramsText = doc.getText(offset + len, fnwpos + 1 - offset - len);
                        if (addSpace && paramsText.length() < 2)
                            text += ' '; // NOI18N
                        len = fnwpos + 1 - offset;
                        text += paramsText;
                    }
                } catch (BadLocationException e) {
                }
                parms = ctr.getParameters();
                if (paramsText == null) {
                    if (addSpace) {
                        text += ' ';
                    }
                    text += '(';

                    if (parms.length > 0) {
                        selectionStartOffset = text.length();
                        text += parms[0].getName();
                        selectionEndOffset = text.length();
                    }
                    if (parms.length == 0 || (addClosingParen && parms.length == 1)) {
                        text += ")"; // NOI18N
                    }
                } else {
                    try {
                        int fnwpos = Utilities.getFirstNonWhiteFwd(doc, offset + len);
                        if (fnwpos > -1 && doc.getChars(fnwpos, 1)[0] == ')') { // NOI18N
                            paramsText = doc.getText(offset + len, fnwpos + 1 - offset - len);
                            len = fnwpos + 1 - offset;
                            if (parms.length > 0) {
                                selectionStartOffset = text.length();
                                text += parms[0].getName();
                                selectionEndOffset = text.length();
                            }
                            text += paramsText;
                        }
                    } catch (BadLocationException e) {
                    }
                }
                break;
            }        
            
            
            if (text != null) {
                // Update the text
                doc.atomicLock();
                try {
                    String textToReplace = doc.getText(offset, len);
                    if (text.equals(textToReplace)) {
                        c.setCaretPosition(offset + len);
                        return false;
                    }
                    doc.remove(offset, len);
                    doc.insertString(offset, text, null);
                    if (selectionStartOffset >= 0) {
                        c.select(offset + selectionStartOffset,
                        offset + selectionEndOffset);
                    }
                    return true;
                } catch (BadLocationException e) {
                    // Can't update
                } finally {
                    doc.atomicUnlock();
                }
            }

            return false;
        }
        
        public String getItemText() {
            return ctr.getClazz().getName();
        }

        protected JCPaintComponent.ConstructorPaintComponent createPaintComponent(){
            return new JCPaintComponent.ConstructorPaintComponent();
        }
        
        public Component getPaintComponent(boolean isSelected) {
            if (ctrComponent == null) {
                ctrComponent = createPaintComponent();
            }
            ctrComponent.setName(getItemText());
            ctrComponent.setModifiers(getModifiers());
            ctrComponent.setParams(getParams());
            ctrComponent.setExceptions(getExceptions());
            ctrComponent.setDeprecated(isDeprecated);
            ctrComponent.setSelected(isSelected);
            return ctrComponent;
        }

        
    }
    
    public static class PackageResultItem extends JCResultItem{
        
        private boolean displayFullPackagePath;
        private JCPackage pkg;
        private String pkgName;
        private static JCPaintComponent.PackagePaintComponent pkgComponent;
        
        public PackageResultItem(JCPackage pkg, boolean displayFullPackagePath){
            super(pkg);
            this.pkg = pkg;
            this.displayFullPackagePath = displayFullPackagePath;
            this.pkgName = pkg.getName();
            
        }
        
        
        public String getItemText() {
            return displayFullPackagePath ? pkg.getName() : pkg.getLastName();
        }

        protected JCPaintComponent.PackagePaintComponent createPaintComponent(){
            return new JCPaintComponent.PackagePaintComponent();
        }
        
        public Component getPaintComponent(boolean isSelected) {
            if (pkgComponent == null) {
                pkgComponent = createPaintComponent();
            }
            pkgComponent.setSelected(isSelected);
            pkgComponent.setPackageName(pkgName);
            pkgComponent.setDisplayFullPackagePath(displayFullPackagePath);
            return pkgComponent;
        }
        
    }
    
    
    
    public static class ClassResultItem extends JCResultItem{
        
        private JCClass cls;
        private boolean isInterface;
        private int classDisplayOffset;
        private boolean isDeprecated;
        private boolean displayFQN;
        
        private static JCPaintComponent.ClassPaintComponent clsComponent = null;
        private static JCPaintComponent.InterfacePaintComponent interfaceComponent = null;
        
        public ClassResultItem(JCClass cls, boolean displayFQN){
            this(cls, 0, displayFQN);
        }
        
        public ClassResultItem(JCClass cls, int classDisplayOffset, boolean displayFQN){
            super(cls);
            this.cls = cls;
            this.isInterface = cls.isInterface();
            this.classDisplayOffset = classDisplayOffset;
            this.isDeprecated = JCUtilities.isDeprecated(cls);
            this.displayFQN = displayFQN;
        }


        protected String getName(){
            return cls.getName();
        }
        
        protected String getReplaceText(){
            String text = getItemText();
            if (classDisplayOffset > 0
            && classDisplayOffset < text.length()
            ) { // Only the last name for inner classes
                text = text.substring(classDisplayOffset);
            }
            return text;
        }
        
        public String getItemText() {
            return displayFQN ? cls.getFullName() : cls.getName();
        }
        

        protected JCPaintComponent.InterfacePaintComponent createInterfacePaintComponent(){
            return new JCPaintComponent.InterfacePaintComponent();
        }
        
        protected JCPaintComponent.ClassPaintComponent createClassPaintComponent(){
            return new JCPaintComponent.ClassPaintComponent();
        }
        
        public Component getPaintComponent(boolean isSelected) {
            if (isInterface){
                if (interfaceComponent == null){
                    interfaceComponent = createInterfacePaintComponent();
                }
                interfaceComponent.setSelected(isSelected);
                interfaceComponent.setDeprecated(isDeprecated);
                interfaceComponent.setFormatClassName(getName());
                return interfaceComponent;
            }else{
                if (clsComponent == null){
                    clsComponent = createClassPaintComponent();
                }
                clsComponent.setSelected(isSelected);
                clsComponent.setDeprecated(isDeprecated);
                clsComponent.setFormatClassName(getName());
                return clsComponent;
            }
        }
        
    }

    
    static class ParamStr {
        private String type, simpleType, prm;
        private Color typeColor;
        public ParamStr(String type, String simpleType, String prm, Color typeColor) {
            this.type = type;
            this.simpleType = simpleType;
            this.prm = prm;
            this.typeColor = typeColor;
        }
        
        public String getTypeName() {
            return type;
        }
        
        public String getSimpleTypeName() {
            return simpleType;
        }

        public String getName() {
            return prm;
        }
        
        public Color getTypeColor() {
            return typeColor;
        }
    }
    
    static class ExcStr {
        private String name;
        private Color typeColor;
        public ExcStr(String name, Color typeColor) {
            this.name = name;
            this.typeColor = typeColor;
        }
        
        public String getName() {
            return name;
        }
        
        public Color getTypeColor() {
            return typeColor;
        }
    }
    
}
