/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.editor.ext.java;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.font.TextAttribute;
import java.lang.reflect.Modifier;
import java.text.AttributedString;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JPanel;

/**
 *
 * @author  Martin Roskanin
 */



public class JCPaintComponent extends JPanel{
    
    protected int drawX;
    
    protected int drawY;
    
    protected int drawHeight;
    
    private Font drawFont;
    
    private int iconTextGap = 5;
    
    private int fontHeight;
    
    private int ascent;
    
    private Map widths;
    
    private FontMetrics fontMetrics;
    
    protected boolean isSelected;
    
    private String text;
    
    protected boolean isDeprecated;
    
    private static final String THROWS = " throws "; // NOI18N
    
    
    private static final String[] frequentWords = new String[] {
        "", " ", "[]", "(", ")", ", ", "String", THROWS // NOI18N
    };
    
    private static final Color KEYWORD_COLOR = Color.darkGray;
    private static final Color TYPE_COLOR = Color.black;
    
    private Icon icon;

    
    public JCPaintComponent(){
        super();
        setOpaque(true);
        setBorder(BorderFactory.createEmptyBorder(0, 3, 0, 3));
    }
    
    protected void setSelected(boolean isSelected){
        this.isSelected = isSelected;
    }
    
    protected void setDeprecated(boolean isDeprecated){
        this.isDeprecated = isDeprecated;
    }

    protected boolean isSelected(){
        return isSelected;
    }

    protected boolean isDeprecated(){
        return isDeprecated;
    }
    
    public void paintComponent(Graphics g) {
        // clear background
        g.setColor(getBackground());
        java.awt.Rectangle r = g.getClipBounds();
        g.fillRect(r.x, r.y, r.width, r.height);
        draw(g);
    }
    
    protected void draw(Graphics g){
    }

    /** @param level JavaCompletion.PUBLIC_LEVEL, JavaCompletion.PROTECTED_LEVEL ...
     */
    protected void setIcon(Icon icon){
       this.icon = icon;
    }

    protected Icon getIcon(){
        return icon;
    }
    
    
    /** Draw the icon if it is valid for the given type.
     * Here the initial drawing assignments are also done.
     */
    protected void drawIcon(Graphics g, Icon icon) {
        Insets i = getInsets();
        if (i != null) {
            drawX = i.left;
            drawY = i.top;
        } else {
            drawX = 0;
            drawY = 0;
        }
        
        if (icon != null) {
            if (g != null) {
                icon.paintIcon(this, g, drawX, drawY);
            }
            drawX += icon.getIconWidth() + iconTextGap;
            drawHeight = Math.max(fontHeight, icon.getIconHeight());
        } else {
            drawHeight = fontHeight;
        }
        if (i != null) {
            drawHeight += i.bottom;
        }
        drawHeight += drawY;
        drawY += ascent;
    }
    
    protected void drawType(Graphics g, JCType typ) {
        drawType(g, typ, false);
    }
    
    protected void drawType(Graphics g, JCType typ, boolean strike) {
        Color c = getTypeColor(typ.getClazz().getName());
        drawString(g, typ.format(false), c, null, strike);
    }
    
    protected void drawString(Graphics g, String s){
        drawString(g, s, false);
    }
    
    /** Draw string using the foreground color */
    protected void drawString(Graphics g, String s, boolean strike) {
        if (g != null) {
            g.setColor(getForeground());
        }
        drawStringToGraphics(g, s, null, strike);
    }
    
    
    /** Draw string with given color which is first possibly modified
     * by calling getColor() method to care about selection etc.
     */
    protected void drawString(Graphics g, String s, Color c) {
        if (g != null) {
            g.setColor(getColor(s, c));
        }
        drawStringToGraphics(g, s);
    }
    
    protected void drawString(Graphics g, String s, Color c, Font font, boolean strike) {
        if (g != null) {
            g.setColor(getColor(s, c));
            g.setFont(font);
        }
        drawStringToGraphics(g, s, font,  strike);
        if (g != null) {
            g.setFont(drawFont);
        }
        
    }
    
    protected void drawStringToGraphics(Graphics g, String s) {
        drawStringToGraphics(g, s, null, false);
    }
    
    protected void drawStringToGraphics(Graphics g, String s, Font font, boolean strike) {
        if (g != null) {
            if (!strike){
                g.drawString(s, drawX, drawY);
            }else{
                Graphics2D g2 = ((Graphics2D)g);
                AttributedString strikeText = new AttributedString(s);
                strikeText.addAttribute(TextAttribute.STRIKETHROUGH, TextAttribute.STRIKETHROUGH_ON);
                strikeText.addAttribute(TextAttribute.FONT, g.getFont());
                g2.drawString(strikeText.getIterator(), drawX, drawY);
            }
        }
        drawX += getWidth(s, font);
    }
    
    protected int getWidth(String s) {
        Integer i = (Integer)widths.get(s);
        if (i != null) {
            return i.intValue();
        } else {
            return fontMetrics.stringWidth(s);
        }
    }
    
    protected int getWidth(String s, Font font) {
        if (font == null) return getWidth(s);
        return getFontMetrics(font).stringWidth(s);
    }
    
    protected Color getColor(String s, Color defaultColor) {
        return isSelected ? getForeground()
        : defaultColor;
    }
    
    private void storeWidth(String s) {
        fontMetrics.stringWidth(s);
    }
    
    public void setFont(Font font) {
        super.setFont(font);
        
        fontMetrics = this.getFontMetrics(font);
        fontHeight = fontMetrics.getHeight();
        ascent = fontMetrics.getAscent();
        if (widths != null) {
            widths.clear();
        } else {
            widths = new HashMap();
        }
        for (int i = 0; i < frequentWords.length; i++) {
            storeWidth(frequentWords[i]);
        }
        Iterator i = JavaCompletion.getPrimitiveClassIterator();
        while (i.hasNext()) {
            storeWidth(((JCClass)i.next()).getName());
        }
        drawFont = font;
    }
    
    protected Font getDrawFont(){
        return drawFont;
    }
    
    protected Color getTypeColor(String s) {
        return (JavaCompletion.isPrimitiveClassName(s))
        ? KEYWORD_COLOR : TYPE_COLOR;
    }
    
    public Dimension getPreferredSize() {
        draw(null);
        Insets i = getInsets();
        if (i != null) {
            drawX += i.right;
        }
        return new Dimension(drawX, drawHeight);
    }
    
    public String toString(){
        StringBuffer sb = new StringBuffer();
        sb.append(text);
        return sb.toString();
    }
    
    //.................. INNER CLASSES .......................
    
    
    public static class ClassPaintComponent extends JCPaintComponent{
        
        String formatClassName;
        private Color CLASS_COLOR = Color.red.darker().darker().darker();
        private boolean displayFQN;
        
        public void setFormatClassName(String formatClassName){
            this.formatClassName = formatClassName;
        }

        protected Color getColor(){
            return CLASS_COLOR;
        }
        
        protected void draw(Graphics g){
            boolean strike = isDeprecated();
            drawIcon(g, getIcon());
            drawString(g, formatClassName, getColor(), null, strike);
        }
        
        public String toString(){
            StringBuffer sb = new StringBuffer();
            sb.append(formatClassName);
            return sb.toString();
        }
    }

    public static class InterfacePaintComponent extends ClassPaintComponent{
        
        private Color INTERFACE_COLOR = Color.darkGray;

        protected Color getColor(){
            return INTERFACE_COLOR;
        }
        
        public InterfacePaintComponent(){
            super();
        }        
    }
    
    public static class PackagePaintComponent extends JCPaintComponent{
        
        private JCPackage pkg;
        private String pkgName;
        private boolean displayFullPackagePath;
        private Color PACKAGE_COLOR = Color.green.darker().darker().darker();
        
        public PackagePaintComponent(){
            super();
        }
        
        public void setPackageName(String pkgName){
            this.pkgName = pkgName;
        }

        public void setDisplayFullPackagePath(boolean displayFullPackagePath){
            this.displayFullPackagePath = displayFullPackagePath;            
        }
        
        protected void draw(Graphics g){
            drawIcon(g, getIcon());
            String name = pkgName;
            if (!displayFullPackagePath) {
                name = name.substring(name.lastIndexOf('.') + 1);
            }
            drawString(g, name, PACKAGE_COLOR);
        }
        
        public String toString(){
            StringBuffer sb = new StringBuffer();
            String name = pkgName;
            if (!displayFullPackagePath) {
                name = name.substring(name.lastIndexOf('.') + 1);
            }
            sb.append(name);
            return sb.toString();
        }
    }
    
    public static class FieldPaintComponent extends JCPaintComponent{
        private Color FIELD_COLOR = Color.blue.darker();
        private String typeName;
        private Color typeColor;
        private String fldName;
        private int modifiers;
        
        public FieldPaintComponent(){
            super();
        }

        public void setName(String fldName){
            this.fldName= fldName;
        }
        
        public void setTypeColor(Color typeColor){
            this.typeColor = typeColor;
        }
        
        public void setTypeName(String typeName){
            this.typeName = typeName;
        }
        
        public void setModifiers(int modifiers){
            this.modifiers = modifiers;
        }
        
        public int getModifiers(){
            return modifiers;
        }
        
        protected void draw(Graphics g){
            
            boolean strike = isDeprecated();
            int level = JavaCompletion.getLevel(modifiers);
            drawIcon(g, getIcon());
            
            drawString(g, typeName, typeColor, null, strike);
            drawString(g, " ", strike); // NOI18N
            if ((modifiers & JavaCompletion.LOCAL_MEMBER_BIT) != 0){
                // it is local field, draw as bold
//                drawString(g, fldName, FIELD_COLOR, getDrawFont().deriveFont(Font.BOLD), strike); // Workaround for issue #55133
                drawString(g, fldName, FIELD_COLOR, new Font(getDrawFont().getName(), getDrawFont().getStyle() | Font.BOLD, getDrawFont().getSize()), strike);
            }else{
                drawString(g, fldName, FIELD_COLOR , null, strike);
            }
        }
        
        public String toString(){
            StringBuffer sb = new StringBuffer();
            sb.append(typeName);
            sb.append(" ");                 //NOI18N
            sb.append(fldName);                
            return sb.toString();
        }
    }
    
    
    public static class ConstructorPaintComponent extends JCPaintComponent{
        
        private Color CONSTRUCTOR_COLOR = Color.orange.darker();
        private Color PARAMETER_NAME_COLOR = Color.magenta.darker();
        private List params = new ArrayList();
        private List excs = new ArrayList();
        private int modifiers;
        private String name;
        
        public ConstructorPaintComponent(){
            super();
        }
        
        public int getMethodModifiers(){
            return modifiers;
        }
        
        public String getName(){
            return name;
        }
        
        public void setModifiers(int modifiers){
            this.modifiers = modifiers;
        }
        
        public int getModifiers(){
            return modifiers;
        }
        
        public void setName(String name){
            this.name = name;
        }
        
        public void setParams(List params){
            this.params = params;
        }
        
        public void setExceptions(List excs){
            this.excs = excs;
        }
        
        protected List getParamList(){
            return params;
        }
        
        protected List getExceptionList(){
            return excs;
        }
        
        protected void drawParameter(Graphics g, JCResultItem.ParamStr prm) {
            drawParameter(g, prm, false);
        }

        protected void drawParameter(Graphics g, JCResultItem.ParamStr prm, boolean strike) {

            //drawType
            drawString(g, prm.getSimpleTypeName(), prm.getTypeColor(), null, strike);
            
            String name = prm.getName();
            if (name != null && name.length() > 0) {
                drawString(g, " ", strike); // NOI18N
                drawString(g, prm.getName(), PARAMETER_NAME_COLOR, null, strike);
            }
        }

        protected void drawParameterList(Graphics g, List prmList) {
            drawParameterList(g, prmList, false);
        }

        protected void drawParameterList(Graphics g, List prmList, boolean strike) {
            drawString(g, "(", strike); // NOI18N
            for (Iterator it = prmList.iterator(); it.hasNext();) {
                drawParameter(g, (JCResultItem.ParamStr)it.next(), strike);
                if (it.hasNext()) {
                    drawString(g, ", ", strike); // NOI18N
                }
            }
            drawString(g, ")", strike); // NOI18N
        }

        protected void drawExceptions(Graphics g, List exc, boolean strike) {
            if (exc.size() > 0) {
                drawString(g, THROWS, KEYWORD_COLOR, null, strike);
                for (Iterator it = exc.iterator(); it.hasNext();) {
                    JCResultItem.ExcStr ex = (JCResultItem.ExcStr) it.next();
                    drawString(g, ex.getName(), ex.getTypeColor(), null, strike);
                    if (it.hasNext()) {
                        drawString(g, ", ", strike); // NOI18N
                    }

                }
            }
        }
        
        protected void draw(Graphics g){
            boolean strike = isDeprecated();
            int level = JavaCompletion.getLevel(getModifiers());
            drawIcon(g, getIcon());
            drawString(g, getName(), CONSTRUCTOR_COLOR, null, strike);
            drawParameterList(g, getParamList(), strike);
            drawExceptions(g, getExceptionList(), strike);
        }
        
        public String toString(){
            StringBuffer sb = new StringBuffer();
            sb.append(getName());
            sb.append(" (");                //NOI18N
            List prmList = getParamList();
            for (Iterator it = prmList.iterator(); it.hasNext();) {
                JCResultItem.ParamStr paramStr = (JCResultItem.ParamStr)it.next();
                sb.append(paramStr.getTypeName());
                sb.append(" ");             //NOI18N
                sb.append(paramStr.getName());
                if (it.hasNext()) {
                    sb.append(", ");        // NOI18N
                }
            }
            sb.append(")");                 //NOI18N
            return sb.toString();
        }
    }
    
    
    public static class MethodPaintComponent extends ConstructorPaintComponent{
        
        private Color PARAMETER_NAME_COLOR = Color.magenta.darker();
        private Color METHOD_COLOR = Color.red.darker().darker();
        private String typeName;
        private Color typeColor;
    
        public MethodPaintComponent(){
            super();
        }
        
        public String getTypeName(){
            return typeName;
        }

        public Color getTypeColor(){
            return typeColor;
        }

        public void setTypeName(String typeName){
            this.typeName = typeName;
        }
        
        public void setTypeColor(Color typeColor){
            this.typeColor = typeColor;
        }
        
        protected void draw(Graphics g){
            boolean strike = isDeprecated();
            int level = JavaCompletion.getLevel(getModifiers());
            drawIcon(g, getIcon());
            
            drawString(g, getTypeName(), getTypeColor(), null, strike);
            drawString(g, " ", strike); // NOI18N
            if ((getModifiers() & JavaCompletion.LOCAL_MEMBER_BIT) != 0){
//                drawString(g, getName(), METHOD_COLOR , getDrawFont().deriveFont(Font.BOLD), strike); // Workaround for issue #55133
                drawString(g, getName(), METHOD_COLOR , new Font(getDrawFont().getName(), getDrawFont().getStyle() | Font.BOLD, getDrawFont().getSize()), strike);
            }else{
                drawString(g, getName(), METHOD_COLOR, null, strike);
            }
            drawParameterList(g, getParamList(), strike);
            drawExceptions(g, getExceptionList(), strike);
        }
        
        public String toString(){
            StringBuffer sb = new StringBuffer();
            sb.append(typeName);
            sb.append(" ");                 //NOI18N
            sb.append(super.getName());
            sb.append(" (");                //NOI18N
            List prmList = getParamList();
            for (Iterator it = prmList.iterator(); it.hasNext();) {
                JCResultItem.ParamStr paramStr = (JCResultItem.ParamStr)it.next();
                sb.append(paramStr.getTypeName());
                sb.append(" ");             //NOI18N
                sb.append(paramStr.getName());
                if (it.hasNext()) {
                    sb.append(", ");        // NOI18N
                }
            }
            sb.append(")");                 //NOI18N
            return sb.toString();
        }
    }
    
}
