/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.editor.ext.java;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import org.netbeans.editor.SettingsUtil;
import org.netbeans.editor.ext.ExtSettingsDefaults;
import org.netbeans.editor.ext.ExtSettingsNames;

/** Compound finder works over more finders and merge the search results.
 *  
 *
 *  @author Martin Roskanin, David Konecny
 *  @since 10/2002
 */
public class CompoundFinder implements JCFinder {
    
    private List finders;
    
    /** Kit class from which for which the settings like case sensitivity etc. will be read */
    private Class kitClass;

    private static JCField artificialField = new JavaCompletion.BaseField(
        JavaCompletion.CLASS_CLASS, "class", JavaCompletion.CLASS_TYPE, Modifier.PUBLIC); // NOI18N

    /** Creates a new instance of CompoundFinder 
     *
     *  @param finders the List of the finders from which search results will be merged
     *  @param kitClass kit class to which this finder is associated and for which
     *     the settings will be read
     */
    public CompoundFinder(List finders, Class kitClass) {
        this.finders = finders;
        this.kitClass = kitClass;
        Iterator it = finders.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if (o instanceof JCBaseFinder) {
                ((JCBaseFinder)o).setParentFinder(this);
            } else {
                //System.out.println("TODO CompoundFinder   ");//assert false : "This should never happen"; // NOI18N
            }
        }
    }

    /** Find classes by name and possibly in some package.
     *  The search results result will be merged from all finders appended to the compound finder.
     * @param pkg package where the classes should be searched for. It can be null
     * @param begining of the name of the class. The package name must be omitted.
     * @param exactMatch whether the given name is the exact requested name
     *   of the class or not.
     * @return list of the matching classes
     *
     */
    public List findClasses(JCPackage pkg, String name, boolean exactMatch) {
        List ret = new ArrayList();
        for (int i = 0; i<finders.size(); i++) {
            JCFinder finder = ((JCFinder)finders.get(i));
            if (pkg == null) {
                ret.addAll(finder.findClasses(null, name, exactMatch));
            } else {
                // Each finder has its own instance of package with associated
                // classes. That's why I cannot use pkg here and has to first find
                // package from the finder and only then ask for its classes.
                List l = finder.findPackages(pkg.getName(), true, false);
                if (l.size() == 1) {
                    JCPackage packageFromFinder = (JCPackage)l.iterator().next();
                    ret.addAll(finder.findClasses(packageFromFinder, name, exactMatch));
                }
            }
        }
        Collections.sort(ret, getNaturalSort() ? JCBaseFinder.NATURAL_MEMBER_NAME_COMPARATOR : JCBaseFinder.CLASS_NAME_COMPARATOR);        
        return ret;
    }
    
    /** Find fields by name in a given class.
     * @param c class which is searched for the fields.
     * @param name start of the name of the field
     * @param exactMatch whether the given name of the field is exact
     * @param staticOnly whether search for the static fields only
     * @param inspectOuterClasses if the given class is inner class of some
     *   outer class, whether the fields of the outer class should be possibly
     *   added or not. This should be false when searching for 'this.'
     * @return list of the matching fields
     *
     */
    public List findFields(JCClass c, String name, boolean exactMatch, boolean staticOnly, boolean inspectOuterClasses) {
        List ret;
        List something = null;
        for (int i=finders.size(); i>0; i--) {
            ret = ((JCFinder)finders.get(i-1)).findFields(c, name, exactMatch, staticOnly, inspectOuterClasses);
            if (ret != null && ret.size() > 0) {
                if (ret.size() == 1 && ret.get(0).equals(artificialField)) {
                    // hmmm, let's try other finders and if no one provide
                    // better results then return at least something
                    something = ret;
                } else {
                    return ret;
                }
            }
        }
        if (something != null) {
            return something;
        }
        return new ArrayList();
    }
    
    /** Find methods by name in a given class.
     * @param c class which is searched for the methods.
     * @param name start of the name of the method
     * @param exactMatch whether the given name of the method is exact
     * @param staticOnly whether search for the static methods only
     * @param inspectOuterClasses if the given class is inner class of some
     *   outer class, whether the methods of the outer class should be possibly
     *   added or not. This should be false when searching for 'this.'
     * @return list of the matching methods
     *
     */
    public List findMethods(JCClass c, String name, boolean exactMatch, boolean staticOnly, boolean inspectOuterClasses) {
        List ret;
        for (int i=finders.size(); i>0; i--) {
            ret = ((JCFinder)finders.get(i-1)).findMethods(c, name, exactMatch, staticOnly, inspectOuterClasses);
            if (ret != null && ret.size() > 0) {
                return ret;
            }
        }
        return new ArrayList();
    }
    
    /** Get the list of packages that start with the given name
     * @param name the start of the requested package(s) name
     * @return list of the matching packages
     *
     */
    public List findPackages(String name, boolean exactMatch, boolean subPackages) {
        TreeSet ret = getNaturalSort() ? new TreeSet(JCBaseFinder.INSENSITIVE_CLASS_NAME_COMPARATOR) : new TreeSet();
        for (int i = 0; i<finders.size(); i++) {
            ret.addAll(((JCFinder)finders.get(i)).findPackages(name, exactMatch, subPackages));
        }
        return new ArrayList(ret);
    }
    
    
    /** Get the class from full name of the class.
     */
    public JCClass getExactClass(String classFullName) {
        JCClass cls;
        for (int i=finders.size(); i>0; i--) {
            cls = ((JCFinder)finders.get(i-1)).getExactClass(classFullName);
            if (cls != null) {
                return cls;
            }
        }
        return null;
   }
    
    /** Get the package from the package name  */
    public JCPackage getExactPackage(String packageName) {
        JCPackage pkg;
        for (int i=finders.size(); i>0; i--) {
            pkg = ((JCFinder)finders.get(i-1)).getExactPackage(packageName);
            if (pkg != null)  {
                return pkg;
            }
        }
        return null;
    }

    private boolean getNaturalSort() {
        return SettingsUtil.getBoolean(kitClass,
            ExtSettingsNames.COMPLETION_NATURAL_SORT,
            ExtSettingsDefaults.defaultCompletionNaturalSort);
    }
    
}
