/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.verification.persistence.rules.clazz;

import org.netbeans.jmi.javamodel.ClassMember;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.modules.j2ee.verification.AbstractRule;
import org.netbeans.modules.j2ee.verification.Problem;
import org.netbeans.modules.j2ee.verification.ProblemFinderContext;
import org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIHelper;
import org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIProblemFinder;
import org.netbeans.modules.j2ee.verification.persistence.BeanAccessType;
import org.netbeans.modules.j2ee.verification.persistence.predicates.clazz.AnnotatedAsIdClassPred;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * all fields found in IdClass must be also defined in the class that uses
 * that class as IdClass.
 *
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class AllFieldsOfIdClassDefinedInEntity extends AbstractRule {

    private List<String> missingFieldNames = new ArrayList<String>();

    public AllFieldsOfIdClassDefinedInEntity() {
        super(null, Arrays.asList(new AnnotatedAsIdClassPred()));
    }

    public String getId() {
        return PersistenceAPIProblemFinder.RuleNames.AllFieldsOfIdClassDefinedInEntity;
    }

    protected String getMessageKey() {
        return "MSG_MissingIdField"; // NOI18N
    }

    @Override protected boolean isApplicable(
            Element subject, ProblemFinderContext ctx) {
        return (ctx.isIsEntityClass() || ctx.isMappedSuperclass()) &&
                ctx.getBeanAccessType() == BeanAccessType.FIELD &&
                super.isApplicable(subject, ctx);
    }

    @Override public Problem apply(Element subject, ProblemFinderContext ctx) {
        if (isApplicable(subject, ctx)) {
            JavaClass javaClass = ctx.getMainJavaClass();
            JavaClass idClass = PersistenceAPIHelper.getIdClass(javaClass);
            if(idClass==null) return null; // unresolved class
            // See, we don't search in super class of idClass
            for (ClassMember cm : (List<ClassMember>) idClass.getContents()) {
                if (cm instanceof Field) {
                    Field idField = Field.class.cast(cm);
                    if (javaClass.getField(idField.getName(), false) == null) { // no inheritance of Id field
                        missingFieldNames.add(idField.getName());
                    }
                }
            }
            if (!missingFieldNames.isEmpty()) {
                return createProblem(ctx, subject);
            }
        }
        return null;
    }

    @Override protected Object[] getMessageParameters() {
        StringBuilder sb = new StringBuilder();
        for(String f : missingFieldNames) {
            sb.append(f).append(" "); // NOI18N
        }
        Object[] result = {sb.toString()};
        return result;
    }

    @Override public void reset() {
        missingFieldNames.clear();
    }
}
