/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.verification.ejb;

import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.j2ee.dd.api.ejb.DDProvider;
import org.netbeans.modules.j2ee.dd.api.ejb.EjbJar;
import org.netbeans.modules.j2ee.dd.api.ejb.EnterpriseBeans;
import org.netbeans.modules.j2ee.dd.api.ejb.Session;
import org.netbeans.modules.javacore.api.JavaModel;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;

import java.io.IOException;
import java.util.*;

import static org.netbeans.modules.j2ee.verification.ejb.EJBAPIAnnotations.*;
import static org.netbeans.modules.j2ee.verification.ProblemFindingUtils.*;
import static org.netbeans.modules.j2ee.verification.JEEVerificationAnnotationProvider.*;

/**
 * Utility class
 *
 * @author Sanjeeb.Sahoo@Sun.COM
 */
public class EJBApiHelper {
    
    public static EjbJar getEjbJar(JavaClass javaClass) {
        final FileObject fileObject = JavaModel.getFileObject(
                javaClass.getResource());
        
        if (fileObject != null){
            try {
                org.netbeans.modules.j2ee.api.ejbjar.EjbJar apiEjbJar =
                        org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(
                        fileObject);
                
                if (apiEjbJar != null){
                    return DDProvider.getDefault().getMergedDDRoot(apiEjbJar.getMetadataUnit());
                }
            } catch (IOException e) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
            }
        }
        
        return null;
    }
    
    /**
     * Is this class a session bean?
     *
     * @param javaClass
     * @return
     */
    public static boolean isSessionBean(JavaClass javaClass) {
        // we don't use DD because that seems noot to be complete at this point
        // of development.
        Set annotations = new TreeSet(Arrays.asList(STATEFUL, STATELESS));
        return findFirstAnnotationFromGivenSet(javaClass, annotations) != null;
    }
    
    public static boolean isStateless(JavaClass javaClass) {
        Set annotations = Collections.singleton(STATELESS);
        return findFirstAnnotationFromGivenSet(javaClass, annotations) != null;
        
        // we don't use DD because that seems noot to be complete at this point
        // of development.
        
//        EjbJar ejbjar = getEjbJar(javaClass);
//        if (ejbjar != null) {
//            final EnterpriseBeans enterpriseBeans = ejbjar.getEnterpriseBeans();
//            if(enterpriseBeans != null) {
//                for (Session s : enterpriseBeans.getSession()) {
//                    if (javaClass.getName().equals(s.getEjbClass())) {
//                        return SESSION_TYPE_STATEFUL.equals(s.getSessionType());
//                    }
//                }
//            }
//        }
//        return false;
    }
    
    public static boolean isStateful(JavaClass javaClass) {
        Set annotations = Collections.singleton(STATEFUL);
        return findFirstAnnotationFromGivenSet(javaClass, annotations) != null;
    }
    
    /**
     * Utility method to find if the given class is used as a remote business
     * interface. The implementation is such that it returns false when
     * javaClass is not an interface.
     *
     * @param javaClass
     * @return true if this class is a remote business interface or not.
     */
    public static boolean isRBI(JavaClass javaClass) {
        // return immediately if this is not an interface
        if (!javaClass.isInterface()) return false;
        boolean isAnnotated = findAnnotation(javaClass, REMOTE) != null;
        if (isAnnotated) {
            return true;
        }
        
        // if it is not annotated, then let's ask the model to see if any
        // session bean uses it as a remote interface.
        EjbJar ejbJar = getEjbJar(javaClass);
        if (ejbJar != null) {
            EnterpriseBeans ejbs = ejbJar.getEnterpriseBeans();
            if (ejbs != null) {
                Session[] sbs = ejbs.getSession();
                if (sbs != null) {
                    String inputName = javaClass.getName();
                    for (Session sb : sbs) {
                        if (inputName.equals(getBusinessRemote(sb))) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }
    
    /**
     * Utility method to find if the given class is used as a local business
     * interface. The implementation is such that it returns false when
     * javaClass is not an interface.
     *
     * @param javaClass
     * @return true if this class is a local business interface or not.
     */
    public static boolean isLBI(JavaClass javaClass) {
        // return immediately if this is not an interface
        if (!javaClass.isInterface()) return false;
        boolean isAnnotated = findAnnotation(javaClass, LOCAL) != null;
        if (isAnnotated) {
            return true;
        }
        
        // if it is not annotated, then let's ask the model to see if any
        // session bean uses it as a local interface.
        EjbJar ejbJar = getEjbJar(javaClass);
        if (ejbJar != null) {
            EnterpriseBeans ejbs = ejbJar.getEnterpriseBeans();
            if (ejbs != null) {
                Session[] sbs = ejbs.getSession();
                if (sbs != null) {
                    String inputName = javaClass.getName();
                    for (Session sb : sbs) {
                        if (inputName.equals(getBusinessLocal(sb))) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }
    
    /**
     * A very important utility method to find out local and remote business
     * interfaces of a 3.0 session bean. As per the spec: the remote and local
     * business interface processing is based on the bean class, not any
     * super-classes. The same goes for the component-defining annotations,
     * RemoteHome, LocalHome, type-level Interceptors. As in prior versions
     * of EJB, inheritance is allowed from the bean class, but only as
     * implementation inheritance.
     *
     * @param javaClass the session bean whose business interfaces will be
     *                  determined
     * @param lbis      the output parameter, which will be populated with the
     *                  names of local business interfaces of this bean
     * @param rbis      the output parameter, which will be populated with the
     *                  names of remote business interfaces of this bean
     */
    public static void getBusinessInterfaces(
            JavaClass javaClass,
            Set<String> lbis,
            Set<String> rbis) {
        List<JavaClass> interfaces = getNonStandardInterfaces(javaClass);
        Annotation lNN = findAnnotation(javaClass, LOCAL);
        Annotation rNN = findAnnotation(javaClass, REMOTE);
        List<JavaClass> lNNValues = null;
        List<JavaClass> rNNValues = null;
        if (lNN != null) lNNValues = readValueForRemoteOrLocal(lNN);
        if (rNN != null) rNNValues = readValueForRemoteOrLocal(rNN);
        switch (interfaces.size()) {
            case 0:
            {
                getBusinessInterfaces0(javaClass, lbis, rbis, lNNValues,
                        rNNValues);
                break;
            }
            case 1:
            {
                getBusinessInterfaces1(javaClass, lbis, rbis, lNNValues,
                        rNNValues, interfaces);
                break;
            }
            default :
            {
                getBusinessInterfacesN(javaClass, lbis, rbis, lNNValues,
                        rNNValues, interfaces);
                break;
            }
        }
        StringBuilder sb = new StringBuilder(javaClass.getName() +
                " has following local business interface(s) [ "); // NOI18N
        for (String jc : lbis) {
            sb.append(jc).append(" "); // NOI18N
        }
        sb.append("] and following remote business interface(s) [ "); // NOI18N
        for (String jc : rbis) {
            sb.append(jc).append(" "); // NOI18N
        }
        sb.append("]"); // NOI18N
        tmpDbg(sb.toString());
    }
    
    private static void getBusinessInterfaces0(
            JavaClass javaClass,
            Set<String> lbis,
            Set<String> rbis,
            List<JavaClass> lNNValues,
            List<JavaClass> rNNValues) {
        // class does not implement any interface.
        // So the what ever is specified in annotation, becomes the value
        // of business interface.
        if (lNNValues != null) {
            for (JavaClass c : lNNValues) lbis.add(c.getName());
        }
        if (rNNValues != null) {
            for (JavaClass c : rNNValues) rbis.add(c.getName());
        }
    }
    
    private static void getBusinessInterfaces1(
            JavaClass javaClass,
            Set<String> lbis,
            Set<String> rbis,
            List<JavaClass> lNNValues,
            List<JavaClass> rNNValues,
            List<JavaClass> interfaces) {
        assert(interfaces.size() == 1);
        JavaClass i = interfaces.get(0);
        String iName = i.getName();
        
        // calculate remote business interfaces
        if (rNNValues != null) { // bean is annotated as @Remote
            if (rNNValues.isEmpty()) { // e.g. @Remote class Bean implements I {...
                rbis.add(iName);
            } else { // e.g. @Remote({I1.class, I2.class}) class Bean implements I {...
                for (JavaClass c : rNNValues) rbis.add(c.getName());
            }
        }
        // check if the interface is annotated as @Remote
        if (findAnnotation(i, REMOTE) != null) { // the interface is annotated as @Remote
            rbis.add(iName);
        }
        
        // calculate local business interfaces
        if (lNNValues != null) { // bean is annotated as @Local
            if (lNNValues.isEmpty()) { // e.g. @Local class Bean implements I {...
                lbis.add(iName);
            } else { // e.g. @Local({I3.class, I4.class}) class Bean implements I {...
                for (JavaClass c : lNNValues) lbis.add(c.getName());
            }
        }
        
        // check if the interface is annotated as @Local
        if (findAnnotation(i, LOCAL) != null) {
            lbis.add(iName);
        } else { // bean implements one interface and that is not annotated as @Local
            if (!rbis.contains(iName)) { // interface is not yet identified as Remote
                // e.g. @Remote(I1.class} class Bean implements I2 {...} where I2 is not annotated as @Remote
                lbis.add(iName);
            }
        }
    }
    
    private static void getBusinessInterfacesN(
            JavaClass javaClass,
            Set<String> lbis,
            Set<String> rbis,
            List<JavaClass> lNNValues,
            List<JavaClass> rNNValues,
            List<JavaClass> interfaces) {
        assert(interfaces.size() > 1);
        
        // calculate remote business interfaces
        if (rNNValues != null) { // bean is annotated as @Remote
            if (rNNValues.isEmpty()) {
                ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL,
                        "@Stateless @Remote class Foo implements I1, I2{} pattern detected"); //NOI18N
            } else {
                for (JavaClass c : rNNValues) rbis.add(c.getName());
            }
        }
        // now go thru' the interfaces and check if any of them has been annotated as @Remote
        for (JavaClass i : interfaces) {
            if (findAnnotation(i, REMOTE) != null) { // the interface is annotated as @Remote
                rbis.add(i.getName());
            }
        }
        
        // calculate local business interfaces
        if (lNNValues != null) { // bean is annotated as @Local
            if (lNNValues.isEmpty()) {
                ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL,
                        "@Stateless @Local class Foo implements I1, I2{} pattern detected"); //NOI18N
            } else {
                for (JavaClass c : lNNValues) lbis.add(c.getName());
            }
        }
        // now go thru' the interfaces and check if any of them has been annotated as @Local
        for (JavaClass i : interfaces) {
            if (findAnnotation(i, LOCAL) != null) { // the interface is annotated as @Local
                lbis.add(i.getName());
            }
        }
    }
    
    /**
     * find out the default value. exclude standard interfaces like
     * java.io.Serializable; java.io.Externalizable; any of the interfaces
     * defined by the javax.ejb package.
     */
    private static List<JavaClass> getNonStandardInterfaces(
            JavaClass javaClass) {
        List<JavaClass> result = new ArrayList<JavaClass>();
        for (JavaClass i : (List<JavaClass>) javaClass.getInterfaces()) {
            String name = i.getName();
            if ("java.io.Serializable".equals(name) || // NOI18N
                    "java.io.Externalizable".equals(name) || // NOI18N
                    name.startsWith("javax.ejb")) { // NOI18N
                continue;
            }
            result.add(i);
        }
        return result;
    }
    
    /**
     * read list class names specified in value attribute for Remote or Local.
     *
     * @param annotation
     * @return
     */
    private static List<JavaClass> readValueForRemoteOrLocal(
            Annotation annotation) {
        List<JavaClass> result = new ArrayList<JavaClass>();
        if (annotation.isValid()) {
            for (AttributeValue attr :
                (List<AttributeValue>) annotation.getAttributeValues()) {
                    // value is optional because it is the only method in this @
                    final InitialValue value = attr.getValue();
                    if (value instanceof ArrayInitialization) {
                        final List elementValues = ArrayInitialization.class.cast(
                                value).getElementValues();
                        for (Object o : elementValues) {
                            if (o instanceof ClassExpression) {
                                final ClassExpression ce =
                                        ClassExpression.class.cast(o);
                                result.add(JavaClass.class.cast(
                                        ce.getClassName().getType()));
                            }
                        }
                    } else if (value instanceof ClassExpression) {
                        final ClassExpression ce = ClassExpression.class.cast(
                                value);
                        result.add(JavaClass.class.cast(
                                ce.getClassName().getType()));
                    }
                }
        } else {
            //invalid code, let user fix the code first
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, "Invalid annotation: " + annotation); //NOI18N
        }
        return result;
    }
    
    /**
     * This method is temporarily required because the EJB Model does not expose
     * getBusinessLocal()/getBusinessRemote() from 3.0 SessionBean.class
     *
     * @param session
     * @return the name of the remote business interface of this session bean,
     *         null if not null.
     */
    private static String getBusinessRemote(Session session) {
        // TODO, place holder method. Wait for EJB 3 model to be available
        ErrorManager.getDefault().log(ErrorManager.ERROR,
                "EJBApiHelper.getBusinessRemote() is not yet implemented."); //NOI18N
        
        return null;
    }
    
    /**
     * This method is temporarily required because the EJB Model does not expose
     * getBusinessLocal()/getBusinessRemote() from 3.0 SessionBean.class
     *
     * @param session
     * @return the name of the local business interface of this session bean,
     *         null if not null.
     */
    private static String getBusinessLocal(Session session) {
        // TODO, place holder method. Wait for EJB 3 model to be available
        ErrorManager.getDefault().log(ErrorManager.ERROR,
                "EJBApiHelper.getBusinessLocal() is not yet implemented."); //NOI18N
        return null;
    }
}
